package com.zzsn.excelexportservice.service;

import com.alibaba.cloud.commons.lang.StringUtils;
import com.zzsn.excelexportservice.dto.*;
import com.zzsn.excelexportservice.exportStrategys.ExportStrategy;
import com.zzsn.excelexportservice.factorys.ExportStrategyFactory;

import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;

import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.*;
import com.zzsn.excelexportservice.dto.ExportDataResponse.Header;
/**
 * @Author: lxj
 * @Date: 2025/9/18 11:07
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class ExcelExportService {
    @Resource
    private CmsSearch cmsSearch;

    private final ExportStrategyFactory exportStrategyFactory;

    public void export(List<ExportReq> exportReqList, HttpServletRequest request, HttpServletResponse response) {
        if (exportReqList == null || exportReqList.isEmpty()) {
            sendError(response, "导出失败: 入参为空");
            return;
        }
        SXSSFWorkbook workbook = new SXSSFWorkbook(1000);
        workbook.setCompressTempFiles(true);

        for (ExportReq exportReq : exportReqList) {
            ExportDataResponse resp = null;
            String serviceName = exportReq.getServiceName();
            Map<String, Object> queryParams = exportReq.getQueryParams();

            ExportStrategy strategy = exportStrategyFactory.getStrategy(serviceName, exportReq.getApiPath());
            if (strategy == null) {
                log.error("No strategy found for service: {}, path: {}", serviceName, exportReq.getApiPath());
                sendError(response, "导出失败: 未找到匹配的服务策略");
                return;
            }

            resp = strategy.execute(request, queryParams, exportReq);

            if (resp == null || resp.getCode() != 0) {
                log.error("导出数据失败, serviceName: {}, apiPath: {}, errorMsg: {}",
                        serviceName, exportReq.getApiPath(), resp != null ? resp.getMsg() : "null");
                sendError(response, resp != null ? resp.getMsg() : "导出失败: 未知错误");
                return;
            }

            // 创建 sheet
            Sheet sheet = workbook.createSheet(
                    StringUtils.isNotBlank(exportReq.getSheetName())
                            ? exportReq.getSheetName()
                            : "Sheet_" + serviceName);

            int startRow = 0;
            int endCol = 13;

            // 判断是否有图片
            if (StringUtils.isNotBlank(exportReq.getBase64Img())) {
                log.info("开始插入图片------->" + System.currentTimeMillis());
                drawImg(workbook, sheet, exportReq.getBase64Img(), startRow, endCol);
                startRow++; // 图片占用一行，从下一行开始写标题
            }

            List<Map<String, Object>> dataList = resp.getDataList();
            List<Header> headers = resp.getHeaders();

            // 写标题
            log.info("开始插入标题------->" + System.currentTimeMillis());
            handlerExcelTitle(workbook, sheet, startRow, headers);

            // 写数据
            log.info("开始写入数据------->" + System.currentTimeMillis());
            handlerExcelData(workbook, sheet, startRow + 1, headers, dataList);

        }

        // 最后写到 response
        log.info("生成excel结束------->" + System.currentTimeMillis());
        writeExcelToResponse("导出", workbook, response);

    }
    @SuppressWarnings("unchecked")
    private <T> T getParam(Map<String, Object> params, String key, T defaultVal) {
        Object val = params.get(key);
        if (val == null) {
            return defaultVal;
        }
        try {
            return (T) val;
        } catch (ClassCastException e) {
            log.warn("参数 {} 类型不正确，期望 {}, 实际 {}", key, defaultVal.getClass(), val.getClass());
            return defaultVal;
        }
    }

private static void sendError(HttpServletResponse response, String msg) {
    response.reset();
    response.setStatus(1);
    response.setContentType("text/plain;charset=UTF-8");
    try {
        response.getWriter().write(msg != null ? msg : "未知错误");
    } catch (IOException e) {
        e.printStackTrace();
    }
}

    // 插入图片（Base64）
    private void drawImg(SXSSFWorkbook workbook, Sheet sheet, String base64Img, int rowIndex, int endCol) {
        Row firstRow = sheet.createRow(rowIndex);
        for (int i = 0; i < endCol; i++) {
            firstRow.createCell(i);
        }
        firstRow.setHeightInPoints(230);
        // 合并单元格
        sheet.addMergedRegion(new CellRangeAddress(rowIndex, rowIndex, 0, endCol));

        if (base64Img != null && !base64Img.isBlank()) {
            int start = base64Img.indexOf(",");
            if (start != -1) {
                base64Img = base64Img.substring(start + 1);
            }
            byte[] imgBytes = Base64.getDecoder().decode(base64Img);

            Drawing<?> drawingPatriarch = sheet.createDrawingPatriarch();
            ClientAnchor anchor = drawingPatriarch.createAnchor(0, 0, 0, 0, 0, rowIndex, endCol, rowIndex + 1);
            int pictureIdx = workbook.addPicture(imgBytes, Workbook.PICTURE_TYPE_PNG);
            drawingPatriarch.createPicture(anchor, pictureIdx);
        }
    }

    // 写入标题
    private void handlerExcelTitle(Workbook workbook, Sheet sheet, int rowIndex, List<Header> headers) {
        Row row = sheet.createRow(rowIndex);
        CellStyle titleStyle = getTitleStyle(workbook);
        if (headers != null && !headers.isEmpty()) {
            for (int i = 0; i < headers.size(); i++) {
                Cell cell = row.createCell(i);
                cell.setCellStyle(titleStyle);
                cell.setCellValue(headers.get(i).getTitle());
            }
        }
    }

    // Title 样式
    private CellStyle getTitleStyle(Workbook workbook) {
        Font font = workbook.createFont();
        font.setFontName("Courier New");
        font.setBold(true);

        CellStyle style = workbook.createCellStyle();
        style.setFont(font);
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        return style;
    }

    // 写入数据
    private void handlerExcelData(Workbook workbook, Sheet sheet, int startRowIndex,
                                  List<Header> headers, List<Map<String, Object>> dataList) {
        CellStyle dataStyle = getDataStyle(workbook);
        if (headers != null && !headers.isEmpty() && dataList != null && !dataList.isEmpty()) {
            for (int j = 0; j < dataList.size(); j++) {
                Row row = sheet.createRow(startRowIndex + j);
                Map<String, Object> dataMap = dataList.get(j);
                for (int i = 0; i < headers.size(); i++) {
                    Cell cell = row.createCell(i);
                    cell.setCellStyle(dataStyle);
                    Object value = dataMap.get(headers.get(i).getField());
                    cell.setCellValue(value == null ? "" : value.toString());
                }
            }
        }
    }


    // 数据样式
    private CellStyle getDataStyle(Workbook workbook) {
        Font font = workbook.createFont();
        font.setFontName("Courier New");

        CellStyle style = workbook.createCellStyle();
        style.setFont(font);
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        return style;
    }

    // 将数据写入 HttpServletResponse（流式，不占用内存）
    private void writeExcelToResponse(String fileNamePrefix, SXSSFWorkbook workbook, HttpServletResponse response) {
        try {
            response.reset();
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            response.setHeader("Content-disposition",
                    "attachment;filename=" + URLEncoder.encode(fileNamePrefix + System.currentTimeMillis() + ".xlsx", "UTF-8"));
            response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");

            workbook.write(response.getOutputStream());
            response.getOutputStream().flush();
        } catch (IOException e) {
            sendError(response, "导出 Excel 异常：" + e.getMessage());
        } finally {
            // 清理临时文件
            try {
                workbook.close();
            } catch (IOException ignored) {
            }
            workbook.dispose();
        }
    }
}
