package com.zzsn.thinktank.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.http.HttpUtil;
import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.zzsn.thinktank.entity.ThinktankBasicInfo;
import com.zzsn.thinktank.service.InfoSourceService;
import com.zzsn.thinktank.service.LeaderCategoryService;
import com.zzsn.thinktank.service.ThinktankBasicInfoService;
import com.zzsn.thinktank.util.ExcelExportUtil;
import com.zzsn.thinktank.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/4/24 10:28
 * @Content:
 */
@Slf4j
@RestController
@RequestMapping("/api/basicinfo")
public class ThinktankBasicInfoController {

    @Autowired
    ThinktankBasicInfoService thinktankBasicInfoService;
    @Autowired
    LeaderCategoryService leaderCategoryService;
    @Autowired
    private InfoSourceService infoSourceService;

    @Value("${cj.url.infoSource.columnDetail:http://1.95.79.85:8823/baseSourceInfo/api/infoSource/columnDetail?columnIds={IDS}}")
    private String CJ_IINFOSOURCE_COLUMNDETAIL;

    /**
     * 新增
     *
     * @param thinktankBasicInfo
     * @return
     */
    @PostMapping("/add")
    public Result<?> add(@RequestBody ThinktankBasicInfo thinktankBasicInfo) {
        log.info("新增信息：{}", thinktankBasicInfo);
        return thinktankBasicInfoService.add(thinktankBasicInfo);
    }

    /**
     * 根据id主键删除
     *
     * @param id
     * @return
     */
    @GetMapping("/del")
    public Result<?> del(@RequestParam(name = "id", required = true) String id) {
        log.info("根据id删除:{}", id);
        return thinktankBasicInfoService.del(id);
    }

    /**
     * 根据id修改信息
     *
     * @param thinktankBasicInfo
     * @return
     */
    @PostMapping("/edit")
    public Result<?> edit(@RequestBody ThinktankBasicInfo thinktankBasicInfo) {
        log.info("根据id修改信息:{}", thinktankBasicInfo);
        if (thinktankBasicInfo.getId() == null)
            return Result.error(400, "id不能为空");
        return thinktankBasicInfoService.edit(thinktankBasicInfo);
    }

    /**
     * 根据id主键查询
     *
     * @param id
     * @return
     */
    @GetMapping("getById")
    public Result<?> getById(@RequestParam(name = "id", required = true) String id) {
        log.info("根据id主键查询:{}", id);
        ThinktankBasicInfo thinktankBasicInfo = thinktankBasicInfoService.getById(id);
        return Result.OK(thinktankBasicInfo);
    }

    @PostMapping("/list")
    public Result<?> getList(@RequestBody ThinktankBasicInfoListVo thinktankBasicInfoListVo) {
        log.info("智库信息条件分页查询:{}", thinktankBasicInfoListVo);
        Integer pageNo = thinktankBasicInfoListVo.getPageNo();
        Integer pageSize = thinktankBasicInfoListVo.getPageSize();
        if (pageNo < 1)
            pageNo = 1;
        if (pageSize < 0)
            pageSize = 10;
        Integer office = pageSize * (pageNo - 1);
        thinktankBasicInfoListVo.setOffset(office);
        return thinktankBasicInfoService.getList(thinktankBasicInfoListVo);
    }

    /**
     * 模板下载
     *
     * @param response
     */
    @GetMapping("/downloadTemplate")
    public void downloadTemplate(HttpServletResponse response) {
        thinktankBasicInfoService.downloadTemplate(response);
    }


    /**
     * 批量导入数据
     *
     * @param file excel文件
     * @author lkg
     * @date 2024/8/22
     */
    @PostMapping("/batchImport")
    public Result<?> batchImport(@RequestParam("file") MultipartFile file) {
        int index = Objects.requireNonNull(file.getOriginalFilename()).lastIndexOf(".");
        String fileSuffix = file.getOriginalFilename().substring(index + 1);
        if ("xls".equals(fileSuffix) || "xlsx".equals(fileSuffix)) {
            boolean flag = checkExcel(file);
            if (!flag) {
                return Result.error("模版有误，请使用正确的模板！");
            }
        } else {
            return Result.error("不支持的文件类型！");
        }
        try {
            ExcelReader reader = ExcelUtil.getReader(file.getInputStream());
            Map<String, String> header = new HashMap<>();
            header.put("智库机构编码", "codeId");
            header.put("* 中文全称", "chineseWhole");
            header.put("中文简称", "chineseSimple");
            header.put("英文全称", "englishWhole");
            header.put("英文简称", "englishSimple");
            header.put("官网", "officialWebsite");
            header.put("* 所属国家", "belongCountry");
            header.put("语言", "lang");
            header.put("运营状态", "operateStatus");
            header.put("营利性质", "profitNature");
            header.put("智库性质", "nature");
            header.put("研究领域", "tagName");
            header.put("影响力和地位", "influencePosition");
            header.put("人员规模", "staffSize");
            header.put("智库规模", "scale");
            header.put("成果类型", "achievementType");
            header.put("资金来源", "fundsSource");
            header.put("原文全称", "originalWhole");
            header.put("原文简称", "originalSimple");
            header.put("简介", "biographicalNotes");
            header.put("成立时间", "establishTime");
            header.put("是否收费(1-是;0-否)", "charge");
            header.put("地址", "address");
            header.put("创办单位/所属单位", "belongUnit");
            header.put("是否展示(1-是;0-否)", "ynDisplay");
            reader.setHeaderAlias(header);
            List<ThinkTankFileVO> thinkTankImportVOList = reader.read(1, 2, ThinkTankFileVO.class);
            if (CollectionUtils.isNotEmpty(thinkTankImportVOList)) {
                boolean requiredFlag = true;
                for (ThinkTankFileVO thinkTankImportVO : thinkTankImportVOList) {
                    if (StringUtils.isEmpty(thinkTankImportVO.getChineseWhole()) || StringUtils.isEmpty(thinkTankImportVO.getBelongCountry())) {
                        requiredFlag = false;
                    }
                        break;
                }
                if (!requiredFlag) {
                    return Result.error("必填项【中文全称和所属国家】有缺失,请核对后再导入。");
                }
                String msg = thinktankBasicInfoService.batchImport(thinkTankImportVOList);
                return Result.OK(msg);
            } else {
                return Result.error("文件数据为空。");
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return Result.OK();
    }

    /**
     * 导出
     *
     * @param one 检索条件
     * @author lkg
     * @date 2025/4/27
     */
    @PostMapping("/batchExport")
    public byte[] batchExport(@RequestBody ThinktankBasicInfoListVo one) {
        byte[] bytes = null;
        try {
            List<ThinktankBasicInfo> thinkTankBasicInfos = thinktankBasicInfoService.exportListByCondition(one.getId(), one.getCodeId(),
                    one.getTypeId(), one.getThinktankName(), one.getTagId(), one.getKeyword());
            if (CollectionUtils.isNotEmpty(thinkTankBasicInfos)) {
                String[] arr = new String[]{"智库机构id", "智库机构编码", "中文全称", "中文简称", "英文全称", "英文简称", "官网", "所属国家", "语言", "经营状态", "营利性质",
                        "智库性质", "研究领域", "影响力和地位", "人员规模", "智库规模", "成果类型", "资金来源", "原文全称", "原文简称", "简介", "成立时间", "是否收费(1-是;0-否)",
                        "地址", "创办单位/所属单位", "是否展示(1-是;0-否)"};
                XSSFWorkbook workbook = new XSSFWorkbook();
                //基本信息
                List<List<String>> rows = new ArrayList<>();
                thinkTankBasicInfos.forEach(each -> rows.add(each.toList()));
                ExcelExportUtil.exportExcelData(workbook, 0, Arrays.asList(arr), rows, "智库基本信息");

                //智库栏目采集统计信息
                String[] header = new String[]{"智库机构id", "智库机构编码", "中文全称", "关联信息源栏目数量", "信息采集总量", "关联信息源栏目编码",
                        "关联信息源名称", "栏目名称", "网址", "启用状态(1-启用;0-禁用)", "信息源采集数量", "信息源近一月采集量", "采集状态"};
                List<ThinkTankSourceVO> thinkTankSourceVOS = thinktankBasicInfoService.thinkTankCollectCount(thinkTankBasicInfos);
                //补充信息源栏目采集状态
                List<String> allIds = thinkTankSourceVOS.stream().map(ThinkTankSourceVO::getSourceId).distinct().collect(Collectors.toList());
                List<List<String>> groupIdList = CollectionUtil.split(allIds, 300);
                Map<String, JSONObject> infoMap = new HashMap<>();
                for (List<String> ids : groupIdList) {
                    String res = HttpUtil.get(CJ_IINFOSOURCE_COLUMNDETAIL.replace("{IDS}", StringUtils.join(ids, ",")));
                    JSONArray result = JSONObject.parseObject(res).getJSONArray("result");
                    for (int i = 0; i < result.size(); i++) {
                        JSONObject record = result.getJSONObject(i);
                        infoMap.put(record.getString("id"), record);
                    }
                }
                for (ThinkTankSourceVO thinkTankSourceVO : thinkTankSourceVOS) {
                    JSONObject record = infoMap.get(thinkTankSourceVO.getSourceId());
                    if (record == null) {
                        continue;
                    }
                    //采集状态
                    switch (record.getString("abnormalCode")) {
                        case "1":
                            thinkTankSourceVO.setAbnormalName("正常采集");
                            break;
                        case "101":
                            thinkTankSourceVO.setAbnormalName("访问异常");
                            break;
                        case "201":
                            thinkTankSourceVO.setAbnormalName("列表页解析异常");
                            break;
                        case "202":
                            thinkTankSourceVO.setAbnormalName("详情页配置异常");
                            break;
                        case "203":
                            thinkTankSourceVO.setAbnormalName("详情规则有差异");
                            break;
                        case "301":
                            thinkTankSourceVO.setAbnormalName("信息源漏采");
                            break;
                        case "503":
                            thinkTankSourceVO.setAbnormalName("账号异常");
                            break;
                        case "601":
                            thinkTankSourceVO.setAbnormalName("微信公众号异堂");
                            break;
                        default:
                            thinkTankSourceVO.setAbnormalName(record.getString("abnormalCode"));
                    }
                }
                List<List<String>> rowList = new ArrayList<>();
                thinkTankSourceVOS.forEach(e -> rowList.add(e.toList()));
                ExcelExportUtil.exportExcelData(workbook, 1, Arrays.asList(header), rowList, "智库栏目采集统计信息");

                //智库机构采集统计信息
                List<String> tIds = thinkTankBasicInfos.stream().map(ThinktankBasicInfo::getId).collect(Collectors.toList());
                exportBindCount(workbook, 2, "智库机构采集统计信息", infoMap, thinkTankBasicInfos, tIds);
                ByteArrayOutputStream outStream = new ByteArrayOutputStream();
                workbook.write(outStream);
                // 将字节流转换为InputStream
                bytes = outStream.toByteArray();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return bytes;
    }

    /**
     * 智库成果列表
     *
     * @param id       智库id
     * @param pageNo   页码
     * @param pageSize 每页返回条数
     * @author lkg
     * @date 2025/8/4
     */
    @GetMapping("/listBaseData")
    public Result<?> getListBaseData(@RequestParam String id,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("智库成果分页查询:{}", id);
        return thinktankBasicInfoService.getListBaseData(id, pageNo, pageSize);
    }

    private boolean checkExcel(MultipartFile multipartFile) {
        Workbook workbook = ExcelExportUtil.getWorkbook(multipartFile);
        if (workbook != null) {
            List<String> list = ExcelExportUtil.getSheetTitles(workbook);
            return list != null
                    && list.contains("智库机构编码")
                    && list.contains("* 中文全称")
                    && list.contains("中文简称")
                    && list.contains("英文全称")
                    && list.contains("英文简称")
                    && list.contains("官网")
                    && list.contains("* 所属国家")
                    && list.contains("语言")
                    && list.contains("营利性质")
                    && list.contains("智库性质")
                    && list.contains("研究领域")
                    && list.contains("影响力和地位")
                    && list.contains("智库规模")
                    && list.contains("成果类型")
                    && list.contains("资金来源")
                    && list.contains("原文全称")
                    && list.contains("原文简称")
                    && list.contains("简介")
                    && list.contains("成立时间")
                    && list.contains("是否收费(1-是;0-否)")
                    && list.contains("地址")
                    && list.contains("创办单位/所属单位")
                    && list.contains("是否展示(1-是;0-否)");
        }
        return false;
    }

    /**
     * 基于信息源组的采集状态统计导出
     *
     * @param workbook
     * @param sheetNum
     * @param sheetName
     * @param infoMap
     * @param thinkTankBasicInfos
     * @param tIds                智库id列表
     */
    private void exportBindCount(XSSFWorkbook workbook, int sheetNum, String sheetName, Map<String,
            JSONObject> infoMap, List<ThinktankBasicInfo> thinkTankBasicInfos, List<String> tIds) {
        //查询所有的
        List<BindSourceDetailVO> bindSourceDetailList = infoSourceService.bindSourceDetail(tIds);
        List<String> headers = new ArrayList<>();
        headers.add("智库机构id");
        headers.add("智库机构编码");
        headers.add("中文全称");
        headers.add("中文简称");
        headers.add("英文全称");
        headers.add("英文简称");
        headers.add("国别");
        headers.add("影响力及地位");
        headers.add("是否展示");

        headers.add("栏目数");
        Map<String, Integer> columnNum = new HashMap<>();

        headers.add("启用数");
        Map<String, Integer> enableNum = new HashMap<>();
        headers.add("未启用数");
        Map<String, Integer> disableNum = new HashMap<>();

        headers.add("正常采集");
        Map<String, Integer> normalNum = new HashMap<>();
        headers.add("访问异常");
        Map<String, Integer> errorNum = new HashMap<>();
        headers.add("列表页解析异常");
        Map<String, Integer> listParseErrorNum = new HashMap<>();
        headers.add("详情页配置异常");
        Map<String, Integer> detailParseErrorNum = new HashMap<>();
        headers.add("详情规则有差异");
        Map<String, Integer> detailParseDiffNum = new HashMap<>();
        headers.add("信息源漏采");
        Map<String, Integer> missNum = new HashMap<>();
        headers.add("账号异常");
        Map<String, Integer> accountErrorNum = new HashMap<>();
        headers.add("微信公众号异堂");
        Map<String, Integer> wechatErrorNum = new HashMap<>();

        headers.add("待配置");
        Map<String, Integer> configNum = new HashMap<>();
        headers.add("待验证");
        Map<String, Integer> verifyNum = new HashMap<>();
        headers.add("验证通过");
        Map<String, Integer> verifyPassNum = new HashMap<>();
        headers.add("验证不通过");
        Map<String, Integer> verifyFailNum = new HashMap<>();
        //查询智库列表
        List<String> hasInfoIds = bindSourceDetailList.stream().map(BindSourceDetailVO::getThinkTankId).distinct().collect(Collectors.toList());
        List<ThinktankBasicInfo> basicInfos = thinkTankBasicInfos.stream().filter(item -> hasInfoIds.contains(item.getId())).collect(Collectors.toList());
        for (ThinktankBasicInfo basicInfo : basicInfos) {
            String tid = basicInfo.getId();
            //查询智库关联的信息源列表
            List<String> sourceList = bindSourceDetailList.stream().filter(item -> tid.equals(item.getThinkTankId())).map(BindSourceDetailVO::getSourceId).collect(Collectors.toList());
            for (String sourceId : sourceList) {
                JSONObject record = infoMap.get(sourceId);
                if (record == null) {
                    continue;
                }
                countAdd(columnNum, tid);
                //是否启用
                if ("1".equals(record.getString("enable"))) {
                    countAdd(enableNum, tid);
                } else {
                    countAdd(disableNum, tid);
                }
                //采集状态
                switch (record.getString("abnormalCode")) {
                    case "1":
                        countAdd(normalNum, tid);
                        break;
                    case "101":
                        countAdd(errorNum, tid);
                        break;
                    case "201":
                        countAdd(listParseErrorNum, tid);
                        break;
                    case "202":
                        countAdd(detailParseErrorNum, tid);
                        break;
                    case "203":
                        countAdd(detailParseDiffNum, tid);
                        break;
                    case "301":
                        countAdd(missNum, tid);
                        break;
                    case "503":
                        countAdd(accountErrorNum, tid);
                        break;
                    case "601":
                        countAdd(wechatErrorNum, tid);
                        break;
                }
                //状态
                switch (record.getString("status")) {
                    case "1":
                        countAdd(configNum, tid);
                        break;
                    case "2":
                        countAdd(verifyNum, tid);
                        break;
                    case "3":
                        countAdd(verifyPassNum, tid);
                        break;
                    case "4":
                        countAdd(verifyFailNum, tid);
                        break;
                }
            }
        }

        List<List<String>> rows = new ArrayList<>();
        for (ThinktankBasicInfo basicInfo : basicInfos) {
            List<String> dataItem = new ArrayList<>();
            String id = basicInfo.getId();
            //智库机构id
            dataItem.add(basicInfo.getId());
            //智库机构编码
            dataItem.add(basicInfo.getCodeId());
            //中文全称
            dataItem.add(basicInfo.getChineseWhole());
            //中文简称
            dataItem.add(basicInfo.getChineseSimple());
            //英文全称
            dataItem.add(basicInfo.getEnglishWhole());
            //英文简称
            dataItem.add(basicInfo.getEnglishSimple());
            //国别
            dataItem.add(basicInfo.getBelongCountry());
            //影响力及地位
            dataItem.add(basicInfo.getInfluencePosition());
            //是否展示
            dataItem.add(String.valueOf(basicInfo.getYnDisplay()));
            //栏目数
            dataItem.add(getMapStr(columnNum, id));
            //是否启用
            dataItem.add(getMapStr(enableNum, id));
            dataItem.add(getMapStr(disableNum, id));
            //采集状态
            dataItem.add(getMapStr(normalNum, id));
            dataItem.add(getMapStr(errorNum, id));
            dataItem.add(getMapStr(listParseErrorNum, id));
            dataItem.add(getMapStr(detailParseErrorNum, id));
            dataItem.add(getMapStr(detailParseDiffNum, id));
            dataItem.add(getMapStr(missNum, id));
            dataItem.add(getMapStr(accountErrorNum, id));
            dataItem.add(getMapStr(wechatErrorNum, id));
            //状态
            dataItem.add(getMapStr(configNum, id));
            dataItem.add(getMapStr(verifyNum, id));
            dataItem.add(getMapStr(verifyPassNum, id));
            dataItem.add(getMapStr(verifyFailNum, id));
            rows.add(dataItem);
        }
        ExcelExportUtil.exportExcelData(workbook, sheetNum, headers, rows, sheetName);
    }

    private void exportBindCount_old(XSSFWorkbook workbook, int sheetNum, String sheetName, Map<String,
            JSONObject> infoMap, List<ThinktankBasicInfo> thinkTankBasicInfos, List<String> tIds) {
        //查询所有的
        List<CountVo> bindAll = thinktankBasicInfoService.bindInfoSource(tIds);
        List<String> headers = new ArrayList<>();
        headers.add("智库机构id");
        headers.add("智库机构编码");
        headers.add("中文全称");
        headers.add("中文简称");
        headers.add("英文全称");
        headers.add("英文简称");
        headers.add("国别");
        headers.add("影响力及地位");
        headers.add("是否展示");

        headers.add("栏目数");
        Map<String, Integer> columnNum = new HashMap<>();

        headers.add("启用数");
        Map<String, Integer> enableNum = new HashMap<>();
        headers.add("未启用数");
        Map<String, Integer> disableNum = new HashMap<>();

        headers.add("正常采集");
        Map<String, Integer> normalNum = new HashMap<>();
        headers.add("访问异常");
        Map<String, Integer> errorNum = new HashMap<>();
        headers.add("列表页解析异常");
        Map<String, Integer> listParseErrorNum = new HashMap<>();
        headers.add("详情页配置异常");
        Map<String, Integer> detailParseErrorNum = new HashMap<>();
        headers.add("详情规则有差异");
        Map<String, Integer> detailParseDiffNum = new HashMap<>();
        headers.add("信息源漏采");
        Map<String, Integer> missNum = new HashMap<>();
        headers.add("账号异常");
        Map<String, Integer> accountErrorNum = new HashMap<>();
        headers.add("微信公众号异堂");
        Map<String, Integer> wechatErrorNum = new HashMap<>();

        headers.add("待配置");
        Map<String, Integer> configNum = new HashMap<>();
        headers.add("待验证");
        Map<String, Integer> verifyNum = new HashMap<>();
        headers.add("验证通过");
        Map<String, Integer> verifyPassNum = new HashMap<>();
        headers.add("验证不通过");
        Map<String, Integer> verifyFailNum = new HashMap<>();
        //查询智库列表
        List<String> hasInfoIds = bindAll.stream().map(CountVo::getId).distinct().collect(Collectors.toList());
        List<ThinktankBasicInfo> basicInfos = thinkTankBasicInfos.stream().filter(item -> hasInfoIds.contains(item.getId())).collect(Collectors.toList());
        for (ThinktankBasicInfo basicInfo : basicInfos) {
            String tid = basicInfo.getId();
            //查询智库关联的信息源列表
            List<String> sourceList = bindAll.stream().filter(item -> tid.equals(item.getId())).map(CountVo::getCode).collect(Collectors.toList());
            for (String sourceId : sourceList) {
                JSONObject record = infoMap.get(sourceId);
                if (record == null) {
                    continue;
                }
                countAdd(columnNum, tid);
                //是否启用
                if ("1".equals(record.getString("enable"))) {
                    countAdd(enableNum, tid);
                } else {
                    countAdd(disableNum, tid);
                }
                //采集状态
                switch (record.getString("abnormalCode")) {
                    case "1":
                        countAdd(normalNum, tid);
                        break;
                    case "101":
                        countAdd(errorNum, tid);
                        break;
                    case "201":
                        countAdd(listParseErrorNum, tid);
                        break;
                    case "202":
                        countAdd(detailParseErrorNum, tid);
                        break;
                    case "203":
                        countAdd(detailParseDiffNum, tid);
                        break;
                    case "301":
                        countAdd(missNum, tid);
                        break;
                    case "503":
                        countAdd(accountErrorNum, tid);
                        break;
                    case "601":
                        countAdd(wechatErrorNum, tid);
                        break;
                }
                //状态
                switch (record.getString("status")) {
                    case "1":
                        countAdd(configNum, tid);
                        break;
                    case "2":
                        countAdd(verifyNum, tid);
                        break;
                    case "3":
                        countAdd(verifyPassNum, tid);
                        break;
                    case "4":
                        countAdd(verifyFailNum, tid);
                        break;
                }
            }
        }

        List<List<String>> rows = new ArrayList<>();
        for (ThinktankBasicInfo basicInfo : basicInfos) {
            List<String> dataItem = new ArrayList<>();
            String id = basicInfo.getId();
            //智库机构id
            dataItem.add(basicInfo.getId());
            //智库机构编码
            dataItem.add(basicInfo.getCodeId());
            //中文全称
            dataItem.add(basicInfo.getChineseWhole());
            //中文简称
            dataItem.add(basicInfo.getChineseSimple());
            //英文全称
            dataItem.add(basicInfo.getEnglishWhole());
            //英文简称
            dataItem.add(basicInfo.getEnglishSimple());
            //国别
            dataItem.add(basicInfo.getBelongCountry());
            //影响力及地位
            dataItem.add(basicInfo.getInfluencePosition());
            //是否展示
            dataItem.add(String.valueOf(basicInfo.getYnDisplay()));
            //栏目数
            dataItem.add(getMapStr(columnNum, id));
            //是否启用
            dataItem.add(getMapStr(enableNum, id));
            dataItem.add(getMapStr(disableNum, id));
            //采集状态
            dataItem.add(getMapStr(normalNum, id));
            dataItem.add(getMapStr(errorNum, id));
            dataItem.add(getMapStr(listParseErrorNum, id));
            dataItem.add(getMapStr(detailParseErrorNum, id));
            dataItem.add(getMapStr(detailParseDiffNum, id));
            dataItem.add(getMapStr(missNum, id));
            dataItem.add(getMapStr(accountErrorNum, id));
            dataItem.add(getMapStr(wechatErrorNum, id));
            //状态
            dataItem.add(getMapStr(configNum, id));
            dataItem.add(getMapStr(verifyNum, id));
            dataItem.add(getMapStr(verifyPassNum, id));
            dataItem.add(getMapStr(verifyFailNum, id));
            rows.add(dataItem);
        }
        ExcelExportUtil.exportExcelData(workbook, sheetNum, headers, rows, sheetName);
    }

    private String getMapStr(Map<String, Integer> map, String id) {
        if (map.containsKey(id)) {
            return map.get(id) + "";
        }
        return "0";
    }

    private String getSumMapStr(Map<String, Integer> map) {
        if (map.isEmpty()) {
            return "0";
        }
        return map.values().stream().mapToInt(Integer::intValue).sum() + "";
    }

    private void countAdd(Map<String, Integer> map, String id) {
        if (map.containsKey(id)) {
            map.put(id, map.get(id) + 1);
        } else {
            map.put(id, 1);
        }
    }

}
