package com.zzsn.thinktank.service.impl;

import com.zzsn.thinktank.service.IGeneratorIdService;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.support.atomic.RedisAtomicLong;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * 雪花算法生成id
 */
@Component
@Slf4j
public class SnowFlakeGeneratorId implements IGeneratorIdService {

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    private final static int MAX_QUEUE = 128;
    private final static int DEFAULT_MACHINE_BIT_NUM = 6; //机器标识占用的位数
    private final static int DEFAULT_PORT_BIT_NUM = 4; //端口占用的位数
    private final static long START_STAMP = 1641286225129L; //起始时间

    // 线程安全队列
    private final ConcurrentLinkedQueue<Long> queue = new ConcurrentLinkedQueue<>();
    private final ConcurrentLinkedQueue<Long> orderQueue = new ConcurrentLinkedQueue<>();

    /**
     * 可分配的位数
     */
    private final static int REMAIN_BIT_NUM = 22;

    private static final String FINANCE = "THINK_TANK_CODE:";
    private final static int NUM = 8;
    private static final String ZKG = "ZKG";


    @Value("${server.port}")
    private String port;

    @PostConstruct
    public void init() {

        int portBitNum = DEFAULT_PORT_BIT_NUM;
        int sequenceBitNum = REMAIN_BIT_NUM - portBitNum - machineBitNum;

        this.maxSequenceValue = ~(-1 << sequenceBitNum);

        machineBitLeftOffset = sequenceBitNum;
        portBitLeftOffset = portBitNum + sequenceBitNum;
        timestampBitLeftOffset = portBitNum + machineBitNum + sequenceBitNum;

        this.portId = Integer.parseInt(port) & 15;
        this.machineId = getMachineId() & 63;
        productIdToQueue(queue, MAX_QUEUE);
        productIdToQueue(orderQueue, MAX_QUEUE);
    }


    public Long getId() {
        Long peek = queue.poll();
        if (queue.size() <= MAX_QUEUE / 2) {
            productIdToQueue(queue, MAX_QUEUE / 2);
        }
        return peek;
    }

    public Long getOrderId() {
        Long peek = orderQueue.poll();
        if (orderQueue.size() <= MAX_QUEUE / 2) {
            productIdToQueue(orderQueue, MAX_QUEUE / 2);
        }
        return peek;
    }

    @Override
    public String getIdNo() {
        return getId(FINANCE, NUM);
    }
    //根据规则生成id(规则：格式化日期(截取后六位)+递增值(8位数))
    private String getId(String prefix, int num){
        String redisKey = getRedisKey(prefix);
        Date expireDate = getExpireDate();
        //返回当前redis中的key的最大值
        long seq = generate(stringRedisTemplate, redisKey, expireDate);
        //获取当天的日期，格式为yyyyMMdd
        String date = new SimpleDateFormat("yyyyMMdd").format(expireDate);
        //生成八位的序列号，如果seq不够四位，seq前面补0，
        //如果seq位数超过了八位，那么无需补0直接返回当前的seq
        String sequence = StringUtils.leftPad(Long.toString(seq), num, "0");
        StringBuffer sb = new StringBuffer()
                .append(ZKG)
                .append("-")
                .append(date)
                .append("-")
                .append(sequence);
        return sb.toString();
    }
    //获取redis--key
    @SneakyThrows
    private String getRedisKey(String prefix){
        if (StringUtils.isEmpty(prefix)){
            throw new Exception("前缀不能为空!");
        }
        //用作存放redis中的key前缀
        String PREFIX_KEY = "CodeGenerateUtil::";
        return PREFIX_KEY + prefix;
    }
    //获取缓存过期时间点
    private Date getExpireDate(){
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        //设置过期时间，这里设置为当天的23:59:59
        return calendar.getTime();
    }
    //获取redis缓存
    private long generate(StringRedisTemplate stringRedisTemplate, String key, Date expireTime) {
        //RedisAtomicLong为原子类，根据传入的key和redis链接工厂创建原子类
        RedisAtomicLong counter = new RedisAtomicLong(key,stringRedisTemplate.getConnectionFactory());
        //设置过期时间
        counter.expireAt(expireTime);
        //返回redis中key的值
        return counter.incrementAndGet();
    }


    /**
     * 产生下一个ID
     */
    public synchronized Long nextId() {
        long currentStamp = getTimeMill();
        if (currentStamp < lastStamp) {
            throw new RuntimeException(String.format("Clock moved backwards. Refusing to generate id for %d milliseconds", lastStamp - currentStamp));
        }

        //新的毫秒，序列从0开始，否则序列自增
        if (currentStamp == lastStamp) {
            sequence = (sequence + 1) & this.maxSequenceValue;
            if (sequence == 0L) {
                //Twitter源代码中的逻辑是循环，直到下一个毫秒
                lastStamp = tilNextMillis();
//                throw new IllegalStateException("sequence over flow");
            }
        } else {
            sequence = 0L;
        }

        lastStamp = currentStamp;

        return (currentStamp - START_STAMP) << timestampBitLeftOffset | portId << portBitLeftOffset | machineId << machineBitLeftOffset | sequence;
    }


    private void productIdToQueue(ConcurrentLinkedQueue<Long> queue, Integer num) {
        new Thread(() -> {
            for (int i = 0; i < num; i++) {
                queue.add(nextId());
            }
        }).start();
    }

    private Integer getMachineId() {
        InetAddress addr = null;
        try {
            addr = InetAddress.getLocalHost();
        } catch (UnknownHostException e) {
            e.printStackTrace();
        }
        /*获取本机IP*/
        assert addr != null;
        String ip = addr.getHostAddress();
        log.info("本机IP ： {}  ", ip);

        return Integer.parseInt(ip.split("\\.")[3]);
    }

    private final int machineBitNum = DEFAULT_MACHINE_BIT_NUM;

    /**
     * idc编号
     */
    private long portId;

    /**
     * 机器编号
     */
    private long machineId;

    /**
     * 当前序列号
     */
    private long sequence = 0L;

    /**
     * 上次最新时间戳
     */
    private long lastStamp = -1L;

    /**
     * idc偏移量：一次计算出，避免重复计算
     */
    private int portBitLeftOffset;

    /**
     * 机器id偏移量：一次计算出，避免重复计算
     */
    private int machineBitLeftOffset;

    /**
     * 时间戳偏移量：一次计算出，避免重复计算
     */
    private int timestampBitLeftOffset;

    /**
     * 最大序列值：一次计算出，避免重复计算
     */
    private int maxSequenceValue;

    private long getTimeMill() {
        return System.currentTimeMillis();
    }

    private long tilNextMillis() {
        long timestamp = getTimeMill();
        while (timestamp <= lastStamp) {
            timestamp = getTimeMill();
        }
        return timestamp;
    }
}