package com.zzsn.thinktank.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.math.MathUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.thinktank.entity.InfoSource;
import com.zzsn.thinktank.service.InfoSourceGroupMapService;
import com.zzsn.thinktank.service.InfoSourceService;
import com.zzsn.thinktank.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.security.auth.Subject;
import java.util.List;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/4/25 9:34
 * @Content: 信息源绑定
 */
@Slf4j
@RestController
@RequestMapping("/api/bind")
public class ThinktankBindController {

    @Autowired
    private InfoSourceService infoSourceService;

    @Autowired
    private InfoSourceGroupMapService infoSourceGroupMapService;

    /**
     * 查询信息源组的绑定列表
     *
     * @param infoSourceVo
     * @param ynBind
     * @param groupId
     * @param pageNo
     * @param pageSize
     * @return
     */
    //@GetMapping(value = "/bindList")
    public Result<?> bindList(InfoSourceVo infoSourceVo,
                              @RequestParam(name = "ynBind", required = true) Integer ynBind,
                              @RequestParam(name = "groupId", required = true) String groupId,
                              @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("查询信息源组的绑定列表:{}", infoSourceVo);
        log.info("信息源组id：{}---{}", groupId, pageNo);
        IPage<InfoSourceVo> pageList = infoSourceService.pageListByGroupId(infoSourceVo, ynBind, groupId, pageNo, pageSize);
        List<InfoSourceVo> list = pageList.getRecords();
        return Result.OK(pageList);
    }


    /**
     * 智库绑定信息源分页列表
     *
     * @param thinkTankId         智库id
     * @param infoSearchCondition 检索条件
     * @param pageNo              页码
     * @param pageSize            每页条数
     * @author lkg
     * @date 2025/8/2
     */
    @GetMapping(value = "/bindInfoSourcePageList")
    public Result<?> bindInfoSourcePageList(InfoSearchCondition infoSearchCondition,
                                            @RequestParam String thinkTankId,
                                            @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                            @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<BindInfoSourceVO> pageList = infoSourceService.bindInfoSourcePageList(thinkTankId, infoSearchCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 智库绑定信息源栏目分页列表
     *
     * @param thinkTankId         智库id
     * @param infoSearchCondition 检索条件
     * @param pageNo              页码
     * @param pageSize            每页条数
     * @author lkg
     * @date 2025/8/2
     */
    @GetMapping(value = "/bindInfoSourceColumnPageList")
    public Result<?> bindInfoSourceColumnPageList(InfoSearchCondition infoSearchCondition,
                                                  @RequestParam String thinkTankId,
                                                  @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                                  @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<BindInfoSourceColumnVO> pageList = infoSourceService.bindInfoSourceColumnPageList(thinkTankId, infoSearchCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }


    /**
     * 智库绑定信息源组列表
     *
     * @param thinkTankId 智库id
     * @author lkg
     * @date 2025/8/2
     */
    @GetMapping(value = "/bindInfoSourceGroupList")
    public Result<?> bindInfoSourceMainGroupList(@RequestParam String thinkTankId) {
        JSONObject infoSourceGroupPages = infoSourceService.bindInfoSourceGroupList(thinkTankId);
        return Result.OK(infoSourceGroupPages);
    }

    /**
     * 智库绑定的信息源列表
     *
     * @param thinkTankId 智库id
     * @author lkg
     * @date 2025/8/2
     */
    @GetMapping(value = "/bindInfoSourceList")
    public Result<?> bindInfoSourceList(@RequestParam String thinkTankId) {
        JSONObject infoSourcePages = infoSourceService.bindInfoSourceList(thinkTankId);
        return Result.OK(infoSourcePages);
    }


    /**
     * 智库绑定信息源栏目组列表
     *
     * @param thinkTankId 智库id
     * @author lkg
     * @date 2025/8/2
     */
    @GetMapping(value = "/bindInfoSourceColumnGroupList")
    public Result<?> bindInfoSourceColumnGroupList(@RequestParam String thinkTankId) {
        JSONObject infoSourceGroupPages = infoSourceService.bindInfoSourceColumnGroupList(thinkTankId);
        return Result.OK(infoSourceGroupPages);
    }

    /**
     * 智库绑定的信息源栏目列表
     *
     * @param thinkTankId 智库id
     * @author lkg
     * @date 2025/8/2
     */
    @GetMapping(value = "/bindInfoSourceColumnList")
    public Result<?> bindInfoSourceColumnList(@RequestParam String thinkTankId) {
        JSONObject infoSourceGroupPages = infoSourceService.bindInfoSourceColumnList(thinkTankId);
        return Result.OK(infoSourceGroupPages);
    }

    /**
     * 智库绑定的信息源/信息源栏目标签列表
     *
     * @param thinkTankId 智库id
     * @param category    类型(1-信息源栏目；2-信息源)
     * @author lkg
     * @date 2025/8/2
     */
    @GetMapping(value = "/bindInfoSourceLabelList")
    public Result<?> bindInfoSourceLabelList(@RequestParam String thinkTankId, @RequestParam Integer category) {
        if (category != 1 && category != 2) {
            return Result.error("参数错误");
        }
        JSONObject infoSourceGroupPages = infoSourceService.bindInfoSourceLabelList(thinkTankId, category);
        return Result.OK(infoSourceGroupPages);
    }

    /**
     * 信息源分页列表查询
     *
     * @param infoSourceVo
     * @param pageNo
     * @param pageSize
     * @param infoSourceTypeId
     * @return
     */
    //@GetMapping(value = "/listByTypeId")
    public Result<?> queryPageListByTypeId(InfoSourceVo infoSourceVo,
                                           @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                           @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                           @RequestParam(name = "infoSourceTypeId", defaultValue = "") String infoSourceTypeId) {

        log.info("信息源分页列表查询:{}", infoSourceVo);
        IPage<InfoSourceVo> pageList = infoSourceService.pageListByTypeId(null, null, infoSourceVo, pageNo, pageSize, infoSourceTypeId);
        return Result.OK(pageList);
    }


    /**
     * 绑定信息源
     *
     * @param bindVO 绑定信息
     * @author lkg
     * @date 2025/8/2
     */
    @PostMapping(value = "/bind")
    public Result<?> bind(@RequestBody BindVO bindVO) {
        List<String> bindIds = bindVO.getBindIds();
        if (StringUtils.isEmpty(bindVO.getThinkTankId())) {
            return Result.error("智库id不能为空");
        }
        if (bindVO.getSourceBindType() == null) {
            return Result.error("绑定类型不能为空");
        }
        if (bindVO.getSourceType() == null) {
            return Result.error("信息源类型不能为空");
        }
        if (CollectionUtil.isEmpty(bindIds) && (bindVO.getSourceType() == 1 || bindVO.getSourceType() == 2)) {
            return Result.error("绑定id不能为空");
        }
        if (CollectionUtil.isEmpty(bindVO.getBindLabels()) && bindVO.getSourceType() == 3) {
            return Result.error("绑定标签不能为空");
        }
        infoSourceGroupMapService.bind(bindVO);
        return Result.OK();
    }

    /**
     * 绑定信息源
     *
     * @param bindVO 绑定信息
     * @author lkg
     * @date 2025/8/2
     */
    @PostMapping(value = "/unBind")
    public Result<?> unBind(@RequestBody BindVO bindVO) {
        List<String> bindIds = bindVO.getBindIds();
        if (StringUtils.isEmpty(bindVO.getThinkTankId())) {
            return Result.error("智库id不能为空");
        }
        if (bindVO.getSourceBindType() == null) {
            return Result.error("解绑类型不能为空");
        }
        if (bindVO.getSourceType() == null) {
            return Result.error("信息源类型不能为空");
        }
        if (CollectionUtil.isEmpty(bindIds) && (bindVO.getSourceType() == 1 || bindVO.getSourceType() == 2)) {
            return Result.error("解绑绑定id不能为空");
        }
        if (CollectionUtil.isEmpty(bindVO.getBindLabels()) && bindVO.getSourceType() == 3) {
            return Result.error("解绑标签不能为空");
        }
        infoSourceGroupMapService.unBind(bindVO);
        return Result.OK();
    }


    /**
     * 绑定信息源
     */
    //@PostMapping(value = "/bind")
    public Result<?> bind(@RequestBody InfoSourceGroupPage infoSourceGroupPage) {
        log.info("绑定信息源:{}", infoSourceGroupPage);
        String message = infoSourceGroupMapService.bind(infoSourceGroupPage.getId(), infoSourceGroupPage.getInfoSourceIds());
        return Result.OK(message);
    }

    /**
     * 解绑信息源
     */
    @PostMapping(value = "/unBound")
    public Result<?> unBound(@RequestBody InfoSourceGroupPage infoSourceGroupPage) {
        log.info("解绑信息源:{}", infoSourceGroupPage);
        infoSourceGroupMapService.unBind(infoSourceGroupPage.getId(), infoSourceGroupPage.getInfoSourceIds());
        return Result.OK("解绑成功!");
    }

    /**
     * 提升信息源优先等级
     * 批量紧急
     *
     * @param groupId 信息源组id
     * @author
     * @date
     */
    @GetMapping("/upLevel")
    public Result<?> upLevel(@RequestParam("groupId") String groupId) {
        log.info("批量紧急:{}", groupId);
        List<String> sourceIdList = infoSourceService.listByGroupId(groupId);
        LambdaUpdateWrapper<InfoSource> update = Wrappers.lambdaUpdate();
        update.set(InfoSource::getSiteLevel, 1).in(InfoSource::getId, sourceIdList);
        infoSourceService.update(update);
        return Result.OK();
    }

}
