package com.zzsn.thinktank.service.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.poi.excel.ExcelUtil;
import cn.hutool.poi.excel.ExcelWriter;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zzsn.thinktank.entity.*;
import com.zzsn.thinktank.mapper.ThinktankBasicInfoMapper;
import com.zzsn.thinktank.service.*;
import com.zzsn.thinktank.util.*;
import com.zzsn.thinktank.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.Functions;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/4/24 10:32
 * @Content:
 */
@Slf4j
@Service
public class ThinktankBasicInfoServiceImpl extends ServiceImpl<ThinktankBasicInfoMapper, ThinktankBasicInfo> implements ThinktankBasicInfoService {

    @Autowired
    ThinktankBasicInfoMapper thinktankBasicInfoMapper;

    //生成id
    @Autowired
    IGeneratorIdService generatorIdService;

    //左侧分类树
    @Autowired
    LeaderCategoryService leaderCategoryService;

    //标签
    @Autowired
    ThinktankTagService thinktankTagService;

    //所属单位
    @Autowired
    EnterpriseService enterpriseService;

    //信息源关联
    @Autowired
    InfoSourceGroupMapService infoSourceGroupMapService;

    @Autowired
    EsUtil esUtil;

    @Resource
    private DataImportService dataImportService;

    //语种字典吗
    @Value("${dict.language:}")
    private String LANGUAGE;
    //营利性质字典吗
    @Value("${dict.think_tank_profit_nature:}")
    private String THINK_TANK_PROFIT_NATURE;
    //智库性质字典吗
    @Value("${dict.think_tank_nature:}")
    private String THINK_TANK_NATURE;
    //研究领域标签id
    @Value("${sys_base_label_type}")
    private String SYS_BASE_LABEL_TYPE;


    @Value("${files.storage}")
    String filesStorage;

    private static final String IMPORT_TEMP = "think-tank-import-temp.xlsx";
    private static final String IMPORT_TEMP_NAME = "智库信息导入模板.xlsx";


    @Override
    public Result<?> add(ThinktankBasicInfo thinktankBasicInfo) {

        String id = Long.toString(generatorIdService.getOrderId());
        thinktankBasicInfo.setId(id);
        String codeId = generatorIdService.getIdNo();
        thinktankBasicInfo.setCodeId(codeId);

        //标签
        String tagId = thinktankBasicInfo.getTagId();
        if (tagId != null && tagId.length() > 0) {
            String[] ids = tagId.split(",");
            List<LabelEntity> list = thinktankTagService.getListByIds(ids);
            List<String> nameList = new ArrayList<>();
            for (LabelEntity labelEntity : list) {
                String name = labelEntity.getName();
                nameList.add(name);
            }
            String tagName = nameList.stream().collect(Collectors.joining(","));
            thinktankBasicInfo.setTagName(tagName);
        }

        //类别
        String belongCountryId = thinktankBasicInfo.getBelongCountryId();
        ThinktankCategoryStructure tcs = leaderCategoryService.getById(belongCountryId);
        if (tcs != null) {
            String typeName = tcs.getTypeName();
            thinktankBasicInfo.setBelongCountry(typeName);
        }

        thinktankBasicInfo.setCreateTime(new Date());
        thinktankBasicInfo.setUpdateTime(new Date());

        boolean b = this.save(thinktankBasicInfo);
        Result result = Result.OK();
        if (!b) {
            result.error500("保存失败！");
        }
        //同步新增智库数据到标签管理
        thinktankTagService.addLabel(thinktankBasicInfo);
        return result;
    }

    @Override
    public Result<?> del(String id) {
        String[] idArr = id.split(",");
        List<String> ids = Arrays.asList(idArr);
        boolean b = this.removeByIds(ids);
        Result result = Result.OK();
        if (!b) {
            result.error500("删除失败！");
        }
        thinktankTagService.delLabel(ids);
        return result;
    }

    @Override
    public Result<?> edit(ThinktankBasicInfo thinktankBasicInfo) {

        //标签
        String tagId = thinktankBasicInfo.getTagId();
        if (tagId != null && tagId.length() > 0) {
            String[] ids = tagId.split(",");
            List<LabelEntity> list = thinktankTagService.getListByIds(ids);
            List<String> nameList = new ArrayList<>();
            for (LabelEntity labelEntity : list) {
                String name = labelEntity.getName();
                nameList.add(name);
            }
            String tagName = nameList.stream().collect(Collectors.joining(","));
            thinktankBasicInfo.setTagName(tagName);
        }

        //类别
        String belongCountryId = thinktankBasicInfo.getBelongCountryId();
        ThinktankCategoryStructure tcs = leaderCategoryService.getById(belongCountryId);
        String typeName = tcs.getTypeName();
        thinktankBasicInfo.setBelongCountry(typeName);

        thinktankBasicInfo.setUpdateTime(new Date());

        boolean b = this.updateById(thinktankBasicInfo);
        Result result = Result.OK();
        if (!b) {
            result.error500("修改失败！");
        }
        thinktankTagService.editLabel(thinktankBasicInfo);
        return result;
    }

    @Override
    public ThinktankBasicInfoVo getInfoById(String id) {
        ThinktankBasicInfo info = this.getById(id);

        if (info == null) {
            return null;
        }

        ThinktankBasicInfoVo infoVo = new ThinktankBasicInfoVo();
        BeanUtils.copyProperties(info, infoVo);
        List<ThinktankBasicInfoVo> list = new ArrayList<>();
        list.add(infoVo);

        List<String> groupIds = new ArrayList<>();
        for (ThinktankBasicInfoVo thinktankBasicInfoVo : list) {
            groupIds.add(thinktankBasicInfoVo.getId());
        }
        List<InfoSourceGroupMap> infoSourceGroupMapList = infoSourceGroupMapService.getSourceGroupMap(groupIds);

        List<InfoSourceGroupCountVo> countVoList = new ArrayList<>();
        List<String> sidList = new ArrayList<>();
        for (InfoSourceGroupMap infoSourceGroupMap : infoSourceGroupMapList) {
            InfoSourceGroupCountVo infoSourceGroupCountVo = new InfoSourceGroupCountVo();
            BeanUtils.copyProperties(infoSourceGroupMap, infoSourceGroupCountVo);
            countVoList.add(infoSourceGroupCountVo);
            sidList.add(infoSourceGroupMap.getSourceId());
        }

        //查询es
        BoolQueryBuilder boolQuerylist = QueryBuilders.boolQuery();
        boolQuerylist.must(QueryBuilders.termsQuery("sid", sidList));
        Map<String, Long> mapGroup = null;
        try {
            mapGroup = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
        } catch (IOException e) {
            log.error("ES查询失败");
            e.printStackTrace();
        }
        if (mapGroup != null && mapGroup.size() > 0) {
            for (InfoSourceGroupCountVo infoSourceGroupCountVo : countVoList) {
                String sourceId = infoSourceGroupCountVo.getSourceId();
                Long countInfo = mapGroup.get(sourceId);
                if (countInfo == null) {
                    countInfo = 0L;
                }
                infoSourceGroupCountVo.setInfoCount(Math.toIntExact(countInfo));
            }
        }

        //统计关联信息源数量
        Map<String, List<InfoSourceGroupCountVo>> studentsByGrade = countVoList.stream()
                .collect(Collectors.groupingBy(InfoSourceGroupCountVo::getGroupId));

        for (ThinktankBasicInfoVo thinktankBasicInfoVo : list) {
//            String id = thinktankBasicInfoVo.getId();
            List<InfoSourceGroupCountVo> listVo = studentsByGrade.get(thinktankBasicInfoVo.getId());
            if (listVo == null) {
                thinktankBasicInfoVo.setCollectionCount(0);
                thinktankBasicInfoVo.setInfoSourceCount(0);
            } else {
                thinktankBasicInfoVo.setInfoSourceCount(listVo.size());
                int count = 0;
                for (InfoSourceGroupCountVo infoSourceGroupCountVo : listVo) {
                    count += infoSourceGroupCountVo.getInfoCount();
                }
                thinktankBasicInfoVo.setCollectionCount(count);
            }
        }

        return list.get(0);
    }

    @Override
    public Result<?> getList(ThinktankBasicInfoListVo thinktankBasicInfoListVo) {
        IPage page = new Page();

        thinktankBasicInfoListVo.setColumn(StringUtil.humpToLine(thinktankBasicInfoListVo.getColumn()));
        //查询列表
        List<ThinktankBasicInfoVo> list = thinktankBasicInfoMapper.getList(thinktankBasicInfoListVo);
        page.setRecords(list);
        //查询总数
        Integer total = thinktankBasicInfoMapper.getCount(thinktankBasicInfoListVo);
        page.setTotal(total);
        Integer pageNo = thinktankBasicInfoListVo.getPageNo();
        Integer pageSize = thinktankBasicInfoListVo.getPageSize();
        page.setCurrent(pageNo);
        page.setSize(pageSize);

        if (list.size() == 0) {
            return Result.OK(page);
        }

        //信息源
        List<String> groupIds = new ArrayList<>();
        for (ThinktankBasicInfoVo thinktankBasicInfoVo : list) {
            groupIds.add(thinktankBasicInfoVo.getId());
        }
        List<InfoSourceGroupMap> infoSourceGroupMapList = infoSourceGroupMapService.getSourceGroupMap(groupIds);

        List<InfoSourceGroupCountVo> countVoList = new ArrayList<>();
        List<String> sidList = new ArrayList<>();
        for (InfoSourceGroupMap infoSourceGroupMap : infoSourceGroupMapList) {
            InfoSourceGroupCountVo infoSourceGroupCountVo = new InfoSourceGroupCountVo();
            BeanUtils.copyProperties(infoSourceGroupMap, infoSourceGroupCountVo);
            countVoList.add(infoSourceGroupCountVo);
            sidList.add(infoSourceGroupMap.getSourceId());
        }

        //查询es
        BoolQueryBuilder boolQuerylist = QueryBuilders.boolQuery();
        boolQuerylist.must(QueryBuilders.termsQuery("sid", sidList));
        Map<String, Long> mapGroup = null;
        try {
            mapGroup = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
        } catch (IOException e) {
            log.error("ES查询失败");
            e.printStackTrace();
        }
        if (mapGroup != null && mapGroup.size() > 0) {
            for (InfoSourceGroupCountVo infoSourceGroupCountVo : countVoList) {
                String sourceId = infoSourceGroupCountVo.getSourceId();
                Long countInfo = mapGroup.get(sourceId);
                if (countInfo == null) {
                    countInfo = 0L;
                }
                infoSourceGroupCountVo.setInfoCount(Math.toIntExact(countInfo));
            }
        }

        //统计关联信息源数量
        Map<String, List<InfoSourceGroupCountVo>> studentsByGrade = countVoList.stream()
                .collect(Collectors.groupingBy(InfoSourceGroupCountVo::getGroupId));

        for (ThinktankBasicInfoVo thinktankBasicInfoVo : list) {
            String id = thinktankBasicInfoVo.getId();
            List<InfoSourceGroupCountVo> listVo = studentsByGrade.get(id);
            if (listVo == null) {
                thinktankBasicInfoVo.setCollectionCount(0);
                thinktankBasicInfoVo.setInfoSourceCount(0);
            } else {
                thinktankBasicInfoVo.setInfoSourceCount(listVo.size());
                int count = 0;
                for (InfoSourceGroupCountVo infoSourceGroupCountVo : listVo) {
                    Integer infoCount = infoSourceGroupCountVo.getInfoCount();
                    if (infoCount != null) {
                        count += infoCount;
                    }
                }
                thinktankBasicInfoVo.setCollectionCount(count);
            }
        }

        //记录热门标签
        String tagId = thinktankBasicInfoListVo.getTagId();
        if (tagId != null && tagId.length() > 0) {
            String[] tagIdArr = tagId.split(",");
            for (String s : tagIdArr) {
                thinktankTagService.addTagTop(s);
            }
        }

        return Result.OK(page);
    }

    @Override
    public void downloadTemplate(HttpServletResponse response) {
        String filePath = filesStorage + IMPORT_TEMP;

        File file = new File(filePath);
        try {
            // 将文件写入输入流
            FileInputStream fileInputStream = new FileInputStream(file);
            InputStream fis = new BufferedInputStream(fileInputStream);
            byte[] buffer = new byte[fis.available()];
            fis.read(buffer);
            fis.close();
            // 清空response
            response.reset();
            // 设置response的Header
            response.setCharacterEncoding("UTF-8");
            //Content-Disposition的作用：告知浏览器以何种方式显示响应返回的文件，用浏览器打开还是以附件的形式下载到本地保存
            //attachment表示以附件方式下载   inline表示在线打开   "Content-Disposition: inline; filename=文件名.mp3"
            // filename表示文件的默认名称，因为网络传输只支持URL编码的相关支付，因此需要将文件名URL编码后进行传输,前端收到后需要反编码才能获取到真正的名称
            response.addHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(IMPORT_TEMP_NAME, "UTF-8"));
            // 告知浏览器文件的大小
            response.addHeader("Content-Length", "" + file.length());
            OutputStream outputStream = new BufferedOutputStream(response.getOutputStream());
            response.setContentType("application/octet-stream");
            outputStream.write(buffer);
            outputStream.flush();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public Result batchImport(MultipartFile multipartFile) {
        int index = multipartFile.getOriginalFilename().lastIndexOf(".");
        String fileSuffix = multipartFile.getOriginalFilename().substring(index + 1);
        if ("xls".equals(fileSuffix) || "xlsx".equals(fileSuffix)) {
            //文件校验
            if (!this.checkExcel(multipartFile)) {
                log.error("excel格式不对");
                return Result.error("模版有误，请使用正确的模板！");
            }
            String msg = doExcel(multipartFile, fileSuffix);
            return Result.OK(msg);
        } else {
            return Result.error("不支持的文件类型！");
        }
    }

    private boolean checkExcel(MultipartFile multipartFile) {
        Workbook workbook = ExcelExportUtil.getWorkbook(multipartFile);
        if (workbook != null) {
            List<String> list = ExcelExportUtil.getSheetTitles(workbook);
            return list != null
                    && list.contains("序号")
                    && list.contains("智库机构编码")
                    && list.contains("中文全称")
                    && list.contains("中文简称")
                    && list.contains("英文全称")
                    && list.contains("英文简称")
                    && list.contains("原文全称")
                    && list.contains("原文简称")
                    && list.contains("官网")
                    && list.contains("所属国家")
                    && list.contains("标签")
                    && list.contains("简介")
                    && list.contains("成立时间")
                    && list.contains("是否收费(1-是;0-否)")
                    && list.contains("地址")
                    && list.contains("创办单位/所属单位")
                    && list.contains("属性")
                    ;
        }
        return false;
    }

    private String doExcel(MultipartFile multipartFile, String fileSuffix) {
        List<List<String>> excelLists = null;
        try {
            byte[] fileData = multipartFile.getBytes();
            excelLists = ExcelExportUtil.readExcel(new ByteArrayInputStream(fileData), 1, 17);
        } catch (IOException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
        String msg = "";
        int i = 0;//记录成功数量
        int j = 0;//记录失败数量
        if (excelLists != null && excelLists.size() > 0) {
            //处理数据
            List<ThinktankBasicInfo> infoList = new ArrayList<>();

            for (List<String> list : excelLists) {
                ThinktankBasicInfo thinktankBasicInfo = new ThinktankBasicInfo();
                if (StringUtils.isNotEmpty(list.get(0))) thinktankBasicInfo.setId(list.get(0));
                if (StringUtils.isNotEmpty(list.get(1))) thinktankBasicInfo.setCodeId(list.get(1));
                if (StringUtils.isNotEmpty(list.get(2))) thinktankBasicInfo.setChineseWhole(list.get(2));
                if (StringUtils.isNotEmpty(list.get(3))) thinktankBasicInfo.setChineseSimple(list.get(3));
                if (StringUtils.isNotEmpty(list.get(4))) thinktankBasicInfo.setEnglishWhole(list.get(4));
                if (StringUtils.isNotEmpty(list.get(5))) thinktankBasicInfo.setEnglishSimple(list.get(5));
                if (StringUtils.isNotEmpty(list.get(6))) thinktankBasicInfo.setOriginalWhole(list.get(6));
                if (StringUtils.isNotEmpty(list.get(7))) thinktankBasicInfo.setOriginalSimple(list.get(7));
                if (StringUtils.isNotEmpty(list.get(8))) thinktankBasicInfo.setOfficialWebsite(list.get(8));
                if (StringUtils.isNotEmpty(list.get(9))) thinktankBasicInfo.setBelongCountry(list.get(9));
                if (StringUtils.isNotEmpty(list.get(10))) thinktankBasicInfo.setTagName(list.get(10));
                if (StringUtils.isNotEmpty(list.get(11))) thinktankBasicInfo.setBiographicalNotes(list.get(11));
                if (StringUtils.isNotEmpty(list.get(12))) thinktankBasicInfo.setEstablishTime(list.get(12));
                //是否收费
                if (StringUtils.isNotEmpty(list.get(13))) {
                    String chargeStr = list.get(13);
                    try {
                        Integer charge = Integer.valueOf(chargeStr);
                        thinktankBasicInfo.setCharge(charge);
                    } catch (Exception e) {
                        thinktankBasicInfo.setCharge(-1);
                        log.error("是否收费转换失败：{}，{}", list.get(1), list.get(13));
                    }
                }
                if (StringUtils.isNotEmpty(list.get(14))) thinktankBasicInfo.setAddress(list.get(14));
                if (StringUtils.isNotEmpty(list.get(15))) thinktankBasicInfo.setBelongUnit(list.get(15));
                if (StringUtils.isNotEmpty(list.get(15))) thinktankBasicInfo.setAttribute(list.get(16));
                infoList.add(thinktankBasicInfo);
            }

            //国家列表
            List<ThinktankCategoryStructure> listCategoryStructure = leaderCategoryService.list();
            Map<String, String> mapCategoryStructure = new HashMap<>();
            for (ThinktankCategoryStructure thinktankCategoryStructure : listCategoryStructure) {
                mapCategoryStructure.put(thinktankCategoryStructure.getTypeName(), thinktankCategoryStructure.getId());
            }
            //标签列表
            Map<String, String> mapTag = new HashMap<>();//标签名称列表
            Result result = thinktankTagService.getTagList(null, null);
            List<SysLabelVo> list = (List<SysLabelVo>) result.getResult();
            for (SysLabelVo labelVo : list) {
                mapTag.put(labelVo.getName(), labelVo.getId());
            }
            StringBuffer sb = new StringBuffer();

            for (ThinktankBasicInfo thinktankBasicInfo : infoList) {
                boolean b = true;
                //序号
                String number = thinktankBasicInfo.getId();
                String codeId = thinktankBasicInfo.getCodeId();
                if (codeId == null || codeId.length() == 0) {
                    thinktankBasicInfo.setCodeId(generatorIdService.getIdNo());
                    thinktankBasicInfo.setId(Long.toString(generatorIdService.getOrderId()));
                    thinktankBasicInfo.setCreateTime(new Date());
                    thinktankBasicInfo.setUpdateTime(new Date());
                } else {
                    LambdaQueryWrapper<ThinktankBasicInfo> lambdaQuery = Wrappers.lambdaQuery();
                    lambdaQuery.eq(ThinktankBasicInfo::getCodeId, codeId);
                    List<ThinktankBasicInfo> listBasicInfo = thinktankBasicInfoMapper.selectList(lambdaQuery);
                    if (listBasicInfo.size() > 0) {
                        thinktankBasicInfo.setId(listBasicInfo.get(0).getId());
                        thinktankBasicInfo.setUpdateTime(new Date());
                    } else {
                        thinktankBasicInfo.setId(Long.toString(generatorIdService.getOrderId()));
                        thinktankBasicInfo.setCreateTime(new Date());
                        thinktankBasicInfo.setUpdateTime(new Date());
                    }
                }

                //国家
                String belongCountry = thinktankBasicInfo.getBelongCountry();
                String belongCountryId = mapCategoryStructure.get(belongCountry);
                if (StringUtils.isEmpty(belongCountryId)) {
                    b = false;
                    sb.append("序号为'" + number + "'的国家’" + belongCountry + "‘数据库没有，导入失败;\n");
                } else {
                    thinktankBasicInfo.setBelongCountryId(belongCountryId);
                }

                //标签
                String tagName = thinktankBasicInfo.getTagName();
                if (tagName != null && tagName.length() > 0) {
                    String[] tagNameArr = tagName.split(",");
                    List<String> tagIdList = new ArrayList<>();
                    for (String s : tagNameArr) {
                        String tagId = mapTag.get(s);
                        if (StringUtils.isEmpty(tagId)) {
                            b = false;
                            sb.append("序号为'" + number + "'的标签'" + s + "'数据库没有，导入失败;\n");
                        } else {
                            tagIdList.add(tagId);
                        }
                    }
                    String tagId = String.join(",", tagIdList);
                    thinktankBasicInfo.setTagId(tagId);
                }

                //所属单位
                String belongUnit = thinktankBasicInfo.getBelongUnit();
                if (belongUnit != null && belongUnit.length() > 0) {
                    List<SysBaseEnterprise> listEnter = enterpriseService.getListByName(belongUnit);
                    if (listEnter.size() > 0) {
                        SysBaseEnterprise enter = listEnter.get(0);
                        String belongUnitCode = enter.getSocialCreditCode();
                        thinktankBasicInfo.setBelongUnitCode(belongUnitCode);
                    } else {
                        b = false;
                        sb.append("序号为'" + number + "'的创办单位/所属单位'" + belongUnit + "'数据库没有，导入失败;\n");
                    }
                }
                //成立时间
                String establishTime = thinktankBasicInfo.getEstablishTime();
                if (!DateUtil.isDate(establishTime)) {
                    b = false;
                    sb.append("序号为'" + number + "'的成立时间'" + belongUnit + "'格式错误，导入失败;\n");
                }

                if (b) {
                    this.saveOrUpdate(thinktankBasicInfo);
                    thinktankTagService.addLabel(thinktankBasicInfo);
                    i++;
                } else {
                    j++;
                }

            }
            sb.append("请重新修改后再次上传。");
            msg = "导入完成!\n"
                    + "导入成功数量：" + i + "\n"
                    + "导入失败数量：" + j + "\n"
                    + "其中：\n"
                    + sb.toString();
        }
        if (j > 0) {
            return msg;
        }
        return "true";
    }

    @Override
    public byte[] batchExport(ThinktankBasicInfoListVo thinktankBasicInfoListVo, HttpServletRequest request, HttpServletResponse response) {
        List<ThinktankBasicInfoExportVo> list = this.getExportList(thinktankBasicInfoListVo);
        log.info("{}", list.size());

        int i = 1;
        for (ThinktankBasicInfoExportVo thinktankBasicInfoVo : list) {
            thinktankBasicInfoVo.setInfoSourceId(String.valueOf(i++));
        }

        try {
            ExcelWriter writer = ExcelUtil.getWriter();
            writer.addHeaderAlias("infoSourceId", "序号");
            writer.addHeaderAlias("id", "智库机构ID");
            writer.addHeaderAlias("codeId", "智库机构编码");
            writer.addHeaderAlias("chineseWhole", "中文全称");
            writer.addHeaderAlias("chineseSimple", "中文简称");
            writer.addHeaderAlias("englishWhole", "英文全称");
            writer.addHeaderAlias("englishSimple", "英文简称");
            writer.addHeaderAlias("originalWhole", "原文全称");
            writer.addHeaderAlias("originalSimple", "原文简称");
            writer.addHeaderAlias("officialWebsite", "官网");
            writer.addHeaderAlias("belongCountry", "所属国家");
            writer.addHeaderAlias("tagName", "标签");
            writer.addHeaderAlias("biographicalNotes", "简介");
            writer.addHeaderAlias("establishTime", "成立时间");
            writer.addHeaderAlias("charge", "是否收费(1-是;0-否)");
            writer.addHeaderAlias("address", "地址");
            writer.addHeaderAlias("belongUnit", "创办单位/所属单位");
            writer.addHeaderAlias("collectionCount", "信息采集数量");
            writer.addHeaderAlias("infoSourceCount", "关联信息源数量");
            writer.addHeaderAlias("infoSourceCode", "关联信息源编码");
            writer.addHeaderAlias("webSiteName", "关联信息源名称");
            writer.addHeaderAlias("siteName", "栏目名称");
            writer.addHeaderAlias("siteUri", "网址");
            writer.addHeaderAlias("status", "启用状态(1-启用;0-禁用)");
            writer.addHeaderAlias("count", "信息源采集数量");

            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

            // 一次性写出内容，使用默认样式，强制输出标题
            writer.write(list, true);
            //out为OutputStream，需要写出到的目标流
            writer.flush(outputStream);
            // 关闭writer，释放内存
            writer.close();

            return outputStream.toByteArray();

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    public List<ThinktankBasicInfoExportVo> getExportList(ThinktankBasicInfoListVo thinktankBasicInfoListVo) {
        thinktankBasicInfoListVo.setColumn(StringUtil.humpToLine(thinktankBasicInfoListVo.getColumn()));

        //查询列表
        List<ThinktankBasicInfoExportVo> list = thinktankBasicInfoMapper.getExportList(thinktankBasicInfoListVo);

        if (list.size() == 0) {
            return list;
        }

        //信息源
        List<String> groupIds = new ArrayList<>();
        for (ThinktankBasicInfoExportVo thinktankBasicInfoVo : list) {
            groupIds.add(thinktankBasicInfoVo.getId());
        }
        List<InfoSourceGroupMap> infoSourceGroupMapList = infoSourceGroupMapService.getSourceGroupMap(groupIds);

        List<InfoSourceGroupCountVo> countVoList = new ArrayList<>();
        List<String> sidList = new ArrayList<>();
        for (InfoSourceGroupMap infoSourceGroupMap : infoSourceGroupMapList) {
            InfoSourceGroupCountVo infoSourceGroupCountVo = new InfoSourceGroupCountVo();
            BeanUtils.copyProperties(infoSourceGroupMap, infoSourceGroupCountVo);
            countVoList.add(infoSourceGroupCountVo);
            sidList.add(infoSourceGroupMap.getSourceId());
        }

        //查询es
        BoolQueryBuilder boolQuerylist = QueryBuilders.boolQuery();
        boolQuerylist.must(QueryBuilders.termsQuery("sid", sidList));

        //boolQuerylist.filter(QueryBuilders.rangeQuery("publishDate").gte("2024-05-18T00:00:01"));

        Map<String, Long> mapGroup = null;
        try {
            mapGroup = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
        } catch (IOException e) {
            log.error("ES查询失败");
            e.printStackTrace();
        }
        if (mapGroup != null && mapGroup.size() > 0) {
            for (InfoSourceGroupCountVo infoSourceGroupCountVo : countVoList) {
                String sourceId = infoSourceGroupCountVo.getSourceId();
                Long countInfo = mapGroup.get(sourceId);
                if (countInfo == null) {
                    countInfo = 0L;
                }
                infoSourceGroupCountVo.setInfoCount(Math.toIntExact(countInfo));
            }
        }

        //统计关联信息源数量
        Map<String, List<InfoSourceGroupCountVo>> studentsByGrade = countVoList.stream()
                .collect(Collectors.groupingBy(InfoSourceGroupCountVo::getGroupId));

        for (ThinktankBasicInfoExportVo thinktankBasicInfoVo : list) {

            String infoSourceId = thinktankBasicInfoVo.getInfoSourceId();
            Long count = mapGroup.get(infoSourceId);
            if (count == null) {
                thinktankBasicInfoVo.setCount(0);
            } else {
                thinktankBasicInfoVo.setCount(Math.toIntExact(count));
            }

            String id = thinktankBasicInfoVo.getId();
            List<InfoSourceGroupCountVo> listVo = studentsByGrade.get(id);
            if (listVo == null) {
                thinktankBasicInfoVo.setCollectionCount(0);
                thinktankBasicInfoVo.setInfoSourceCount(0);
            } else {
                thinktankBasicInfoVo.setInfoSourceCount(listVo.size());
                int sum = 0;
                for (InfoSourceGroupCountVo infoSourceGroupCountVo : listVo) {
                    sum += infoSourceGroupCountVo.getInfoCount();
                }
                thinktankBasicInfoVo.setCollectionCount(sum);
            }
        }

        return list;
    }

    @Override
    public Result<?> getListBaseData(String id, Integer pageNo, Integer pageSize, Integer office) {
        List<InfoSourceGroupMap> infoSourceGroupMapList = infoSourceGroupMapService.getSourceGroupMap(Arrays.asList(id));

        List<String> sidList = new ArrayList<>();
        for (InfoSourceGroupMap infoSourceGroupMap : infoSourceGroupMapList) {
            sidList.add(infoSourceGroupMap.getSourceId());
        }

        //设置分页参数
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
//        searchSourceBuilder.size(pageSize);
//        searchSourceBuilder.from(office);
        searchSourceBuilder.trackTotalHits(true);
        //排序
        searchSourceBuilder.sort("createDate", SortOrder.ASC);

        //查询es
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("sid", sidList));

        //查询数据
        searchSourceBuilder.query(boolQuery);
        Page<TransferData> page = null;

        try {
            page = esUtil.queryPage("basedata", searchSourceBuilder, TransferData.class, pageNo, pageSize);
        } catch (IOException e) {
            e.printStackTrace();
        }
        for (TransferData transferData : page.getRecords()) {
            transferData.setCreateDate(EsDateUtil.esFieldDateMapping(transferData.getCreateDate()));
            transferData.setPublishDate(EsDateUtil.esFieldDateMapping(transferData.getPublishDate()));
        }
        return Result.OK(page);
    }

    @Override
    @Transactional
    public String batchImport(List<ThinkTankFileVO> thinkTankImportVOList) {
        //国家列表
        List<ThinktankCategoryStructure> listCategoryStructure = leaderCategoryService.list();
        Map<String, String> mapCategoryStructure = listCategoryStructure.stream().collect(Collectors.toMap(ThinktankCategoryStructure::getTypeName, ThinktankCategoryStructure::getId, (value1, value2) -> value1));
        //语种
        List<DataChangeVO> languageList = dataImportService.dictItemList(LANGUAGE);
        Map<String, String> languageMap = languageList.stream().collect(Collectors.toMap(DataChangeVO::getDescription, DataChangeVO::getValue, (value1, value2) -> value1));
        //营利性质
        List<DataChangeVO> profitNatureList = dataImportService.dictItemList(THINK_TANK_PROFIT_NATURE);
        Map<String, String> profitNatureMap = profitNatureList.stream().collect(Collectors.toMap(DataChangeVO::getDescription, DataChangeVO::getValue, (value1, value2) -> value1));
        //智库性质
        List<DataChangeVO> natureList = dataImportService.dictItemList(THINK_TANK_NATURE);
        Map<String, String> natureMap = natureList.stream().collect(Collectors.toMap(DataChangeVO::getDescription, DataChangeVO::getValue, (value1, value2) -> value1));
        //研究领域
        List<DataChangeVO> researchFieldList = dataImportService.labelEntityList(SYS_BASE_LABEL_TYPE);
        Map<String, String> researchFieldMap = researchFieldList.stream().collect(Collectors.toMap(DataChangeVO::getDescription, DataChangeVO::getValue, (value1, value2) -> value1));
        StringBuilder stringBuilder = new StringBuilder();
        for (ThinkTankFileVO thinkTankImportVO : thinkTankImportVOList) {
            ThinktankBasicInfo thinktankBasicInfo = new ThinktankBasicInfo();
            BeanUtils.copyProperties(thinkTankImportVO, thinktankBasicInfo);
            String codeId = thinkTankImportVO.getCodeId();
            if (StringUtils.isEmpty(codeId)) {
                codeId = generatorIdService.getIdNo();
            }
            thinktankBasicInfo.setCodeId(codeId);
            String belongCountry = thinkTankImportVO.getBelongCountry();
            String belongCountryId = mapCategoryStructure.get(belongCountry);
            if (StringUtils.isEmpty(belongCountryId)) {
                stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的国家【").append(belongCountry).append("】,数据库中没有，导入失败;\n");
                continue;
            } else {
                thinktankBasicInfo.setBelongCountryId(belongCountryId);
            }
            String lang = thinkTankImportVO.getLang();
            if (StringUtils.isNotEmpty(lang)) {
                StringBuilder noExist = new StringBuilder();
                StringBuilder langCode = new StringBuilder();
                for (String value : lang.split(",")) {
                    String langValue = languageMap.get(value);
                    if (StringUtils.isEmpty(langValue)) {
                        noExist.append(",").append(value);
                    } else {
                        langCode.append(",").append(langValue);
                    }
                }
                if (StringUtils.isNotEmpty(noExist)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的语言【").append(noExist.substring(1)).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    thinktankBasicInfo.setLang(langCode.substring(1));
                }
            }
            String profitNature = thinkTankImportVO.getProfitNature();
            if (StringUtils.isNotEmpty(profitNature)) {
                String profitNatureValue = profitNatureMap.get(profitNature);
                if (StringUtils.isEmpty(profitNatureValue)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的营利性质【").append(profitNature).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    thinktankBasicInfo.setProfitNature(profitNatureValue);
                }
            }
            String nature = thinkTankImportVO.getNature();
            if (StringUtils.isNotEmpty(nature)) {
                String natureValue = natureMap.get(nature);
                if (StringUtils.isEmpty(natureValue)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的智库性质【").append(nature).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    thinktankBasicInfo.setNature(natureValue);
                }
            }
            String tagName = thinkTankImportVO.getTagName();
            if (StringUtils.isNotEmpty(tagName)) {
                StringBuilder noExist = new StringBuilder();
                StringBuilder tagId = new StringBuilder();
                for (String name : tagName.split(",")) {
                    String value = researchFieldMap.get(name);
                    if (StringUtils.isEmpty(value)) {
                        noExist.append(",").append(name);
                    } else {
                        tagId.append(",").append(value);
                    }
                }
                if (StringUtils.isNotEmpty(noExist)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的研究领域【").append(noExist.substring(1)).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    thinktankBasicInfo.setTagId(tagId.substring(1));
                }
            }
            String belongUnit = thinkTankImportVO.getBelongUnit();
            if (StringUtils.isNotEmpty(belongUnit)) {
                List<SysBaseEnterprise> listEnter = enterpriseService.getListByName(belongUnit);
                if (CollectionUtils.isEmpty(listEnter)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的创办单位/所属单位【").append(belongUnit).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    SysBaseEnterprise enter = listEnter.get(0);
                    thinktankBasicInfo.setBelongUnitCode(enter.getSocialCreditCode());
                }
            }
            LambdaQueryWrapper<ThinktankBasicInfo> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(ThinktankBasicInfo::getCodeId, codeId);
            ThinktankBasicInfo one = thinktankBasicInfoMapper.selectOne(queryWrapper);
            if (one != null) {
                thinktankBasicInfo.setId(one.getId());
                thinktankBasicInfoMapper.updateById(thinktankBasicInfo);
                //编辑标签数据
                thinktankTagService.editLabel(thinktankBasicInfo);
            } else {
                thinktankBasicInfo.setId(Long.toString(generatorIdService.getOrderId()));
                thinktankBasicInfoMapper.insert(thinktankBasicInfo);
                //增加标签数据
                thinktankTagService.addLabel(thinktankBasicInfo);
            }
        }
        return stringBuilder.length() > 0 ? stringBuilder.toString() : null;
    }

    @Override
    public List<ThinktankBasicInfo> exportListByCondition(String id, String codeId, String typeId, String thinktankName, String tagId, String keyword) {
        List<String> ids = new ArrayList<>();
        if (StringUtils.isNotEmpty(id)) {
            ids = Arrays.asList(id.split(","));
        }
        List<ThinktankBasicInfo> thinkTankBasicInfos = thinktankBasicInfoMapper.exportListByCondition(ids, codeId, typeId, thinktankName, tagId, keyword);
        if (CollectionUtils.isNotEmpty(thinkTankBasicInfos)) {
            //国家列表
            List<ThinktankCategoryStructure> listCategoryStructure = leaderCategoryService.list();
            Map<String, String> mapCategoryStructure = listCategoryStructure.stream().collect(Collectors.toMap(ThinktankCategoryStructure::getId, ThinktankCategoryStructure::getTypeName, (value1, value2) -> value1));
            //语种
            List<DataChangeVO> languageList = dataImportService.dictItemList(LANGUAGE);
            Map<String, String> languageMap = languageList.stream().collect(Collectors.toMap(DataChangeVO::getValue, DataChangeVO::getDescription, (value1, value2) -> value1));
            //营利性质
            List<DataChangeVO> profitNatureList = dataImportService.dictItemList(THINK_TANK_PROFIT_NATURE);
            Map<String, String> profitNatureMap = profitNatureList.stream().collect(Collectors.toMap(DataChangeVO::getValue, DataChangeVO::getDescription, (value1, value2) -> value1));
            //智库性质
            List<DataChangeVO> natureList = dataImportService.dictItemList(THINK_TANK_NATURE);
            Map<String, String> natureMap = natureList.stream().collect(Collectors.toMap(DataChangeVO::getValue, DataChangeVO::getDescription, (value1, value2) -> value1));
            //研究领域
            List<DataChangeVO> researchFieldList = dataImportService.labelEntityList(SYS_BASE_LABEL_TYPE);
            Map<String, String> researchFieldMap = researchFieldList.stream().collect(Collectors.toMap(DataChangeVO::getValue, DataChangeVO::getDescription, (value1, value2) -> value1));
            for (ThinktankBasicInfo thinkTankBasicInfo : thinkTankBasicInfos) {
                thinkTankBasicInfo.setBelongCountry(mapCategoryStructure.get(thinkTankBasicInfo.getBelongCountryId()));
                String lang = thinkTankBasicInfo.getLang();
                if (StringUtils.isNotEmpty(lang)) {
                    StringBuilder sb = new StringBuilder();
                    for (String s : lang.split(",")) {
                        String value = languageMap.get(s);
                        sb.append(",").append(value);
                    }
                    thinkTankBasicInfo.setLang(sb.substring(1));
                }
                String profitNature = thinkTankBasicInfo.getProfitNature();
                if (StringUtils.isNotEmpty(profitNature)) {
                    thinkTankBasicInfo.setProfitNature(profitNatureMap.get(profitNature));
                }
                String nature = thinkTankBasicInfo.getNature();
                if (StringUtils.isNotEmpty(nature)) {
                    thinkTankBasicInfo.setNature(natureMap.get(nature));
                }
                String tagIds = thinkTankBasicInfo.getTagId();
                if (StringUtils.isNotEmpty(tagIds)) {
                    StringBuilder sb = new StringBuilder();
                    for (String s : tagIds.split(",")) {
                        String value = researchFieldMap.get(s);
                        sb.append(",").append(value);
                    }
                    thinkTankBasicInfo.setTagName(sb.substring(1));
                }
                String belongUnitCode = thinkTankBasicInfo.getBelongUnitCode();
                if (StringUtils.isNotEmpty(belongUnitCode)) {
                    LambdaQueryWrapper<SysBaseEnterprise> queryWrapper = Wrappers.lambdaQuery();
                    queryWrapper.select(SysBaseEnterprise::getName).eq(SysBaseEnterprise::getSocialCreditCode, belongUnitCode);
                    SysBaseEnterprise one = enterpriseService.getOne(queryWrapper);
                    if (one != null) {
                        thinkTankBasicInfo.setBelongUnit(one.getName());
                    }
                }
            }
        }
        return thinkTankBasicInfos;
    }

    @Override
    public List<ThinkTankSourceVO> thinkTankCollectCount(List<ThinktankBasicInfo> dataList) {
        Map<String, ThinktankBasicInfo> map = dataList.stream().collect(Collectors.toMap(ThinktankBasicInfo::getId, Function.identity()));
        List<String> idList = new ArrayList<>();
        dataList.forEach(e -> idList.add(e.getId()));
        List<ThinkTankSourceVO> thinkTankSourceVOS = infoSourceGroupMapService.listOfThinkTank(idList);
        List<String> sourceIds = new ArrayList<>();
        for (ThinkTankSourceVO thinkTankSourceVO : thinkTankSourceVOS) {
            ThinktankBasicInfo thinktankBasicInfo = map.get(thinkTankSourceVO.getGroupId());
            thinkTankSourceVO.setCodeId(thinktankBasicInfo.getCodeId());
            thinkTankSourceVO.setChineseWhole(thinktankBasicInfo.getChineseWhole());
            //信息源id集合
            sourceIds.add(thinkTankSourceVO.getSourceId());
        }
        //查询es
        BoolQueryBuilder boolQuerylist = QueryBuilders.boolQuery();
        boolQuerylist.must(QueryBuilders.termsQuery("sid", sourceIds));
        Map<String, Long> totalCountMap = new HashMap<>();
        Map<String, Long> monthlyCountMap = new HashMap<>();
        try {
            //所有
            totalCountMap = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
            //近一个月
            DateTime yesterday = cn.hutool.core.date.DateUtil.yesterday();
            DateTime startTime = cn.hutool.core.date.DateUtil.offsetDay(yesterday, -30);
            boolQuerylist.filter(QueryBuilders.rangeQuery("createDate")
                    .gte(EsDateUtil.esFieldDateFormat(cn.hutool.core.date.DateUtil.format(startTime, "yyyy-MM-dd 00:00:00")))
                    .lte(EsDateUtil.esFieldDateFormat(cn.hutool.core.date.DateUtil.format(yesterday, "yyyy-MM-dd 23:59:59"))));
            monthlyCountMap = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
        } catch (IOException e) {
            log.error("ES查询失败");
            e.printStackTrace();
        }
        //采集量统计
        for (ThinkTankSourceVO thinkTankSourceVO : thinkTankSourceVOS) {
            String sourceId = thinkTankSourceVO.getSourceId();
            Long infoCount = totalCountMap.get(sourceId);
            if (infoCount == null) {
                infoCount = 0L;
            }
            thinkTankSourceVO.setInfoCount(infoCount.intValue());
            Long infoMonthlyCount = monthlyCountMap.get(sourceId);
            if (infoMonthlyCount == null) {
                infoMonthlyCount = 0L;
            }
            thinkTankSourceVO.setInfoMonthlyCount(infoMonthlyCount.intValue());
        }
        //数据合并
        List<ThinkTankSourceVO> list = new ArrayList<>();
        Map<String, List<ThinkTankSourceVO>> groupCountMap = thinkTankSourceVOS.stream().collect(Collectors.groupingBy(ThinkTankSourceVO::getGroupId));
        for (Map.Entry<String, List<ThinkTankSourceVO>> entry : groupCountMap.entrySet()) {
            List<ThinkTankSourceVO> value = entry.getValue();
            int groupCount = value.stream().mapToInt(ThinkTankSourceVO::getInfoCount).sum();
            ThinkTankSourceVO thinkTankSourceVO = value.get(0);
            thinkTankSourceVO.setGroupCount(groupCount);
            thinkTankSourceVO.setInfoSourceCount(value.size());
            list.addAll(value);
        }
        return list;
    }
}
