package com.zzsn.thinktank.service.impl;

import cn.hutool.core.map.MapUtil;
import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelUtil;
import cn.hutool.core.date.DateTime;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.zzsn.thinktank.constants.CommonConstants;
import com.zzsn.thinktank.entity.*;
import com.zzsn.thinktank.mapper.ThinktankBasicInfoMapper;
import com.zzsn.thinktank.mapper.ThinktankOttInfoMapper;
import com.zzsn.thinktank.service.*;
import com.zzsn.thinktank.util.EsDateUtil;
import com.zzsn.thinktank.util.EsUtil;
import com.zzsn.thinktank.util.SimRedisUtil;
import com.zzsn.thinktank.util.StringUtil;
import com.zzsn.thinktank.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.util.*;
import java.util.function.Function;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/4/24 10:32
 * @Content:
 */
@Slf4j
@Service
public class ThinktankBasicInfoServiceImpl extends ServiceImpl<ThinktankBasicInfoMapper, ThinktankBasicInfo> implements ThinktankBasicInfoService {

    @Autowired
    ThinktankBasicInfoMapper thinktankBasicInfoMapper;

    @Autowired
    ThinktankOttInfoMapper thinktankOttInfoMapper;

    //生成id
    @Autowired
    IGeneratorIdService generatorIdService;

    //左侧分类树
    @Autowired
    LeaderCategoryService leaderCategoryService;

    //标签
    @Autowired
    ThinktankTagService thinktankTagService;

    //所属单位
    @Autowired
    EnterpriseService enterpriseService;

    //信息源关联
    @Autowired
    InfoSourceGroupMapService infoSourceGroupMapService;

    //区域国家
    @Autowired
    ISysBaseRegionService sysBaseRegionService;

    @Autowired
    private ISysDictItemService sysDictItemService;

    @Autowired
    private ISysDictService sysDictService;

    @Autowired
    EsUtil esUtil;

    @Resource
    private DataImportService dataImportService;

    //语种字典吗
    @Value("${dict.language:}")
    private String LANGUAGE;
    //营利性质字典吗
    @Value("${dict.think_tank_profit_nature:}")
    private String THINK_TANK_PROFIT_NATURE;
    //智库性质字典吗
    @Value("${dict.think_tank_nature:}")
    private String THINK_TANK_NATURE;
    //研究领域标签id
    @Value("${sys_base_label_type}")
    private String SYS_BASE_LABEL_TYPE;


    @Value("${files.storage}")
    String filesStorage;
    @Resource
    private SimRedisUtil simRedisUtil;
    private final static String KEY = "thinkTank:getList";
    private static final String IMPORT_TEMP = "think-tank-import-temp.xlsx";
    private static final String IMPORT_TEMP_NAME = "智库信息导入模板.xlsx";


    @Override
    public Result<?> add(ThinktankBasicInfo thinktankBasicInfo) {

        String id = Long.toString(generatorIdService.getOrderId());
        thinktankBasicInfo.setId(id);
        String codeId = generatorIdService.getIdNo();
        thinktankBasicInfo.setCodeId(codeId);

        //标签
        String tagId = thinktankBasicInfo.getTagId();
        if (tagId != null && tagId.length() > 0) {
            String[] ids = tagId.split(",");
            List<LabelEntity> list = thinktankTagService.getListByIds(ids);
            List<String> nameList = new ArrayList<>();
            for (LabelEntity labelEntity : list) {
                String name = labelEntity.getName();
                nameList.add(name);
            }
            String tagName = nameList.stream().collect(Collectors.joining(","));
            thinktankBasicInfo.setTagName(tagName);
        }

        //类别
        String belongCountryId = thinktankBasicInfo.getBelongCountryId();
        ThinktankCategoryStructure tcs = leaderCategoryService.getById(belongCountryId);
        if (tcs != null) {
            String typeName = tcs.getTypeName();
            thinktankBasicInfo.setBelongCountry(typeName);
        }

        thinktankBasicInfo.setCreateTime(new Date());
        thinktankBasicInfo.setUpdateTime(new Date());

        boolean b = this.save(thinktankBasicInfo);
        Result result = Result.OK();
        if (!b) {
            result.error500("保存失败！");
        }
        //同步新增智库数据到标签管理
        thinktankTagService.addLabel(thinktankBasicInfo);
        return result;
    }

    @Override
    public Result<?> del(String id) {
        String[] idArr = id.split(",");
        List<String> ids = Arrays.asList(idArr);
        boolean b = this.removeByIds(ids);
        Result result = Result.OK();
        if (!b) {
            result.error500("删除失败！");
        }
        thinktankTagService.delLabel(ids);
        return result;
    }

    @Override
    public Result<?> edit(ThinktankBasicInfo thinktankBasicInfo) {

        //标签
        String tagId = thinktankBasicInfo.getTagId();
        if (tagId != null && tagId.length() > 0) {
            String[] ids = tagId.split(",");
            List<LabelEntity> list = thinktankTagService.getListByIds(ids);
            List<String> nameList = new ArrayList<>();
            for (LabelEntity labelEntity : list) {
                String name = labelEntity.getName();
                nameList.add(name);
            }
            String tagName = nameList.stream().collect(Collectors.joining(","));
            thinktankBasicInfo.setTagName(tagName);
        }

        //类别
        String belongCountryId = thinktankBasicInfo.getBelongCountryId();
        ThinktankCategoryStructure tcs = leaderCategoryService.getById(belongCountryId);
        String typeName = tcs.getTypeName();
        thinktankBasicInfo.setBelongCountry(typeName);

        thinktankBasicInfo.setUpdateTime(new Date());

        boolean b = this.updateById(thinktankBasicInfo);
        Result result = Result.OK();
        if (!b) {
            result.error500("修改失败！");
        }
        thinktankTagService.editLabel(thinktankBasicInfo);
        return result;
    }

    @Override
    public ThinktankBasicInfoVo getInfoById(String id) {
        ThinktankBasicInfo info = this.getById(id);

        if (info == null) {
            return null;
        }

        ThinktankBasicInfoVo infoVo = new ThinktankBasicInfoVo();
        BeanUtils.copyProperties(info, infoVo);
        List<ThinktankBasicInfoVo> list = new ArrayList<>();
        list.add(infoVo);

        List<String> groupIds = new ArrayList<>();
        for (ThinktankBasicInfoVo thinktankBasicInfoVo : list) {
            groupIds.add(thinktankBasicInfoVo.getId());
        }
        List<InfoSourceGroupMap> infoSourceGroupMapList = infoSourceGroupMapService.getSourceGroupMap(groupIds);

        List<InfoSourceGroupCountVo> countVoList = new ArrayList<>();
        List<String> sidList = new ArrayList<>();
        for (InfoSourceGroupMap infoSourceGroupMap : infoSourceGroupMapList) {
            InfoSourceGroupCountVo infoSourceGroupCountVo = new InfoSourceGroupCountVo();
            BeanUtils.copyProperties(infoSourceGroupMap, infoSourceGroupCountVo);
            countVoList.add(infoSourceGroupCountVo);
            sidList.add(infoSourceGroupMap.getSourceId());
        }

        //查询es
        BoolQueryBuilder boolQuerylist = QueryBuilders.boolQuery();
        boolQuerylist.must(QueryBuilders.termsQuery("sid", sidList));
        Map<String, Long> mapGroup = null;
        try {
            mapGroup = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
        } catch (IOException e) {
            log.error("ES查询失败");
            e.printStackTrace();
        }
        if (mapGroup != null && mapGroup.size() > 0) {
            for (InfoSourceGroupCountVo infoSourceGroupCountVo : countVoList) {
                String sourceId = infoSourceGroupCountVo.getSourceId();
                Long countInfo = mapGroup.get(sourceId);
                if (countInfo == null) {
                    countInfo = 0L;
                }
                infoSourceGroupCountVo.setInfoCount(Math.toIntExact(countInfo));
            }
        }

        //统计关联信息源数量
        Map<String, List<InfoSourceGroupCountVo>> studentsByGrade = countVoList.stream()
                .collect(Collectors.groupingBy(InfoSourceGroupCountVo::getGroupId));

        for (ThinktankBasicInfoVo thinktankBasicInfoVo : list) {
//            String id = thinktankBasicInfoVo.getId();
            List<InfoSourceGroupCountVo> listVo = studentsByGrade.get(thinktankBasicInfoVo.getId());
            if (listVo == null) {
                thinktankBasicInfoVo.setCollectionCount(0);
                thinktankBasicInfoVo.setInfoSourceCount(0);
            } else {
                thinktankBasicInfoVo.setInfoSourceCount(listVo.size());
                int count = 0;
                for (InfoSourceGroupCountVo infoSourceGroupCountVo : listVo) {
                    count += infoSourceGroupCountVo.getInfoCount();
                }
                thinktankBasicInfoVo.setCollectionCount(count);
            }
        }

        return list.get(0);
    }

    /**
     * 获取智库详情
     * @param id 智库id
     * @param userId 用户id
     * @return
     */
    @Override
    public ThinktankInfoVo getThinktankBasicInfoById(String id, String userId) {
        ThinktankInfoVo thinktankInfoVo = new ThinktankInfoVo();
        // 获取基础信息
        ThinktankBasicInfo baseInfo = this.getById(id);
        setDictValue(baseInfo);
        thinktankInfoVo.setBasicInfo(baseInfo);
        // 获取OTT信息
        ThinktankOttInfo ottInfo = thinktankOttInfoMapper.getOttInfoByBaseCode(baseInfo.getCodeId());
        thinktankInfoVo.setOttInfo(ottInfo);
        // 获取收藏状态
        int collectInfo = thinktankOttInfoMapper.getCollectInfo(baseInfo.getCodeId(), userId);
        if (collectInfo > 0) {
            thinktankInfoVo.setCollect("1");
        } else {
            thinktankInfoVo.setCollect("0");
        }
        return thinktankInfoVo;
    }


    /**
     * 关联字典值
     */
    public void setDictValue(ThinktankBasicInfo baseInfo) {
        if(baseInfo!=null && StringUtils.isNotBlank(baseInfo.getNature())){
            QueryWrapper query = new QueryWrapper();
            query.eq("item_value",baseInfo.getNature());
            List<SysDictItem> sysDictItemList = sysDictItemService.list(query);
            if(sysDictItemList!=null && sysDictItemList.size()>0){
                baseInfo.setNatureName(sysDictItemList.get(0).getItemText());
            }
        }
        if(baseInfo!=null && StringUtils.isNotBlank(baseInfo.getLang())){
            String langName = "";
            for(String str:Arrays.asList(baseInfo.getLang().split(","))){
                QueryWrapper query = new QueryWrapper();
                query.in("item_value",str);
                List<SysDictItem> sysDictItemList = sysDictItemService.list(query);
                if(sysDictItemList!=null && sysDictItemList.size()>0){
                    if(langName.equals("")){
                        langName = sysDictItemList.get(0).getItemText();
                    }else{
                        langName = langName  + "," + sysDictItemList.get(0).getItemText();
                    }
                }
            }
            if(!langName.equals("")){
                baseInfo.setLangName(langName);
            }
        }
        if(baseInfo!=null && StringUtils.isNotBlank(baseInfo.getInfluencePosition())){
            String influencePosition = "";
            for(String str:Arrays.asList(baseInfo.getInfluencePosition().split(","))){
                if(StringUtils.isNotBlank(str) && str.equals(CommonConstants.INFLUENCE_3)){
                    continue;//特殊需求,过滤颠覆性重点智库
                }
                QueryWrapper query = new QueryWrapper();
                query.in("item_value",str);
                List<SysDictItem> sysDictItemList = sysDictItemService.list(query);
                if(sysDictItemList!=null && sysDictItemList.size()>0){
                    if(influencePosition.equals("")){
                        influencePosition = sysDictItemList.get(0).getItemText();
                    }else{
                        influencePosition = influencePosition  + "," + sysDictItemList.get(0).getItemText();
                    }
                }
            }
            if(!influencePosition.equals("")){
                baseInfo.setInfluencePosition(influencePosition);
            }
        }
    }


    /**
     * 收藏/取消收藏智库
     * @param id 智库id
     * @param userId 用户id
     */
    @Override
    public void collect(String id, String userId) {
        // 获取智库基础信息
        ThinktankBasicInfo baseInfo = this.getById(id);
        // 获取收藏状态
        int collectInfo = thinktankOttInfoMapper.getCollectInfo(baseInfo.getCodeId(), userId);
        if (collectInfo > 0) {
            thinktankOttInfoMapper.cancelCollect(baseInfo.getCodeId(),userId);
        } else {
            thinktankOttInfoMapper.collect(baseInfo.getCodeId(),userId,new Date());
        }
    }


    /**
     * 智库关联ott
     * @param baseCode 智库编码
     * @param ottCode ott表主键
     */
    @Override
    public void relation(String baseCode, String ottCode) {
        String relation = thinktankOttInfoMapper.getRelation(baseCode);
        if (relation != null || !"".equals(relation)) {
            // 已关联
            thinktankOttInfoMapper.updateRelation(baseCode,ottCode);
        } else {
            // 未关联
            thinktankOttInfoMapper.addRelation(baseCode,ottCode);
        }
    }

    @Override
    public List<ThinktankBasicInfo> collectList(String userId) {
        List<ThinktankBasicInfo> collectList = thinktankOttInfoMapper.getCollectList(userId);
        return collectList;
    }

    @Override
    public Result<?> getList(ThinktankBasicInfoListVo thinktankBasicInfoListVo) {
        // 缓存的type类型
        String cacheTypeStr = thinktankBasicInfoListVo.getTypeId();
        if (StringUtils.equals(thinktankBasicInfoListVo.getQueryType(), "all")) {
            cacheTypeStr = "all";
            thinktankBasicInfoListVo.setTypeId(null);
            // 排除中国的
            thinktankBasicInfoListVo.setExcludeTypeIds(Lists.newArrayList(CommonConstants.CHINA_ID));
        } else if (StringUtils.equals(thinktankBasicInfoListVo.getQueryType(), "other")) {
            // 其它-排除热门的
            cacheTypeStr = "other" + thinktankBasicInfoListVo.getExcludeTypeIds().toString();
            thinktankBasicInfoListVo.setTypeId(null);
            thinktankBasicInfoListVo.getExcludeTypeIds().add(CommonConstants.CHINA_ID);
        } else {
            thinktankBasicInfoListVo.setExcludeTypeIds(null);
        }

        if(StringUtils.isNotBlank(cacheTypeStr)
                && StringUtils.isBlank(thinktankBasicInfoListVo.getTagId())){
            IPage cacheObject = (IPage)simRedisUtil.getCacheObject(KEY + "_" + cacheTypeStr
                    + "_" + thinktankBasicInfoListVo.getColumn() + "_" + thinktankBasicInfoListVo.getOrder());
            if(cacheObject!=null){
                return Result.OK(cacheObject);
            }
        }

        IPage page = new Page();

        thinktankBasicInfoListVo.setColumn(StringUtil.humpToLine(thinktankBasicInfoListVo.getColumn()));
        //查询列表
        List<ThinktankBasicInfoVo> list = thinktankBasicInfoMapper.getList(thinktankBasicInfoListVo);
        page.setRecords(list);
        //查询总数
        Integer total = thinktankBasicInfoMapper.getCount(thinktankBasicInfoListVo);
        page.setTotal(total);
        Integer pageNo = thinktankBasicInfoListVo.getPageNo();
        Integer pageSize = thinktankBasicInfoListVo.getPageSize();
        page.setCurrent(pageNo);
        page.setSize(pageSize);

        if (list.size() == 0) {
            return Result.OK(page);
        }

        //信息源
        List<String> groupIds = new ArrayList<>();
        for (ThinktankBasicInfoVo thinktankBasicInfoVo : list) {
            groupIds.add(thinktankBasicInfoVo.getId());
        }
        List<InfoSourceGroupMap> infoSourceGroupMapList = infoSourceGroupMapService.getSourceGroupMap(groupIds);

        List<InfoSourceGroupCountVo> countVoList = new ArrayList<>();
        List<String> sidList = new ArrayList<>();
        for (InfoSourceGroupMap infoSourceGroupMap : infoSourceGroupMapList) {
            InfoSourceGroupCountVo infoSourceGroupCountVo = new InfoSourceGroupCountVo();
            BeanUtils.copyProperties(infoSourceGroupMap, infoSourceGroupCountVo);
            countVoList.add(infoSourceGroupCountVo);
            sidList.add(infoSourceGroupMap.getSourceId());
        }

        //查询es
        BoolQueryBuilder boolQuerylist = QueryBuilders.boolQuery();
        boolQuerylist.must(QueryBuilders.termsQuery("sid", sidList));
        Map<String, Long> mapGroup = null;
        try {
            mapGroup = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
        } catch (IOException e) {
            log.error("ES查询失败");
            e.printStackTrace();
        }
        if (mapGroup != null && mapGroup.size() > 0) {
            for (InfoSourceGroupCountVo infoSourceGroupCountVo : countVoList) {
                String sourceId = infoSourceGroupCountVo.getSourceId();
                Long countInfo = mapGroup.get(sourceId);
                if (countInfo == null) {
                    countInfo = 0L;
                }
                infoSourceGroupCountVo.setInfoCount(Math.toIntExact(countInfo));
            }
        }

        //统计关联信息源数量
        Map<String, List<InfoSourceGroupCountVo>> studentsByGrade = countVoList.stream()
                .collect(Collectors.groupingBy(InfoSourceGroupCountVo::getGroupId));

        for (ThinktankBasicInfoVo thinktankBasicInfoVo : list) {
            String id = thinktankBasicInfoVo.getId();
            List<InfoSourceGroupCountVo> listVo = studentsByGrade.get(id);
            if (listVo == null) {
                thinktankBasicInfoVo.setCollectionCount(0);
                thinktankBasicInfoVo.setInfoSourceCount(0);
            } else {
                thinktankBasicInfoVo.setInfoSourceCount(listVo.size());
                int count = 0;
                for (InfoSourceGroupCountVo infoSourceGroupCountVo : listVo) {
                    Integer infoCount = infoSourceGroupCountVo.getInfoCount();
                    if (infoCount != null) {
                        count += infoCount;
                    }
                }
                thinktankBasicInfoVo.setCollectionCount(count);
            }
        }

        //记录热门标签
        String tagId = thinktankBasicInfoListVo.getTagId();
        if (tagId != null && tagId.length() > 0) {
            String[] tagIdArr = tagId.split(",");
            for (String s : tagIdArr) {
                thinktankTagService.addTagTop(s);
            }
        }
        if(StringUtils.isNotBlank(cacheTypeStr) && StringUtils.isBlank(thinktankBasicInfoListVo.getTagId())){
            if (Arrays.asList("all", "other").contains(cacheTypeStr) || cacheTypeStr.contains("other")) {
                // all 和other数据量过大，只存id
                List<Map<String, Object>> ids = (List<Map<String, Object>>) page.getRecords().stream()
                        .map(v -> {
                            Map<String, Object> map = Maps.newHashMap();
                            map.put("id", ((ThinktankBasicInfoVo) v).getId());
                            return map;
                        }).collect(Collectors.toList());
                page.setRecords(ids);
            }
            simRedisUtil.setCacheObject(KEY + "_" + cacheTypeStr
                    + "_" + thinktankBasicInfoListVo.getColumn() + "_" + thinktankBasicInfoListVo.getOrder(), page,10L, TimeUnit.DAYS);
        }
        return Result.OK(page);
    }

    @Override
    public void downloadTemplate(HttpServletResponse response) {
        String filePath = filesStorage + IMPORT_TEMP;

        File file = new File(filePath);
        try {
            // 将文件写入输入流
            FileInputStream fileInputStream = new FileInputStream(file);
            InputStream fis = new BufferedInputStream(fileInputStream);
            byte[] buffer = new byte[fis.available()];
            fis.read(buffer);
            fis.close();
            // 清空response
            response.reset();
            // 设置response的Header
            response.setCharacterEncoding("UTF-8");
            //Content-Disposition的作用：告知浏览器以何种方式显示响应返回的文件，用浏览器打开还是以附件的形式下载到本地保存
            //attachment表示以附件方式下载   inline表示在线打开   "Content-Disposition: inline; filename=文件名.mp3"
            // filename表示文件的默认名称，因为网络传输只支持URL编码的相关支付，因此需要将文件名URL编码后进行传输,前端收到后需要反编码才能获取到真正的名称
            response.addHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(IMPORT_TEMP_NAME, "UTF-8"));
            // 告知浏览器文件的大小
            response.addHeader("Content-Length", "" + file.length());
            OutputStream outputStream = new BufferedOutputStream(response.getOutputStream());
            response.setContentType("application/octet-stream");
            outputStream.write(buffer);
            outputStream.flush();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public Result<?> getListBaseData(String id, Integer pageNo, Integer pageSize, Integer office) {
        List<InfoSourceGroupMap> infoSourceGroupMapList = infoSourceGroupMapService.getSourceGroupMap(Arrays.asList(id));

        List<String> sidList = new ArrayList<>();
        for (InfoSourceGroupMap infoSourceGroupMap : infoSourceGroupMapList) {
            sidList.add(infoSourceGroupMap.getSourceId());
        }

        //设置分页参数
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
//        searchSourceBuilder.size(pageSize);
//        searchSourceBuilder.from(office);
        searchSourceBuilder.trackTotalHits(true);
        //排序
        searchSourceBuilder.sort("createDate", SortOrder.ASC);

        //查询es
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("sid", sidList));

        //查询数据
        searchSourceBuilder.query(boolQuery);
        Page<TransferData> page = null;

        try {
            page = esUtil.queryPage("basedata", searchSourceBuilder, TransferData.class, pageNo, pageSize);
        } catch (IOException e) {
            e.printStackTrace();
        }
        for (TransferData transferData : page.getRecords()) {
            transferData.setCreateDate(EsDateUtil.esFieldDateMapping(transferData.getCreateDate()));
            transferData.setPublishDate(EsDateUtil.esFieldDateMapping(transferData.getPublishDate()));
        }
        return Result.OK(page);
    }

    @Override
    @Transactional
    public String batchImport(List<ThinkTankFileVO> thinkTankImportVOList) {
        //国家列表
        List<ThinktankCategoryStructure> listCategoryStructure = leaderCategoryService.list();
        Map<String, String> mapCategoryStructure = listCategoryStructure.stream().collect(Collectors.toMap(ThinktankCategoryStructure::getTypeName, ThinktankCategoryStructure::getId, (value1, value2) -> value1));
        //语种
        List<DataChangeVO> languageList = dataImportService.dictItemList(LANGUAGE);
        Map<String, String> languageMap = languageList.stream().collect(Collectors.toMap(DataChangeVO::getDescription, DataChangeVO::getValue, (value1, value2) -> value1));
        //营利性质
        List<DataChangeVO> profitNatureList = dataImportService.dictItemList(THINK_TANK_PROFIT_NATURE);
        Map<String, String> profitNatureMap = profitNatureList.stream().collect(Collectors.toMap(DataChangeVO::getDescription, DataChangeVO::getValue, (value1, value2) -> value1));
        //智库性质
        List<DataChangeVO> natureList = dataImportService.dictItemList(THINK_TANK_NATURE);
        Map<String, String> natureMap = natureList.stream().collect(Collectors.toMap(DataChangeVO::getDescription, DataChangeVO::getValue, (value1, value2) -> value1));
        //研究领域
        List<DataChangeVO> researchFieldList = dataImportService.labelEntityList(SYS_BASE_LABEL_TYPE);
        Map<String, String> researchFieldMap = researchFieldList.stream().collect(Collectors.toMap(DataChangeVO::getDescription, DataChangeVO::getValue, (value1, value2) -> value1));
        StringBuilder stringBuilder = new StringBuilder();
        for (ThinkTankFileVO thinkTankImportVO : thinkTankImportVOList) {
            ThinktankBasicInfo thinktankBasicInfo = new ThinktankBasicInfo();
            BeanUtils.copyProperties(thinkTankImportVO, thinktankBasicInfo);
            String codeId = thinkTankImportVO.getCodeId();
            if (StringUtils.isEmpty(codeId)) {
                codeId = generatorIdService.getIdNo();
            }
            thinktankBasicInfo.setCodeId(codeId);
            String belongCountry = thinkTankImportVO.getBelongCountry();
            String belongCountryId = mapCategoryStructure.get(belongCountry);
            if (StringUtils.isEmpty(belongCountryId)) {
                stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的国家【").append(belongCountry).append("】,数据库中没有，导入失败;\n");
                continue;
            } else {
                thinktankBasicInfo.setBelongCountryId(belongCountryId);
            }
            String lang = thinkTankImportVO.getLang();
            if (StringUtils.isNotEmpty(lang)) {
                StringBuilder noExist = new StringBuilder();
                StringBuilder langCode = new StringBuilder();
                for (String value : lang.split(",")) {
                    String langValue = languageMap.get(value);
                    if (StringUtils.isEmpty(langValue)) {
                        noExist.append(",").append(value);
                    } else {
                        langCode.append(",").append(langValue);
                    }
                }
                if (StringUtils.isNotEmpty(noExist)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的语言【").append(noExist.substring(1)).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    thinktankBasicInfo.setLang(langCode.substring(1));
                }
            }
            String profitNature = thinkTankImportVO.getProfitNature();
            if (StringUtils.isNotEmpty(profitNature)) {
                String profitNatureValue = profitNatureMap.get(profitNature);
                if (StringUtils.isEmpty(profitNatureValue)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的营利性质【").append(profitNature).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    thinktankBasicInfo.setProfitNature(profitNatureValue);
                }
            }
            String nature = thinkTankImportVO.getNature();
            if (StringUtils.isNotEmpty(nature)) {
                String natureValue = natureMap.get(nature);
                if (StringUtils.isEmpty(natureValue)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的智库性质【").append(nature).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    thinktankBasicInfo.setNature(natureValue);
                }
            }
            String tagName = thinkTankImportVO.getTagName();
            if (StringUtils.isNotEmpty(tagName)) {
                StringBuilder noExist = new StringBuilder();
                StringBuilder tagId = new StringBuilder();
                for (String name : tagName.split(",")) {
                    String value = researchFieldMap.get(name);
                    if (StringUtils.isEmpty(value)) {
                        noExist.append(",").append(name);
                    } else {
                        tagId.append(",").append(value);
                    }
                }
                if (StringUtils.isNotEmpty(noExist)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的研究领域【").append(noExist.substring(1)).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    thinktankBasicInfo.setTagId(tagId.substring(1));
                }
            }
            String belongUnit = thinkTankImportVO.getBelongUnit();
            if (StringUtils.isNotEmpty(belongUnit)) {
                List<SysBaseEnterprise> listEnter = enterpriseService.getListByName(belongUnit);
                if (CollectionUtils.isEmpty(listEnter)) {
                    stringBuilder.append("智库【").append(thinktankBasicInfo.getChineseWhole()).append("】的创办单位/所属单位【").append(belongUnit).append("】,数据库中没有，导入失败;\n");
                    continue;
                } else {
                    SysBaseEnterprise enter = listEnter.get(0);
                    thinktankBasicInfo.setBelongUnitCode(enter.getSocialCreditCode());
                }
            }
            LambdaQueryWrapper<ThinktankBasicInfo> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(ThinktankBasicInfo::getCodeId, codeId);
            ThinktankBasicInfo one = thinktankBasicInfoMapper.selectOne(queryWrapper);
            if (one != null) {
                thinktankBasicInfo.setId(one.getId());
                thinktankBasicInfoMapper.updateById(thinktankBasicInfo);
                //编辑标签数据
                thinktankTagService.editLabel(thinktankBasicInfo);
            } else {
                thinktankBasicInfo.setId(Long.toString(generatorIdService.getOrderId()));
                thinktankBasicInfoMapper.insert(thinktankBasicInfo);
                //增加标签数据
                thinktankTagService.addLabel(thinktankBasicInfo);
            }
        }
        return stringBuilder.length() > 0 ? stringBuilder.toString() : null;
    }
    @Override
    public IPage<ThinktankBasicInfo> pageList(Integer pageNo,Integer pageSize,ThinktankBasicInfoListVo vo, HttpServletRequest req) throws Exception {
        QueryWrapper<ThinktankBasicInfo> queryWrapper = new QueryWrapper<>();
        if(vo!=null){
            List<String> fieldTypeId = vo.getFieldTypeId();//领域标签id
            List<String> regionalCountriesTypeId = vo.getRegionalCountriesTypeId();//区域国家标签id
            List<String> langList = vo.getLangList();//语言
            List<String> natureList = vo.getNatureList();//智库性质
            List<String> influencePositionList = vo.getInfluencePositionList();//影响力和地位
            List<String> scaleList = vo.getScaleList();//智库规模
            List<String> staffSizeList = vo.getStaffSizeList();//人员规模
            List<String> operateStatusList = vo.getOperateStatusList();//运营状态
            List<String> regionalCountriesTypeIds = new ArrayList<>();
            queryWrapper.eq("status",1);
            queryWrapper.eq("yn_display",1);//yjzx-是否展示到列表（0否 1是）
            if(StringUtils.isNotBlank(vo.getKeyword())){
                queryWrapper.and(wreapper->{
                    wreapper.like("chinese_whole", vo.getKeyword());
                    wreapper.or();
                    wreapper.like("english_whole", vo.getKeyword());
                    wreapper.or();
                    wreapper.like("tag_name", vo.getKeyword());
                    wreapper.or();
                    wreapper.like("belong_country", vo.getKeyword());
                    wreapper.or();
                    wreapper.like("upper(english_simple)", vo.getKeyword().toUpperCase());
                });
            }
            if(fieldTypeId!=null && fieldTypeId.size()>0 && StringUtils.isNotBlank(fieldTypeId.get(0))){
                queryWrapper.and(wreapper->{
                    for(int i=0;i<fieldTypeId.size();i++){
                        wreapper.like("tag_id",fieldTypeId.get(i));
                        if(i != fieldTypeId.size()-1){
                            wreapper.or();
                        }
                    }
                });
            }
            if(regionalCountriesTypeId!=null && regionalCountriesTypeId.size()>0 && StringUtils.isNotBlank(regionalCountriesTypeId.get(0))){
                for(String id:regionalCountriesTypeId){
                    SysBaseRegion region = sysBaseRegionService.getById(id);
                    if(region!=null && region.getLevelId()!=null && region.getLevelId().equals("country")){
                        regionalCountriesTypeIds.add(id);
                    }else if(region!=null && region.getLevelId()!=null && region.getLevelId().equals("continent")){
                        QueryWrapper<SysBaseRegion> query = new QueryWrapper<>();
                        query.eq("pid",id);
                        List<SysBaseRegion> list = sysBaseRegionService.list(query);
                        for(SysBaseRegion obj:list){
                            regionalCountriesTypeIds.add(obj.getId());
                        }
                    }
                }
            }else{
                //查询国外智库
                queryWrapper.ne("belong_country_id","1405761925932109826");//查询非中国智库
            }
            if(regionalCountriesTypeIds!=null && regionalCountriesTypeIds.size()>0){
                queryWrapper.and(wreapper->{
                    for(int i=0;i<regionalCountriesTypeIds.size();i++){
                        wreapper.eq("belong_country_id",regionalCountriesTypeIds.get(i));
                        if(i != regionalCountriesTypeIds.size()-1){
                            wreapper.or();
                        }
                    }
                });
            }
            if(langList!=null && langList.size()>0 && StringUtils.isNotBlank(langList.get(0))){
                queryWrapper.and(wreapper->{
                    for(int i=0;i<langList.size();i++){
                        wreapper.like("lang",langList.get(i));
                        if(i != langList.size()-1){
                            wreapper.or();
                        }
                    }
                });
            }
            if(natureList!=null && natureList.size()>0 && StringUtils.isNotBlank(natureList.get(0))){
                queryWrapper.and(wreapper->{
                    for(int i=0;i<natureList.size();i++){
                        wreapper.like("nature",natureList.get(i));
                        if(i != natureList.size()-1){
                            wreapper.or();
                        }
                    }
                });
            }
            if(influencePositionList!=null && influencePositionList.size()>0 && StringUtils.isNotBlank(influencePositionList.get(0))){
                queryWrapper.and(wreapper->{
                    for(int i=0;i<influencePositionList.size();i++){
                        wreapper.like("influence_position",influencePositionList.get(i));
                        if(i != influencePositionList.size()-1){
                            wreapper.or();
                        }
                    }
                });
            }
            if(scaleList!=null && scaleList.size()>0 && StringUtils.isNotBlank(scaleList.get(0))){
                queryWrapper.and(wreapper->{
                    for(int i=0;i<scaleList.size();i++){
                        wreapper.like("scale",scaleList.get(i));
                        if(i != scaleList.size()-1){
                            wreapper.or();
                        }
                    }
                });
            }
            if(staffSizeList!=null && staffSizeList.size()>0 && StringUtils.isNotBlank(staffSizeList.get(0))){
                queryWrapper.and(wreapper->{
                    for(int i=0;i<staffSizeList.size();i++){
                        wreapper.like("staff_size",staffSizeList.get(i));
                        if(i != staffSizeList.size()-1){
                            wreapper.or();
                        }
                    }
                });
            }
            if(operateStatusList!=null && operateStatusList.size()>0 && StringUtils.isNotBlank(operateStatusList.get(0))){
                queryWrapper.and(wreapper->{
                    for(int i=0;i<operateStatusList.size();i++){
                        wreapper.like("operate_status",operateStatusList.get(i));
                        if(i != operateStatusList.size()-1){
                            wreapper.or();
                        }
                    }
                });
            }
            if (!CollectionUtils.isEmpty(vo.getFirstWordList())) {
                queryWrapper.and(wrapper->{
                    for (int i = 0; i < vo.getFirstWordList().size(); i++) {
                        String firstWord = vo.getFirstWordList().get(i);
                        wrapper.likeRight("english_whole", firstWord);
                        if(i != vo.getFirstWordList().size() - 1){
                            wrapper.or();
                        }
                    }
                });
            }
        }

        queryWrapper.orderByAsc("SUBSTRING(english_whole, 1, 1)");
        queryWrapper.orderByDesc("update_time");
        Integer total = this.getBaseMapper().selectCount(queryWrapper);

        Integer office = pageSize * (pageNo - 1);
        queryWrapper.last("limit  "+office+", "+pageSize);
        List<ThinktankBasicInfo> list = this.getBaseMapper().selectList(queryWrapper);

        IPage<ThinktankBasicInfo> page = new Page<>();
        page.setRecords(list);
        page.setTotal(total);
        page.setSize(list.size());
        page.setCurrent(pageNo);
        return page;
    }

    @Override
    public List<Map<String,Object>> queryStatisticalAnalysis(String type,Page page, ThinktankBasicInfoListVo vo, HttpServletRequest req){
        List<Map<String,Object>> mapList = new ArrayList<>();
        QueryWrapper<ThinktankBasicInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.select("id","code_id","chinese_whole","belong_country_id","belong_country","tag_id","tag_name",
                "lang","profit_nature","nature","influence_position","scale","achievement_type","funds_source","operate_status","staff_size",
                "(SELECT name_cn from brpa_base_region where id = (SELECT pid from brpa_base_region where id=belong_country_id) ) as continentName",
                "(SELECT pid from brpa_base_region where id=belong_country_id) as continentId");
        List<ThinktankBasicInfo> list = this.getBaseMapper().selectPage(page,queryWrapper).getRecords();
        switch (type) {
            case "continent":
                mapList = processGrouping(list, ThinktankBasicInfo::getContinentName,false,null);//洲
                break;
            case "country":
                mapList = processGrouping(list, ThinktankBasicInfo::getBelongCountry,false,null);//国家
                break;
            case "tag":
                mapList = processGrouping(list, ThinktankBasicInfo::getTagName, true,",");//领域标签
                break;
            case "lang":
                mapList = processGroupingByLang(list, ThinktankBasicInfo::getLang, true,",");//语言
                // 取排名前20
                mapList.sort((m1, m2) -> MapUtil.getInt(m2, "num") - MapUtil.getInt(m1, "num"));
                mapList = mapList.stream().limit(20).collect(Collectors.toList());
                break;
            case "nature":
                mapList = processGrouping(list, ThinktankBasicInfo::getNature, false,null);//性质
                List<SysDictItem> dictItemList = sysDictItemService.selectByItemCode(CommonConstants.DIC_THINK_TANK_NATURE);
                // 方式1：保留第一个出现的重复值（推荐，除非明确需要覆盖）
                Map<String, String> dictMap = dictItemList.stream()
                        .collect(Collectors.toMap(
                                SysDictItem::getItemValue,
                                SysDictItem::getItemText,
                                (existingValue, newValue) -> existingValue  // 遇到重复键时保留旧值
                        ));
                mapList.forEach(map -> {
                    String dictKey = MapUtil.getStr(map, "data");
                    if (MapUtil.isNotEmpty(dictMap) && dictMap.containsKey(dictKey)) {
                        map.put("data", dictMap.get(dictKey));
                    }
                });
                break;
            case "influencePosition":
                List<Map<String,Object>> mapListFilter = processGrouping(list, ThinktankBasicInfo::getInfluencePosition, true,",");//影响力和地位
                List<SysDictItem> dictItemListPos = sysDictItemService.selectByItemCode(CommonConstants.DIC_INFLUENCE);
                Map<String, String> dictMapPos = dictItemListPos.stream().collect(Collectors.toMap(SysDictItem::getItemValue, SysDictItem::getItemText));
                List<Map<String, Object>> finalMapList = mapList;
                mapListFilter.forEach(map -> {
                    String dictKey = MapUtil.getStr(map, "data");
                    if (!dictKey.equals(CommonConstants.INFLUENCE_3) && MapUtil.isNotEmpty(dictMapPos) && dictMapPos.containsKey(dictKey)) {
                        map.put("data", dictMapPos.get(dictKey));
                        finalMapList.add(map);
                    }
                });
                break;
            case "scale":
                mapList = processGrouping(list, ThinktankBasicInfo::getScale, false,null);//智库规模
                break;
            case "staffSize":
                mapList = processGrouping(list, ThinktankBasicInfo::getStaffSize, false,null);//人员规模
                break;
            case "operateStatus":
                mapList = processGrouping(list, ThinktankBasicInfo::getOperateStatus, false,null);//运营状态
                break;
        }
        return mapList;
    }
    // 提取的分组和计数方法，添加了两个参数来处理是否需要拆分标签、以及用什么符号拆分
    private List<Map<String, Object>> processGrouping(List<ThinktankBasicInfo> list, Function<ThinktankBasicInfo, String> classifier, boolean splitTags,String split) {
        Stream<String> stream = list.stream().map(classifier)
                .filter(item -> item != null && !item.isEmpty()); // 过滤掉null和空字符串
        if (splitTags) {
            stream = stream.flatMap(info -> Arrays.stream(info.split(split)))
                    .map(String::trim)
                    .filter(Objects::nonNull); // 过滤掉null元素;
        }
        return stream.collect(Collectors.groupingBy(
                        Function.identity(),
                        Collectors.counting()
                )).entrySet().stream()
                .map(entry -> createMap(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());
    }


    //语言聚合单独处理
    private List<Map<String, Object>> processGroupingByLang(List<ThinktankBasicInfo> list, Function<ThinktankBasicInfo, String> classifier, boolean splitTags,String split) {
        List<Map<String, Object>> listMapNew = new ArrayList<>();
        Stream<String> stream = list.stream().map(classifier)
                .filter(item -> item != null && !item.isEmpty()); // 过滤掉null和空字符串
        if (splitTags) {
            stream = stream.flatMap(info -> Arrays.stream(info.split(split)))
                    .map(String::trim)
                    .filter(Objects::nonNull); // 过滤掉null元素;
        }
        List<Map<String, Object>> listMap = stream.collect(Collectors.groupingBy(
                        Function.identity(),
                        Collectors.counting()
                )).entrySet().stream()
                .map(entry -> createMap(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());

        QueryWrapper query = new QueryWrapper();
        query.eq("dict_code","language");
        SysDict sysDict = sysDictService.getOne(query);
        if(sysDict!=null){
            QueryWrapper queryItem = new QueryWrapper();
            queryItem.eq("dict_id",sysDict.getId());
            List<SysDictItem> sysDictItemList = sysDictItemService.list(queryItem);

            // 创建映射关系
            Map<String, String> dataToNameMap = new HashMap<>();
            for (SysDictItem item : sysDictItemList) {
                dataToNameMap.put(item.getItemValue(), item.getItemText());
            }

            // 替换list1中的data值为name值
            for (Map<String, Object> map : listMap) {
                String dataValue = (String) map.get("data");
                String nameValue = getIgnoreCase(dataToNameMap, dataValue);//忽略大小写
                if (nameValue != null) {
                    map.put("data", nameValue);
                    listMapNew.add(map);
                }
            }
        }
        return listMapNew;
    }
    private static String getIgnoreCase(Map<String, String> map, String key) {
        for (String mapKey : map.keySet()) {
            if (mapKey.equalsIgnoreCase(key)) {
                return map.get(mapKey);
            }
        }
        return null;
    }
    // 创建Map的方法
    private Map<String, Object> createMap(String key, Long value) {
        Map<String, Object> map = new HashMap<>();
        map.put("data", key);
        map.put("num", value);
        return map;
    }
    /**
     * 智库维护
     *
     * @param file
     */
    @Override
    public void thinkTank(MultipartFile file) {
        try {
            InputStream inputStream = file.getInputStream();
            ExcelReader reader = ExcelUtil.getReader(inputStream);
            Sheet sheet = reader.getSheet();
            for (int i = 1; i <= sheet.getLastRowNum(); i++) {
                Row row = sheet.getRow(i);
                if (row != null) {
//                    Cell idCell = row.getCell(1);
//                    Cell codeCell = row.getCell(2);
//                    Cell cell1 = row.getCell(11);
//                    Cell cell2 = row.getCell(13);
//                    Cell cell3 = row.getCell(14);
//                    Cell cell4 = row.getCell(16);
//                    Cell cell5 = row.getCell(17);
                    Cell idCell = row.getCell(0);
                    Cell influencePositionCell = row.getCell(10);
                    if (idCell != null) {
                        String id = idCell.getStringCellValue();
//                        String stringCellValue1 = cell1.getStringCellValue();
//                        String stringCellValue2 = cell2.getStringCellValue();
//                        String stringCellValue3 = cell3.getStringCellValue();
//                        String stringCellValue4 = cell4.getStringCellValue();
//                        String stringCellValue5 = cell5.getStringCellValue();
//                        thinktankBasicInfoMapper.updateThinkTankBasicInfoById(id,stringCellValue1,stringCellValue2,stringCellValue3,stringCellValue4,stringCellValue5);
                        try {
                            String stringCellValue = influencePositionCell.getStringCellValue();
                            if (stringCellValue != null && !"".equals(stringCellValue)) {
                                thinktankBasicInfoMapper.updateThinkTankInfluencePositionById(id,stringCellValue);
                                log.info("success,{}",i);
                            }
                        } catch (Exception e) {
                            log.info("读取数据异常,{}",i);
                        }


                    }
                }

            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public List<ThinktankBasicInfo> exportListByCondition(String id, String codeId, String typeId, String thinktankName, String tagId, String keyword) {
        List<String> ids = new ArrayList<>();
        if (StringUtils.isNotEmpty(id)) {
            ids = Arrays.asList(id.split(","));
        }
        List<ThinktankBasicInfo> thinkTankBasicInfos = thinktankBasicInfoMapper.exportListByCondition(ids, codeId, typeId, thinktankName, tagId, keyword);
        if (CollectionUtils.isNotEmpty(thinkTankBasicInfos)) {
            //国家列表
            List<ThinktankCategoryStructure> listCategoryStructure = leaderCategoryService.list();
            Map<String, String> mapCategoryStructure = listCategoryStructure.stream().collect(Collectors.toMap(ThinktankCategoryStructure::getId, ThinktankCategoryStructure::getTypeName, (value1, value2) -> value1));
            //语种
            List<DataChangeVO> languageList = dataImportService.dictItemList(LANGUAGE);
            Map<String, String> languageMap = languageList.stream().collect(Collectors.toMap(DataChangeVO::getValue, DataChangeVO::getDescription, (value1, value2) -> value1));
            //营利性质
            List<DataChangeVO> profitNatureList = dataImportService.dictItemList(THINK_TANK_PROFIT_NATURE);
            Map<String, String> profitNatureMap = profitNatureList.stream().collect(Collectors.toMap(DataChangeVO::getValue, DataChangeVO::getDescription, (value1, value2) -> value1));
            //智库性质
            List<DataChangeVO> natureList = dataImportService.dictItemList(THINK_TANK_NATURE);
            Map<String, String> natureMap = natureList.stream().collect(Collectors.toMap(DataChangeVO::getValue, DataChangeVO::getDescription, (value1, value2) -> value1));
            //研究领域
            List<DataChangeVO> researchFieldList = dataImportService.labelEntityList(SYS_BASE_LABEL_TYPE);
            Map<String, String> researchFieldMap = researchFieldList.stream().collect(Collectors.toMap(DataChangeVO::getValue, DataChangeVO::getDescription, (value1, value2) -> value1));
            for (ThinktankBasicInfo thinkTankBasicInfo : thinkTankBasicInfos) {
                thinkTankBasicInfo.setBelongCountry(mapCategoryStructure.get(thinkTankBasicInfo.getBelongCountryId()));
                String lang = thinkTankBasicInfo.getLang();
                if (StringUtils.isNotEmpty(lang)) {
                    StringBuilder sb = new StringBuilder();
                    for (String s : lang.split(",")) {
                        String value = languageMap.get(s);
                        sb.append(",").append(value);
                    }
                    thinkTankBasicInfo.setLang(sb.substring(1));
                }
                String profitNature = thinkTankBasicInfo.getProfitNature();
                if (StringUtils.isNotEmpty(profitNature)) {
                    thinkTankBasicInfo.setProfitNature(profitNatureMap.get(profitNature));
                }
                String nature = thinkTankBasicInfo.getNature();
                if (StringUtils.isNotEmpty(nature)) {
                    thinkTankBasicInfo.setNature(natureMap.get(nature));
                }
                String tagIds = thinkTankBasicInfo.getTagId();
                if (StringUtils.isNotEmpty(tagIds)) {
                    StringBuilder sb = new StringBuilder();
                    for (String s : tagIds.split(",")) {
                        String value = researchFieldMap.get(s);
                        sb.append(",").append(value);
                    }
                    thinkTankBasicInfo.setTagName(sb.substring(1));
                }
                String belongUnitCode = thinkTankBasicInfo.getBelongUnitCode();
                if (StringUtils.isNotEmpty(belongUnitCode)) {
                    LambdaQueryWrapper<SysBaseEnterprise> queryWrapper = Wrappers.lambdaQuery();
                    queryWrapper.select(SysBaseEnterprise::getName).eq(SysBaseEnterprise::getSocialCreditCode, belongUnitCode);
                    SysBaseEnterprise one = enterpriseService.getOne(queryWrapper);
                    if (one != null) {
                        thinkTankBasicInfo.setBelongUnit(one.getName());
                    }
                }
            }
        }
        return thinkTankBasicInfos;
    }

    @Override
    public List<ThinkTankSourceVO> thinkTankCollectCount(List<ThinktankBasicInfo> dataList) {
        Map<String, ThinktankBasicInfo> map = dataList.stream().collect(Collectors.toMap(ThinktankBasicInfo::getId, Function.identity()));
        List<String> idList = new ArrayList<>();
        dataList.forEach(e -> idList.add(e.getId()));
        List<ThinkTankSourceVO> thinkTankSourceVOS = infoSourceGroupMapService.listOfThinkTank(idList);
        List<String> sourceIds = new ArrayList<>();
        for (ThinkTankSourceVO thinkTankSourceVO : thinkTankSourceVOS) {
            ThinktankBasicInfo thinktankBasicInfo = map.get(thinkTankSourceVO.getGroupId());
            thinkTankSourceVO.setCodeId(thinktankBasicInfo.getCodeId());
            thinkTankSourceVO.setChineseWhole(thinktankBasicInfo.getChineseWhole());
            //信息源id集合
            sourceIds.add(thinkTankSourceVO.getSourceId());
        }
        //查询es
        BoolQueryBuilder boolQuerylist = QueryBuilders.boolQuery();
        boolQuerylist.must(QueryBuilders.termsQuery("sid", sourceIds));
        Map<String, Long> totalCountMap = new HashMap<>();
        Map<String, Long> monthlyCountMap = new HashMap<>();
        try {
            //所有
            totalCountMap = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
            //近一个月
            DateTime yesterday = cn.hutool.core.date.DateUtil.yesterday();
            DateTime startTime = cn.hutool.core.date.DateUtil.offsetDay(yesterday, -30);
            boolQuerylist.filter(QueryBuilders.rangeQuery("createDate")
                    .gte(EsDateUtil.esFieldDateFormat(cn.hutool.core.date.DateUtil.format(startTime, "yyyy-MM-dd 00:00:00")))
                    .lte(EsDateUtil.esFieldDateFormat(cn.hutool.core.date.DateUtil.format(yesterday, "yyyy-MM-dd 23:59:59"))));
            monthlyCountMap = esUtil.groupBy("basedata", "sid", boolQuerylist, 100);
        } catch (IOException e) {
            log.error("ES查询失败");
            e.printStackTrace();
        }
        //采集量统计
        for (ThinkTankSourceVO thinkTankSourceVO : thinkTankSourceVOS) {
            String sourceId = thinkTankSourceVO.getSourceId();
            Long infoCount = totalCountMap.get(sourceId);
            if (infoCount == null) {
                infoCount = 0L;
            }
            thinkTankSourceVO.setInfoCount(infoCount.intValue());
            Long infoMonthlyCount = monthlyCountMap.get(sourceId);
            if (infoMonthlyCount == null) {
                infoMonthlyCount = 0L;
            }
            thinkTankSourceVO.setInfoMonthlyCount(infoMonthlyCount.intValue());
        }
        //数据合并
        List<ThinkTankSourceVO> list = new ArrayList<>();
        Map<String, List<ThinkTankSourceVO>> groupCountMap = thinkTankSourceVOS.stream().collect(Collectors.groupingBy(ThinkTankSourceVO::getGroupId));
        for (Map.Entry<String, List<ThinkTankSourceVO>> entry : groupCountMap.entrySet()) {
            List<ThinkTankSourceVO> value = entry.getValue();
            int groupCount = value.stream().mapToInt(ThinkTankSourceVO::getInfoCount).sum();
            ThinkTankSourceVO thinkTankSourceVO = value.get(0);
            thinkTankSourceVO.setGroupCount(groupCount);
            thinkTankSourceVO.setInfoSourceCount(value.size());
            list.addAll(value);
        }
        return list;
    }
}
