package com.zzsn.leaderbase.controller;

import com.zzsn.leaderbase.service.CharacterInfoService;
import com.zzsn.leaderbase.vo.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/2/29 9:26
 * @Content: 人物信息操作
 */
@RestController
@RequestMapping("/api/character")
public class CharacterInfoController {

    @Autowired
    CharacterInfoService characterInfoService;

    /**
     * 新增人物基本信息和职务变动
     * @param characterVo
     * @return
     */
    @PostMapping("/add")
    public Result<?> add(@RequestBody CharacterVo characterVo){
        return characterInfoService.add(characterVo);
    }

    /**
     * 根据人物uid查询人物信息
     * @param uid
     * @return
     */
    @GetMapping("getByUid")
    public Result<?> getByUid(@RequestParam(name="uid",required=true) String uid){
        return characterInfoService.getByUid(uid);
    }

    /**
     * 根据人物id主键查询
     * @param id
     * @return
     */
    @GetMapping
    public Result<?> getById(@RequestParam(name="id",required=true) String id){
        return characterInfoService.getById(id);
    }

    /**
     * 根据人物id修改信息
     * @param characterVo
     * @return
     */
    @PostMapping("/edit")
    public Result<?> edit(@RequestBody CharacterVo characterVo){
        return characterInfoService.edit(characterVo);
    }

    /**
     * 根据人物id主键删除
     * @param id
     * @return
     */
    @GetMapping("/del")
    public Result<?> del(@RequestParam(name="id",required=true) String id){
        return characterInfoService.del(id);
    }

    /**
     * 条件分页查询
     * @param characterListVo
     * @return
     */
    @PostMapping("/list")
    public Result<?> getList(@RequestBody CharacterListVo characterListVo){
        //参数校验
        Integer pageNo = characterListVo.getPageNo();
        Integer pageSize= characterListVo.getPageSize();
        if(pageNo < 1)
            return Result.error("参数pageNo的值不能小于1！");
        if(pageSize < 0)
            return Result.error("参数pageSize的值不能小于0！");
        Integer office = pageSize * (pageNo - 1);
        characterListVo.setOffset(office);
        return characterInfoService.getList(characterListVo);
    }


    /**
     * 审核界面根据关联资讯id 查询资讯关联的人物
     * @param
     * @return
     */
    @GetMapping("/checkList")
    public Result<?> checkList(@RequestParam(name="articleId",required=true) String articleId,
                               @RequestParam(name="checkResultArr",required=false) String[] checkResultArr){
        if(articleId == null && articleId.length() == 0)
            return Result.error("资讯id不能为空！");
        return characterInfoService.checkList(articleId,checkResultArr);
    }

    /**
     * 审核界面根据人物名称查询人物列表，排除当条数据
     * @param
     * @return
     */
    @GetMapping("/getListByName")
    public Result<?> getListByName(@RequestParam(name="id",required=true) String id,
                                   @RequestParam(name="name",required=true) String name,
                                   @RequestParam(name="articleId",required=true) String articleId){

        if(id == null && id.length() == 0)
            return Result.error("id不能为空！");
        if(id == null)
            return Result.error("name不能为空！");

        return characterInfoService.getListByName(id, name, articleId);
    }



    /**
     * 审核
     * @param characterCheckVo
     * @return
     */
    @PostMapping("/checkUpdate")
    public Result<?> checkUpdate(@RequestBody CharacterCheckVo characterCheckVo){
        if(characterCheckVo.getOriginalCharacter() == null || characterCheckVo.getCharacter() == null)
            return Result.error("参数有误：历史人物信息或新人物信息不能为空！");
        return characterInfoService.checkUpdate(characterCheckVo);
    }

    /**
     * 根据id 取消人物  审核
     * @param id
     * @return
     */
    @GetMapping("/checkCancelUpdate")
    public Result<?> checkCancelUpdate(@RequestParam(name="id",required=true) String id){
        if(id == null || id.length() ==0)
            return Result.error("id不能为空！");
        return characterInfoService.checkCancelUpdate(id);
    }


    /**
     * 人物合并
     * 操作类型，1合并人物  2新增人物  3追加历任职务
     * @param characterMergeVo
     * @return
     */
    @PostMapping("/merge")
    public Result<?> merge(@RequestBody CharacterMergeVo characterMergeVo){
        return characterInfoService.merge(characterMergeVo);
    }

    /**
     * 取消人物合并
     * 操作类型，1合并人物  2新增人物  3追加历任职务
     * @param id
     * @return
     */
    @GetMapping("/mergeCancel")
    public Result<?> mergeCancel(@RequestParam(name="id",required=true) String id){
        return characterInfoService.mergeCancel(id);
    }

}
