package com.zzsn.leaderbase.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zzsn.leaderbase.entity.CharacterBasicInfo;
import com.zzsn.leaderbase.entity.CharacterCategoryStructure;
import com.zzsn.leaderbase.entity.CharacterTagMap;
import com.zzsn.leaderbase.feign.api.RemoteSystemService;
import com.zzsn.leaderbase.feign.entity.SysDictItem;
import com.zzsn.leaderbase.mapper.CharacterBasicInfoMapper;
import com.zzsn.leaderbase.service.*;
import com.zzsn.leaderbase.util.ExcelUtil;
import com.zzsn.leaderbase.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * @Version 1.0
 * @Author: ZhangJingKun
 * @Date: 2024/2/29 10:35
 * @Content: 人物基本信息
 */
@Slf4j
@Service
public class CharacterBasicInfoServiceImpl extends ServiceImpl<CharacterBasicInfoMapper, CharacterBasicInfo> implements CharacterBasicInfoService {

    //生成id
    @Autowired
    IGeneratorIdService generatorIdService;

    @Autowired
    CharacterTagService characterTagService;

    @Autowired
    CharacterTagMapService characterTagMapService;

    @Autowired
    LeaderCategoryService leaderCategoryService;

    @Autowired(required = false)
    private RemoteSystemService remoteSystemService;

    @Override
    public CharacterVo getByUid(String uid) {
        return baseMapper.getByUid(uid);
    }

    @Override
    public CharacterBasicInfo getOneById(String id) {
        CharacterBasicInfo basicInfo = this.getById(id);
        return basicInfo;
    }

    @Override
    public IPage<CharacterBasicInfo> getList(BasicInfoListVo basicInfoListVo) {
        //多级数据字典标签，获取当前选择标签层级之后的相差3级以内的标签
        List<LabelVO> labelCodeList = basicInfoListVo.getLabelList();
        if (CollUtil.isNotEmpty(labelCodeList)) {
            List<LabelVO> realList = getRealList(labelCodeList);
            basicInfoListVo.setLabelList(realList);
        }
        List<CharacterBasicInfo> list = baseMapper.getList(basicInfoListVo);
        Integer total = baseMapper.getListCount(basicInfoListVo);
        List<CharacterBasicInfo> listNew =setListValues(list,basicInfoListVo);
        IPage page = new Page();
        page.setRecords(listNew);
        page.setTotal(total);
        Integer pageNo = basicInfoListVo.getPageNo();
        Integer pageSize= basicInfoListVo.getPageSize();
        page.setCurrent(pageNo);
        page.setSize(pageSize);
        return page;
    }

    /**
     * 多级数据字典标签，获取当前选择标签层级之后的相差3级以内的标签
     *
     * @param labelCodeList 标签参数
     * @author lkg
     * @date 2024/12/10
     */
    private List<LabelVO> getRealList(List<LabelVO> labelCodeList) {
        Set<LabelVO> set = new HashSet<>();
        Map<String, List<SysDictItem>> dictMap = new HashMap<>();
        //若有数据字典类数据进行检索，提前准备好字典数据
        Map<String, List<LabelVO>> collect = labelCodeList.stream()
                .filter(e -> e.getDictCode() != null)
                .collect(Collectors.groupingBy(LabelVO::getDictCode));
        if (MapUtil.isNotEmpty(collect)) {
            for (Map.Entry<String, List<LabelVO>> entry : collect.entrySet()) {
                String dictCode = entry.getKey();
                //关联数据字典，展示字典树
                List<SysDictItem> sysDictItems = JSON.parseArray(remoteSystemService.sysDictItemList(dictCode),SysDictItem.class);
                dictMap.put(dictCode, sysDictItems);
            }
        }
        for (LabelVO LabelVO : labelCodeList) {
            Integer level = LabelVO.getLevel();
            if (level != null) {
                String labelCode = LabelVO.getLabelCode();
                String labelItemId = LabelVO.getLabelItemId();
                String dictCode = LabelVO.getDictCode();
                List<SysDictItem> sysDictItems = dictMap.get(dictCode);
                List<SysDictItem> children = sysDictItems.stream().filter(e -> e.getPathIds().contains(labelItemId) && e.getLevel() <= (level + 2)).collect(Collectors.toList());
                Set<LabelVO> followSet = new HashSet<>();
                if (CollUtil.isNotEmpty(children)) {
                    for (SysDictItem child : children) {
                        LabelVO map = new LabelVO();
                        map.setLabelCode(labelCode);
                        map.setLabelItemCode(child.getItemValue());
                        followSet.add(map);
                    }
                    set.addAll(followSet);
                }
            } else {
                set.add(LabelVO);
            }
        }
        return new ArrayList<>(set);
    }

    private List<CharacterBasicInfo> setListValues(List<CharacterBasicInfo> list, BasicInfoListVo basicInfoListVo) {
        if(CollectionUtil.isNotEmpty(list)){
            List<String> ids=list.stream().map(CharacterBasicInfo::getId).collect(Collectors.toList());
            Set<String> uIds = list.stream().map(CharacterBasicInfo::getUid).collect(Collectors.toSet());
            LambdaQueryWrapper<CharacterBasicInfo> queryWrapper=new LambdaQueryWrapper<>();
            queryWrapper.in(CharacterBasicInfo::getUid,uIds);
            queryWrapper.notIn(CharacterBasicInfo::getId,ids);
            //queryWrapper.select(CharacterBasicInfo::getUid);
            queryWrapper.orderByDesc(CharacterBasicInfo::getCreateTime);
            List<CharacterBasicInfo> subList = this.list(queryWrapper);
            Set<String> containsUIds= subList.stream().map(CharacterBasicInfo::getUid).collect(Collectors.toSet());
            Map<CharacterBasicInfo, CharacterBasicInfo> replaceMap = new HashMap<>();
            list.forEach(t->{
                if(containsUIds.contains(t.getUid())){
                    t.setHasChild(1);
                    //有关联人物且 当前为历任的，切换显示最新在任数据
                    if("历任".equals(t.getTakeOfficeStatus())){
                        List<CharacterBasicInfo> collect = subList.stream().filter(e -> e.getUid().equals(t.getUid()) && "在职".equals(e.getTakeOfficeStatus())).collect(Collectors.toList());
                        if(!collect.isEmpty()){
                            collect.get(0).setHasChild(1);
                            //添加到待替换列表
                            replaceMap.put(t,collect.get(0));
                        }
                    }
                }
            });
            replaceMap.forEach((k,v)->{
                //删除指定数据并返回所在索引位置
                int index = list.indexOf(k);
                if(index != -1){
                    list.remove(index);
                    list.add(index,v);
                }
            });
        }
        //记录热门标签
        String tagId = basicInfoListVo.getTagId();
        if(tagId != null && tagId.length() >0){
            String[] tagIdArr = tagId.split(",");
            for (String s : tagIdArr) {
                characterTagService.addTagTop(s);
            }
        }
        return list;
    }

    //判断是否有关联数据
    private List<CharacterBasicInfo> hasChild(List<CharacterBasicInfo> list){
        if(list != null && list.size() > 0 ){
            List<String> uidList = new ArrayList<>();
            list.forEach(info -> { uidList.add(info.getUid()); });
            QueryWrapper<CharacterBasicInfo> queryWrapper = new QueryWrapper<>();
            queryWrapper.in("uid", uidList)
                    .ne("main_entry",1).select("id","uid");
            List<CharacterBasicInfo> hasChildList = this.list(queryWrapper);
            if(hasChildList.size() > 0){
                Map<String, List<CharacterBasicInfo>> groupList = hasChildList.stream().collect(Collectors.groupingBy(e -> e.getUid()));
                list.forEach(info -> {
                    String uid = info.getUid();
                    List<CharacterBasicInfo> groupUid = groupList.get(uid);
                    if(groupUid != null && groupUid.size() != 0){
                        info.setHasChild(1);
                    }
                });
            }
        }
        return list;
    }


    @Override
    public List<CharacterVo> checkList(String articleId, String checkResult) {
        return baseMapper.checkList(articleId,checkResult);
    }

    @Override
    public List<CharacterBasicInfo> getListByName(String name) {
        return baseMapper.getListByName(name);
    }

    @Override
    public Integer deduplication(String duty, String company, String name) {
        Integer count = baseMapper.deduplication(duty, company, name);
        return count;
    }

    @Override
    public List<CharacterBasicInfo> getData(String time) {
        LambdaQueryWrapper<CharacterBasicInfo> lambdaQuery = Wrappers.lambdaQuery();
        if(!"".equals(time)){
            lambdaQuery.ge(CharacterBasicInfo::getCreateTime, time);
        }
        lambdaQuery.orderByAsc(CharacterBasicInfo::getCreateTime);
        lambdaQuery.select(CharacterBasicInfo::getUid,CharacterBasicInfo::getName);
        List<CharacterBasicInfo> list = baseMapper.selectList(lambdaQuery);
        return list;
    }

    @Override
    public List<CharacterBasicInfo> getPreviousDuty(String uid,String id,String category) {
        return baseMapper.getPreviousDuty(uid,id,category);
    }

    @Override
    public List<String> getCorrelationByUid(String uid) {
        return baseMapper.getCorrelationByUid(uid);
    }

    @Override
    public List<CharacterVo> getByOriginalId(String id) {
        return baseMapper.getByOriginalId(id);
    }

    @Override
    public Result<?> add(CharacterBasicInfo characterBasicInfo) {
        String id = Long.toString(generatorIdService.getOrderId());
        characterBasicInfo.setId(id);
        characterBasicInfo.setCreateTime(new Date());
        characterBasicInfo.setUpdateTime(new Date());
        String uid = generatorIdService.getIdNo();
        characterBasicInfo.setUid(uid);
        characterBasicInfo.setMainEntry(1);
        try{
            log.info("新增专家===开始====主题标签、领域标签关联character_tag_map表");
            //新增专家时，主题标签、领域标签关联character_tag_map表
            List<CharacterTagMap> mapList = getTagMap(characterBasicInfo);
            if(mapList!=null && mapList.size()>0){
                characterTagMapService.saveBatch(mapList);
            }
        }catch (Exception e){
            log.error("新增专家===主题标签、领域标签关联character_tag_map表失败,异常:{}",e.toString());
        }

        Boolean res = this.save(characterBasicInfo);
        return Result.OK(res);
    }

    @Override
    public Result<?> edit(CharacterBasicInfo characterBasicInfo) {
        characterBasicInfo.setUpdateTime(new Date());
        try{
            log.info("编辑专家===开始====主题标签、领域标签关联character_tag_map表");
            //编辑专家时，主题标签、领域标签关联character_tag_map表
            List<CharacterTagMap> sourceList = getTagMap(characterBasicInfo);
            if(sourceList!=null && sourceList.size()>0){
                QueryWrapper queryWrapper = new QueryWrapper();
                queryWrapper.eq("character_uid",characterBasicInfo.getUid());
                List<CharacterTagMap> targetList = characterTagMapService.list(queryWrapper);
                List<CharacterTagMap> recordsToDelete = new ArrayList<>();
                syncCharacterTagMaps(sourceList,targetList,recordsToDelete);
                log.info("编辑专家===开始====targetList数量:{},recordsToDelete数量:{}",targetList.size(),recordsToDelete.size());
                if(targetList!=null && targetList.size()>0){
                    characterTagMapService.saveOrUpdateBatch(targetList);
                }
                if(recordsToDelete!=null && recordsToDelete.size()>0){
                    List<Long> deletedIds = recordsToDelete.stream()
                            .map(CharacterTagMap::getId)
                            .collect(Collectors.toList());
                    characterTagMapService.removeByIds(deletedIds);
                }
            }
        }catch (Exception e){
            log.error("编辑专家===主题标签、领域标签关联character_tag_map表失败,异常:{}",e.toString());
        }
        Boolean res = this.updateById(characterBasicInfo);
        return Result.OK(res);
    }


    public List<CharacterTagMap> getTagMap(CharacterBasicInfo characterBasicInfo) {
        List<SysDictItem> sysDictItemsKeyword = JSON.parseArray(remoteSystemService.sysDictItemList("ai_character_keyword"),SysDictItem.class);
        List<SysDictItem> sysDictItemsArea = JSON.parseArray(remoteSystemService.sysDictItemList("ai_character_area"),SysDictItem.class);
        log.info("ai_character_keyword数量:{}",sysDictItemsKeyword.size());
        log.info("ai_character_area数量:{}",sysDictItemsArea.size());
        List<CharacterTagMap> mapList = new ArrayList<>();
        if(StringUtils.isNotBlank(characterBasicInfo.getTagId())){
            for(String tagId:Arrays.asList(characterBasicInfo.getTagId().split(","))){
                SysDictItem targetItemKeyword = sysDictItemsKeyword.stream()
                        .filter(item -> Objects.equals(item.getId(), tagId))
                        .findFirst()
                        .orElse(null);
                if (targetItemKeyword != null) {
                    CharacterTagMap map = new CharacterTagMap();
                    map.setCharacterUid(characterBasicInfo.getUid());
                    map.setTagId(targetItemKeyword.getId());
                    map.setTagName(targetItemKeyword.getItemText());
                    map.setTagType("ai_character_keyword");
                    map.setScore(new BigDecimal("10.0"));//默认分数
                    map.setCreateTime(new Date());
                    mapList.add(map);
                }
                SysDictItem targetItemArea = sysDictItemsArea.stream()
                        .filter(item -> Objects.equals(item.getId(), tagId))
                        .findFirst()
                        .orElse(null);
                if (targetItemArea != null) {
                    CharacterTagMap map = new CharacterTagMap();
                    map.setCharacterUid(characterBasicInfo.getUid());
                    map.setTagId(targetItemArea.getId());
                    map.setTagName(targetItemArea.getItemText());
                    map.setTagType("ai_character_area");
                    map.setScore(new BigDecimal("10.0"));//默认分数
                    map.setCreateTime(new Date());
                    mapList.add(map);
                }
            }
            log.info("mapList数量:{}",mapList.size());
        }
        return mapList;
    }


    public static void syncCharacterTagMaps(List<CharacterTagMap> source, List<CharacterTagMap> target,List<CharacterTagMap> recordsToDelete) {
        // 1. 将source集合转换为以tagId为键的Map，便于快速查找
        Map<String, CharacterTagMap> sourceMap = source.stream()
                .collect(Collectors.toMap(CharacterTagMap::getTagId, item -> item));
        // 2. 找出target中source不存在的记录（即需要删除的记录）
        for (CharacterTagMap item : target) {
            if (!sourceMap.containsKey(item.getTagId())) {
                recordsToDelete.add(item);
            }
        }
        // 遍历target集合，删除source中不存在的记录
        target.removeIf(item -> !sourceMap.containsKey(item.getTagId()));
        // 3. 遍历source集合，将target中不存在的记录添加进去
        for (CharacterTagMap item : source) {
            boolean exists = target.stream()
                    .anyMatch(t -> t.getTagId().equals(item.getTagId()));
            if (!exists) {
                target.add(item);
            }
        }
    }

    @Override
    public Boolean del(String id) {
        List<String> ids = Arrays.asList(id.split(","));
        return this.removeByIds(ids);
    }

    @Override
    public Result<?> correlation(CharacterBasicInfo characterBasicInfo) {
        //传过来的id是操作的数据id
        String id = characterBasicInfo.getId();
        CharacterBasicInfo newInfo = this.getById(id);

        String newId = Long.toString(generatorIdService.getOrderId());
        characterBasicInfo.setId(newId);
        characterBasicInfo.setMainEntry(1);
        characterBasicInfo.setCategory(newInfo.getCategory());
        characterBasicInfo.setCreateTime(new Date());
        characterBasicInfo.setUpdateTime(new Date());
        Boolean res = this.save(characterBasicInfo);

        //主条目重新设置
        newInfo.setMainEntry(0);
        this.updateById(newInfo);
        return Result.OK(res);
    }

    @Override
    public Result<?> correlationEdit(CharacterBasicInfo characterBasicInfo) {
        characterBasicInfo.setUpdateTime(new Date());
        Boolean res = this.updateById(characterBasicInfo);
        return Result.OK(res);
    }

    @Override
    public Result<?> correlationMerge(CheckVo checkVo){
        String suspectedId = checkVo.getSuspectedId();
        List<String> subSuspectedIds= checkVo.getSubSuspectedIds();
        CharacterBasicInfo characterBasicInfo=this.getById(suspectedId);
        LambdaUpdateWrapper<CharacterBasicInfo> updateWrapper=new LambdaUpdateWrapper<>();
        updateWrapper.set(CharacterBasicInfo::getUid, characterBasicInfo.getUid());
        updateWrapper.set(CharacterBasicInfo::getUpdateTime,new Date());
        updateWrapper.in(CharacterBasicInfo::getId,subSuspectedIds);
        this.update(updateWrapper);
        return Result.OK();
    }


    /**
     * 根据社会信用代码查询公司名称
     * @param socialCreditCode
     * @return
     */
    @Override
    public String getDepartmentCreditCode(String socialCreditCode) {
        return baseMapper.getDepartmentCreditCode(socialCreditCode);
    }

    /**
     * 根据社会信用代码查询相关联高管人员
     * @param socialCreditCode
     * @return
     */
    @Override
    public List<CharacterBasicInfo> getBasicInfoByCreditCode(String socialCreditCode) {
        return baseMapper.getBasicInfoByCreditCode(socialCreditCode);
    }

    @Override
    public List<Map<String, String>> getDepartment(String socialCreditCode, String department, String keyword) {
        if(keyword != null && keyword.length() > 0){
            if((keyword.length() == 18 || keyword.length() == 19) && keyword.matches("[0-9].*|[Zz].*")){
                socialCreditCode = keyword;
            } else {
                department = keyword;
            }
            keyword = null;
        }
        return baseMapper.getDepartment(socialCreditCode, department, keyword);
    }

    @Override
    public List<CharacterBasicInfo> getAllList(BasicInfoListVo basicInfoListVo) {
        List<CharacterBasicInfo> list = baseMapper.getAllList(basicInfoListVo);
        List<CharacterBasicInfo> listNew =setListValues(list,basicInfoListVo);
        return listNew;
    }

    @Override
    public CharacterBasicInfo getBasicInfoByUid(String category,String uid) {
        LambdaQueryWrapper<CharacterBasicInfo> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(CharacterBasicInfo::getUid,uid);
        queryWrapper.eq(CharacterBasicInfo::getCategory,category);
        return baseMapper.selectOne(queryWrapper);
    }

    @Override
    public CharacterBasicInfo getByNameAndByDepartment(String category, String typeId, String name, String department) {
        LambdaQueryWrapper<CharacterBasicInfo> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(CharacterBasicInfo::getTypeId,typeId);
        queryWrapper.eq(CharacterBasicInfo::getCategory,category);
        queryWrapper.eq(CharacterBasicInfo::getName,name);
        queryWrapper.eq(CharacterBasicInfo::getDepartment,department);
        List<CharacterBasicInfo> characterBasicInfos = baseMapper.selectList(queryWrapper);
        if(characterBasicInfos!=null && characterBasicInfos.size()>0){
            return characterBasicInfos.get(0);
        }
        return null;
    }

    @Override
    public String importExcel(String filePath, String category, Map<String, String> importFieldMap) {
        List<String> sheetNameList = new ArrayList<>();
        Workbook workbook = null;
        InputStream in = null;
        String result = "";
        try {
            // 获取输入流
            in = Files.newInputStream(Paths.get(filePath));
            // 判断excel版本
            if (ExcelUtil.judegExcelEdition(filePath)) {
                workbook = new XSSFWorkbook(in);
            } else {
                workbook = new HSSFWorkbook(in);
            }
            Iterator<Sheet> it = workbook.sheetIterator();
            while (it.hasNext()) {
                Sheet sheetTmp = it.next();
                String sheetName = sheetTmp.getSheetName();
                sheetNameList.add(sheetName);
            }
            Map<String, String> typeNameKeyMap = getTypeMapper(category).stream().collect(Collectors.toMap(CharacterCategoryStructure::getTypeName, CharacterCategoryStructure::getId));
            List<SysDictItem> dictItemList = dictItemList("Region_dic");
            Map<String, List<SysDictItem>> dictValueMap = dictItemList.stream().collect(Collectors.groupingBy(SysDictItem::getItemText, Collectors.toList()));
            Map<String, List<SysDictItem>> dictPathIdsMap = dictItemList.stream().collect(Collectors.groupingBy(SysDictItem::getPathIds, Collectors.toList()));
            //人物类别映射集合，用来转化typeId和类别名称
            for (String sheetName : sheetNameList) {
                Sheet sheet = workbook.getSheet(sheetName);
                //获取标题行
                Row titleRow = sheet.getRow(0);
                List<CharacterBasicInfo> list = new ArrayList<>();
                for (int i = 1; i < sheet.getPhysicalNumberOfRows(); i++) {
                    // 循环获取工作表的每一行
                    Row sheetRow = sheet.getRow(i);
                    if (sheetRow == null) {
                        continue;
                    }
                    Map<String, Object> beanMap = new HashMap<>();
                    for (Cell cell : titleRow) {
                        String titleName = cell.getStringCellValue();
                        if (importFieldMap.containsKey(titleName)) {
                            try {
                                String stringCellValue =Objects.toString(sheetRow.getCell(cell.getColumnIndex()),"");
                                //如果是分类列，转换成id
                                if ("分类".equals(titleName) && "typeId".equals(importFieldMap.get(titleName))) {
                                    stringCellValue = typeNameKeyMap.get(stringCellValue);
                                }
                                //籍贯 转换
                                if ("籍贯".equals(titleName) && "nativePlace".equals(importFieldMap.get(titleName))) {
                                    stringCellValue = nativePlaceChange(stringCellValue,dictValueMap,dictPathIdsMap);
                                }
                                //将导入文件中不为null的列放入beanMap，key为导入列在数据库中对应的表列名
                                if (StringUtils.isNotEmpty(stringCellValue)) {
                                    beanMap.put(importFieldMap.get(titleName), stringCellValue);
                                }
                            } catch (Exception e) {
                                log.debug("数据导入excel解析{}行{}字段为空", cell.getColumnIndex(), titleName);
                            }
                        }
                    }
                    //过滤掉根据导入文档类型中文名找不到对应类别的数据（这些不做处理）
                    if (StringUtils.isNotEmpty((CharSequence) beanMap.get("typeId"))) {
                        list.add(BeanUtil.mapToBean(beanMap, CharacterBasicInfo.class, true, CopyOptions.create().setIgnoreError(true)));
                    }
                }
                //需要新增的数据列表
                List<CharacterBasicInfo> saveList = new ArrayList<>();
                //需要更新的数据列表
                List<CharacterBasicInfo> updateList = new ArrayList<>();
                for (CharacterBasicInfo basicInfo : list) {
                    if (basicInfo == null || StringUtils.isEmpty(basicInfo.getName()) || StringUtils.isEmpty(basicInfo.getDepartment()) || StringUtils.isEmpty(basicInfo.getTypeId())) {
                        log.info("数据为空，跳过");
                        continue;
                    }
                    basicInfo.setCategory(category);
                    if (StringUtils.isNotEmpty(basicInfo.getBirthday())) {
                        basicInfo.setBirthday(basicInfo.getBirthday().replaceAll("[./]", "-"));
                    }
                    if (StringUtils.isNotEmpty(basicInfo.getTechnicalDate())) {
                        basicInfo.setTechnicalDate(basicInfo.getTechnicalDate().replaceAll("[./]", "-"));
                    }
                    if (StringUtils.isNotEmpty(basicInfo.getWorkDate())) {
                        basicInfo.setWorkDate(basicInfo.getWorkDate().replaceAll("[./]", "-"));
                    }
                    //检查是否已有数据
                    CharacterBasicInfo oldBasicInfo = null;
                    if (StringUtils.isNotEmpty(basicInfo.getUid())) {
                        oldBasicInfo = getBasicInfoByUid(category, basicInfo.getUid());
                    } else {
                        oldBasicInfo = getByNameAndByDepartment(category, basicInfo.getTypeId(), basicInfo.getName(), basicInfo.getDepartment());
                    }
                    if (oldBasicInfo != null) {
                        //合并两个对象
                        oldBasicInfo.setSex(basicInfo.getSex());
                        oldBasicInfo.setBirthday(basicInfo.getBirthday());
                        oldBasicInfo.setNation(basicInfo.getNation());
                        oldBasicInfo.setNativePlace(basicInfo.getNativePlace());
                        oldBasicInfo.setSchool(basicInfo.getSchool());
                        oldBasicInfo.setSchoolSpeciality(basicInfo.getSchoolSpeciality());
                        oldBasicInfo.setEducation(basicInfo.getEducation());
                        oldBasicInfo.setDepart(basicInfo.getDepart());
                        oldBasicInfo.setDuty(basicInfo.getDuty());
                        oldBasicInfo.setTakeOfficeTime(basicInfo.getTakeOfficeTime());
                        oldBasicInfo.setTakeOfficeTimeEnd(basicInfo.getTakeOfficeTimeEnd());
                        oldBasicInfo.setExpertType(basicInfo.getExpertType());
                        oldBasicInfo.setResearchField(basicInfo.getResearchField());
                        oldBasicInfo.setSpeciality(basicInfo.getSpeciality());
                        oldBasicInfo.setTalentPlanning(basicInfo.getTalentPlanning());
                        oldBasicInfo.setTechnicalTitles(basicInfo.getTechnicalTitles());
                        oldBasicInfo.setTechnicalDate(basicInfo.getTechnicalDate());
                        oldBasicInfo.setWorkDate(basicInfo.getWorkDate());
                        oldBasicInfo.setUpdateTime(new Date());
                        updateList.add(oldBasicInfo);
                    } else {
                        String id = Long.toString(generatorIdService.getOrderId());
                        basicInfo.setId(id);
                        basicInfo.setCreateTime(new Date());
                        String uid = generatorIdService.getIdNo();
                        basicInfo.setUid(uid);
                        basicInfo.setMainEntry(1);
                        saveList.add(basicInfo);
                    }
                }
                if (!saveList.isEmpty()) {
                    saveBatch(saveList);
                    log.info("新增{}条数据", saveList.size());
                    result = "新增" + saveList.size() + "条数据";
                }
                if (!updateList.isEmpty()) {
                    updateBatchById(updateList);
                    log.info("更新{}条数据", updateList.size());
                    if (!"".equals(result)) {
                        result += ";";
                    }
                    result += "更新" + updateList.size() + "条数据";
                }
                if(saveList.isEmpty()&&updateList.isEmpty())
                {
                    result = "无可处理数据(如有数据代表数据不规范)";
                }
            }
        } catch (Exception e) {
            log.info("导入专家信息异常：{}", e.getMessage(), e);
            result = "导入专家信息异常：" + e.getMessage();
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
            try {
                if (workbook != null) {
                    workbook.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return result;
    }


    private String nativePlaceChange(String stringCellValue,Map<String, List<SysDictItem>> dictValueMap,Map<String, List<SysDictItem>> dictPathIdsMap){
        if (StringUtils.isNotBlank(stringCellValue)) {
            if (stringCellValue.contains("/")){
                String[] split = stringCellValue.split("/");
                //判断节点是否全部包含
                boolean match = true;
                for (String s : split) {
                    boolean b = dictValueMap.containsKey(s);
                    if (!b) {
                        match = false;
                        break;
                    }
                }
                if (match) {
                    StringBuilder stringBuilder = new StringBuilder();
                    for (int j = 0; j < split.length; j++) {
                        if (j == 0) {
                            List<SysDictItem> sysDictItems = dictValueMap.get(split[j]);
                            stringBuilder.append(",").append(sysDictItems.get(0).getId());
                        } else {
                            List<SysDictItem> sysDictItems = dictValueMap.get(split[j]);
                            for (SysDictItem sysDictItem : sysDictItems) {
                                if (sysDictItem.getPathIds().contains(stringBuilder.substring(1))) {
                                    stringBuilder.append(",").append(sysDictItem.getId());
                                    break;
                                }
                            }
                        }
                    }
                    String substring = stringBuilder.substring(1);
                    if (dictPathIdsMap.containsKey(substring)) {
                        stringCellValue = dictPathIdsMap.get(substring).get(0).getItemValue();
                    } else {
                        stringCellValue = null;
                    }
                } else {
                    stringCellValue = null;
                }
            } else {
                if (dictValueMap.containsKey(stringCellValue)){
                    List<SysDictItem> sysDictItems = dictValueMap.get(stringCellValue);
                    stringCellValue = sysDictItems.get(0).getItemValue();
                } else {
                    stringCellValue = null;
                }
            }
        }
        return stringCellValue;
    }


    private List<CharacterCategoryStructure> getTypeMapper(String category) {
        //根据名称模糊查询
        LambdaQueryWrapper<CharacterCategoryStructure> characterCategoryStructureLambdaQueryWrapper = Wrappers.lambdaQuery();
        characterCategoryStructureLambdaQueryWrapper.eq(StrUtil.isNotBlank(category), CharacterCategoryStructure::getCategory, category);
        List<CharacterCategoryStructure> characterCategoryWrapper = leaderCategoryService.list(characterCategoryStructureLambdaQueryWrapper);
        return characterCategoryWrapper;
    }

    @Override
    public ResponseEntity<byte[]> exportExcel(String category, List<CharacterBasicInfo> characterBasicInfos, Map<String, String> importFieldMap) throws IOException {
        return exportExcelByList(category, characterBasicInfos, importFieldMap);
    }

    @Override
    public ResponseEntity<byte[]> exportExcelByIds(String category, String ids, Map<String, String> importFieldMap) throws IOException {
        List<CharacterBasicInfo> characterBasicInfos = listByIds(Arrays.asList(ids.split(",")));
        return exportExcelByList(category, characterBasicInfos, importFieldMap);
    }

    @Override
    public ResponseEntity<byte[]> exportTemplate(Map<String, String> importFieldMap) throws IOException {
        return generateTemplate(importFieldMap);
    }

    @Override
    public List<CharacterBasicInfo> getDutyList(String uid, String socialCreditCode, String name) {
        LambdaQueryWrapper<CharacterBasicInfo> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(CharacterBasicInfo::getUid, uid);
        queryWrapper.eq(CharacterBasicInfo::getSocialCreditCode, socialCreditCode);
        queryWrapper.eq(CharacterBasicInfo::getName, name);
        queryWrapper.eq(CharacterBasicInfo::getTakeOfficeStatus, "在职");
        queryWrapper.orderByDesc(CharacterBasicInfo::getCreateTime);
        return this.baseMapper.selectList(queryWrapper);
    }

    private ResponseEntity<byte[]> generateTemplate(Map<String, String> importFieldMap) throws IOException {
        // 创建Excel工作簿
        Workbook workbook = new XSSFWorkbook();
        Sheet sheet = workbook.createSheet("人物信息");
        // 填充数据到Excel工作表
        Row titlerow = sheet.createRow(0);
        // 创建字体样式
        Font headerFont = workbook.createFont();
        headerFont.setFontName("Arial");
        headerFont.setFontHeightInPoints((short) 12);
        headerFont.setBold(true);
        // 创建单元格样式并应用字体
        CellStyle titleCellStyle = workbook.createCellStyle();
        titleCellStyle.setFont(headerFont);
        titleCellStyle.setAlignment(HorizontalAlignment.CENTER);
        titleCellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        titleCellStyle.setFillForegroundColor(IndexedColors.BRIGHT_GREEN.getIndex());
        titleCellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        AtomicInteger colNum = new AtomicInteger();
        importFieldMap.forEach((key, value) -> {
            int andIncrement = colNum.getAndIncrement();
            Cell cell = titlerow.createCell(andIncrement);
            cell.setCellValue(key);
            cell.setCellStyle(titleCellStyle);
            sheet.setColumnWidth(andIncrement, 15 * 256);
        });
        // 将Excel工作簿写入字节数组输出流
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        workbook.write(byteArrayOutputStream);
        // 准备HTTP响应头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.parseMediaType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"));
        // 对文件名进行编码
        String encodedFileName = URLEncoder.encode("人物导入模板.xlsx", "utf-8");
        headers.add("Content-Disposition", "attachment; filename=" + encodedFileName);
        // 返回Excel文件作为字节数组响应
        return new ResponseEntity<>(byteArrayOutputStream.toByteArray(), headers, HttpStatus.OK);
    }

    private ResponseEntity<byte[]> exportExcelByList(String category,List<CharacterBasicInfo> characterBasicInfos, Map<String, String> importFieldMap) throws IOException {
        // 创建Excel工作簿
        Workbook workbook = new XSSFWorkbook();
        Sheet sheet = workbook.createSheet("人物信息");
        // 填充数据到Excel工作表
        Row titlerow = sheet.createRow(0);
        // 创建字体样式
        Font headerFont = workbook.createFont();
        headerFont.setFontName("Arial");
        headerFont.setFontHeightInPoints((short) 12);
        headerFont.setBold(true);
        // 创建单元格样式并应用字体
        CellStyle titleCellStyle = workbook.createCellStyle();
        titleCellStyle.setFont(headerFont);
        titleCellStyle.setAlignment(HorizontalAlignment.CENTER);
        titleCellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        titleCellStyle.setFillForegroundColor(IndexedColors.BRIGHT_GREEN.getIndex());
        titleCellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        AtomicInteger colNum = new AtomicInteger();
        importFieldMap.forEach((key, value) -> {
            int andIncrement = colNum.getAndIncrement();
            Cell cell = titlerow.createCell(andIncrement);
            cell.setCellValue(key);
            cell.setCellStyle(titleCellStyle);
            sheet.setColumnWidth(andIncrement, 15 * 256);
        });
        int rowNum = 1;
        Map<String,String> typeIdeKeyMap=getTypeMapper(category).stream().collect(Collectors.toMap(CharacterCategoryStructure::getId, CharacterCategoryStructure::getTypeName));
        List<SysDictItem> dictItemList = dictItemList("Region_dic");
        Map<String, SysDictItem> dictValueMap = dictItemList.stream().collect(Collectors.toMap(SysDictItem::getItemValue, e -> e));
        Map<String, SysDictItem> dictItemIdMap = dictItemList.stream().collect(Collectors.toMap(SysDictItem::getId, e -> e));
        //数据填充到Excel工作表
        for (CharacterBasicInfo characterBasicInfo : characterBasicInfos) {
            Row row = sheet.createRow(rowNum++);
            //总列数
            int columns = importFieldMap.size();
            for (int i = 0; i < columns; i++) {
                Cell cell = row.createCell(i);
                // 列名获取
                String columnName = importFieldMap.get(titlerow.getCell(i).getStringCellValue());
                // 对象转化为map
                Map<String, Object> characterMap = BeanUtil.beanToMap(characterBasicInfo);
                if (columnName != null) {
                    Object value = characterMap.get(columnName);
                    if(ObjectUtils.isNotEmpty(value) && "typeId".equals(columnName)){
                        value=typeIdeKeyMap.get(value);
                    }
                    if (ObjectUtils.isNotEmpty(value)  && "nativePlace".equals(columnName)) {
                        StringBuilder stringBuilder = new StringBuilder();
                        String pathIds = dictValueMap.get(value).getPathIds();
                        for (String itemId : pathIds.split(",")) {
                            stringBuilder.append("/").append(dictItemIdMap.get(itemId).getItemText());
                        }
                        value = stringBuilder.substring(1);
                    }
                    if (value != null) {
                        if (value instanceof String) {
                            cell.setCellValue((String) value);
                        } else if (value instanceof Number) {
                            cell.setCellValue(((Number) value).doubleValue());
                        } else if (value instanceof Date) {
                            cell.setCellValue((Date) value);
                        } else {
                            // 处理其他类型的值，例如布尔值等
                            cell.setCellValue(value.toString());
                        }
                    } else {
                        // 如果值为 null，可以设置为空字符串或其他默认值
                        cell.setCellValue("");
                    }
                } else {
                    // 如果 columnName 不存在于 importFieldMap 中，可以设置为空字符串或其他默认值
                    cell.setCellValue("");
                }
            }
        }
        // 将Excel工作簿写入字节数组输出流
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        workbook.write(byteArrayOutputStream);
        // 准备HTTP响应头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.parseMediaType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"));
        // 对文件名进行编码
        String encodedFileName = URLEncoder.encode("人物信息.xlsx", "utf-8");
        headers.add("Content-Disposition", "attachment; filename=" + encodedFileName);
        // 返回Excel文件作为字节数组响应
        return new ResponseEntity<>(byteArrayOutputStream.toByteArray(), headers, HttpStatus.OK);
    }

    private List<SysDictItem> dictItemList(String dictCode){
        String s = remoteSystemService.sysDictItemList(dictCode);
        return JSON.parseArray(s, SysDictItem.class);
        /*List<SysDictItem> list = new ArrayList<>();
        SysDictItem item = new SysDictItem();
        item.setId("1865339606107422721");
        item.setItemText("国内地域");
        item.setItemValue("Domestic");
        item.setPathIds("1865339606107422721");
        list.add(item);
        SysDictItem item1 = new SysDictItem();
        item1.setId("1865339606120005633");
        item1.setItemText("中国");
        item1.setItemValue("China");
        item1.setPathIds("1865339606107422721,1865339606120005633");
        list.add(item1);
        SysDictItem item2 = new SysDictItem();
        item2.setId("1865339606518464514");
        item2.setItemText("河南省");
        item2.setItemValue("410000000000");
        item2.setPathIds("1865339606107422721,1865339606120005633,1865339606518464514");
        list.add(item2);
        SysDictItem item3 = new SysDictItem();
        item3.setId("1865339641427656705");
        item3.setItemText("焦作市");
        item3.setItemValue("410800000000");
        item3.setPathIds("1865339606107422721,1865339606120005633,1865339606518464514,1865339641427656705");
        list.add(item3);
        return list;*/
    }
}
