package com.zzsn.leaderbase.controller;

import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.leaderbase.entity.CharacterBasicInfo;
import com.zzsn.leaderbase.entity.ExecutiveNew;
import com.zzsn.leaderbase.service.CharacterBasicInfoService;
import com.zzsn.leaderbase.service.IGeneratorIdService;
import com.zzsn.leaderbase.util.SortUtil;
import com.zzsn.leaderbase.util.SyncUtil;
import com.zzsn.leaderbase.vo.BasicInfoListVo;
import com.zzsn.leaderbase.vo.CheckVo;
import com.zzsn.leaderbase.vo.Result;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
@RestController
@RequestMapping("/api/character/basic")
public class CharacterBasicInfoController {

    @Autowired
    CharacterBasicInfoService characterBasicInfoService;
    @Autowired
    IGeneratorIdService       generatorIdService;
    //category 分类类型(1-领导人、2-专家、3-企业高管) 末尾数字对应导入导出数据集类别 例：importFieldMap2代表专家导入导出的字段map集合
    private static Map<String, String> importFieldMap1;
    private static Map<String, String> importFieldMap2;
    private static Map<String, String> importFieldMap3;

    /**
     * 资讯抽取人物信息分页查询
     * 人物审核列表
     *
     * @param
     * @return
     */
    @PostMapping("/list")
    public Result<?> getList(@RequestBody BasicInfoListVo basicInfoListVo) {
        log.info("人物管理列表:{}", basicInfoListVo);
        Integer office = basicInfoListVo.getPageSize() * (basicInfoListVo.getPageNo() - 1);
        basicInfoListVo.setOffset(office);
        basicInfoListVo.setColumn(SortUtil.humpToLine(basicInfoListVo.getColumn()));
        IPage page = characterBasicInfoService.getList(basicInfoListVo);
        return Result.OK(page);
    }

    @GetMapping("/getById")
    public Result<?> getById(@RequestParam(name = "id", required = true) String id) {
        log.info("人物审核根据人物id查询:{}", id);
        CharacterBasicInfo info = characterBasicInfoService.getOneById(id);
        return Result.OK(info);
    }

    /**
     * 根据姓名查询 疑似人物查询
     *
     * @param name
     * @return
     */
    @GetMapping("/getListByName")
    public Result<?> getListByName(@RequestParam(name = "name", required = true) String name) {
        log.info("根据人物名称查询疑似人物:{}", name);
        List<CharacterBasicInfo> list = characterBasicInfoService.getListByName(name);
        return Result.OK(list);
    }

    /**
     * 新增人物
     */
    @PostMapping("/add")
    public Result<?> add(@RequestBody CharacterBasicInfo characterBasicInfo) {
        log.info("新增人物:{}", characterBasicInfo);
        Result<?> add = characterBasicInfoService.add(characterBasicInfo);
        characterBasicInfoService.syncToCaoji(characterBasicInfo);
        return add;
    }

    /**
     * 修改人物
     */
    @PostMapping("/edit")
    public Result<?> edit(@RequestBody CharacterBasicInfo characterBasicInfo) {
        log.info("新增人物:{}", characterBasicInfo);
        String id = characterBasicInfo.getId();
        if (id == null || id.length() == 0){
            return Result.error("人物id不能为空");
        }
        Result<?> edit = characterBasicInfoService.edit(characterBasicInfo);
        characterBasicInfoService.syncToCaoji(characterBasicInfo);
        return edit;
    }

    /**
     * 根据id删除人物
     *
     * @param id
     * @return
     */
    @GetMapping("/del")
    public Result<?> del(@RequestParam(name = "id", required = true) String id) {
        log.info("根据id删除人物:{}", id);
        CharacterBasicInfo byId = characterBasicInfoService.getById(id);
        if(byId!=null){
            byId.setTakeOfficeStatus("历任");
            Boolean res = characterBasicInfoService.del(id);
            characterBasicInfoService.syncToCaoji(byId);
            return Result.OK(res);
        }
        return Result.error("人物不存在");
    }

    /**
     * 根据uid查询人物的历任职务
     *
     * @param uid
     * @return
     */
    @GetMapping("/getPreviousDuty")
    public Result<?> getPreviousDuty(@RequestParam(name = "uid") String uid,
                                     @RequestParam(name = "id") String id,@RequestParam(name = "category") String category) {
        log.info("根据uid查询人物的历任职务:{}", uid);
        List<CharacterBasicInfo> list = characterBasicInfoService.getPreviousDuty(uid,id,category);
        return Result.OK(list);
    }

    /**
     * 关联
     *
     * @param characterBasicInfo
     * @return
     */
    @PostMapping("/correlation")
    public Result<?> correlation(@RequestBody CharacterBasicInfo characterBasicInfo) {
        log.info("关联:{}", characterBasicInfo.getId());
        Result<?> result = characterBasicInfoService.correlation(characterBasicInfo);
        characterBasicInfoService.syncToCaoji(characterBasicInfo);
        return result;
    }

    /**
     * 修改关联人物
     *
     * @param characterBasicInfo
     * @return
     */
    @PostMapping("/correlationEdit")
    public Result<?> correlationEdit(@RequestBody CharacterBasicInfo characterBasicInfo) {
        log.info("修改关联人物:{}", characterBasicInfo.getId());
        Result<?> edit = characterBasicInfoService.correlationEdit(characterBasicInfo);
        characterBasicInfoService.syncToCaoji(characterBasicInfo);
        return edit;
    }

    /**
     * 关联人物合并
     * @param checkVo
     * @return
     */
    @PostMapping("/correlationMerge")
    public Result<?> correlationMerge(@RequestBody CheckVo checkVo) {
        log.info("关联人物合并:{}", checkVo.getSuspectedId());
        return characterBasicInfoService.correlationMerge(checkVo);
    }

    /**
     * 企业查询
     *
     * @param socialCreditCode 社会信用代码
     * @param department
     * @return
     */
    @GetMapping("/getDepartment")
    public Result<?> getDepartment(@RequestParam(name = "socialCreditCode", required = false) String socialCreditCode,
                                   @RequestParam(name = "department", required = false) String department,
                                   @RequestParam(name = "keyword", required = false) String keyword) {
        log.info("查询企业名称:{}---{}---{}", socialCreditCode, department, keyword);
        List<Map<String, String>> list = characterBasicInfoService.getDepartment(socialCreditCode, department, keyword);
        return Result.OK(list);
    }

    /**
     * @param basicInfoListVo 条件封装
     * @描述 查询专家列表
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/listAll")
    public Result<?> listAll(@RequestBody BasicInfoListVo basicInfoListVo) {
        //类别过滤为专家
        basicInfoListVo.setCategory("2");
        basicInfoListVo.setColumn(SortUtil.humpToLine(basicInfoListVo.getColumn()));
        List<CharacterBasicInfo> list = characterBasicInfoService.getAllList(basicInfoListVo);
        return Result.OK(list);
    }

    /**
     * @param basicInfoListVo 条件封装
     * @描述 查询专家列表（分页，产业链关联专家时查询用）
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/pageListAll")
    public Result<?> pageListAll(@RequestBody BasicInfoListVo basicInfoListVo) {
        //类别过滤为专家
        basicInfoListVo.setCategory("2");
        Integer office = basicInfoListVo.getPageSize() * (basicInfoListVo.getPageNo() - 1);
        basicInfoListVo.setOffset(office);
        basicInfoListVo.setColumn(SortUtil.humpToLine(basicInfoListVo.getColumn()));
        IPage page = characterBasicInfoService.getList(basicInfoListVo);
        return Result.OK(page);
    }

    @PostMapping("/importExcel")
    public Result<?> importExcel(@RequestParam(value = "file", required = false) MultipartFile file, @RequestParam String category) {
        if (file.isEmpty()) {
            return Result.error("文件为空");
        }
        File tempFile = null;
        // 获取文件的二进制数据
        try (InputStream inputStream = file.getInputStream()) {
            tempFile = new File(file.getOriginalFilename());
            Files.copy(inputStream, Paths.get(tempFile.getAbsolutePath()), StandardCopyOption.REPLACE_EXISTING);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        //category 分类类型(1-领导人、2-专家、3-企业高管)
        // 根据 category 选择对应的 importFieldMap
        Map<String, String> importFieldMap = getImportFieldMap(category);
        String result = characterBasicInfoService.importExcel(tempFile.getAbsolutePath(), category, importFieldMap);
        return Result.OK(result);
    }

    @PostMapping("/exportExcel")
    public ResponseEntity<byte[]> exportExcel(@RequestBody BasicInfoListVo basicInfoListVo) {
        String category = basicInfoListVo.getCategory();
        basicInfoListVo.setColumn(SortUtil.humpToLine(basicInfoListVo.getColumn()));
        List<CharacterBasicInfo> list = characterBasicInfoService.getAllList(basicInfoListVo);
        Map<String, String> importFieldMap = getImportFieldMap(category);
        ResponseEntity<byte[]> responseEntity;
        try {
            responseEntity = characterBasicInfoService.exportExcel(category,list,importFieldMap);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return responseEntity;
    }

    @GetMapping("/exportExcelByIds")
    public ResponseEntity<byte[]> exportExcelByIds(@RequestParam String ids,@RequestParam String category) {
        Map<String, String> importFieldMap = getImportFieldMap(category);
        ResponseEntity<byte[]> responseEntity;
        try {
            responseEntity = characterBasicInfoService.exportExcelByIds(category,ids,importFieldMap);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return responseEntity;
    }

    @GetMapping("/exportTemplate")
    public ResponseEntity<byte[]> exportTemplate(@RequestParam String category) {
        Map<String, String> importFieldMap = getImportFieldMap(category);
        ResponseEntity<byte[]> responseEntity;
        try {
            responseEntity = characterBasicInfoService.exportTemplate(importFieldMap);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return responseEntity;
    }

    private Map<String, String> getImportFieldMap(String category) {
        Map<String, String> importFieldMap;
        switch (category) {
            case "1":
                importFieldMap = importFieldMap1;
                break;
            case "2":
                importFieldMap = importFieldMap2;
                break;
            case "3":
                importFieldMap = importFieldMap3;
                break;
            default:
                throw new IllegalArgumentException("无效的 category 值: " + category);
        }
        return importFieldMap;
    }

    static {
        importFieldMap1 = new LinkedHashMap<>();
        importFieldMap1.put("人物id", "uid");
        importFieldMap1.put("姓名", "name");
        importFieldMap1.put("性别", "sex");
        importFieldMap1.put("出生年月", "birthday");
        importFieldMap1.put("民族", "nation");
        importFieldMap1.put("籍贯", "nativePlace");
        importFieldMap1.put("毕业院校", "school");
        importFieldMap1.put("学历", "education");
        importFieldMap1.put("单位名称", "department");
        importFieldMap1.put("职务", "duty");
        importFieldMap1.put("任职开始时间", "takeOfficeTime");
        importFieldMap1.put("任职结束时间", "takeOfficeTimeEnd");
        importFieldMap1.put("分类", "typeId");
    }

    static {
        importFieldMap2 = new LinkedHashMap<>();
        importFieldMap2.put("人物id", "uid");
        importFieldMap2.put("姓名", "name");
        importFieldMap2.put("性别", "sex");
        importFieldMap2.put("出生年月", "birthday");
        importFieldMap2.put("民族", "nation");
        importFieldMap2.put("籍贯", "nativePlace");
        importFieldMap2.put("毕业院校", "school");
        importFieldMap2.put("毕业专业", "schoolSpeciality");
        importFieldMap2.put("学历", "education");
        importFieldMap2.put("单位名称", "department");
        importFieldMap2.put("所在部门", "depart");
        importFieldMap2.put("职务", "duty");
        importFieldMap2.put("任职开始时间", "takeOfficeTime");
        importFieldMap2.put("任职结束时间", "takeOfficeTimeEnd");
        importFieldMap2.put("顶尖专家/领军专家", "expertType");
        importFieldMap2.put("研究领域", "researchField");
        importFieldMap2.put("专业方向", "speciality");
        importFieldMap2.put("已入选的人才计划", "talentPlanning");
        importFieldMap2.put("专业技术职称", "technicalTitles");
        importFieldMap2.put("专业技术职称取得时间", "technicalDate");
        importFieldMap2.put("参加工作时间", "workDate");
        importFieldMap2.put("分类", "typeId");
    }

    static {
        importFieldMap3 = new LinkedHashMap<>();
        importFieldMap3.put("人物id", "uid");
        importFieldMap3.put("姓名", "name");
        importFieldMap3.put("性别", "sex");
        importFieldMap3.put("出生年月", "birthday");
        importFieldMap3.put("民族", "nation");
        importFieldMap3.put("籍贯", "nativePlace");
        importFieldMap3.put("毕业院校", "school");
        importFieldMap3.put("学历", "education");
        importFieldMap3.put("单位名称", "department");
        importFieldMap3.put("职务", "duty");
        importFieldMap3.put("任职开始时间", "takeOfficeTime");
        importFieldMap3.put("任职结束时间", "takeOfficeTimeEnd");
        importFieldMap3.put("分类", "typeId");
    }
}

