package com.zzsn.knowbase.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.knowbase.entity.KbAuthorizedUser;
import com.zzsn.knowbase.entity.KnowFile;
import com.zzsn.knowbase.entity.Knowledge;
import com.zzsn.knowbase.entity.KnowledgeExcel;
import com.zzsn.knowbase.service.IKnowledgeService;
import com.zzsn.knowbase.service.ILocalFileService;
import com.zzsn.knowbase.util.*;
import com.zzsn.knowbase.vo.IntelligentQaParam;
import com.zzsn.knowbase.vo.KnowledgeParam;
import com.zzsn.knowbase.vo.KnowledgeVO;
import com.zzsn.knowbase.vo.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

/**
 * @Description: 知识
 * @Author: chenshiqiang
 * @Version: V1.0
 */
@Api(tags = "知识")
@RestController
@RequestMapping("/api/knowledge")
@Slf4j
public class KnowledgeController {
    @Autowired
    private IKnowledgeService knowledgeService;

    @Value("${python.intelligentQaUrl:}")
    private String intelligentQaUrl;

    @Autowired
    private ILocalFileService localFileService;

    @Autowired
    private RedisUtil redisUtil;

    /**
     * 分页列表查询
     */
    @GetMapping(value = "/list")
    public Result<?> queryPageList(KnowledgeParam knowledgeParam,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                   @RequestParam(name = "column", defaultValue = "common") String column,
                                   @RequestParam(name = "order", defaultValue = "desc") String order) {
        if (null != knowledgeParam.getSearchInfo() && Integer.valueOf("2").equals(knowledgeParam.getSearchScope())) {
            IPage<KnowledgeVO> pageList = knowledgeService.listFromPython(knowledgeParam, pageNo, pageSize, column, order);
            return Result.OK(pageList);
        }
        IPage<KnowledgeVO> pageList = knowledgeService.queryPageList(knowledgeParam, pageNo, pageSize, column, order);
        return Result.OK(pageList);
    }

    /**
     * 详情
     */
    @GetMapping(value = "/getById")
    public Result<?> getById(@RequestParam String id) {
        log.info("{}===query begin", new Date());
        KnowledgeVO knowledgeVO = knowledgeService.getById(id);
        if (null != knowledgeVO.getPublishDate() && (!knowledgeVO.getPublishDate().isEmpty())) {
            knowledgeVO.setPublishDate(EsDateUtil.esFieldDateMapping(knowledgeVO.getPublishDate()));
        }
        if (null != knowledgeVO.getVerifyTime() && (!knowledgeVO.getVerifyTime().isEmpty())) {
            knowledgeVO.setVerifyTime(EsDateUtil.esFieldDateMapping(knowledgeVO.getVerifyTime()));
        }
        return Result.OK(knowledgeVO);
    }


    @GetMapping(value = "/listFromPython")
    public Result<?> listFromPython(KnowledgeParam knowledgeParam,
                                    @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                    @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                    @RequestParam(name = "column", defaultValue = "common") String column,
                                    @RequestParam(name = "order", defaultValue = "desc") String order) {
        IPage<KnowledgeVO> pageList = knowledgeService.listFromPython(knowledgeParam, pageNo, pageSize, column, order);
        return Result.OK(pageList);
    }

    /**
     * 添加
     *
     * @param knowledge
     * @return
     */
    @ApiOperation(value = "知识-添加", notes = "知识-添加")
    @PostMapping(value = "/uploadKnowledge")
    public Result<?> uploadKnowledge(HttpServletRequest request, Knowledge knowledge) {
        MultipartHttpServletRequest multipartRequest = WebUtils.getNativeRequest(request, MultipartHttpServletRequest.class);

        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        Result<List<KnowFile>> result = localFileService.upload(fileMap);
        log.info("upload result:{}", result);
        if (!Integer.valueOf("200").equals(result.getCode())) {
            return Result.error(result.getMessage());
        }
        List<KnowFile> resultList = result.getResult();
        KbAuthorizedUser userInfo = SpringContextUtils.getUserInfo();
        knowledge.setImportData(0);
        CompletableFuture.runAsync(() -> this.add(resultList, knowledge, userInfo));
        return Result.OK("添加成功！");
    }

    private void add(List<KnowFile> resultList, Knowledge knowledge, KbAuthorizedUser userInfo) {

        if (null == resultList || resultList.isEmpty()) {
            return;
        }
        for (KnowFile knowFileResult : resultList) {
            knowledge.setId(null);
            knowledge.setTitle(null);
            knowledgeService.addKnowledge(knowFileResult, knowledge, userInfo);
        }
    }

    /**
     * 编辑
     *
     * @param knowledge
     * @return
     */
    @ApiOperation(value = "信息源组类别-编辑", notes = "信息源组类别-编辑")
    @PutMapping(value = "/edit")
    public Result<?> edit(@RequestBody Knowledge knowledge) {
        knowledgeService.updateKnowledge(knowledge);
        return Result.OK("编辑成功!");
    }

    /**
     * 通过id删除
     *
     * @param
     * @return
     */
    @PostMapping(value = "/delete")
    public Result<?> delete(@RequestBody DeleteParam deleteParam) {
        knowledgeService.deleteKnowledge(deleteParam.getKnowledgeList());
        return Result.OK("删除成功!");
    }

    @GetMapping(value = "/parse")
    public Result<?> parse(@RequestParam(name = "filePath") String filePath) throws IOException {
        String s = DocUtil.docParseHtml(filePath);
        return Result.OK(s);
    }

    /**
     * 智能问答
     *
     * @param intelligentQaParam
     * @return
     */
    @PostMapping(value = "/IntelligentQa")
    public Result<?> intelligentQa(@RequestBody IntelligentQaParam intelligentQaParam) throws IOException {
        JSONObject params = new JSONObject();
        params.put("score_threshold", intelligentQaParam.getScoreThreshold());
        params.put("question", intelligentQaParam.getQuestion());
        params.put("knowledge_base_id", intelligentQaParam.getKbKnowledgeIds().split(","));
        params.put("size", intelligentQaParam.getSize());
        String result = HttpUtil.doPost(intelligentQaUrl, params, 120000);
        if (!result.isEmpty()) {
            JSONObject jsonObject = JSON.parseObject(result);
            return Result.OK(jsonObject.get("result"));
        }
        return Result.error("响应失败");
    }

    /**
     * 通过excel导入
     *
     * @return
     */
    @RequestMapping(value = "/importInfo", method = RequestMethod.POST)
    public Result<?> importInfo(HttpServletRequest request, HttpServletResponse response) {
        KbAuthorizedUser userInfo = SpringContextUtils.getUserInfo();
        assert userInfo != null;
        return knowledgeService.doImportInfo(request, userInfo.getId());

    }

    /**
     * 下载导入模板
     */
    @RequestMapping(value = "/download")
    public Result<?> download(HttpServletRequest request, HttpServletResponse response) {
        try {
            File fileDir = ExcelExportUtil.getFileDir();
            List<KnowledgeExcel> dataList = new ArrayList<>();
            String filePath = fileDir.getAbsolutePath() + File.separator + "info.xlsx";
            ExcelExportUtil.writeExcelFront(dataList, "知识库列表导入模板", filePath, KnowledgeExcel.class);
            ExcelExportUtil.download(response, filePath, true);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Result.OK();
    }

    /*
     * 导出excel（专题-已采集）
     *
     * @param request
     * @param
     * @update by wp
     * 异步调用
     *
     */
    @PostMapping(value = "/exportCollectionXls")
    public Result<?> exportCollectionXls(HttpServletRequest request, HttpServletResponse response, KnowledgeVO knowledgeVO,
                                         String ids,
                                         @RequestParam(name = "isAll", defaultValue = "0") String isAll) {
        //获取数据
        if (redisUtil.get(ids) == null) {
            try {
                String[] arr = new String[]{"标题", "正文", "作者", "来源", "发布时间"};
                List<KnowledgeVO> informationList = knowledgeService.listByIds(ids);
                if (CollectionUtils.isNotEmpty(informationList)) {
                    //动态补充表头
                    List<String> headers = Arrays.asList(arr);
                    //每个sheet页最多5000条数据
                    List<List<KnowledgeVO>> partition = ListUtils.partition(informationList, 5000);
                    XSSFWorkbook workbook = new XSSFWorkbook();
                    for (int i = 0; i < partition.size(); i++) {
                        List<List<String>> rows = formatData(partition.get(i));
                        ExcelExportUtil.exportExcelData(workbook, i, headers, rows, "sheet" + (i + 1));
                    }
                    File fileDir = ExcelExportUtil.getFileDir();
                    String filePath = fileDir.getAbsolutePath() + File.separator + UUID.randomUUID().toString().replace("-","") + "info.xlsx";
                    FileOutputStream outputStream = new FileOutputStream(filePath);
                    workbook.write(outputStream);
                    redisUtil.set(ids, filePath, 3600*24);
                }
            } catch (Exception e) {
                e.printStackTrace();
                return Result.OK(false);
            }
        }
        return Result.OK(true);
    }

    /*
     * 导出excel（专题-已入库）
     *
     * @param request
     * @param
     * @update by wp
     * 异步调用
     *
     */

    @RequestMapping(value = "/downloadXls")
    public void downloadXls(HttpServletResponse response,String ids) {
        //获取下载路径
        Object object = redisUtil.get(ids);
        if (object != null) {
            String filePath = object.toString();
            try {
                redisUtil.del(ids);
                ExcelExportUtil.download(response, filePath, true);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private List<List<String>> formatData(List<KnowledgeVO> knowledgeVOS) {
        List<List<String>> list = new ArrayList<>(knowledgeVOS.size());
        for (KnowledgeVO knowledgeVO : knowledgeVOS) {
            List<String> innerList=new ArrayList<>();
            innerList.add(knowledgeVO.getTitle());
            innerList.add(knowledgeVO.getContentAll());
            innerList.add(knowledgeVO.getAuthor());
            innerList.add(knowledgeVO.getOrigin());
            innerList.add(knowledgeVO.getPublishDate());
            list.add(innerList);
        }
        return list;
    }
}

@Data
class DeleteParam {
    private List<Knowledge> knowledgeList;
}