package com.zzsn.knowbase.util;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Snowflake;
import io.minio.*;
import io.minio.http.Method;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.URLDecoder;
import java.util.Date;

/**
 * minio文件上传工具类
 */
@Slf4j
public class MinioUtil {
    private static String minioUrl;
    private static String minioName;
    private static String minioPass;
    private static String bucketName;

    public static void setMinioUrl(String minioUrl) {
        MinioUtil.minioUrl = minioUrl;
    }

    public static void setMinioName(String minioName) {
        MinioUtil.minioName = minioName;
    }

    public static void setMinioPass(String minioPass) {
        MinioUtil.minioPass = minioPass;
    }

    public static void setBucketName(String bucketName) {
        MinioUtil.bucketName = bucketName;
    }

    public static String getMinioUrl() {
        return minioUrl;
    }

    public static String getBucketName() {
        return bucketName;
    }

    private static MinioClient minioClient = null;

    /**
     * 上传文件
     *
     * @param file         文件
     * @param bizPath      文件夹名称
     * @param customBucket 自定义桶的名称
     * @author lkg
     * @date 2023/5/22
     */
    public static String upload(MultipartFile file, String bizPath, String customBucket) {
        String file_url = "";
        //过滤上传文件夹名特殊字符，防止攻击
        bizPath = StrAttackFilter.filter(bizPath);
        String newBucket = bucketName;
        if (StringUtils.isNotEmpty(customBucket)) {
            newBucket = customBucket;
        }
        try {
            initMinio(minioUrl, minioName, minioPass);
            // 检查存储桶是否已经存在
            if (minioClient.bucketExists(BucketExistsArgs.builder().bucket(newBucket).build())) {
                log.info("Bucket already exists.");
            } else {
                // 创建一个名为ota的存储桶
                minioClient.makeBucket(MakeBucketArgs.builder().bucket(newBucket).build());
                log.info("create a new bucket.");
            }
            InputStream stream = file.getInputStream();
            // 获取文件名
            String orgName = file.getOriginalFilename();
            if (StringUtils.isEmpty(orgName)) {
                orgName = file.getName();
            }
            orgName = getFileName(orgName);
            String objectName = bizPath + "/" + orgName.substring(0, orgName.lastIndexOf(".")) + "_" + System.currentTimeMillis() + orgName.substring(orgName.lastIndexOf("."));
            // 使用putObject上传一个本地文件到存储桶中。
            if (objectName.startsWith("/")) {
                objectName = objectName.substring(1);
            }
            PutObjectArgs objectArgs = PutObjectArgs.builder().object(objectName)
                    .bucket(newBucket)
                    .contentType("application/octet-stream")
                    .stream(stream, stream.available(), -1).build();
            minioClient.putObject(objectArgs);
            stream.close();
            file_url = newBucket + "/" + objectName;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return file_url;
    }

    /**
     * 文件上传
     *
     * @param file    文件
     * @param bizPath 文件夹名称
     * @return
     */
    public static String upload(MultipartFile file, String bizPath) {
        return upload(file, bizPath, null);
    }

    /**
     * 获取文件流
     *
     * @param bucketName 桶名称
     * @param objectName 文件路径：文件夹/文件   例如：test/test.docx、test/123/1234.docx
     * @return
     */
    public static InputStream getMinioFile(String bucketName, String objectName) {
        InputStream inputStream = null;
        try {
            initMinio(minioUrl, minioName, minioPass);
            GetObjectArgs objectArgs = GetObjectArgs.builder().object(objectName)
                    .bucket(bucketName).build();
            inputStream = minioClient.getObject(objectArgs);
        } catch (Exception e) {
            log.info("文件获取失败" + e.getMessage());
        }
        return inputStream;
    }

    /**
     * 删除文件
     *
     * @param bucketName 桶名称
     * @param objectName 文件路径  格式： 文件夹/文件   例如：test/test.docx、test/123/1234.docx
     * @throws Exception
     */
    public static void removeObject(String bucketName, String objectName) {
        try {
            initMinio(minioUrl, minioName, minioPass);
            RemoveObjectArgs objectArgs = RemoveObjectArgs.builder().object(objectName)
                    .bucket(bucketName).build();
            minioClient.removeObject(objectArgs);
        } catch (Exception e) {
            log.info("文件删除失败" + e.getMessage());
        }
    }

    /**
     * 获取文件外链/预览链接(仅支持pdf文件和图片)
     *
     * @param bucketName 桶名称
     * @param objectName 文件路径 格式：文件夹/文件   例如：test/test.docx、test/123/1234.docx
     * @param expires    失效时长
     * @return
     */
    public static String getPreviewUrl(String bucketName, String objectName, Integer expires) {
        initMinio(minioUrl, minioName, minioPass);
        try {
            GetPresignedObjectUrlArgs objectArgs;
            if (expires != null) {
                objectArgs = GetPresignedObjectUrlArgs.builder()
                        .object(objectName)
                        .bucket(bucketName)
                        .expiry(expires)
                        .method(Method.GET)
                        .build();
            } else {
                objectArgs = GetPresignedObjectUrlArgs.builder()
                        .object(objectName)
                        .bucket(bucketName)
                        .method(Method.GET)
                        .build();
            }
            String url = minioClient.getPresignedObjectUrl(objectArgs);
            return URLDecoder.decode(url, "UTF-8");
        } catch (Exception e) {
            log.info("文件路径获取失败" + e.getMessage());
        }
        return null;
    }

    /**
     * 初始化客户端
     *
     * @param minioUrl  地址
     * @param minioName 账号
     * @param minioPass 密码
     */
    private static void initMinio(String minioUrl, String minioName, String minioPass) {
        if (minioClient == null) {
            try {
                minioClient = MinioClient.builder()
                        .endpoint(minioUrl)
                        .credentials(minioName, minioPass)
                        .build();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 上传文件到minio
     *
     * @param stream       文件流
     * @param relativePath 文件路径
     */
    public static String upload(InputStream stream, String relativePath) throws Exception {
        initMinio(minioUrl, minioName, minioPass);
        if (minioClient.bucketExists(BucketExistsArgs.builder().bucket(bucketName).build())) {
            log.info("Bucket already exists.");
        } else {
            // 创建一个名为ota的存储桶
            minioClient.makeBucket(MakeBucketArgs.builder().bucket(bucketName).build());
            log.info("create a new bucket.");
        }
        PutObjectArgs objectArgs = PutObjectArgs.builder().object(relativePath)
                .bucket(bucketName)
                .contentType("application/octet-stream")
                .stream(stream, stream.available(), -1).build();
        minioClient.putObject(objectArgs);
        stream.close();
        return "/" + bucketName + "/" + relativePath;
    }


    /**
     * 上传文件
     *
     * @param file         文件
     * @param contentType  类型，如果pdf文件或者图片，需要预览功能的话，必填.示例：pdf application/pdf；图片  image/png；image/jpeg
     * @param bizPath      文件夹名称
     * @param customBucket 自定义桶的名称
     * @author lkg
     * @date 2023/5/22
     */
    public static String upload(MultipartFile file, String contentType, String bizPath, String customBucket) {
        String fileUrl = null;
        try {
            if (StringUtils.isEmpty(bizPath)) {
                bizPath = "report";
            } else {
                //过滤上传文件夹名特殊字符，防止攻击
                bizPath = StrAttackFilter.filter(bizPath);
            }
            InputStream stream = file.getInputStream();
            // 获取文件名
            String orgName = file.getOriginalFilename();
            if (StringUtils.isEmpty(orgName)) {
                orgName = file.getName();
            }
            String objectName = bizPath + "/" + orgName;
            // 使用putObject上传一个本地文件到存储桶中。
            if (objectName.startsWith("/")) {
                objectName = objectName.substring(1);
            }
            fileUrl = uploadByStream(stream,contentType, customBucket,objectName);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return fileUrl;
    }


    /**
     * 通过文件流上传文件
     *
     * @param stream       文件流
     * @param contentType  类型，如果pdf文件或者图片，需要预览功能的话，必填.示例：pdf application/pdf；图片  image/png；image/jpeg
     * @param customBucket 自定义桶名称
     * @param relativePath 文件路径
     * @author lkg
     * @date 2023/5/22
     */
    public static String uploadByStream(InputStream stream, String contentType, String customBucket, String relativePath) throws Exception {
        initMinio(minioUrl, minioName, minioPass);
        if (StringUtils.isEmpty(customBucket)) {
            customBucket = bucketName;
        }
        if (!minioClient.bucketExists(BucketExistsArgs.builder().bucket(customBucket).build())) {
            minioClient.makeBucket(MakeBucketArgs.builder().bucket(customBucket).build());
        }
        if (StringUtils.isEmpty(contentType)) {
            contentType = "application/octet-stream";
        }
        int index = relativePath.lastIndexOf(".");
        relativePath = relativePath.substring(0,index) + "_" + System.currentTimeMillis() + relativePath.substring(index);
        if (relativePath.startsWith("/")) {
            relativePath = relativePath.substring(1);
        }
        PutObjectArgs objectArgs = PutObjectArgs.builder().object(relativePath)
                .bucket(customBucket)
                .contentType(contentType)
                .stream(stream, stream.available(), -1).build();
        minioClient.putObject(objectArgs);
        stream.close();
        return "/" + customBucket + "/" + relativePath;
    }

    /**
     * 通过字节上传文件
     *
     * @param bytes        字节
     * @param contentType  类型，如果pdf文件或者图片，需要预览功能的话，必填.示例：pdf application/pdf；图片  image/png；image/jpeg
     * @param customBucket 自定义桶名称
     * @param relativePath 文件路径
     * @author lkg
     * @date 2023/5/22
     */
    public static String uploadByByte(byte[] bytes, String contentType, String customBucket, String relativePath) throws Exception {
        ByteArrayInputStream stream = new ByteArrayInputStream(bytes);
        return uploadByStream(stream, contentType, customBucket, relativePath);
    }

    /**
     * 获取永久预览链接
     *
     * @param filePath 文件路径(带桶) 桶/文件夹/文件名称  例如 jxyq/report/会议纪要.pdf
     * @author lkg
     * @date 2023/5/23
     */
    public static String foreverPreviewUrl(String filePath){
        int index = filePath.indexOf("/");
        String bucketName = filePath.substring(0,index);
        String filename = filePath.substring(index);
        return getPreviewUrl(bucketName,filename,null);
    }

    /**
     * 根据文件名生成文件存储路径(yyyy/yyyyMMdd/id)
     * @param fileName
     * @return
     */
    public static String getRelativePath(String fileName) {
        if(fileName == null || StringUtils.isEmpty(fileName) ){
            return null;
        }
        String suffix = fileName;
        if(fileName.lastIndexOf(".") > -1){
            suffix = fileName.substring(fileName.lastIndexOf(".") + 1);
        }
        String date = DateUtil.formatDate(new Date()).replaceAll("-", "");
        return date.substring(0,4)+"/"+date+"/"+ new Snowflake(1L,1L).nextIdStr()+"."+suffix;
    }
    /**
     * 企业logo存储路径获取
     * @param fileName
     * @return
     */
    public static String getEnterpriseLogoPath(String fileName) {
        if(fileName == null || StringUtils.isEmpty(fileName) ){
            return null;
        }
        String suffix = fileName;
        if(fileName.lastIndexOf(".") > -1){
            suffix = fileName.substring(fileName.lastIndexOf(".") + 1);
        }
        return "logo/enterprise/"+ new Snowflake(1L,1L).nextIdStr()+"."+suffix;
    }
    public static String getFileName(String fileName){
        //判断是否带有盘符信息
        // Check for Unix-style path
        int unixSep = fileName.lastIndexOf('/');
        // Check for Windows-style path
        int winSep = fileName.lastIndexOf('\\');
        // Cut off at latest possible point
        int pos = (winSep > unixSep ? winSep : unixSep);
        if (pos != -1)  {
            // Any sort of path separator found...
            fileName = fileName.substring(pos + 1);
        }
        //替换上传文件名字的特殊字符
        fileName = fileName.replace("=","").replace(",","").replace("&","").replace("#", "");
        //替换上传文件名字中的空格
        fileName=fileName.replaceAll("\\s","");
        return fileName;
    }
}
