package com.zzsn.event.controller.common;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.EventAnalysisVersionRecord;
import com.zzsn.event.entity.EventLlmConfig;
import com.zzsn.event.enums.AnalysisColumnEnum;
import com.zzsn.event.service.AnalysisService;
import com.zzsn.event.service.EventAnalysisVersionRecordService;
import com.zzsn.event.service.EventLlmConfigService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.vo.EventLlmConfigModifyVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Objects;

/**
 * 大模型配置
 *
 * @author lkg
 * @date 2025/7/28
 */
@Slf4j
@RestController
@RequestMapping("/llm/config")
public class LLmConfigController {

    @Autowired
    private EventLlmConfigService llmConfigService;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private EventAnalysisVersionRecordService eventAnalysisVersionRecordService;
    @Autowired
    private IEventService eventService;

    /**
     * 事件下大模型配置信息
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/28
     */
    @GetMapping("/list")
    public Result<?> list(@RequestParam String eventId) {
        List<EventLlmConfig> configList = llmConfigService.getConfigList(eventId);
        return Result.OK(configList);
    }

    /**
     * 批量编辑事件的大模型配置信息
     *
     * @param eventLlmConfigList 大模型配置信息
     * @author lkg
     * @date 2025/7/28
     */
    @PostMapping("/modifyBatch")
    public Result<?> modifyBatch(@RequestBody List<EventLlmConfig> eventLlmConfigList) {
        llmConfigService.modifyBatch(eventLlmConfigList);
        return Result.OK();
    }

    /**
     * 编辑事件栏目的大模型配置信息,同时触发大模型更新结果
     *
     * @param eventLlmConfigModifyVO 大模型配置信息
     * @author lkg
     * @date 2025/7/28
     */
    @PostMapping("/modifySingle")
    public Result<?> modifySingle(@RequestBody EventLlmConfigModifyVO eventLlmConfigModifyVO) {
        String llmResult = null;
        String versionId = eventLlmConfigModifyVO.getVersionId();
        if (StringUtils.isEmpty(versionId)) {
            return Result.FAIL("请选择要修改的版本");
        }
        EventLlmConfig eventLlmConfig = new EventLlmConfig();
        BeanUtils.copyProperties(eventLlmConfigModifyVO, eventLlmConfig);
        llmConfigService.modifySingle(eventLlmConfig);
        Integer columnCode = eventLlmConfig.getColumnCode();
        AnalysisColumnEnum analysisColumnEnum = AnalysisColumnEnum.getByCode(columnCode);
        if (analysisColumnEnum != null) {
            String eventId = eventLlmConfig.getEventId();
            Event event = eventService.getById(eventId);
            String eventName = event.getEventName();
            StringBuilder content = new StringBuilder("事件标题；" + eventName + "\n事件摘要:" + event.getEventDescribe());
            if (StringUtils.isNotEmpty(event.getKeyIssue())) {
                content.append("\n关心问题:").append(event.getKeyIssue());
            }
            //调用大模型更细结果
            //应对建议需要把 驱动因素和影响评估作为输入
            specialProcess(content, versionId, columnCode);
            llmResult = analysisService.llmResult(eventId, content.toString(), null, null, columnCode);
            log.info("{}-事件分析【{}】重新生成逻辑完成。", eventName, analysisColumnEnum.getName());
        } else {
            log.info("未找到对应的事件栏目【{}】", columnCode);
        }
        //更新返回结果
        eventAnalysisVersionRecordService.modify(versionId, analysisColumnEnum, llmResult);
        return Result.OK(llmResult);
    }


    /**
     * 特殊处理流程
     * 应对建议需要把 驱动因素和影响评估作为输入
     *
     * @param content    输入内同
     * @param versionId  版本id
     * @param columnCode 栏目编码
     * @author lkg
     * @date 2025/9/17
     */
    private void specialProcess(StringBuilder content, String versionId, Integer columnCode) {
        if (!Objects.equals(columnCode, AnalysisColumnEnum.MEASURE_SUGGEST.getCode())) {
            return;
        }
        List<EventAnalysisVersionRecord> versionDataList = eventAnalysisVersionRecordService.getVersionDataList(versionId, AnalysisColumnEnum.IMPACT_ASSESSMENT.getCode(), AnalysisColumnEnum.DRIVING_FACTORS.getCode());
        if (CollectionUtils.isEmpty(versionDataList)) {
            return;
        }
        for (EventAnalysisVersionRecord record : versionDataList) {
            if (record.getColumnCode().equals(AnalysisColumnEnum.IMPACT_ASSESSMENT.getCode())) {
                String str = parseImpactAssessment(record.getReturnData());
                if (StringUtils.isEmpty(str)) {
                    continue;
                }
                content.append("\n影响评估:").append(str);
            } else if (record.getColumnCode().equals(AnalysisColumnEnum.DRIVING_FACTORS.getCode())) {
                String str = parseDrivingFactors(record.getReturnData());
                if (StringUtils.isEmpty(str)) {
                    continue;
                }
                content.append("\n驱动因素:").append(str);
            }
        }
    }

    /**
     * 作为应对建议的输入时，解析影响评估数据
     *
     * @param impactAssessment 影响评估数据
     * @author lkg
     * @date 2025/9/17
     */
    private String parseImpactAssessment(String impactAssessment) {
        String result = null;
        try {
            List<JSONObject> jsonObjects = JSON.parseArray(impactAssessment, JSONObject.class);
            for (JSONObject jsonObject : jsonObjects) {
                jsonObject.remove("impactDetail");
            }
            result = jsonObjects.toString();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * 作为应对建议的输入时，解析驱动因素数据
     *
     * @param drivingFactors 驱动因素数据
     * @author lkg
     * @date 2025/9/17
     */
    private String parseDrivingFactors(String drivingFactors) {
        String result = null;
        try {
            JSONObject jsonObject = JSONObject.parseObject(drivingFactors);
            result = jsonObject.get("drivingFactor").toString();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }
}
