package com.zzsn.event.service.impl;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.*;
import com.zzsn.event.enums.BindTypeEnum;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.es.EsService;
import com.zzsn.event.mapper.EventMapper;
import com.zzsn.event.service.*;
import com.zzsn.event.util.*;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.util.tree.TreeUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.entity.XxlJobInfo;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.ParsedStringTerms;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.ParsedTopHits;
import org.elasticsearch.search.aggregations.metrics.TopHitsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.jsoup.Jsoup;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.IOException;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @Description: 事件
 * @Author: jeecg-boot
 * @Date: 2024-03-14
 * @Version: V1.0
 */
@Slf4j
@Service
public class EventServiceImpl extends ServiceImpl<EventMapper, Event> implements IEventService {
    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private ISubjectTypeMapService iSubjectTypeMapService;
    @Autowired
    private ISubjectInfoSourceMapService iSubjectInfoSourceMapService;
    @Autowired
    private ISubjectKeywordsMapService iSubjectKeywordsMapService;
    @Autowired
    private IEventTagService eventTagService;
    @Autowired
    private EventRegionMapService eventRegionMapService;
    @Autowired
    private IProjectSubjectMapService iProjectSubjectMapService;
    @Autowired
    private ISubjectTypeService iSubjectTypeService;
    @Autowired
    private IEventCategoryService eventCategoryService;
    @Autowired
    private ISubjectSearchEnginesMapService subjectSearchEnginesMapService;
    @Autowired
    private ISubjectModelMapService subjectModelMapService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private EventAnalysisReportService eventAnalysisReportService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;
    @Autowired
    private CommonService commonService;
    @Resource
    private RestHighLevelClient client;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private EsService esService;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private PythonUtil pythonUtil;
    @Autowired
    private EsOpUtil esOpUtil;

    @Value("${kafka.topic.event.run:}")
    private String EVENT_MODEL_KAFKA_CHANNEL;

    @Override
    public List<SubjectKafkaVo> progressList(Date disableDate) {
        return this.baseMapper.processList(disableDate);
    }

    @Override
    public List<SubjectKafkaVo> eventSubjectList() {
        return this.baseMapper.eventSubjectList();
    }

    @Override
    public EventVO queryInfo(String eventId) {
        EventVO eventVO = baseMapper.queryInfo(eventId);
        String relationEvents = eventVO.getRelationEvents();
        if (null != relationEvents) {
            List<String> split = Arrays.asList(relationEvents.split(","));
            List<EventVO> relationEventList = this.eventList(split);
            eventVO.setRelatedEventList(relationEventList);
        }
        EventTag one = eventTagService.getOne(new LambdaQueryWrapper<EventTag>()
                .eq(EventTag::getEventId, eventId)
                .last(" limit 1"));
        eventVO.setEventTag(one);
        List<RegionVO> regionList = eventRegionMapService.regionList(eventId);
        eventVO.setRegionList(regionList);
        KeywordsVO keywordsVO = keyWordsService.keywordInfoByEventId(eventId);
        eventVO.setKeywordsVO(keywordsVO);
        return eventVO;
    }

    @Override
    public IPage<EventManageVO> pageList(String eventName, Integer eventType, Integer facePublic, Integer publishStatus, String startTime, String endTime, String order, String orderType, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        List<EventManageVO> pageList = baseMapper.pageList(eventName, eventType, facePublic, publishStatus, startTime, endTime, order, orderType, offset, pageSize);
        //获取总条数
        Integer count = baseMapper.totalCount(eventName, eventType, facePublic, publishStatus, startTime, endTime);
        IPage<EventManageVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public IPage<EventFrontVO> frontAllPageList(String projectId,String eventId, String eventName, String eventType, String labelField, String labelName, String order, String orderType, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<String> eventTypes = new ArrayList<>();
        if (StringUtils.isNotEmpty(eventType)) {
            List<Node> nodes = eventCategoryService.categoryList();
            eventTypes = TreeUtil.belowList(nodes, eventType, true);
        }
        //获取当前登录人信息
        UserVo currentUser = UserUtil.getLoginUser();
        List<EventFrontVO> pageList = baseMapper.frontAllPageList(projectId, currentUser.getUsername(), eventId,eventName, eventTypes, labelField, labelName, type, order, orderType, offset, pageSize);
        if (CollectionUtils.isNotEmpty(pageList)) {
            Map<String, String> map = getFirstMap(pageList);
            if (MapUtil.isNotEmpty(map)) {
                for (EventFrontVO eventFrontVO : pageList) {
                    //获取专题资讯的首发来源
                    String firstOrigin = map.get(eventFrontVO.getId());
                    eventFrontVO.setFirstOrigin(firstOrigin);
                }
            }
            for (EventFrontVO eventFrontVO : pageList) {
                String createBy = eventFrontVO.getCreateBy();
                eventFrontVO.setOwner(StringUtils.isNotBlank(createBy) && createBy.equals(currentUser.getUsername()));
            }
        }
        //获取总条数
        Integer count = baseMapper.frontAllCount(projectId, currentUser.getUsername(), eventId,eventName, eventTypes, labelField, labelName, type);
        IPage<EventFrontVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public IPage<EventFrontVO> frontOwnerPageList(String projectId, String eventName, String eventType, String labelField, String labelName, String order, String orderType, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<String> eventTypes = new ArrayList<>();
        if (StringUtils.isNotEmpty(eventType)) {
            List<Node> nodes = eventCategoryService.categoryList();
            eventTypes = TreeUtil.belowList(nodes, eventType, true);
        }
        //获取当前登录人信息
        UserVo currentUser = UserUtil.getLoginUser();
        List<EventFrontVO> pageList = baseMapper.frontOwnerPageList(projectId, currentUser.getUsername(), eventName, eventTypes, labelField, labelName, type, order, orderType, offset, pageSize);
        if (CollectionUtils.isNotEmpty(pageList)) {
            Map<String, String> map = getFirstMap(pageList);
            if (MapUtil.isNotEmpty(map)) {
                for (EventFrontVO eventFrontVO : pageList) {
                    //获取专题资讯的首发来源
                    String firstOrigin = map.get(eventFrontVO.getId());
                    eventFrontVO.setFirstOrigin(firstOrigin);
                }
            }
        }
        //获取总条数
        Integer count = baseMapper.frontOwnerCount(projectId, currentUser.getUsername(), eventName, eventTypes, labelField, labelName, type);
        IPage<EventFrontVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public Page<EventNewPlatVO> newPlatPageList(SubjectCondition subjectCondition, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        //查询类别id的所有明细id
        List<String> typeIds = new ArrayList<>();
        String subjectTypeId = subjectCondition.getSubjectTypeId();
        if (StringUtils.isNotEmpty(subjectTypeId) && !"0".equals(subjectTypeId)) {
            typeIds = iSubjectTypeService.belowIdList(subjectTypeId, 2);
        }
        subjectCondition.setTypeIds(typeIds);
        //在根据所有明细节点查出专题列表
        List<EventNewPlatVO> pageList = baseMapper.newPlatPageList(subjectCondition, offset, pageSize);
        long t1 = System.currentTimeMillis();
        count(pageList);
        log.info("数量统计总共耗时====" + (System.currentTimeMillis() - t1));
        //获取总条数
        Integer count = baseMapper.newPlatCount(subjectCondition);
        Page<EventNewPlatVO> pageData = new Page<>(pageNo, pageSize,count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public IPage<EventNewPlatVO> newPlatCustomerPageList(SubjectCondition subjectCondition, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        List<EventNewPlatVO> pageList = baseMapper.newPlatCustomerPageList(subjectCondition, offset, pageSize);
        count(pageList);
        Integer count = baseMapper.newPlatCustomerCount(subjectCondition);
        IPage<EventNewPlatVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public List<EventTopVO> topEventList(String projectId, String createBy,String startTime, String endTime, Integer type, Integer pageSize) {
        return baseMapper.topEventList(projectId,createBy,startTime, endTime, type, 0, pageSize);
    }

    @Override
    public List<EventRegionVO> listByRegion(String projectId, String createBy,Integer type, String name) {
        return baseMapper.listByRegion(projectId,createBy,type, name);
    }

    private Map<String, String> getFirstMap(List<EventFrontVO> pageList) {
        Map<String, String> map = new HashMap<>();
        List<String> eventIdList = new ArrayList<>();
        pageList.forEach(e -> eventIdList.add(e.getId()));
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        //根据设备查询设备的相关信息
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.from(0);
        searchSourceBuilder.size(0);
        searchSourceBuilder.sort("publishDate", SortOrder.ASC);
        BoolQueryBuilder boolQueryBuilder = new BoolQueryBuilder();
        boolQueryBuilder.must(QueryBuilders.termsQuery("subjectId.keyword", eventIdList));
        //聚合搜索
        TermsAggregationBuilder one = AggregationBuilders.terms("one").field("subjectId.keyword").size(eventIdList.size());
        //ES分组取每组第一条Java写法
        TopHitsAggregationBuilder topHitsAggregationBuilder = AggregationBuilders.topHits("top_docs")
                .sort("publishDate", SortOrder.ASC)
                .size(1);
        one.subAggregation(topHitsAggregationBuilder);
        searchSourceBuilder.aggregation(one);
        searchSourceBuilder.query(boolQueryBuilder);
        searchRequest.source(searchSourceBuilder);
        // 查询ES
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            ParsedStringTerms oneTerms = searchResponse.getAggregations().get("one");
            List<? extends Terms.Bucket> onebuckets = oneTerms.getBuckets();
            for (Terms.Bucket threebucket : onebuckets) {
                ParsedTopHits topDetail = threebucket.getAggregations().get("top_docs");
                SearchHit[] hits = topDetail.getHits().getHits();
                Map<String, Object> latestDocument = hits[0].getSourceAsMap();
                map.put(latestDocument.get("subjectId").toString(), latestDocument.get("origin").toString());
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return map;
    }

    @Override
    @Transactional
    public Event saveMain(AddEventParam addEventParam) {
        Event event = new Event();
        BeanUtils.copyProperties(addEventParam, event);
        event.setStatus(1);
        Date endTime = event.getEndTime();
        if (endTime == null) {
            endTime = DateUtil.addDate(new Date(), 15);
        }
        event.setEndTime(endTime);
        //事件专题的默认分析规则参数-必填
        event.setIncreAnaRule(20);
        event.setTotalAnaRule(50);
        event.setTimeAnaRule(5);
        String subjectCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.EVENT_DEFAULT.getValue());
        event.setEventCode(subjectCode);
        //默认发布
        event.setPublishStatus(1);
        event.setPublishDate(DateUtil.dateToString(new Date()));
        event.setUnit("2");
        event.setSpace(1);
        String cron = CronUtil.generateCron("2", 1);
        event.setCron(cron);
        baseMapper.insert(event);
        String eventId = event.getId();
        CompletableFuture.runAsync(() -> {
            //插入专题-类别、项目的绑定关系
            saveMapMain(event, addEventParam);
            //事件标签
            eventTagService.save(EventTag.builder().eventId(eventId).build());
            //地缘关系绑定
            List<RegionVO> regionList = addEventParam.getRegionList();
            addRegionMap(eventId, regionList);
            //默认绑定搜索引擎、模型
            saveDefaultMap(eventId);
        });
        return event;
    }

    @Override
    @Transactional
    public void updateMain(AddEventParam addEventParam) {
        Event oldEvent = this.getById(addEventParam.getId());
        Event event = new Event();
        BeanUtils.copyProperties(addEventParam, event);
        String eventId = event.getId();
        baseMapper.updateById(event);
        CompletableFuture.runAsync(() -> {
            //专题-类别、项目的绑定关系处理
            //删除专题-类别绑定关系
            iSubjectTypeMapService.deleteBySubjectId(event.getId());
            //删除专题-项目的绑定关系
            iProjectSubjectMapService.deleteBySubjectId(event.getId());
            saveMapMain(event, addEventParam);
            //地域关系处理
            LambdaQueryWrapper<EventRegionMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(EventRegionMap::getEventId, eventId);
            eventRegionMapService.remove(queryWrapper);
            List<RegionVO> regionList = addEventParam.getRegionList();
            addRegionMap(eventId, regionList);
            //修改redis缓存，，用于向事件里补充数据
            updateRedisCache(event, oldEvent);
        });
    }

    @Override
    @Transactional
    public void deleteMain(String id) {
        baseMapper.deleteById(id);
        CompletableFuture.runAsync(() -> {
            //删除与信息源的关联关系
            iSubjectInfoSourceMapService.delete(id);
            //删除与关键词组的关联关系
            iSubjectKeywordsMapService.delete(id);
            //删除地域关系
            eventRegionMapService.remove(Wrappers.<EventRegionMap>lambdaQuery().eq(EventRegionMap::getEventId, id));
            //删除专题-类别绑定关系
            iSubjectTypeMapService.deleteBySubjectId(id);
            //删除专题-项目的绑定关系
            iProjectSubjectMapService.deleteBySubjectId(id);
            //删除事件标签
            eventTagService.remove(Wrappers.<EventTag>lambdaQuery().eq(EventTag::getEventId, id));
            //删除事件-模型关系
            subjectModelMapService.remove(Wrappers.<SubjectModelMap>lambdaQuery().eq(SubjectModelMap::getSubjectId, id));
            //删除事件-搜索引擎关系
            subjectSearchEnginesMapService.remove(Wrappers.<SubjectSearchEnginesMap>lambdaQuery().eq(SubjectSearchEnginesMap::getSubjectId, id));
            //删除事件 观点分析、事件脉络数据
            subjectAnalysisService.remove(Wrappers.<SubjectAnalysis>lambdaQuery().eq(SubjectAnalysis::getSubjectId,id));
            //删除事件 分析报告数据
            eventAnalysisReportService.delete(id);
        });
    }

    @Override
    public Boolean configVerification(AddEventParam addEventParam) {
        String eventId = addEventParam.getId();
        EventVO eventDetail = queryInfo(eventId);
        Date firstOpenTime = eventDetail.getFirstOpenTime();
        if (firstOpenTime == null) {
            return true;
        } else {
            //判断时间是否变化--只需判断开始时间，若范围缩小则校验不通过
            if (addEventParam.getStartTime().after(DateUtil.stringToDate(eventDetail.getStartTime(), "yyyy-MM-dd HH:mm:ss"))) {
                return false;
            }
            //判断关键词是否变化
            KeywordsVO keywordsOld = eventDetail.getKeywordsVO();
            KeywordsVO keywordsNew = addEventParam.getKeywordsVO();
            boolean keywordsChange = !keywordsOld.getKeyword().equals(keywordsNew.getKeyword());
            boolean exclusionWordChange;
            if (StringUtils.isEmpty(keywordsOld.getExclusionWord())) {
                exclusionWordChange = StringUtils.isNotEmpty(keywordsNew.getExclusionWord());
            } else {
                if (StringUtils.isEmpty(keywordsNew.getExclusionWord())) {
                    exclusionWordChange = true;
                } else {
                    exclusionWordChange = !keywordsNew.getExclusionWord().equals(keywordsOld.getExclusionWord());
                }
            }
            return !keywordsChange && !exclusionWordChange;
        }
    }

    @Override
    public void updateStatus(String eventId, Integer status) {
        Event event = this.getById(eventId);
        LambdaUpdateWrapper<Event> updateWrapper = Wrappers.lambdaUpdate();
        updateWrapper.set(Event::getStatus, status).eq(Event::getId, eventId);
        if (status == 1 && event.getFirstOpenTime() == null) {
            //判断是否第一次启用
            updateWrapper.set(Event::getFirstOpenTime, new Date());
        }
        this.update(updateWrapper);
        CompletableFuture.runAsync(() -> {
            iXxlJobInfoService.update(Wrappers.<XxlJobInfo>lambdaUpdate().eq(XxlJobInfo::getInfoSourceCode, event.getEventCode())
                    .set(XxlJobInfo::getTriggerStatus, status));
            //关键词
            KeywordsVO keywordsVO = keyWordsService.keywordInfoByEventId(eventId);
            iXxlJobInfoService.update(Wrappers.<XxlJobInfo>lambdaUpdate().eq(XxlJobInfo::getInfoSourceCode, keywordsVO.getWordsCode())
                    .set(XxlJobInfo::getTriggerStatus, status));
            if (1 == status) {
                kafkaTemplate.send(EVENT_MODEL_KAFKA_CHANNEL, event.getEventCode());
            }
        });
    }

    @Override
    public void clearSubjectData(String eventId) {
        LambdaUpdateWrapper<Event> updateWrapper = Wrappers.lambdaUpdate();
        updateWrapper.eq(Event::getId, eventId)
                .set(Event::getStatus, 0)
                .set(Event::getFirstOpenTime, null);
        this.update(updateWrapper);
        CompletableFuture.runAsync(() -> {
            //调用python接口
            pythonUtil.clearDuplicateHistory(Collections.singletonList(eventId));
            //清空专题数据
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", eventId));
            esOpUtil.docDeleteByQuery(Constants.SUBJECT_INDEX, boolQuery);
        });
    }

    @Override
    public List<ModelVO> modelList() {
        return baseMapper.modelList();
    }

    @Override
    public List<EventVO> eventList(List<String> eventIdList) {
        return baseMapper.eventList(eventIdList);
    }

    @Override
    public List<Node> projectList(String userId, String customerId) {
        return baseMapper.projectList(userId, customerId);
    }

    @Override
    public List<String> selectSubjectByTypeIds(String userId, List<String> typeIds) {
        return baseMapper.selectSubjectByTypeIds(userId, typeIds);
    }

    @Override
    public List<String> codesByLabels(List<String> labelIds) {
        return baseMapper.codesByLabels(labelIds);
    }

    @Override
    public List<LabelModelVo> selectLabelModelBySubjectId(List<String> subjectIds) {
        return baseMapper.selectLabelModelBySubjectId(subjectIds);
    }

    @Override
    public List<String> bindKeyWordsIdList(List<String> subjectIds) {
        return baseMapper.bindKeyWordsIdList(subjectIds);
    }

    @Override
    public List<String> selectSubjectWithCustomer(String userId, String customerId) {
        return baseMapper.selectSubjectWithCustomer(userId, customerId);
    }

    @Override
    public List<EventExcelVO> frontAllList(String projectId, String createBy, List<String> eventIdList,String eventName,
                                           String eventType, String labelField,String labelName, Integer size) {
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<String> eventTypes = new ArrayList<>();
        if (StringUtils.isNotEmpty(eventType)) {
            List<Node> nodes = eventCategoryService.categoryList();
            eventTypes = TreeUtil.belowList(nodes, eventType, true);
        }
        return baseMapper.frontAllList(projectId,createBy,eventIdList,eventName,eventTypes,labelField,labelName,type, size);
    }

    @Override
    public List<EventExcelVO> frontOwnerList(String projectId, String createBy, List<String> eventIdList,String eventName,
                                             String eventType, String labelField,String labelName, Integer size) {
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<String> eventTypes = new ArrayList<>();
        if (StringUtils.isNotEmpty(eventType)) {
            List<Node> nodes = eventCategoryService.categoryList();
            eventTypes = TreeUtil.belowList(nodes, eventType, true);
        }
        return baseMapper.frontOwnerList(projectId,createBy,eventIdList,eventName,eventTypes,labelField,labelName,type, size);
    }

    @Override
    public List<StatisticsKeyWordVo> hotWords(String index, String id, Integer number) {
        SubjectDataVo subjectDataVo = esService.queryInfo(index, id);
        String content = subjectDataVo.getContent();
        List<Map.Entry<String, Integer>> keywordsList = HanlpUtil.extractKeyWordsByText(Jsoup.parse(content).text(), number);
        List<StatisticsKeyWordVo> rn = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(keywordsList)) {
            for (Map.Entry<String, Integer> entry : keywordsList) {
                StatisticsKeyWordVo statisticsKeyWordVo = new StatisticsKeyWordVo();
                statisticsKeyWordVo.setName(entry.getKey());
                statisticsKeyWordVo.setValue(entry.getValue());
                rn.add(statisticsKeyWordVo);
            }
        }
        return rn;
    }

    @Override
    public void directBindInfoSource(DirectBindInfoVO directBindInfoVO) {
        String subjectId = directBindInfoVO.getSubjectId();
        List<String> bindList = directBindInfoVO.getBindList();
        Integer type = BindTypeEnum.INFO_SOURCE.getvalue();
        for (String infoSourceId : bindList) {
            /*
             * 1.先删除该信息源在专题下的所有数据
             * 2.再判断该信息源在专题已绑定的信息源组下，若不在则新增绑定关系；若在则跳过
             */
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId);
            iSubjectInfoSourceMapService.remove(queryWrapper);
            int num = baseMapper.ynBelowBindGroup(subjectId, infoSourceId);
            if (num == 0) {
                SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                subjectInfoSourceMap.setSourceId(infoSourceId);
                subjectInfoSourceMap.setSubjectId(subjectId);
                subjectInfoSourceMap.setType(type);
                iSubjectInfoSourceMapService.save(subjectInfoSourceMap);
            }
        }
    }

    @Override
    public void directRemoveInfoSource(DirectBindInfoVO directBindInfoVO) {
        String subjectId = directBindInfoVO.getSubjectId();
        List<String> excludeList = directBindInfoVO.getExcludeList();
        for (String infoSourceId : excludeList) {
            /*
             * 1.先删除该信息源在专题下的所有数据
             * 2.再判断该信息源在专题已排除的信息源组下，若不在则新增绑定关系；若在则跳过
             */
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId);
            iSubjectInfoSourceMapService.remove(queryWrapper);
            int num = baseMapper.ynBelowExcludeGroup(subjectId, infoSourceId);
            if (num == 0) {
                SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                subjectInfoSourceMap.setSourceId(infoSourceId);
                subjectInfoSourceMap.setSubjectId(subjectId);
                subjectInfoSourceMap.setType(BindTypeEnum.EXCLUDE_INFO_SOURCE.getvalue());
                iSubjectInfoSourceMapService.save(subjectInfoSourceMap);
            }
        }
    }

    @Override
    public void removeUnBindInfoSource(String subjectId, String infoSourceId) {
        int num = baseMapper.ynBelowExcludeGroup(subjectId, infoSourceId);
        if (num == 0) {
            LambdaUpdateWrapper<SubjectInfoSourceMap> update = Wrappers.lambdaUpdate();
            iSubjectInfoSourceMapService.update(update.eq(SubjectInfoSourceMap::getSubjectId, subjectId)
                    .eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getType, 3)
                    .set(SubjectInfoSourceMap::getType, 1));
        } else {
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId)
                    .eq(SubjectInfoSourceMap::getType, 3);
            iSubjectInfoSourceMapService.remove(queryWrapper);
        }
    }

    @Override
    public List<List<String>> subjectStatistics(List<String> subjectIds, String startDate, String endDate) {
        List<String> dateList = DateUtil.betweenDate(startDate, endDate);
        List<List<String>> dataList = new ArrayList<>();
        //专题关联的信息源集合
        List<SubjectStatisticsVo> infoSourceList = iSubjectInfoSourceMapService.subjectRealBindInfoSources(subjectIds);
        Map<String, Map<String, Long>> stringMapMap = esService.regetFromEs(infoSourceList.stream().map(SubjectStatisticsVo::getId).collect(Collectors.toList()), startDate, endDate);
        //封装导出excel需要的数据集合
        for (SubjectStatisticsVo subjectStatisticsVo : infoSourceList) {
            Map<String, Long> stringLongMap = stringMapMap.get(subjectStatisticsVo.getId());
            List<SubjectStatisticsVo> value = new ArrayList<>();
            int sum = 0;
            if (null != stringLongMap) {
                for (String s : stringLongMap.keySet()) {
                    SubjectStatisticsVo subjectStatisticsVoTemp = new SubjectStatisticsVo();
                    subjectStatisticsVoTemp.setId(subjectStatisticsVo.getId());
                    subjectStatisticsVoTemp.setSiteName(subjectStatisticsVo.getSiteName());
                    subjectStatisticsVoTemp.setDate(s);
                    int num = Math.toIntExact(stringLongMap.get(s));
                    subjectStatisticsVoTemp.setCollectCount(num);
                    value.add(subjectStatisticsVoTemp);
                    sum += Math.toIntExact(stringLongMap.get(s));
                }
            }
            List<String> list = new ArrayList<>(subjectStatisticsVo.toExcelList());
            //信息源在时间段内的采集总量
            list.add(String.valueOf(sum));
            //信息源采集到数据的日期集合
            Set<String> collectDates = value.stream().collect(Collectors.groupingBy(SubjectStatisticsVo::getDate)).keySet();
            //信息源未采集到数据的日期集合
            List<String> noCollectDates = new ArrayList<>(dateList);
            noCollectDates.removeAll(collectDates);
            for (String date : noCollectDates) {
                SubjectStatisticsVo vo = new SubjectStatisticsVo();
                BeanUtils.copyProperties(subjectStatisticsVo, vo);
                vo.setCollectCount(0);
                vo.setDate(date);
                value.add(vo);
            }
            //根据采集时间排序(正序)
            value.sort(Comparator.comparing(SubjectStatisticsVo::getDate));
            value.forEach(e -> list.add(String.valueOf(e.getCollectCount())));
            dataList.add(list);
        }
        return dataList;
    }

    @Override
    public IPage<KeyWordsPage> bindKeyWordsList(List<String> subjectIds, String groupName, String wordName, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        List<KeyWordsPage> wordsList = baseMapper.bindKeyWordsList(subjectIds, groupName, wordName, offset, pageSize);
        Long count = baseMapper.bindKeyWordsCount(subjectIds, groupName, wordName);
        IPage<KeyWordsPage> page = new Page<>(pageNo, pageSize, count);
        page.setRecords(wordsList);
        return page;
    }

    private void addRegionMap(String eventId, List<RegionVO> regionList) {
        if (CollectionUtils.isNotEmpty(regionList)) {
            List<EventRegionMap> dataList = new ArrayList<>();
            for (RegionVO regionVO : regionList) {
                EventRegionMap eventRegionMap = new EventRegionMap();
                eventRegionMap.setEventId(eventId);
                eventRegionMap.setRegionId(regionVO.getId());
                eventRegionMap.setTopRegionId(regionVO.getTopId());
                eventRegionMap.setType(regionVO.getType());
                dataList.add(eventRegionMap);
            }
            eventRegionMapService.saveBatch(dataList);
        }
    }

    private void saveMapMain(Event subject, AddEventParam addEventParam) {
        if (StringUtils.isNotEmpty(addEventParam.getSubjectTypeId())) {
            SubjectTypeMap subjectTypeMap = new SubjectTypeMap();
            subjectTypeMap.setSubjectId(subject.getId());
            subjectTypeMap.setUpdateBy(subject.getUpdateBy());
            subjectTypeMap.setUpdateTime(subject.getUpdateTime());
            subjectTypeMap.setTypeId(addEventParam.getSubjectTypeId());
            iSubjectTypeMapService.save(subjectTypeMap);
        }
        if (StringUtils.isNotEmpty(addEventParam.getProjectId())) {
            ProjectSubjectMap projectSubjectMap = new ProjectSubjectMap();
            projectSubjectMap.setProjectId(addEventParam.getProjectId());
            projectSubjectMap.setSubjectId(subject.getId());
            iProjectSubjectMapService.save(projectSubjectMap);
        }
    }

    @Override
    public void saveDefaultMap(String eventId) {
        //默认引擎
        Map<String, String> defaultSearchEngines = Constants.DEFAULT_SEARCH_ENGINE;
        List<SubjectSearchEnginesMap> enginesMaps = new ArrayList<>();
        for (Map.Entry<String, String> entry : defaultSearchEngines.entrySet()) {
            SubjectSearchEnginesMap enginesMap = new SubjectSearchEnginesMap();
            enginesMap.setSubjectId(eventId);
            enginesMap.setSearchEngineId(entry.getKey());
            enginesMaps.add(enginesMap);
        }
        subjectSearchEnginesMapService.saveBatch(enginesMaps);
        //默认模型
        Map<String, String> defaultModels = Constants.DEFAULT_MODEL;
        List<SubjectModelMap> modelMaps = new ArrayList<>();
        for (Map.Entry<String, String> entry : defaultModels.entrySet()) {
            SubjectModelMap modelMap = new SubjectModelMap();
            modelMap.setSubjectId(eventId);
            modelMap.setModelId(entry.getKey());
            modelMap.setType(entry.getValue());
            modelMap.setSign(1);
            modelMaps.add(modelMap);
        }
        subjectModelMapService.saveBatch(modelMaps);
    }


    //专题实际绑定信息源的集合
    private List<SubjectSourceVO> subjectBindSourceList(List<String> subjectIds) {
        List<SubjectSourceVO> bindList = baseMapper.bindSourceList(subjectIds);
        List<SubjectSourceVO> excludeList = baseMapper.excludeSourceList(subjectIds);
        bindList.removeAll(excludeList);
        return bindList;
    }

    //查询每个专题的数量
    private Map<String, Integer> subjectInfoCountMap(List<String> subjectIdList, List<Integer> checkStatusList) {
        Map<String, Integer> map = new HashMap<>(subjectIdList.size());
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        //默认查询没删除的
        boolQuery.must(QueryBuilders.matchQuery("deleteFlag", 0));
        if (checkStatusList != null && checkStatusList.size() > 0) {
            boolQuery.must(QueryBuilders.termsQuery("checkStatus", checkStatusList));
        }
        searchSourceBuilder.query(boolQuery);
        //分组
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_subjectId").field("subjectId.keyword");
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取id分组集合
            Terms parsedStringTerms = aggregations.get("group_subjectId");
            List<? extends Terms.Bucket> buckets = parsedStringTerms.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String subjectId = bucket.getKeyAsString();
                long collectCount = bucket.getDocCount();
                map.put(subjectId, (int) collectCount);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return map;
    }

    private void count(List<EventNewPlatVO> pageList) {
        if (CollectionUtils.isNotEmpty(pageList)) {
            List<String> idList = pageList.stream().map(EventNewPlatVO::getId).collect(Collectors.toList());
            Map<String, Integer> infoSourceNumMap = new HashMap<>();
            long t1 = System.currentTimeMillis();
            Map<String, Integer> keyWordsNumMap = commonService.bindKeyWordsCountList(idList).stream().collect(Collectors.toMap(SubjectPage::getId, SubjectPage::getKeyWordsNum));
            long t2 = System.currentTimeMillis();
            log.info("关键词数量查询耗时==={}",t2-t1);
            Map<String, List<SubjectSourceVO>> collect = commonService.bindSourceList(idList).stream().collect(Collectors.groupingBy(SubjectSourceVO::getSubjectId));
            long t3 = System.currentTimeMillis();
            log.info("信息源数量查询耗时==={}",t3-t2);
            for (Map.Entry<String, List<SubjectSourceVO>> entry : collect.entrySet()) {
                String subjectId = entry.getKey();
                List<SubjectSourceVO> value = entry.getValue();
                infoSourceNumMap.put(subjectId, value.size());
            }
            long t4 = System.currentTimeMillis();
            Map<String, Integer> subjectInfoCountMap = subjectInfoCountMap(idList, null);
            log.info("资讯数量查询耗时==={}",System.currentTimeMillis()-t4);
            for (EventNewPlatVO newPlatVO : pageList) {
                int keyWordsNum = null == keyWordsNumMap.get(newPlatVO.getId()) ? 0 : keyWordsNumMap.get(newPlatVO.getId());
                int infoSourceNum = null == infoSourceNumMap.get(newPlatVO.getId()) ? 0 : infoSourceNumMap.get(newPlatVO.getId());
                //查询每个专题现有的信息数量
                int subjectInfoCount = null == subjectInfoCountMap.get(newPlatVO.getId()) ? 0 : subjectInfoCountMap.get(newPlatVO.getId());
                newPlatVO.setSubjectInfoNum(subjectInfoCount);
                newPlatVO.setInfoSourceNum(infoSourceNum);
                newPlatVO.setKeyWordsNum(keyWordsNum);
            }
        }
    }

    /**
     * 修改事件时间缓存，用于补充数据（只有启用过的事件才会起作用）
     *
     * @param event    新事件信息
     * @param oldEvent 旧事件信息
     * @author lkg
     * @date 2025/2/7
     */
    private void updateRedisCache(Event event, Event oldEvent) {
        Date firstOpenTime = oldEvent.getFirstOpenTime();
        if (firstOpenTime != null) {
            Date oldTimeEnable = oldEvent.getStartTime();
            LocalDate oldStart = oldTimeEnable.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            Date timeEnable = event.getStartTime();
            List<String> newDateList = new ArrayList<>();
            LocalDate start = timeEnable.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            if (start.isBefore(oldStart)) {
                for (LocalDate date = start; date.isBefore(oldStart); date = date.plusDays(1)) {
                    // 在这里处理每一天的逻辑
                    //格式化date成字符串
                    String format = date.format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
                    newDateList.add(format);
                }
            }
            if (CollectionUtils.isNotEmpty(newDateList)) {
                redisUtil.rpushMultipleValues(Constants.HISTORY_EVENT_DATE_QUEUE + event.getEventCode(), newDateList.toArray(new String[0]));
            }
        }
    }
}
