package com.zzsn.event.controller.plat;

import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Event;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.IInfoSourceService;
import com.zzsn.event.service.ISubjectTypeService;
import com.zzsn.event.util.*;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.DisplayInfo;
import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 平台事件管理页
 *
 * @author lkg
 * @date 2024/5/7
 */
@Slf4j
@Api(tags = "平台事件管理")
@RestController
@RequestMapping("/plat/manage")
public class PlatEventManageController {


    @Autowired
    private IEventService eventService;
    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private EsOpUtil esOpUtil;

    /**
     * 事件页左侧树
     *
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/leftTypeTree")
    public Result<?> leftTree() {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        String userId = null;
        String customerId = null;
        if (category.equals(Constants.COMMON_USER)) {
            userId = currentUser.getUserId();
        } else if (category.equals(Constants.ADMIN_USER)) {
            customerId = currentUser.getCustomerId();
        }
        List<SubjectTreeVO> tree = subjectTypeService.eventAndTypeTree(userId,customerId);
        return Result.OK(tree);
    }

    /**
     * 事件页左侧树
     *
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/leftCustomerTree")
    public Result<?> leftCustomerTree() {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        String userId = null;
        String customerId = null;
        if (category.equals(Constants.COMMON_USER)) {
            userId = currentUser.getUserId();
        } else if (category.equals(Constants.ADMIN_USER)) {
            customerId = currentUser.getCustomerId();
        }
        List<SubjectTreeVO> tree = subjectTypeService.subjectAndCustomerTree(userId, customerId);
        return Result.OK(tree);
    }


    /**
     * 分页列表-新平台管理
     *
     * @param subjectCondition 筛选条件
     * @param pageNo           当前页
     * @param pageSize         返回条数
     * @author lkg
     * @date 2024/4/28
     */
    @GetMapping("/newPlatPageList")
    public Result<?> newPlatPageList(SubjectCondition subjectCondition,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        if (category.equals(Constants.COMMON_USER)) {
            subjectCondition.setUserId(currentUser.getUserId());
        } else if (category.equals(Constants.ADMIN_USER)) {
            subjectCondition.setCustomerId(currentUser.getCustomerId());
        }
        IPage<EventNewPlatVO> pageList = eventService.newPlatPageList(subjectCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 分页列表-新平台管理
     *
     * @param subjectCondition 筛选条件
     * @param pageNo           当前页
     * @param pageSize         返回条数
     * @author lkg
     * @date 2024/4/28
     */
    @GetMapping("/newPlatCustomerPageList")
    public Result<?> newPlatCustomerPageList(SubjectCondition subjectCondition,
                                             @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                             @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        if (category.equals(Constants.COMMON_USER)) {
            subjectCondition.setUserId(currentUser.getUserId());
            subjectCondition.setCustomerId(currentUser.getCustomerId());
        } else if (category.equals(Constants.ADMIN_USER)) {
            if (StringUtils.isEmpty(subjectCondition.getCustomerId()) || "0".equals(subjectCondition.getCustomerId())) {
                subjectCondition.setCustomerId(currentUser.getCustomerId());
            }
        }
        if ("0".equals(subjectCondition.getCustomerId())) {
            subjectCondition.setCustomerId(null);
        }
        IPage<EventNewPlatVO> pageList = eventService.newPlatCustomerPageList(subjectCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 专题绑定的信息源
     *
     * @param infoSourceCondition 筛选条件
     * @param subjectId           专题id/专题分类id
     * @param pageNo              当前页
     * @param pageSize            返回条数
     * @author lkg
     * @date 2024/5/7
     */
    @GetMapping(value = "/bindInfoSourceList")
    public Result<?> bindInfoSourceList(InfoSourceCondition infoSourceCondition,
                                        @RequestParam(name = "subjectId", defaultValue = "0") String subjectId,
                                        @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                        @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<String> subjectIdList = new ArrayList<>();
        if (!"0".equals(subjectId)) {
            //查询类别id的所有明细id
            if (StringUtils.isNotEmpty(subjectId)) {
                List<String> typeIds = subjectTypeService.belowIdList(subjectId, null);
                subjectIdList = eventService.selectSubjectByTypeIds(null, typeIds);
            }
            subjectIdList.add(subjectId);
        }
        IPage<InfoSourceVo> pageList = infoSourceService.bindInfoSourceList(infoSourceCondition, subjectIdList, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 专题直接绑定信息源-批量
     *
     * @param directBindInfoVO 参数封装
     * @author lkg
     * @date 2024/4/24
     */
    @PostMapping("/directBindInfoSource")
    public Result<?> directBindInfoSource(@RequestBody DirectBindInfoVO directBindInfoVO) {
        eventService.directBindInfoSource(directBindInfoVO);
        return Result.OK();
    }

    /**
     * 专题直接解绑信息源-批量
     *
     * @param directBindInfoVO 参数封装
     * @author lkg
     * @date 2024/4/24
     */
    @PostMapping("/directRemoveInfoSource")
    public Result<?> directRemoveInfoSource(@RequestBody DirectBindInfoVO directBindInfoVO) {
        eventService.directRemoveInfoSource(directBindInfoVO);
        return Result.OK();
    }

    /**
     * 专题直接屏蔽的信息源列表(不包括专题屏蔽的信息源组下的信息源)
     *
     * @param infoSourceCondition 筛选条件
     * @param pageNo              当前页
     * @param pageSize            返回条数
     * @author lkg
     * @date 2024/4/30
     */
    @GetMapping("/unBindInfoSourceList")
    public Result<?> unBindInfoSourceList(InfoSourceCondition infoSourceCondition,
                                          @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                          @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<InfoSourceVo> pageList = infoSourceService.unBindSourcePageList(infoSourceCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 删除专题下已屏蔽的信息源(即恢复绑定关系)
     *
     * @param subjectId 专题id
     * @param sourceId  信息源id
     * @author lkg
     * @date 2024/4/30
     */
    @GetMapping("/removeUnBindInfoSource")
    public Result<?> removeUnBindInfoSource(@RequestParam String subjectId, @RequestParam String sourceId) {
        eventService.removeUnBindInfoSource(subjectId, sourceId);
        return Result.OK();
    }


    /**
     * 专题绑定的关键词组-分页列表
     *
     * @param id 题id/专题分类id
     * @author lkg
     * @date 2024/5/7
     */
    @GetMapping(value = "/bindKeyWordsList")
    public Result<?> bindKeyWordsList(@RequestParam String id,
                                      @RequestParam(required = false) String groupName,
                                      @RequestParam(required = false) String wordName,
                                      @RequestParam(defaultValue = "1") Integer pageNo,
                                      @RequestParam(defaultValue = "10") Integer pageSize) {
        List<String> subjectIdList = new ArrayList<>();
        try {
            if (StringUtils.isNotEmpty(id) && !"0".equals(id)) {
                List<String> typeIds = subjectTypeService.belowIdList(id, null);
                subjectIdList = eventService.selectSubjectByTypeIds(null, typeIds);
            }
            subjectIdList.add(id);
        } catch (Exception e) {
            e.printStackTrace();
        }
        IPage<KeyWordsPage> page = eventService.bindKeyWordsList(subjectIdList, groupName, wordName, pageNo, pageSize);
        return Result.OK(page);
    }


    /**
     * 通过id查询(采集库)
     *
     * @param id    资讯id
     * @param index 索引名称
     * @author lkg
     * @date 2024/5/8
     */
    @GetMapping(value = "/collectionQueryById")
    public Result<?> collectionQueryById(@RequestParam String id, @RequestParam String index) {
        DisplayInfo info = (DisplayInfo) esOpUtil.getInfoById(index, id, DisplayInfo.class);
        if (StringUtils.isNotEmpty(info.getContentWithTag())) {
            String contentNoTag = Utility.TransferHTML2TextWithImg(info.getContentWithTag());
            String contentNoTag2 = Utility.dealImg(contentNoTag);
            info.setContent(contentNoTag2);
            info.setContentWithTag(null);
        }
        info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
        return Result.OK(info);
    }

    /**
     * 通过id查询(专题库)
     *
     * @param id    资讯id
     * @param index 索引名称
     * @author lkg
     * @date 2024/5/8
     */
    @GetMapping(value = "/queryById")
    public Result<?> queryById(@RequestParam String id, @RequestParam String index) {
        DisplayInfo displayInfo = (DisplayInfo) esOpUtil.getInfoById(index, id, DisplayInfo.class);
        addReadNum(displayInfo,index);
        if (StringUtils.isNotEmpty(displayInfo.getContentWithTag())) {
            String contentNoTag = Utility.TransferHTML2TextWithImg(displayInfo.getContentWithTag());
            String contentNoTag2 = Utility.dealImg(contentNoTag);
            displayInfo.setContent(contentNoTag2);
        }
        displayInfo.setPublishDate(EsDateUtil.esFieldDateMapping(displayInfo.getPublishDate()));
        return Result.OK(displayInfo);
    }

    private void addReadNum(DisplayInfo displayInfo,String index){
        if (displayInfo != null) {
            if (displayInfo.getReadNum() == 0) {
                displayInfo.setReadNum(1L);
            } else {
                displayInfo.setReadNum(displayInfo.getReadNum() + 1);
            }
            esOpUtil.docUpdateById(index, displayInfo.getId(), JSON.toJSONString(displayInfo));
        }
    }

    
    /**
     * @param event 条件封装
     * @描述 查询事件列表
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/listAll")
    public Result<?> listAll(@RequestBody Event event) {
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        setListAllQueryWrapper(event, queryWrapper);
        List<Event> list = eventService.list(queryWrapper);
        return Result.OK(list);
    }
    
    /**
     * @param event    条件封装
     * @param pageNo
     * @param pageSize
     * @描述 查询事件列表（分页，产业链关联事件时查询用）
     * @作者 hejinyu
     * @创建时间 2024/12/26
     **/
    @PostMapping("/pageListAll")
    public Result<?> pageListAll(@RequestBody Event event, @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        setListAllQueryWrapper(event, queryWrapper);
        Page<Event> page = new Page<>(pageNo, pageSize);
        IPage<Event> pageList = eventService.page(page, queryWrapper);
        return Result.OK(pageList);
    }
    
    private void setListAllQueryWrapper(@RequestBody Event event, LambdaQueryWrapper<Event> queryWrapper) {
        String eventType = event.getEventType();
        String startDate = event.getStartDate();
        String endDate = event.getEndDate();
        String keyword = event.getKeyword();
        queryWrapper.eq(StringUtils.isNotEmpty(eventType), Event::getEventType, eventType);
        if (StringUtils.isNotEmpty(startDate)) {
            queryWrapper.ge(Event::getCreateTime, startDate);
        }
        if (StringUtils.isNotEmpty(endDate)) {
            queryWrapper.le(Event::getCreateTime, endDate);
        }
        if (StringUtils.isNotEmpty(keyword)) {
            queryWrapper.and(q -> {
                q.eq(Event::getEventType, keyword).or().like(Event::getEventName, keyword).or().eq(Event::getEventDescribe, keyword);
            });
        }
    }
}
