package com.zzsn.event.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zzsn.clb.common.model.task.dto.titr.KeyWordsDTO;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.*;
import com.zzsn.event.enums.BindTypeEnum;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.mapper.SubjectMapper;
import com.zzsn.event.service.*;
import com.zzsn.event.util.*;
import com.zzsn.event.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.TotalHits;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import cn.hutool.core.util.ObjectUtil;
import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @Description: 专题表
 * @Author: jeecg-boot
 * @Date: 2021-12-01
 * @Version: V1.0
 */
@Service
@Slf4j
public class SubjectServiceImpl extends ServiceImpl<SubjectMapper, Subject> implements SubjectService {

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private RedisUtil redisUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private IProjectSubjectMapService projectSubjectMapService;
    @Autowired
    private ISubjectSearchEnginesMapService subjectSearchEnginesMapService;
    @Autowired
    private ISubjectKeywordsMapService subjectKeywordsMapService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private ISubjectModelMapService subjectModelMapService;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private ClbSubjectModelDetailTaskService subjectModelDetailTaskService;
    @Autowired
    private IAlgorithmModelService modelService;


    @Value("${python.subjectProcessorUrl}")
    private String subjectProcessorUrl;


    @Override
    public Page<SubjectPage> pageList(SubjectCondition subjectCondition, Integer pageNo, Integer pageSize) {
        //查询类别id的所有明细id
        List<String> typeIds = new ArrayList<>();
        String subjectTypeId = subjectCondition.getSubjectTypeId();
        if (StringUtils.isNotEmpty(subjectTypeId) && !"0".equals(subjectTypeId)) {
            typeIds = subjectTypeService.belowIdList(subjectTypeId, 1);
        }
        subjectCondition.setTypeIds(typeIds);
        Page<SubjectPage> page = new Page<>(pageNo, pageSize);
        Page<SubjectPage> pageList = baseMapper.pageList(subjectCondition, page);
        List<SubjectPage> records = pageList.getRecords();
        if (CollUtil.isNotEmpty(records)) {
            List<String> idList = records.stream().map(SubjectPage::getId).collect(Collectors.toList());
            Map<String, Integer> infoSourceNumMap = new HashMap<>();
            Map<String, Integer> keyWordsNumMap = commonService.bindKeyWordsCountList(idList).stream().collect(Collectors.toMap(SubjectPage::getId, SubjectPage::getKeyWordsNum));
            Map<String, List<SubjectInfoSourceLabelTypeVo>> collect = commonService.bindSourceList(idList).stream().collect(Collectors.groupingBy(SubjectInfoSourceLabelTypeVo::getSubjectId));
            for (Map.Entry<String, List<SubjectInfoSourceLabelTypeVo>> entry : collect.entrySet()) {
                String subjectId = entry.getKey();
                List<SubjectInfoSourceLabelTypeVo> value = entry.getValue();
                infoSourceNumMap.put(subjectId, value.size());
            }
            List<Integer> checkStatus = new ArrayList<>();
            checkStatus.add(0);
            Map<String, Integer> unCheckCountMap = subjectInfoCountMap(idList, checkStatus);
            Map<String, Integer> subjectInfoCountMap = subjectInfoCountMap(idList, null);
            //查询每个专题绑定的信息源数量及关键词数量
            for (SubjectPage subjectPage1 : records) {
                int keyWordsNum = null == keyWordsNumMap.get(subjectPage1.getId()) ? 0 : keyWordsNumMap.get(subjectPage1.getId());
                int infoSourceNum = null == infoSourceNumMap.get(subjectPage1.getId()) ? 0 : infoSourceNumMap.get(subjectPage1.getId());
                //查询每个专题现有的信息数量
                int subjectInfoCount = null == subjectInfoCountMap.get(subjectPage1.getId()) ? 0 : subjectInfoCountMap.get(subjectPage1.getId());
                int unCheckCount = null == unCheckCountMap.get(subjectPage1.getId()) ? 0 : unCheckCountMap.get(subjectPage1.getId());
                subjectPage1.setUnCheckNum(unCheckCount);
                subjectPage1.setSubjectInfoNum(subjectInfoCount);
                subjectPage1.setInfoSourceNum(infoSourceNum);
                subjectPage1.setKeyWordsNum(keyWordsNum);
            }
        }
        return pageList;
    }

    @Override
    @Transactional
    public Subject saveMain(SubjectPage subjectPage) {
        Subject subject = new Subject();
        //事件专题的默认分析规则参数-必填
        BeanUtils.copyProperties(subjectPage, subject);
        String cron;
        String subjectCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.SUBJECT_DEFAULT.getValue());
        subject.setSubjectCode(subjectCode);
        if (StringUtils.isEmpty(subject.getUnit()) || null == subject.getSpace()) {
            cron = CronUtil.getRandomCron();
        } else {
            cron = CronUtil.generateCron(subject.getUnit(), subject.getSpace());
        }
        subject.setCron(cron);
        baseMapper.insert(subject);
        //插入专题-类别、项目的绑定关系
        saveMapMain(subject, subjectPage);
        return null;
    }

    @Override
    @Transactional
    public void updateMain(SubjectPage subjectPage) {
        Subject subject = new Subject();
        BeanUtils.copyProperties(subjectPage, subject);
        //先记录老的信息状态
        Subject oldSubject = baseMapper.selectById(subject.getId());
        baseMapper.updateById(subject);

        //删除专题-类别绑定关系
        subjectTypeMapService.deleteBySubjectId(subject.getId());
        //删除专题-项目的绑定关系
        projectSubjectMapService.deleteBySubjectId(subject.getId());
        //插入新的
        saveMapMain(subject, subjectPage);

        //判断专题状态是否启用（启用时控制xxljob调度启动，关闭时控制xxljob关闭），随着状态的改变发消息给python
        if (!subject.getStatus().equals(oldSubject.getStatus()) && 1 == subjectPage.getSubjectType()) {
            if (subject.getStatus() == 1) {
                send(subject.getId(), "1");
            } else if (subject.getStatus() == 0) {
                //向python发起停止处理请求
                send(subject.getId(), "0");
            }
        }

        //判断开始时间和结束时间是否发生变动
        Date newTimeEnable = subjectPage.getTimeEnable();
        Date oldTimeEnable = oldSubject.getTimeEnable();
        if (!((newTimeEnable == null && oldTimeEnable == null) ||
                (newTimeEnable != null && oldTimeEnable != null && oldTimeEnable.compareTo(newTimeEnable) == 0))) {
            //查询出该专题绑定的关键词组
            List<KeyWordsPage> keyWordsPages = commonService.selectKeyWordsListById(subject.getId());
            List<String> keyWordIds = new ArrayList<>();
            for (KeyWordsPage keyWordsPage : keyWordsPages) {
                keyWordIds.add(keyWordsPage.getId());
            }
            //更新redis中关键词时间
            updateRedisKeyWordsDate(subjectPage.getId(), keyWordIds);
        }
    }

    @Override
    public void deleteMain(String subjectId) {
        baseMapper.deleteById(subjectId);
        CompletableFuture.runAsync(()->{
            //删除与类别的映射
            subjectTypeMapService.deleteBySubjectId(subjectId);
            //删除与信息源的关联关系
            subjectInfoSourceMapService.delete(subjectId);
            //删除与关键词组的关联关系
            subjectKeywordsMapService.delete(subjectId);
            //删除专题-项目的绑定关系
            projectSubjectMapService.deleteBySubjectId(subjectId);
            //删除事件-模型关系
            subjectModelMapService.remove(Wrappers.<SubjectModelMap>lambdaQuery().eq(SubjectModelMap::getSubjectId, subjectId));
            //删除事件-搜索引擎关系
            subjectSearchEnginesMapService.remove(Wrappers.<SubjectSearchEnginesMap>lambdaQuery().eq(SubjectSearchEnginesMap::getSubjectId, subjectId));
            //向python发送消息结束
            send(subjectId, "-1");
        });
    }

    @Override
    public SubjectPage getSubjectById(String subjectId) {
        SubjectPage subjectPage = baseMapper.selectSubjectById(subjectId);
        int keyWordsNum = keyWordsService.bindCount(subjectPage.getId());
        List<String> subjectIds = new ArrayList<>();
        subjectIds.add(subjectPage.getId());
        int infoSourceNum = infoSourceService.bindSourceCount(subjectIds);
        //查询每个专题现有的信息数量
        int subjectInfoCount = subjectInfoCount(subjectPage.getId(), null);
        //未审核资讯
        List<Integer> checkStatus = new ArrayList<>();
        checkStatus.add(0);
        int unCheckCount = subjectInfoCount(subjectId, checkStatus);
        subjectPage.setSubjectInfoNum(subjectInfoCount);
        subjectPage.setInfoSourceNum(infoSourceNum);
        subjectPage.setKeyWordsNum(keyWordsNum);
        subjectPage.setUnCheckNum(unCheckCount);
        return subjectPage;
    }

    @Override
    public void directBindInfoSource(DirectBindInfoVO directBindInfoVO) {
        String subjectId = directBindInfoVO.getSubjectId();
        List<String> bindList = directBindInfoVO.getBindList();
        Integer type = BindTypeEnum.INFO_SOURCE.getvalue();
        for (String infoSourceId : bindList) {
            /*
             * 1.先删除该信息源在专题下的所有数据
             * 2.再判断该信息源在专题已绑定的信息源组下，若不在则新增绑定关系；若在则跳过
             */
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId);
            subjectInfoSourceMapService.remove(queryWrapper);
            int num = baseMapper.ynBelowBindGroup(subjectId, infoSourceId);
            if (num == 0) {
                SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                subjectInfoSourceMap.setSourceId(infoSourceId);
                subjectInfoSourceMap.setSubjectId(subjectId);
                subjectInfoSourceMap.setType(type);
                subjectInfoSourceMapService.save(subjectInfoSourceMap);
            }
        }
    }

    @Override
    public void directRemoveInfoSource(DirectBindInfoVO directBindInfoVO) {
        String subjectId = directBindInfoVO.getSubjectId();
        List<String> excludeList = directBindInfoVO.getExcludeList();
        for (String infoSourceId : excludeList) {
            /*
             * 1.先删除该信息源在专题下的所有数据
             * 2.再判断该信息源在专题已排除的信息源组下，若不在则新增绑定关系；若在则跳过
             */
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId);
            subjectInfoSourceMapService.remove(queryWrapper);
            int num = baseMapper.ynBelowExcludeGroup(subjectId, infoSourceId);
            if (num == 0) {
                SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                subjectInfoSourceMap.setSourceId(infoSourceId);
                subjectInfoSourceMap.setSubjectId(subjectId);
                subjectInfoSourceMap.setType(BindTypeEnum.EXCLUDE_INFO_SOURCE.getvalue());
                subjectInfoSourceMapService.save(subjectInfoSourceMap);
            }
        }
    }

    @Override
    public void removeUnBindInfoSource(String subjectId, String infoSourceId) {
        int num = baseMapper.ynBelowExcludeGroup(subjectId, infoSourceId);
        if (num == 0) {
            LambdaUpdateWrapper<SubjectInfoSourceMap> update = Wrappers.lambdaUpdate();
            subjectInfoSourceMapService.update(update.eq(SubjectInfoSourceMap::getSubjectId, subjectId)
                    .eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getType, 3)
                    .set(SubjectInfoSourceMap::getType, 1));
        } else {
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId)
                    .eq(SubjectInfoSourceMap::getType, 3);
            subjectInfoSourceMapService.remove(queryWrapper);
        }
    }

    @Override
    public void keyWordsBind(SubjectPage subjectPage) {
        List<String> idList = subjectPage.getKeyWordsIds();
        List<SubjectKeywordsMap> mapList = new ArrayList<>();
        if (idList != null && !idList.isEmpty()) {
            for (String keyWordsId : idList) {
                SubjectKeywordsMap subjectKeywordsMap = new SubjectKeywordsMap();
                subjectKeywordsMap.setKeywordsId(keyWordsId);
                subjectKeywordsMap.setSubjectId(subjectPage.getId());
                subjectKeywordsMap.setType(subjectPage.getType());
                subjectKeywordsMap.setBindingType(subjectPage.getBindingType());
                mapList.add(subjectKeywordsMap);
            }
            subjectKeywordsMapService.saveBatch(mapList);
            //更新redis缓存（查询每个词组绑定专题的最小时间，作为爬取依据）
            updateRedisKeyWordsDate(subjectPage.getId(), idList);
        }
    }

    @Override
    public void deleteBindKeyWords(SubjectPage subjectPage) {
        List<String> deleteKeyWordsIds = subjectPage.getKeyWordsIds();
        if (deleteKeyWordsIds != null && !deleteKeyWordsIds.isEmpty()) {
            subjectKeywordsMapService.remove(Wrappers.<SubjectKeywordsMap>lambdaQuery()
                    .eq(SubjectKeywordsMap::getSubjectId,subjectPage.getId()).in(SubjectKeywordsMap::getKeywordsId,deleteKeyWordsIds)
                    .eq(StrUtil.isNotBlank(subjectPage.getBindingType()),SubjectKeywordsMap::getBindingType,subjectPage.getBindingType())
            );
            //清楚redis里面的绑定关系（包括专题信息，以及搜索引擎）
            for (String keyWordsId : deleteKeyWordsIds) {
                KeyWordsPage keyWordsPage = keyWordsService.getKeyWordsById(keyWordsId);
                KeyWordsDTO keyWordsDTO = (KeyWordsDTO) redisUtil.get(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWordsPage.getWordsCode());
                keyWordsDTO.setStartTime(null);
                keyWordsDTO.setEndTime(null);
                keyWordsDTO.setSearchEngines(null);
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWordsPage.getWordsCode(), keyWordsDTO);
            }
        }
    }

    @Override
    public void searchEngineBind(SubjectPage subjectPage) {
        //先删除后绑定
        subjectSearchEnginesMapService.remove(Wrappers.<SubjectSearchEnginesMap>lambdaQuery().eq(SubjectSearchEnginesMap::getSubjectId, subjectPage.getId()));
        List<String> idList = subjectPage.getSearchEngineIds();
        List<SubjectSearchEnginesMap> mapList = new ArrayList<>();
        if (CollUtil.isNotEmpty(idList)) {
            for (String searchEngineId : idList) {
                SubjectSearchEnginesMap subjectSearchEnginesMap = new SubjectSearchEnginesMap();
                subjectSearchEnginesMap.setSearchEngineId(searchEngineId);
                subjectSearchEnginesMap.setSubjectId(subjectPage.getId());
                mapList.add(subjectSearchEnginesMap);
            }
            subjectSearchEnginesMapService.saveBatch(mapList);
        }
        //查询出该专题绑定的关键词组
        List<KeyWordsPage> keyWordsPages = commonService.selectKeyWordsListById(subjectPage.getId());
        List<String> keyWordIds = new ArrayList<>();
        for (KeyWordsPage keyWordsPage : keyWordsPages) {
            keyWordIds.add(keyWordsPage.getId());
        }
        //更新redis中关键词时间
        updateRedisKeyWordsDate(subjectPage.getId(), keyWordIds);
    }

    @Override
    public List<SearchEnginesVo> bindSearchEngineList(SearchEnginesVo searchEnginesVo) {
        List<SearchEnginesVo> searchEnginesVoList = subjectSearchEnginesMapService.bindSearchEngineList(searchEnginesVo);
        for (SearchEnginesVo searchEnginesVo1 : searchEnginesVoList) {
            if (StringUtils.isNotEmpty(searchEnginesVo1.getSubjectId())) {
                searchEnginesVo1.setYn("1");
            }
        }
        return searchEnginesVoList;
    }

    @Override
    public Result<?> modelBind(SubjectPage subjectPage) {
        List<SubjectModelMap> mapList = new ArrayList<>();
        List<String> hisModelids = new ArrayList<>();
        List<PythonModelVo> models = subjectPage.getModels();
        if (CollUtil.isEmpty(models)){
            return Result.FAIL(500,"请至少配置一个模型");
        }
        for (PythonModelVo model : models) {
            Integer detailType = model.getDetailType();
            if (cn.hutool.core.util.ObjectUtil.isEmpty(detailType)){
                return Result.FAIL(501,"请勾选"+model.getModelName()+"的作用范围");
            }
        }
        if (CollUtil.isNotEmpty(models)) {
            List<PythonModelVo> zengLiang = models.stream().filter(f -> ObjectUtil.isNotEmpty(f.getDetailType()) && (f.getDetailType() == 0 || f.getDetailType() == 2)).collect(Collectors.toList());
            List<PythonModelVo> liShi = models.stream().filter(f -> ObjectUtil.isNotEmpty(f.getDetailType()) && (f.getDetailType() == 1 || f.getDetailType() == 2)).collect(Collectors.toList());
            //检查是否有正在处理的模型任务
            List<ClbSubjectModelDetailTask> list = subjectModelDetailTaskService.list(Wrappers.<ClbSubjectModelDetailTask>lambdaQuery()
                    .eq(ClbSubjectModelDetailTask::getSubjectId, subjectPage.getId()).eq(ClbSubjectModelDetailTask::getTaskStatus, 1)
            );

            if (CollUtil.isNotEmpty(list)){
                //有正在处理的任务
                for (ClbSubjectModelDetailTask clbSubjectModelDetailTask : list) {
                    String modelIds = clbSubjectModelDetailTask.getModelIds();
                    List<String> list1 = Arrays.asList(modelIds.split(","));
                    for (PythonModelVo model : liShi) {
                        if (list1.contains(model.getId())){
                            return Result.FAIL(201, JSONUtil.toJsonStr(clbSubjectModelDetailTask));
                        }
                    }
                }
            }
            List<SubjectModelMap> bind = subjectModelMapService.list(Wrappers.<SubjectModelMap>lambdaQuery()
                    .select(SubjectModelMap::getModelId)
                    .eq(SubjectModelMap::getSubjectId, subjectPage.getId())
            );
            //判断提取任务模型中的增量和全量数据，插入关系配置表
            if (CollectionUtil.isNotEmpty(zengLiang)){
                if (CollectionUtil.isNotEmpty(bind)){
                    List<PythonModelVo> saves = zengLiang.stream().filter(f -> !bind.stream().map(SubjectModelMap::getModelId).collect(Collectors.toList()).contains(f.getId())).collect(Collectors.toList());
                    if (CollectionUtil.isNotEmpty(saves)) {
                        saves.forEach(e->{
                            SubjectModelMap subjectModelMap = new SubjectModelMap();
                            subjectModelMap.setModelId(e.getId());
                            subjectModelMap.setSubjectId(subjectPage.getId());
                            subjectModelMap.setType(e.getType());
                            subjectModelMap.setSign(1);
                            mapList.add(subjectModelMap);
                        });
                        subjectModelMapService.saveBatch(mapList);
                    }
                }else {
                    zengLiang.forEach(e->{
                        SubjectModelMap subjectModelMap = new SubjectModelMap();
                        subjectModelMap.setModelId(e.getId());
                        subjectModelMap.setSubjectId(subjectPage.getId());
                        subjectModelMap.setType(e.getType());
                        subjectModelMap.setSign(1);
                        mapList.add(subjectModelMap);
                    });
                    subjectModelMapService.saveBatch(mapList);
                }

            }

            if (CollectionUtil.isNotEmpty(liShi)) {
                liShi.forEach(e->{
                    if (ObjectUtil.isNotEmpty(e.getDetailType()) && e.getDetailType() != 0){
                        hisModelids.add(e.getId());
                    }
                });
            }

        }
        //模型中如有历史数据处理任务则发起任务
        if (CollectionUtil.isNotEmpty(hisModelids)){
            List<AlgorithmModel> modelInfos = modelService.list(Wrappers.<AlgorithmModel>lambdaQuery()
                    .select(AlgorithmModel::getId,AlgorithmModel::getType,AlgorithmModel::getName,AlgorithmModel::getServiceName)
                    .in(AlgorithmModel::getId, hisModelids));
            //处理历史
            ClbSubjectModelDetailTask task = new ClbSubjectModelDetailTask();
            task.setSubjectId(subjectPage.getId());
            task.setTaskStatus(0);
            task.setModelIds(String.join(",", hisModelids));

            Date now = new Date();
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(now);
            // 添加半小时
            calendar.add(Calendar.MINUTE, 30);
            Date halfHourLater = calendar.getTime();

            task.setDetailToTime(halfHourLater);
            if (ObjectUtil.isNotEmpty(subjectPage.getTimeEnable())) {
                task.setDetailFromTime(subjectPage.getTimeEnable());
            }
            if (ObjectUtil.isNotEmpty(subjectPage.getTimeDisable())) {
                task.setDetailToTime(subjectPage.getTimeDisable());
            }
            task.setTaskCode(subjectPage.getTaskCode());
            task.setTaskParam(JSONUtil.toJsonStr(modelInfos));
            task.setAllParam(JSONUtil.toJsonStr(subjectPage));
            subjectModelDetailTaskService.save(task);
            kafkaTemplate.send("subjectModelDetail",JSONUtil.toJsonStr(task));
            return Result.OK(task);
        }
        return Result.OK();
    }

    private Map<String,Integer> codeMap = new HashMap<>();

    @Override
    public Result<?> checkIsBreak(SubjectPage subjectPage) {
        List<PythonModelVo> models = subjectPage.getModels();
        if (CollectionUtil.isEmpty(models)) {
            return Result.FAIL(500,"请至少配置一个模型");
        }
        for (PythonModelVo model : models) {
            Integer detailType = model.getDetailType();
            if (ObjectUtil.isEmpty(detailType)) {
                return Result.FAIL(501,"请勾选" + model.getModelName() + "的作用范围");
            }
        }
        List<PythonModelVo> liShi = models.stream().filter(f -> ObjectUtil.isNotEmpty(f.getDetailType()) && (f.getDetailType() == 1 || f.getDetailType() == 2)).collect(Collectors.toList());
        //检查是否有正在处理的模型任务
        List<ClbSubjectModelDetailTask> list = subjectModelDetailTaskService.list(Wrappers.<ClbSubjectModelDetailTask>lambdaQuery()
                .eq(ClbSubjectModelDetailTask::getSubjectId, subjectPage.getId()).eq(ClbSubjectModelDetailTask::getTaskStatus, 1)
        );

        String code = "";
        String s = cn.hutool.core.date.DateUtil.formatDate(new Date());
        if (ObjectUtil.isEmpty(codeMap.get(s))){
            codeMap = new HashMap<>();
            codeMap.put(s,1);
            code = s+" 00"+1;
        }else {
            codeMap.put(s,codeMap.get(s)+1);
            code = s +" 00" + +codeMap.get(s);
        }
        if (CollectionUtil.isNotEmpty(list)){
            //有正在处理的任务
            for (ClbSubjectModelDetailTask clbSubjectModelDetailTask : list) {
                String modelIds = clbSubjectModelDetailTask.getModelIds();
                List<String> list1 = Arrays.asList(modelIds.split(","));
                for (PythonModelVo model : liShi) {
                    if (list1.contains(model.getId())){
                        Result<Object> error = Result.FAIL(201, "此次任务中的模型处理，有部分模型已经存在于其他任务中，并且未执行完毕");
                        error.setResult(clbSubjectModelDetailTask);
                        return error;
                    }
                }
            }
        }
        ClbSubjectModelDetailTask clbSubjectModelDetailTask = new ClbSubjectModelDetailTask();
        clbSubjectModelDetailTask.setTaskCode(code);
        return Result.OK(clbSubjectModelDetailTask);
    }

    @Override
    public void send(String subjectId, String status) {
        Subject subjectById = this.getById(subjectId);
        if(ObjectUtil.isEmpty(subjectById) || subjectById.getCreateTime().after(DateTime.of("2023-11-30","yyyy-MM-dd"))){
            return;
        }
        JSONObject jsonObject = commonService.siteInfo(subjectId, null,1);
        if ("1".equals(status)) {
            jsonObject.put("subjectStatus", "1");
        } else if ("0".equals(status)) {
            jsonObject.put("subjectStatus", "0");
        } else if ("-1".equals(status)) {
            jsonObject.put("subjectStatus", "-1");
        }
        jsonObject.put("subjectStatusDetails", "");
        //调用python接口
        try {
            Map<String,String> headers = new HashMap<>();
            headers.put("Content-Type", "application/json;charset=UTF-8");
            headers.put("Accept", "application/json");
            headers.put("Authorization", "!0gwY$5S@5V&A_+XEu)");
            HttpUtil.doPostWithHeader(subjectProcessorUrl, jsonObject, 3000,headers);
        } catch (IOException e) {
            log.error("传递专题配置信息异常!{}",e.getMessage(),e);
        }
    }

    //更新专题绑定的关键词组的缓存信息
    private void updateRedisKeyWordsDate(String subjectId, List<String> keyWordsIds) {
        for (String keyWordsId : keyWordsIds) {
            KeyWordsDTO keyWordsDTO = subjectKeywordsMapService.selectMinByKeyWordsId(keyWordsId);
            KeyWordsDTO redisKeyWordsDTO = (KeyWordsDTO) redisUtil.get(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWordsDTO.getWordsCode());
            int count = subjectKeywordsMapService.selectCountByKeyWordsId(keyWordsId);
            KeyWordsDTO keyWordsDTO1;
            if (count <= 0) {
                keyWordsDTO1 = subjectKeywordsMapService.selectMaxByKeyWordsId(keyWordsId);
                redisKeyWordsDTO.setStartTime(keyWordsDTO.getStartTime());
                redisKeyWordsDTO.setEndTime(keyWordsDTO1.getEndTime());
            } else {
                if (redisKeyWordsDTO == null) {
                    redisKeyWordsDTO = keyWordsDTO;
                } else {
                    redisKeyWordsDTO.setEndTime(null);
                    redisKeyWordsDTO.setStartTime(keyWordsDTO.getStartTime());
                }
            }
            //查询出该专题绑定了哪些搜索引擎
            List<String> stringList = subjectSearchEnginesMapService.querySearchList(subjectId);
            redisKeyWordsDTO.setSearchEngines(stringList);
            redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + redisKeyWordsDTO.getWordsCode(), redisKeyWordsDTO);
            //立即执行一次
            kafkaTemplate.send("keyWordsCrawl", JSON.toJSONString(redisKeyWordsDTO));
        }
    }

    private void saveMapMain(Subject subject, SubjectPage subjectPage) {
        if (StringUtils.isNotEmpty(subjectPage.getSubjectTypeId())) {
            SubjectTypeMap subjectTypeMap = new SubjectTypeMap();
            subjectTypeMap.setSubjectId(subject.getId());
            subjectTypeMap.setCreateBy(subject.getCreateBy());
            subjectTypeMap.setCreateTime(subject.getCreateTime());
            subjectTypeMap.setUpdateBy(subject.getUpdateBy());
            subjectTypeMap.setUpdateTime(subject.getUpdateTime());
            subjectTypeMap.setSysOrgCode(subject.getSysOrgCode());
            subjectTypeMap.setTypeId(subjectPage.getSubjectTypeId());
            subjectTypeMapService.save(subjectTypeMap);
        }
        if (StringUtils.isNotEmpty(subjectPage.getProjectId())) {
            ProjectSubjectMap projectSubjectMap = new ProjectSubjectMap();
            projectSubjectMap.setProjectId(subjectPage.getProjectId());
            projectSubjectMap.setSubjectId(subject.getId());
            projectSubjectMapService.save(projectSubjectMap);
        }
    }

    //查询每个专题的数量
    private Map<String, Integer> subjectInfoCountMap(List<String> subjectIdList, List<Integer> checkStatusList) {
        String[] indexs = EsIndexUtil.getIndexLatelyTwoYear(Constants.SUBJECT_INDEX);
        SearchRequest searchRequest = new SearchRequest(indexs);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        //默认查询没删除的
        boolQuery.must(QueryBuilders.matchQuery("deleteFlag", 0));
        if (checkStatusList != null && !checkStatusList.isEmpty()) {
            boolQuery.must(QueryBuilders.termsQuery("checkStatus", checkStatusList));
        }
        searchSourceBuilder.query(boolQuery);
        //分组
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_subjectId").field("subjectId.keyword");
        searchSourceBuilder.aggregation(aggregationBuilder);
        //不返回文本内容
        searchSourceBuilder.fetchSource(false);
        searchRequest.source(searchSourceBuilder);
        Map<String, Integer> map = new HashMap<>(subjectIdList.size());
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取id分组集合
            Terms parsedStringTerms = aggregations.get("group_subjectId");
            List<? extends Terms.Bucket> buckets = parsedStringTerms.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String subjectId = bucket.getKeyAsString();
                long collectCount = bucket.getDocCount();
                map.put(subjectId, (int) collectCount);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        return map;
    }

    //查询每个专题的数量
    private Integer subjectInfoCount(String subjectId, List<Integer> checkStatusList) {
        int count = 0;
        String[] indexs = EsIndexUtil.getIndexLatelyTwoYear(Constants.SUBJECT_INDEX);
        SearchRequest searchRequest = new SearchRequest(indexs);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        //默认查询没删除的
        boolQuery.must(QueryBuilders.matchQuery("deleteFlag", 0));
        if (checkStatusList != null && !checkStatusList.isEmpty()) {
            boolQuery.must(QueryBuilders.termsQuery("checkStatus", checkStatusList));
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            TotalHits totalHits = searchResponse.getHits().getTotalHits();
            count = (int) totalHits.value;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return count;
    }
}

