package com.zzsn.event.controller;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.obs.services.model.PutObjectResult;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.*;
import com.zzsn.event.es.EsService;
import com.zzsn.event.feign.api.RemoteModelService;
import com.zzsn.event.service.*;
import com.zzsn.event.service.impl.CaiJiCenterHttpService;
import com.zzsn.event.service.impl.ConfigurationMessageService;
import com.zzsn.event.util.*;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.SpecialInformation;
import com.zzsn.event.xxljob.entity.XxlJobInfo;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hwpf.HWPFDocument;
import org.apache.poi.hwpf.extractor.WordExtractor;
import org.apache.poi.xwpf.extractor.XWPFWordExtractor;
import org.apache.poi.xwpf.usermodel.XWPFDocument;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * 平台事件管理页
 *
 * @author lkg
 * @date 2024/5/7
 */
@Slf4j
@RestController
@RequestMapping("/subject/manage")
public class SubjectManageController {

    @Autowired
    private SubjectService subjectService;
    @Autowired
    private IEventService eventService;
    @Autowired
    private ISubjectKeywordsMapService subjectKeywordsMapService;
    @Autowired
    private IXxlJobInfoService xxlJobInfoService;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private ISubjectModelMapService subjectModelMapService;
    @Autowired
    private EsService esService;
    @Autowired
    private SubjectKeywordsGroupRelationService subjectKeywordsGroupRelationService;
    @Autowired
    private ScoreModelService scoreModelService;
    @Autowired
    private ClbLabelService clbLabelService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private SubjectSampleFileService subjectSampleFileService;
    @Autowired
    private SubjectInfoSourceMiddleMapService subjectInfoSourceMiddleMapService;
    @Autowired
    private SysDictItemService sysDictItemService;
    @Autowired
    private ConfigurationMessageService configurationMessageService;
    @Autowired
    private SubjectStatisticInfoService subjectStatisticInfoService;

    @Autowired
    private ObsUtil obsUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private PythonUtil pythonUtil;
    @Autowired(required = false)
    private RemoteModelService remoteModelService;

    @Value("${kafka.topic.subject.run:}")
    private String SUBJECT_MODEL_KAFKA_CHANNEL;
    @Autowired
    private CaiJiCenterHttpService caiJiCenterHttpService;

    /**
     * 专题列表-资讯转换时使用
     *
     * @author lkg
     * @date 2024/12/24
     */
    @GetMapping("/listAll")
    public Result<?> subjectList() {
        List<Subject> list = subjectService.list();
        return Result.OK(list);
    }

    /**
     * 数据字典下字典值信息列表
     *
     * @param dictCode 字典编码
     * @author lkg
     * @date 2025/1/7
     */
    @GetMapping("/dictItemList")
    public Result<?> dictItemList(@RequestParam(defaultValue = "Thematic_Library")String dictCode){
        List<SysDictItem> dictItemList = sysDictItemService.listByDictCode(dictCode);
        return Result.OK(dictItemList);
    }

    /**
     * 分页列表查询-研究中心
     *
     * @param subjectCondition 搜索条件
     * @param pageNo           当前页
     * @param pageSize         每页返回条数
     * @author lkg
     * @date 2025/1/7
     */
    @GetMapping(("/pageList"))
    public Result<?> researchCenterPageList(SubjectCondition subjectCondition,
                                            @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                            @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                            @RequestParam(required = false) String flagCode,HttpServletRequest request) {
        UserVo loginUser = UserUtil.getLoginUser();
        Page<SubjectPage> pageList = subjectService.researchCenterPageList(subjectCondition, pageNo, pageSize,flagCode,request,loginUser.getUsername());
        return Result.OK(pageList);
    }

    /**
     * 当前用户可见的专题列表-研究中心
     *
     * @param type        分类(1-个人;2-公开;3-个人+公开)
     * @param subjectName 专题名称
     * @param environment    环境 1-测试 2-正式
     * @param flagCode    数据权限code
     * @param pageNo      当前页
     * @param pageSize    每页返回条数
     * @author lkg
     * @date 2025/2/8
     */
    @GetMapping("/visiblePageList")
    public Result<?> visiblePageList(@RequestParam Integer type,
                                     @RequestParam(required = false) String subjectName,
                                     @RequestParam(required = false) String environment,
                                     @RequestParam(required = false) String flagCode,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                     HttpServletRequest request) {
        UserVo loginUser = UserUtil.getLoginUser();
        Page<Node> pageList = subjectService.visiblePageList(loginUser.getUsername(), type, subjectName, pageNo, pageSize,environment,request,flagCode);
        return Result.OK(pageList);
    }


    /**
     * 当前用户可见的专题列表-研究中心-不验证
     *
     * @param type        分类(1-个人;2-公开;3-个人+公开)
     * @param subjectName 专题名称
     * @param environment    环境 1-测试 2-正式
     * @param flagCode    数据权限code
     * @param pageNo      当前页
     * @param pageSize    每页返回条数
     * @author lkg
     * @date 2025/2/8
     */
    @GetMapping("/visiblePageListNoSign")
    public Result<?> visiblePageListNoSign(@RequestParam Integer type,
                                     @RequestParam(required = false) String subjectName,
                                     @RequestParam(required = false) String environment,
                                     @RequestParam(required = false) String flagCode,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                     HttpServletRequest request) {
        UserVo loginUser = UserUtil.getLoginUser();
        Page<Node> pageList = subjectService.visiblePageList(loginUser.getUsername(), type, subjectName, pageNo, pageSize,environment,request,flagCode);
        return Result.OK(pageList);
    }

    /**
     * 分页列表查询
     *
     * @param subjectCondition 搜索条件
     * @param pageNo           当前页
     * @param pageSize         每页返回条数
     * @return
     */
    @GetMapping(value = "/listByTypeId")
    public Result<?> pageList(SubjectCondition subjectCondition,
                              @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<SubjectPageVO> pageList = subjectService.pageList_new(subjectCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 专题统计信息
     *
     * @param subjectIds 专题id集合
     * @author lkg
     * @date 2025/3/25
     */
    @GetMapping(value = "/statisticInfo")
    private Result<?> statisticInfo(@RequestParam List<String> subjectIds){
        List<SubjectStatisticInfo> statisticInfoList = subjectService.statisticInfo(subjectIds);
        //异步更新专题统计信息表的数据
        CompletableFuture.runAsync(() -> {
            if (CollectionUtils.isNotEmpty(statisticInfoList)) {
                subjectStatisticInfoService.batchModify(statisticInfoList);
            }
        });
        return Result.OK(statisticInfoList);
    }

    /**
     * 推荐关键词
     *
     * @param files 眼里文章
     * @param words 输入词
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/recommendWordList")
    public Result<?> recommendWordList(MultipartFile[] files, String words) {
        JSONObject jsonObject = new JSONObject();
        boolean flag = true;
        if (files != null && files.length > 0) {
            for (MultipartFile file : files) {
                String originalFilename = file.getOriginalFilename();
                if (!(originalFilename.endsWith(".txt") || originalFilename.endsWith(".doc") || originalFilename.endsWith(".docx"))) {
                    flag = false;
                    break;
                }
            }
            if (!flag) {
                return Result.FAIL("包含不支持的文件类型");
            }
            List<StatisticsKeyWordVo> statisticsKeyWordVos = new ArrayList<>();
            try {
                statisticsKeyWordVos = articleWords(files);
            } catch (Exception e) {
                e.printStackTrace();
            }
            jsonObject.put("article", statisticsKeyWordVos);
        }
        if (StringUtils.isNotBlank(words)) {
            //List<StatisticsKeyWordVo> statisticsKeyWordVos = extractWords(words);
            List<StatisticsKeyWordVo> statisticsKeyWordVos = new ArrayList<>();
            List<String> list = LLMUtil.wordRecommend(words);
            for (String s : list) {
                StatisticsKeyWordVo statisticsKeyWordVo = new StatisticsKeyWordVo();
                statisticsKeyWordVo.setName(s);
                statisticsKeyWordVo.setValue(0);
                statisticsKeyWordVos.add(statisticsKeyWordVo);
            }
            jsonObject.put("word", statisticsKeyWordVos);
        }
        return Result.OK(jsonObject);
    }

    /**
     * 添加专题样例文章
     *
     * @param files 样例文章
     * @return
     */
    @PostMapping(value = "/addSampleFile")
    public Result<?> addSampleFile(MultipartFile[] files, String subjectId) {
        if (StringUtils.isBlank(subjectId)) {
            return Result.FAIL("专题id不能为空");
        }
        boolean flag = true;
        if (files != null && files.length > 0) {
            for (MultipartFile file : files) {
                String originalFilename = file.getOriginalFilename();
                if (!(originalFilename.endsWith(".txt") || originalFilename.endsWith(".doc") || originalFilename.endsWith(".docx"))) {
                    flag = false;
                    break;
                }
            }
            if (!flag) {
                return Result.FAIL("包含不支持的文件类型");
            }
            List<SubjectSampleFile> sampleFiles = new ArrayList<>();
            try {
                for (MultipartFile file : files) {
                    InputStream inputStream = file.getInputStream();
                    PutObjectResult putObjectResult = obsUtil.uploadFile("subjectSampleFile/" + file.getOriginalFilename(), inputStream);
                    String filePath = putObjectResult.getObjectKey();
                    SubjectSampleFile subjectSampleFile = new SubjectSampleFile();
                    subjectSampleFile.setSubjectId(subjectId);
                    subjectSampleFile.setFileName(file.getOriginalFilename());
                    subjectSampleFile.setFilePath(filePath);
                    sampleFiles.add(subjectSampleFile);
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
            subjectSampleFileService.saveBatch(sampleFiles);
        } else {
            return Result.FAIL("请选择上传文件");
        }
        return Result.OK();
    }

    /**
     * 获取专题详情(包含样例文章信息)
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/1/8
     */
    @GetMapping("/queryInfo")
    public Result<?> queryInfo(@RequestParam String subjectId) {
        SubjectDetailVO info = subjectService.queryInfo(subjectId);
        return Result.OK(info);
    }

    /**
     * 专题详情(包含专题分类、专题项目)
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/1/8
     */
    @GetMapping("/getById")
    public Result<?> detailById(@RequestParam String subjectId) {
        return Result.OK(subjectService.getInfo(subjectId));
    }

    /**
     * 添加
     *
     * @param subjectPage
     * @return
     */
    @PostMapping(value = "/add")
    public Result<?> add(@RequestBody SubjectPage subjectPage) {
        Subject subject = subjectService.saveMain(subjectPage);
        //插入xxlJob
        xxlJobInfoService.subjectInsert(subject);
        return Result.OK(subject.getId());
    }

    /**
     * 编辑
     *
     * @param subjectPage
     * @return
     */
    @PostMapping(value = "/edit")
    public Result<?> edit(@RequestBody SubjectPage subjectPage) {
        subjectService.updateMain(subjectPage);
        CompletableFuture.runAsync(() -> {
            //更新xxljob
            if (StrUtil.isNotBlank(subjectPage.getUnit()) && ObjectUtil.isNotEmpty(subjectPage.getSpace())) {
                String cron = CronUtil.generateCron(subjectPage.getUnit(), subjectPage.getSpace());
                xxlJobInfoService.cronUpdate(subjectPage.getSubjectCode(), cron);
            }
        });
        return Result.OK();
    }

    /**
     * 更新状态
     *
     * @param subject
     * @author lkg
     * @date 2025/1/9
     */
    @PostMapping("/updateStatus")
    public Result<?> updateStatus(@RequestBody Subject subject) {
        subjectService.updateStatus(subject);
        Subject byId = subjectService.getById(subject.getId());
        List<XxlJobInfo> list = xxlJobInfoService.list(Wrappers.<XxlJobInfo>lambdaQuery().eq(XxlJobInfo::getInfoSourceCode, byId.getSubjectCode()));
        if (CollectionUtils.isNotEmpty(list)) {
            xxlJobInfoService.update(Wrappers.<XxlJobInfo>lambdaUpdate()
                    .eq(XxlJobInfo::getInfoSourceCode, byId.getSubjectCode())
                    .set(XxlJobInfo::getTriggerStatus, byId.getStatus()));
        }else{
            xxlJobInfoService.subjectInsert(byId);
        }

        if (subject.getStatus() == 1) {
            kafkaTemplate.send(SUBJECT_MODEL_KAFKA_CHANNEL, byId.getSubjectCode());
            configurationMessageService.bindInfoSourceSend(subject.getId());
            configurationMessageService.bindKeyWordsSend(subject.getId());
            configurationMessageService.subjectEnterpriseSourceSync(subject.getId());
        }
        try {
            String res = caiJiCenterHttpService.subjectStatusEdit(subject.getStatus(), subject.getId());
            cn.hutool.json.JSONObject entries = JSONUtil.parseObj(res);
            //获取code状态码
            Integer code = entries.getInt("code");
            if (ObjectUtil.isEmpty(code) ||  code != 200) {
                log.error("专题状态同步采集失败{}",res);
            }
        } catch (Exception e) {
            log.error("专题状态同步采集失败{}",e.getMessage(),e);
        }


        return Result.OK();
    }
    /**
     * 更新状态
     *
     * @author lkg
     * @date 2025/1/9
     */
    @PostMapping("/updateSubjectAllToCaiji")
    public Result<?> updateSubjectAllToCaiji() {
        List<Subject> list = subjectService.list(Wrappers.<Subject>lambdaQuery().eq(Subject::getSubjectType,1));
        for (int i = 0; i < list.size(); i++) {
            Subject subject = list.get(i);
            log.info("专题状态同步采集{},index:{},total:{}",subject.getSubjectName(),i,list.size());
            if (subject.getStatus()!=null && subject.getStatus() == 1) {
                configurationMessageService.bindInfoSourceSend(subject.getId());
                configurationMessageService.bindKeyWordsSend(subject.getId());
                caiJiCenterHttpService.subjectStatusEdit(subject.getStatus(), subject.getId());
                configurationMessageService.subjectEnterpriseSourceSync(subject.getId());
                log.info("专题配置信息同步采集成功:{}",subject.getSubjectName());
            }else{
                caiJiCenterHttpService.delSubject(subject.getId());
                log.info("通知采集删除专题成功:{}",subject.getSubjectName());
            }
        }
        return Result.OK();
    }
    /**
     * 更新状态
     *
     * @author lkg
     * @date 2025/1/9
     */
    @PostMapping("/updateEventAllToCaiji")
    public Result<?> updateEventAllToCaiji() {
        List<Event> list = eventService.list();
        for (int i = 0; i < list.size(); i++) {
            Event event = list.get(i);
            log.info("事件状态同步采集{},index:{},total:{}",event.getEventName(),i,list.size());
            if (event.getStatus()!=null && event.getStatus() == 1) {
                configurationMessageService.bindKeyWordsEventSend(event.getId());
                caiJiCenterHttpService.subjectStatusEdit(event.getStatus(), event.getId());
                log.info("事件配置信息同步采集成功:{}",event.getEventName());
            }else{
                caiJiCenterHttpService.delSubject(event.getId());
                log.info("通知采集删除事件成功:{}",event.getEventName());
            }
        }
        return Result.OK();
    }

    /**
     * 通过id删除
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/delete")
    public Result<?> delete(@RequestParam(name = "id") String id) {
        subjectService.deleteMain(id);
        CompletableFuture.runAsync(() -> {
            //删除xxljob里面的信息
            LambdaQueryWrapper<Subject> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(Subject::getId, id);
            Subject subject = subjectService.getOne(queryWrapper);
            xxlJobInfoService.deleteByInfosourceCode(subject.getSubjectCode());
        });
        try {
            String res = caiJiCenterHttpService.delSubject(id);
            cn.hutool.json.JSONObject entries = JSONUtil.parseObj(res);
            //获取code状态码
            Integer code = entries.getInt("code");
            if (ObjectUtil.isEmpty(code) ||  code != 200) {
                log.error("专题删除同步采集失败{}",res);
            }
        } catch (Exception e) {
            log.error("专题删除同步采集失败{}",e.getMessage(),e);
        }
        return Result.OK();
    }

    /**
     * 批量删除
     *
     * @param ids
     * @return
     */
    @GetMapping(value = "/deleteBatch")
    public Result<?> deleteBatch(@RequestParam(name = "ids") String ids) {
        String[] idArr = ids.split(",");
        for (String id : idArr) {
            subjectService.deleteMain(id);
        }
        CompletableFuture.runAsync(() -> {
            for (String id : idArr) {
                //删除xxljob里面的信息
                LambdaQueryWrapper<Subject> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.eq(Subject::getId, id);
                Subject subject = subjectService.getOne(queryWrapper);
                xxlJobInfoService.deleteByInfosourceCode(subject.getSubjectCode());

                try {
                    String res = caiJiCenterHttpService.delSubject(id);
                    cn.hutool.json.JSONObject entries = JSONUtil.parseObj(res);
                    //获取code状态码
                    Integer code = entries.getInt("code");
                    if (ObjectUtil.isEmpty(code) ||  code != 200) {
                        log.error("专题删除同步采集失败{}",res);
                    }
                } catch (Exception e) {
                    log.error("专题删除同步采集失败{}",e.getMessage(),e);
                }

            }
        });

        return Result.OK();
    }


    /**
     * 专题直接绑定信息源-批量(专题列表页)
     *
     * @param directBindInfoVO 参数封装
     * @author lkg
     * @date 2024/4/24
     */
    @PostMapping("/directBindInfoSource")
    public Result<?> directBindInfoSource(@RequestBody DirectBindInfoVO directBindInfoVO) {
        subjectService.directBindInfoSource(directBindInfoVO);
        return Result.OK();
    }

    /**
     * 专题直接解绑信息源-批量(专题列表页)
     *
     * @param directBindInfoVO 参数封装
     * @author lkg
     * @date 2024/4/24
     */
    @PostMapping("/directRemoveInfoSource")
    public Result<?> directRemoveInfoSource(@RequestBody DirectBindInfoVO directBindInfoVO) {
        subjectService.directRemoveInfoSource(directBindInfoVO);
        return Result.OK();
    }

    /**
     * 专题直接屏蔽的信息源列表(不包括专题屏蔽的信息源组下的信息源)(专题列表页)
     *
     * @param infoSourceCondition 筛选条件
     * @param pageNo              当前页
     * @param pageSize            返回条数
     * @author lkg
     * @date 2024/4/30
     */
    @GetMapping("/unBindInfoSourceList")
    public Result<?> unBindInfoSourceList(InfoSourceCondition infoSourceCondition,
                                          @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                          @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<InfoSourceVo> pageList = infoSourceService.unBindSourcePageList(infoSourceCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 删除专题下已屏蔽的信息源(即恢复绑定关系)(专题列表页)
     *
     * @param subjectId 专题id
     * @param sourceId  信息源id
     * @author lkg
     * @date 2024/4/30
     */
    @GetMapping("/removeUnBindInfoSource")
    public Result<?> removeUnBindInfoSource(@RequestParam String subjectId, @RequestParam String sourceId) {
        subjectService.removeUnBindInfoSource(subjectId, sourceId);
        return Result.OK();
    }

    /**
     * 专题绑定关键词组列表-专题配置页
     *
     * @param id          专题id
     * @param bindingType 词组类型
     * @return
     */
    @GetMapping(value = "/bindKeyWordsList")
    public Result<?> bindKeyWordsList(@RequestParam(name = "id") String id,@RequestParam(name = "bindingType",required = false) String bindingType) {
        List<KeyWordsPage> bindKeyWordsList = subjectKeywordsMapService.bindKeyWordsList(id,bindingType);
        return Result.OK(bindKeyWordsList);
    }

    /**
     * 专题绑定关键词-专题配置页
     *
     * @param subjectPage 参数
     * @return
     */
    @PostMapping(value = "/keyWordsBind")
    public Result<?> keyWordsBind(@RequestBody SubjectPage subjectPage) {
        subjectService.keyWordsBind(subjectPage);
        return Result.OK();
    }

    /**
     * 专题绑定关键词的作用范围-专题配置页
     *
     * @param subjectKeywordsMap 参数
     * @return
     */
    @PostMapping(value = "/keyWordsBindScope")
    public Result<?> keyWordsBindScope(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
        subjectKeywordsMapService.updateById(subjectKeywordsMap);
        //同步配置到采集
        configurationMessageService.bindKeyWordsSend(subjectKeywordsMap.getSubjectId());
        return Result.OK();
    }

    /**
     * 新增/编辑词组关系
     *
     * @param subjectKeywordsGroupRelation 词组关系
     * @author lkg
     * @date 2024/12/24
     */
    @PostMapping("/keyword/relation/save")
    public Result<?> add(@RequestBody SubjectKeywordsGroupRelation subjectKeywordsGroupRelation) {
        return subjectKeywordsGroupRelationService.add(subjectKeywordsGroupRelation);
    }

    /**
     * 获取不通类型的词组之间的关系数据
     *
     * @param subjectId    专题id
     * @param relationType 1：采集词词组关系，2：过滤词词组关系，3排除词组词组关系
     * @author lkg
     * @date 2024/12/24
     */
    @GetMapping("/keyword/relation/getBySubjectIdAndType")
    public Result<?> getBySubjectIdAndType(String subjectId, String relationType) {
        return Result.OK(subjectKeywordsGroupRelationService.getOne(Wrappers.<SubjectKeywordsGroupRelation>lambdaQuery()
                .eq(SubjectKeywordsGroupRelation::getSubjectId, subjectId)
                .eq(SubjectKeywordsGroupRelation::getRelationType, relationType)
        ));
    }

    /**
     * 获取专题下所有的词组关系
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/24
     */
    @GetMapping("/keyword/relation/getBySubjectId")
    public Result<?> getBySubjectId(String subjectId) {
        return Result.OK(subjectKeywordsGroupRelationService.list(Wrappers.<SubjectKeywordsGroupRelation>lambdaQuery()
                .eq(SubjectKeywordsGroupRelation::getSubjectId, subjectId)
        ));
    }

    /**
     * 删除专题和关键词的绑定关系-专题配置页
     *
     * @param subjectPage
     * @return
     */
    @PostMapping(value = "/deleteKeyWordsBind")
    public Result<?> deleteKeyWordsBind(@RequestBody SubjectPage subjectPage) {
        subjectService.deleteBindKeyWords(subjectPage);
        return Result.OK();
    }

    /**
     * 专题搜索引擎绑定-专题配置页
     *
     * @param subjectPage
     * @return
     */
    @PostMapping(value = "/searchEngineBind")
    public Result<?> searchEngineBind(@RequestBody SubjectPage subjectPage) {
        subjectService.searchEngineBind(subjectPage);
        return Result.OK("编辑成功!");
    }

    /**
     * 专题绑定的搜索引擎列表-专题配置页
     *
     * @return
     */
    @GetMapping(value = "/bindSearchEngineList")
    public Result<?> bindSearchEngineList(SearchEnginesVo searchEnginesVo) {
        List<SearchEnginesVo> searchEngineList = subjectService.bindSearchEngineList(searchEnginesVo);
        return Result.OK(searchEngineList);
    }


    /**
     * 专题配置页左侧列表-专题配置页
     *
     * @param subjectType
     * @return
     */
    @GetMapping(value = "/modelList")
    public Result<?> modelList(@RequestParam(name = "subjectType", defaultValue = "0") String subjectType) {
        List<ModelVO> allList = commonService.allList(subjectType);
        return Result.OK(allList);
    }

    /**
     * 模型绑定（包含去重、筛选、标签模型）-专题配置页
     *
     * @param subjectPage 参数
     * @return
     */
    @PostMapping(value = "/modelBind")
    public Result<?> modelBind(@RequestBody SubjectPage subjectPage) {
        return subjectService.modelBind(subjectPage);
    }

    /**
     * 专题绑定的模型信息列表-专题配置页
     *
     * @param subjectId 专题id
     * @param type      模型分类
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/bindModelList")
    public Result<?> bindModelList(@RequestParam String subjectId, @RequestParam(required = false) String type) {
        List<PythonModelVo> pythonModelVoList = subjectModelMapService.bindModelByType(subjectId, type);
        return Result.OK(pythonModelVoList);
    }

    /**
     * 配置打分模型
     *
     * @param scoreModelVo 配置信息
     * @author lkg
     * @date 2024/12/27
     */
    @PostMapping("/configScoreModel")
    public Result<?> configScoreModel(@RequestBody ScoreModelVo scoreModelVo) {
        UserVo loginUser = UserUtil.getLoginUser();
        scoreModelService.addOrUpdate(scoreModelVo, 1);
        //调用接口，重新打分
        String subjectId = scoreModelVo.getSubjectId();
        HttpHeaders headers = new HttpHeaders();
        remoteModelService.reScoreRequest(subjectId, "1887695061047672834", headers);
        log.info("专题-{}，通用打分配置修改，重新打分请求发送成功", subjectId);
        return Result.OK();
    }



    /**
     * 专题打分模型配置信息详情
     *
     * @param subjectId 专题id
     * @param type      打分模型种类（1：通用 2：专家观点 3：案例）
     * @author lkg
     * @date 2024/12/27
     */
    @GetMapping("/scoreModelInfo")
    public Result<?> scoreModelInfo(@RequestParam String subjectId, @RequestParam(defaultValue = "1") String type) {
        JSONObject jsonObject = scoreModelService.scoreModelInfo(subjectId, type);
        return Result.OK(jsonObject);
    }

    /**
     * 数据资源绑定-专题配置页
     *
     * @param subjectPage 参数
     * @author lkg
     * @date 2024/12/18
     */
    @PostMapping(value = "/sourceBind")
    public Result<?> sourceBindOrUpdate(@RequestBody SubjectPage subjectPage) {
        Subject subject = new Subject();
        BeanUtils.copyProperties(subjectPage, subject);
        subjectService.updateById(subject);
        return Result.OK();
    }

    /**
     * 模型绑定（包含去重、筛选、标签模型）--检查是否需要中断-专题配置页
     * todo
     *
     * @param subjectPage 参数
     * @return
     */
    @PostMapping(value = "/checkIsBreak")
    public Result<?> checkIsBreak(@RequestBody SubjectPage subjectPage) {
        return subjectService.checkIsBreak(subjectPage);
    }

    /**
     * 前端发送请求，传参数给python
     * todo
     */
    @PostMapping(value = "/sendPythonInfo")
    public Result<?> sendPythonInfo(@RequestBody SubjectPage subjectPage) {
        subjectService.send(subjectPage.getId(), null);
        return Result.OK();
    }

    /**
     * 获取分类下的标签信息
     *
     * @param labelTypeId 标签分类id
     * @author lkg
     * @date 2025/1/2
     */
    @GetMapping("/labelList")
    public Result<?> labelList(@RequestParam(defaultValue = "1874739452451004417") String labelTypeId) {
        List<InfoSourceLabelVO> labelList = clbLabelService.labelInfoByType(labelTypeId);
        return Result.OK(labelList);
    }

    /**
     * 专题绑定的信息源标签列表
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/1/13
     */
    @GetMapping("/subjectBindLabelList")
    public Result<?> subjectBindLabelList(@RequestParam String subjectId) {
        List<InfoSourceLabelVO> bindList = clbLabelService.bindInfoSourceLabelInfo(subjectId);
        return Result.OK(bindList);
    }
    /**
     * 更新subject的数据范围
     * @param subjectId 专题id
     */
    @GetMapping("/subjectUpdateScope")
    public Result<?> subjectUpdateScope(@RequestParam(name = "subjectId") String subjectId,
                                          @RequestParam(name = "type") String type,
                                          @RequestParam(name = "scop" , defaultValue = "") String scop
                                          ) {
        Subject subject = subjectService.subjectUpdateScope(subjectId,type,scop);
        return Result.OK(subject);
    }

    /**
     * 绑定信息源标签数据
     *
     * @param subjectSourceTagVO 参数
     * @author lkg
     * @date 2025/1/3
     */
    @PostMapping("/bindInfoSourceLabel")
    public Result<?> bindInfoSourceLabel(@RequestBody SubjectSourceTagVO subjectSourceTagVO) {
        List<InfoSourceLabelVO> labelList = subjectSourceTagVO.getLabelList();
        if (CollectionUtils.isEmpty(labelList)) {
            return Result.OK();
        }
        String subjectId = subjectSourceTagVO.getSubjectId();
        if (StringUtils.isEmpty(subjectId)) {
            return Result.FAIL("专题id不能为空");
        }
        List<SubjectInfoSourceMap> dataList = new ArrayList<>();
        for (InfoSourceLabelVO infoSourceLabelVO : labelList) {
            List<InfoSourceLabelItemVO> infoSourceLabelItemList = infoSourceLabelVO.getInfoSourceLabelItemList();
            if (CollectionUtils.isEmpty(infoSourceLabelItemList)) {
                continue;
            }
            String labelCode = infoSourceLabelVO.getLabelCode();
            for (InfoSourceLabelItemVO infoSourceLabelItemVO : infoSourceLabelItemList) {
                String labelItemCode = infoSourceLabelItemVO.getLabelItemCode();
                SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                subjectInfoSourceMap.setSubjectId(subjectId);
                subjectInfoSourceMap.setSourceId(labelCode);
                subjectInfoSourceMap.setSourceItemId(labelItemCode);
                subjectInfoSourceMap.setType(12);
                dataList.add(subjectInfoSourceMap);
            }
        }
        LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectInfoSourceMap::getSubjectId, subjectId).eq(SubjectInfoSourceMap::getType, 12);
        int count = subjectInfoSourceMapService.count(queryWrapper);
        if (count > 0) {
            subjectInfoSourceMapService.remove(queryWrapper);
        }
        subjectInfoSourceMapService.saveBatch(dataList);
        //同步配置到采集
        configurationMessageService.bindInfoSourceSend(subjectId);
        return Result.OK();
    }


    /**
     * 专题绑定的定向信息源分页列表(专题绑定定向信息源时使用-研究中心)
     *
     * @param subjectId 专题id
     * @param pageNo    当前页
     * @param pageSize  返回条数
     * @author lkg
     * @date 2025/1/4
     */
    @GetMapping("/subjectBindDirectSourcePageList")
    public Result<?> subjectBindDirectSourcePageList(@RequestParam String subjectId,
                                                     @RequestParam(defaultValue = "1") Integer pageNo,
                                                     @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<SubjectBindDirectSourceVO> page = infoSourceService.subjectBindDirectSourcePageList(subjectId, pageNo, pageSize);
        return Result.OK(page);
    }

    /**
     * 信息源分页列表(专题绑定定向信息源时使用-研究中心)
     *
     * @param searchWord 搜索词
     * @param pageNo     当前页
     * @param pageSize   返回条数
     * @author lkg
     * @date 2025/1/4
     */
    @GetMapping("/directInfoSourcePageList")
    public Result<?> directSourcePageList(@RequestParam(required = false) String searchWord,
                                          @RequestParam(defaultValue = "1") Integer pageNo,
                                          @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<SubjectBindDirectSourceVO> page = infoSourceService.directSourcePageList(searchWord, pageNo, pageSize);
        return Result.OK(page);
    }

    /**
     * 添加定向信息源(专题绑定定向信息源时使用-研究中心)
     *
     * @param subjectSourceVO 参数
     * @author lkg
     * @date 2025/1/4
     */
    @PostMapping("/bindDirectInfoSource")
    public Result<?> bindDirectInfoSource(@RequestBody SubjectSourceVO subjectSourceVO) {
        List<String> sourceIdList = subjectSourceVO.getSourceIdList();
        if (CollectionUtils.isEmpty(sourceIdList)) {
            LambdaQueryWrapper<InfoSource> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.select(InfoSource::getId,InfoSource::getSiteName,InfoSource::getSiteUri)
                    .in(InfoSource::getWebSiteName, subjectSourceVO.getWebSiteNameList())
                    .groupBy(InfoSource::getId);
            List<InfoSource> infoSources = infoSourceService.list(queryWrapper);
            if (CollectionUtils.isNotEmpty(infoSources)) {
                sourceIdList = infoSources.stream().map(InfoSource::getId).collect(Collectors.toList());
            }
        }
        if (CollectionUtils.isNotEmpty(sourceIdList)) {
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSubjectId, subjectSourceVO.getSubjectId())
                    .in(SubjectInfoSourceMap::getSourceId, sourceIdList)
                    .eq(SubjectInfoSourceMap::getType, 1);
            List<SubjectInfoSourceMap> list = subjectInfoSourceMapService.list(queryWrapper);
            if (CollectionUtils.isNotEmpty(list)) {
                List<String> existSourceIdList = list.stream().map(SubjectInfoSourceMap::getSourceId).collect(Collectors.toList());
                sourceIdList.removeAll(existSourceIdList);
            }
            if (CollectionUtils.isNotEmpty(sourceIdList)) {
                List<SubjectInfoSourceMap> dataList = new ArrayList<>();
                for (String sourceId : sourceIdList) {
                    SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                    subjectInfoSourceMap.setSubjectId(subjectSourceVO.getSubjectId());
                    subjectInfoSourceMap.setSourceId(sourceId);
                    subjectInfoSourceMap.setType(1);
                    dataList.add(subjectInfoSourceMap);
                }
                subjectInfoSourceMapService.saveBatch(dataList);
            }
        }
        //同步配置到采集
        configurationMessageService.bindInfoSourceSend(subjectSourceVO.getSubjectId());
        return Result.OK();
    }


    /**
     * 新增信息源-研究中心
     *
     * @param subjectInfoSourceMiddleMap 参数
     * @author lkg
     * @date 2025/1/4
     */
    @PostMapping("/addInfoSource")
    public Result<?> addInfoSource(@RequestBody SubjectInfoSourceMiddleMap subjectInfoSourceMiddleMap) {
        //专题信息源关系数据
        Set<SubjectInfoSourceMap> mapDataSet = new HashSet<>();
        //待配置信息源新增列表
        Set<SubjectInfoSourceMiddleMap> configDataSet = new HashSet<>();
        String subjectId = subjectInfoSourceMiddleMap.getSubjectId();
        String infoSourceName = subjectInfoSourceMiddleMap.getInfoSourceName();
        String url = subjectInfoSourceMiddleMap.getUrl();
        singleSupplyData(subjectId, infoSourceName, url, mapDataSet, configDataSet);
        if (CollectionUtils.isNotEmpty(mapDataSet)) {
            subjectInfoSourceMapService.saveBatch(mapDataSet);
        }
        if (CollectionUtils.isNotEmpty(configDataSet)) {
            subjectInfoSourceMiddleMapService.saveBatch(configDataSet);
        }
        return Result.OK();
    }

    /**
     * 添加定向信息源-推荐信息源(专题绑定定向信息源时使用-研究中心)
     *
     * @param params 参数
     * @author lkg
     * @date 2025/1/4
     */
    @PostMapping("/bindRecommendInfoSource")
    public Result<?> bindRecommendInfoSource(@RequestBody List<SubjectInfoSourceMap> params) {
        if (CollectionUtils.isNotEmpty(params)) {
            params.forEach(item -> item.setType(1));
            subjectInfoSourceMapService.saveBatch(params);
            //同步配置到采集
            configurationMessageService.bindInfoSourceSend(params.get(0).getSubjectId());
        }
        return Result.OK();
    }

    /**
     * 删除定向信息源(专题绑定定向信息源时使用-研究中心)
     *
     * @param params 参数
     * @author lkg
     * @date 2025/1/9
     */
    @PostMapping("/removeDirectInfoSource")
    public Result<?> removeDirectInfoSource(@RequestBody List<SubjectBindDirectSourceVO> params){
        for (SubjectBindDirectSourceVO param : params) {
            String subjectId = param.getSubjectId();
            String webSiteName = param.getWebSiteName();
            Integer status = param.getStatus();
            if (status == 1) {
                List<String> sourceIdList;
                List<InfoSource> children = param.getChildren();
                if (CollectionUtils.isNotEmpty(children)) {
                    sourceIdList = children.stream().map(InfoSource::getId).collect(Collectors.toList());
                } else {
                    List<String> webSiteNameList = new ArrayList<>();
                    webSiteNameList.add(webSiteName);
                    List<InfoSource> infoSources = infoSourceService.subjectBindSourceColumnList(subjectId,webSiteNameList);
                    sourceIdList = infoSources.stream().map(InfoSource::getId).collect(Collectors.toList());
                }
                subjectInfoSourceMapService.remove(new LambdaQueryWrapper<SubjectInfoSourceMap>()
                        .eq(SubjectInfoSourceMap::getSubjectId, subjectId)
                        .in(SubjectInfoSourceMap::getSourceId, sourceIdList));
            } else if (status == 0) {
                subjectInfoSourceMiddleMapService.remove(new LambdaQueryWrapper<SubjectInfoSourceMiddleMap>()
                        .eq(SubjectInfoSourceMiddleMap::getSubjectId, subjectId)
                        .eq(SubjectInfoSourceMiddleMap::getInfoSourceName, webSiteName));
            }
        }
        //同步配置到采集
        configurationMessageService.bindInfoSourceSend(params.get(0).getSubjectId());
        return Result.OK();
    }

    /**
     * 信息源导入模板下载-研究中心
     *
     * @author lkg
     * @date 2024/06/21
     */
    @GetMapping("/downloadInfoSourceTemplate")
    public void downloadTemplate(HttpServletResponse response) {
        String filePath = "infoSourceImport/导入模板.xlsx";
        commonService.downloadTemplate(response, filePath);
    }

    /**
     * 导入信息源-研究中心
     *
     * @param file      excel文件
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/1/4
     */
    @PostMapping("/importDirectInfoSource")
    public Result<?> importDirectInfoSource(MultipartFile file, String subjectId) throws Exception {
        if (StringUtils.isBlank(subjectId)) {
            return Result.FAIL("专题id不能为空");
        }
        if (file == null) {
            return Result.FAIL("请选择上传文件");
        }
        String originalFilename = file.getOriginalFilename();
        if (originalFilename.endsWith(".xlsx") || originalFilename.endsWith(".xls")) {
            List<List<String>> dataList;
            if (originalFilename.endsWith(".xls")) {
                dataList = ExcelExportUtil.readExcelXls(file.getInputStream(), false,1, 2);
            } else {
                dataList = ExcelExportUtil.readExcelXlsx(file.getInputStream(), false,1, 2);
            }
            if (CollectionUtils.isNotEmpty(dataList)) {
                boolean emptyFlag = false;
                for (List<String> strings : dataList) {
                    String name = strings.get(0);
                    String url = strings.get(1);
                    if (StringUtils.isEmpty(name) || StringUtils.isEmpty(url)) {
                        emptyFlag = true;
                        break;
                    }
                }
                if (emptyFlag) {
                    return Result.FAIL("存在信息源名称/信息源地址为空的数据,请核对后在导入!");
                }
                //专题信息源关系数据
                Set<SubjectInfoSourceMap> mapDataSet = new HashSet<>();
                //待配置信息源新增列表
                Set<SubjectInfoSourceMiddleMap> configDataSet = new HashSet<>();
                for (List<String> data : dataList) {
                    String webSiteName = data.get(0);
                    String siteUri = data.get(1);
                    singleSupplyData(subjectId, webSiteName, siteUri, mapDataSet, configDataSet);
                }
                if (CollectionUtils.isNotEmpty(mapDataSet)) {
                    subjectInfoSourceMapService.saveBatch(mapDataSet);
                }
                if (CollectionUtils.isNotEmpty(configDataSet)) {
                    subjectInfoSourceMiddleMapService.saveBatch(configDataSet);
                }
            } else {
                return Result.FAIL("上传的文件没有内容");
            }
        } else {
            return Result.FAIL("不支持的文件格式");
        }
        return Result.OK();
    }

    /**
     * 单挑数据填充
     *
     * @param subjectId     专题id
     * @param webSiteName   信息源名称
     * @param siteUri       信息源地址
     * @param mapDataSet    专题信息源关系数据
     * @param configDataSet 待配置信息源数据
     * @author lkg
     * @date 2025/1/4
     */
    private void singleSupplyData(String subjectId, String webSiteName, String siteUri, Set<SubjectInfoSourceMap> mapDataSet, Set<SubjectInfoSourceMiddleMap> configDataSet) {
        String domain = Utility.domainURL(siteUri);
        if (StringUtils.isNotEmpty(domain)) {
            String replace = siteUri.replace("https://", "").replace("http://", "");
            if (replace.equalsIgnoreCase(domain) || replace.equalsIgnoreCase(domain + "/")) {
                List<InfoSource> infoSources = matchInfoSourceByUri(domain, 2);
                supplyData(subjectId, webSiteName, siteUri, mapDataSet, configDataSet, infoSources);
            } else {
                List<InfoSource> infoSources = matchInfoSourceByUri(siteUri, 1);
                supplyData(subjectId, webSiteName, siteUri, mapDataSet, configDataSet, infoSources);
            }
        } else {
            List<InfoSource> infoSources = matchInfoSourceByUri(siteUri, 1);
            supplyData(subjectId, webSiteName, siteUri, mapDataSet, configDataSet, infoSources);
        }
    }

    /**
     * 填充数据
     *
     * @param subjectId     专题id
     * @param webSiteName   信息源名称
     * @param siteUri       信息源地址
     * @param mapDataSet    专题信息源关系数据
     * @param configDataSet 待配置信息源数据
     * @param infoSources   匹配到的信息源数据
     * @author lkg
     * @date 2025/1/4
     */
    private void supplyData(String subjectId, String webSiteName, String siteUri, Set<SubjectInfoSourceMap> mapDataSet, Set<SubjectInfoSourceMiddleMap> configDataSet, List<InfoSource> infoSources) {
        if (CollectionUtils.isNotEmpty(infoSources)) {
            for (InfoSource source : infoSources) {
                SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                subjectInfoSourceMap.setSubjectId(subjectId);
                subjectInfoSourceMap.setSourceId(source.getId());
                subjectInfoSourceMap.setType(1);
                mapDataSet.add(subjectInfoSourceMap);
            }
        } else {
            SubjectInfoSourceMiddleMap middleMap = new SubjectInfoSourceMiddleMap();
            middleMap.setInfoSourceName(webSiteName);
            middleMap.setUrl(siteUri);
            middleMap.setSubjectId(subjectId);
            configDataSet.add(middleMap);
        }
    }

    /**
     * 推荐信息源
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2024/12/26
     */
    @PostMapping("/recommendSourceList")
    public Result<?> recommendSourceList(@RequestBody InfoDataSearchCondition searchCondition) {
        List<InfoSource> infoSources = new ArrayList<>();
        String[] fetchFields = new String[]{"id", "sid"};
        searchCondition.setFetchFields(fetchFields);
        List<SearchWordVO> searchWordList = new ArrayList<>();
        for (String words : searchCondition.getWordsList()) {
            SearchWordVO searchWord = new SearchWordVO();
            searchWord.setSearchAccuracy("精确");
            searchWord.setSearchScope(1);
            searchWord.setSearchLogicRelationship("OR");
            searchWord.setSearchInfo(words);
            searchWordList.add(searchWord);
        }
        searchCondition.setSearchWordList(searchWordList);
        List<CountVO> countVOS = esService.groupBySourceId(searchCondition);
        if (CollectionUtils.isNotEmpty(countVOS)) {
            List<String> sourceIdList = new ArrayList<>();
            for (CountVO countVO : countVOS) {
                sourceIdList.add(countVO.getName());
            }
            LambdaQueryWrapper<InfoSource> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.select(InfoSource::getId,InfoSource::getSiteName,InfoSource::getSiteUri).in(InfoSource::getId,sourceIdList);
            infoSources = infoSourceService.list(queryWrapper);
        }
        return Result.OK(infoSources);
    }


    /**
     * 热点地图-专题分析页
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/subjectRegionStatistic")
    public Result<?> subjectRegionStatistic(@RequestParam String subjectId) {
        List<CountVO> countVoList = esService.getRegionStatistic(subjectId);
        return Result.OK(countVoList);
    }

    /**
     * 来源占比-专题分析页
     *
     * @param subjectId 专题id
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @param size      返回个数
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/originRatio")
    public Result<?> originRatio(@RequestParam(value = "size", defaultValue = "10") int size,
                                 @RequestParam(value = "startDate", required = false) String startDate,
                                 @RequestParam(value = "endDate", required = false) String endDate,
                                 @RequestParam(value = "subjectId", required = false) String subjectId) {
        List<CountVO> countVOS = esService.groupByOrigin(subjectId, "publishDate", startDate, endDate, size);
        return Result.OK(countVOS);
    }

    /**
     * @param startDate 统计开始日期
     * @param endDate   统计结束日期
     * @param subjectId 专题id
     * @描述 获取专题统计信息
     * @作者 hejinyu
     * @创建时间 2024/12/13
     **/
    @GetMapping("/listSubjectCountInfo")
    public Result<?> listSubjectCountInfo(@RequestParam String startDate, @RequestParam String endDate, @RequestParam String subjectId) {
        //获取统计天数
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        LocalDate localStartDate = LocalDate.parse(startDate, formatter);
        //查询有查结束日期当日的数据，所以获取日期数组时得多算一天
        LocalDate localEndDate = LocalDate.parse(endDate, formatter).plusDays(1);
        long days = ChronoUnit.DAYS.between(localStartDate, localEndDate);
        String type = "day";
        //参数日期间隔只有一天时，做日报表
        if (days == 1) {
            type = "hour";
        }
        if (days > 180) {
            type = "month";
        }
        Map<String, List<CountVO>> resultMap = new HashMap<>();
        //已采集量
        List<CountVO> collectedData = esService.getCollectedData(startDate, endDate, type, subjectId);
        resultMap.put("collectedData", collectedData);
        //已入库量
        List<CountVO> stockdata = esService.getSubjectDatabaseByTime(startDate, endDate, type, subjectId, null, null);
        resultMap.put("stockdata", stockdata);
        //免审核入库的数据
        List<CountVO> freeCheckdata = esService.getSubjectDatabaseByTime(startDate, endDate, type, subjectId, 1, null);
        resultMap.put("freeCheckdata", freeCheckdata);
        //审核通过的数据
        List<CountVO> checkeddata = esService.getSubjectDatabaseByTime(startDate, endDate, type, subjectId, null, 1);
        resultMap.put("checkeddata", checkeddata);
        return Result.OK(resultMap);
    }

    /*
     * 热词词云-专题分析页
     *
     * @param subjectId
     * @param size
     * @author lkg
     * @date 2024/12/27
     */
    @GetMapping("/keywordsCount")
    public Result<?> keywordsCount(@RequestParam String subjectId, @RequestParam(defaultValue = "100") Integer size) {
        Set<String> excludeKeywords = commonService.getExcludeKeywords(subjectId);
        List<CountVO> countVOS = esService.keywordsCount(subjectId, new ArrayList<>(excludeKeywords), size);
        return Result.OK(countVOS);
    }


    private List<StatisticsKeyWordVo> extractWords(String words) {
        List<StatisticsKeyWordVo> wordList = new ArrayList<>();
        String[] fetchFields = new String[]{"title", "content"};
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setFetchFields(fetchFields);
        List<SearchWordVO> searchWordList = new ArrayList<>();
        SearchWordVO searchWord = new SearchWordVO();
        searchWord.setSearchAccuracy("精确");
        searchWord.setSearchScope(1);
        searchWord.setSearchLogicRelationship("AND");
        searchWord.setSearchInfo(words);
        searchWordList.add(searchWord);
        searchCondition.setSearchWordList(searchWordList);
        searchCondition.setColumn("score");
        searchCondition.setOrder("desc");
        try {
            IPage<SpecialInformation> page = esService.pageListByCondition(searchCondition, null);
            if (page.getTotal() > 0) {
                StringBuilder text = new StringBuilder();
                List<SpecialInformation> records = page.getRecords();
                for (SpecialInformation information : records) {
                    text.append(information.getTitle()).append(information.getContent());
                }
                //List<String> wordsList = pythonUtil.extractKeyword(text.toString(), 10);
                //wordList = formatWordInfo(text.toString(), wordsList);
                List<Map.Entry<String, Integer>> extractKeyWordsByText = HanlpUtil.extractKeyWordsByText(text.toString(), 10);
                if (CollectionUtils.isNotEmpty(extractKeyWordsByText)) {
                    for (Map.Entry<String, Integer> entry : extractKeyWordsByText) {
                        StatisticsKeyWordVo statisticsKeyWordVo = new StatisticsKeyWordVo();
                        statisticsKeyWordVo.setName(entry.getKey());
                        statisticsKeyWordVo.setValue(entry.getValue());
                        wordList.add(statisticsKeyWordVo);
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return wordList.stream().sorted(Comparator.comparing(StatisticsKeyWordVo::getValue).reversed()).collect(Collectors.toList());
    }

    /**
     * 根据样例文章抽取热词
     *
     * @param files 样例文章
     * @author lkg
     * @date 2025/1/3
     */
    private List<StatisticsKeyWordVo> articleWords(MultipartFile[] files) {
        StringBuilder text = new StringBuilder();
        for (MultipartFile file : files) {
            String originalFilename = file.getOriginalFilename();
            if (originalFilename.endsWith(".txt")) {
                parseTxt(text, file);
            } else if (originalFilename.endsWith(".docx")){
                parseWordDocx(text, file);
            } else if (originalFilename.endsWith(".doc")) {
                parseWordDoc(text, file);
            }
        }
        List<StatisticsKeyWordVo> articleWordList = new ArrayList<>();
        if (StringUtils.isNotEmpty(text)) {
            List<String> wordsList = pythonUtil.extractKeyword(text.toString(), 10);
            articleWordList = formatWordInfo(text.toString(), wordsList);
            /*List<Map.Entry<String, Integer>> keywordsList = HanlpUtil.extractKeyWordsByText(text.toString(), 10);
            if (CollectionUtils.isNotEmpty(keywordsList)) {
                for (Map.Entry<String, Integer> entry : keywordsList) {
                    StatisticsKeyWordVo statisticsKeyWordVo = new StatisticsKeyWordVo();
                    statisticsKeyWordVo.setName(entry.getKey());
                    statisticsKeyWordVo.setValue(entry.getValue());
                    articleWordList.add(statisticsKeyWordVo);
                }
            }*/
        }
        return articleWordList.stream().sorted(Comparator.comparing(StatisticsKeyWordVo::getValue).reversed()).collect(Collectors.toList());
    }


    private List<StatisticsKeyWordVo> formatWordInfo(String text,List<String> wordsList){
        List<StatisticsKeyWordVo> articleWordList = new ArrayList<>();
        Map<String, Integer> hitWordsAndTimes = HanlpUtil.getHitWordsAndTimes(wordsList, text);
        for (Map.Entry<String, Integer> entry : hitWordsAndTimes.entrySet()) {
            StatisticsKeyWordVo statisticsKeyWordVo = new StatisticsKeyWordVo();
            statisticsKeyWordVo.setName(entry.getKey());
            statisticsKeyWordVo.setValue(entry.getValue());
            articleWordList.add(statisticsKeyWordVo);
        }
        return articleWordList;
    }

    /**
     * 解析word文档,docx后缀，获取纯文本内容
     *
     * @param text 内容
     * @param file word文件
     * @author lkg
     * @date 2025/1/3
     */
    private void parseWordDocx(StringBuilder text, MultipartFile file) {
        try {
            InputStream inputStream = file.getInputStream();
            XWPFDocument docx = new XWPFDocument(inputStream);
            XWPFWordExtractor extractor = new XWPFWordExtractor(docx);
            text.append(extractor.getText());
            inputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 解析word文档,doc后缀，获取纯文本内容
     *
     * @param text 内容
     * @param file word文件
     * @author lkg
     * @date 2025/1/3
     */
    private void parseWordDoc(StringBuilder text, MultipartFile file) {
        try {
            InputStream inputStream = file.getInputStream();
            HWPFDocument doc = new HWPFDocument(inputStream);
            WordExtractor wordExtractor = new WordExtractor(doc);
            text.append(wordExtractor.getText());
            inputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 解析txt文件，获取纯文本内容
     *
     * @param text 内容
     * @param file word文件
     * @author lkg
     * @date 2025/1/3
     */
    private void parseTxt(StringBuilder text, MultipartFile file) {
        try {
            InputStreamReader reader = new InputStreamReader(file.getInputStream(), "GBK");
            BufferedReader buffReader = new BufferedReader(reader);
            String strTmp;
            while ((strTmp = buffReader.readLine()) != null) {
                text.append(strTmp);
            }
            buffReader.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 根据地址匹配信息源列表
     *
     * @param uri  地址/域名
     * @param type 1-地址精确匹配;2-域名模糊匹配
     * @author lkg
     * @date 2025/1/4
     */
    private List<InfoSource> matchInfoSourceByUri(String uri, Integer type) {
        LambdaQueryWrapper<InfoSource> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.select(InfoSource::getId);
        if (type == 1) {
            queryWrapper.eq(InfoSource::getSiteUri, uri);
        } else {
            queryWrapper.like(InfoSource::getSiteUri, uri);
        }
        return infoSourceService.list(queryWrapper);
    }
}
