package com.zzsn.event.service;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.util.EsIndexUtil;
import com.zzsn.event.vo.DisplayInfo;
import com.zzsn.event.vo.RepeatHold;
import com.zzsn.event.vo.SpecialInformation;
import com.zzsn.event.vo.SubjectInfoVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.collapse.CollapseBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class SubjectDisplayServive {

    @Autowired
    private RestHighLevelClient client;


    /**
     * 分页获取数据
     *
     * @param subjectInfo          查询条件封装
     * @param offset               页数
     * @param pageSize             每页条数
     * @param labelIds
     * @param socialCreditCodeList 企业信用代码集合
     * @param sourceId
     * @throws Exception 异常
     */
    public IPage<DisplayInfo> frontListByPage(SubjectInfoVo subjectInfo, String video, int offset, int pageSize,
                                              String column, String order, String crawler, String labelIds, List<String> socialCreditCodeList, String sourceId) throws Exception {
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((offset - 1) * pageSize);
        //默认按照置顶以及时间倒序排列
        //根据topNum正序查找，查询置顶数据
        searchSourceBuilder.sort("topNum", SortOrder.DESC);
        if (column.equals("score")) {
            if (order.equals("asc")) {
                searchSourceBuilder.sort("score", SortOrder.ASC);
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else if (order.equals("desc")) {
                searchSourceBuilder.sort("score", SortOrder.DESC);
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                searchSourceBuilder.sort("score", SortOrder.DESC);
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                searchSourceBuilder.sort("score", SortOrder.ASC);
            }
        }
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();


        boolQuery.must(QueryBuilders.matchQuery("subjectId", subjectInfo.getSubjectId()));

        if (StringUtils.isNotEmpty(subjectInfo.getSearch())) {
            boolQuery.must(QueryBuilders.multiMatchQuery(subjectInfo.getSearch(), "title", "content"));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getContent())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("content", subjectInfo.getContent()));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getTitle())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("title", subjectInfo.getTitle()));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getOrigin())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("origin", subjectInfo.getOrigin()));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getSourceAddress())) {
            boolQuery.must(QueryBuilders.termQuery("sourceAddress", subjectInfo.getSourceAddress()));
        }
        if (StringUtils.isNotEmpty(subjectInfo.getId())) {
            boolQuery.must(QueryBuilders.termQuery("id", subjectInfo.getId()));
        }
        if (subjectInfo.getCheckStatusList() != null && subjectInfo.getCheckStatusList().size() > 0) {
            List<Integer> list = new ArrayList<>();
            list = subjectInfo.getCheckStatusList();
            if (list.size() == 1 && list.contains(4)) {
                //
            } else {
                boolQuery.must(QueryBuilders.termsQuery("checkStatus", subjectInfo.getCheckStatusList()));
            }
        }
        if (StringUtils.isNotBlank(video)) {
            boolQuery.must(QueryBuilders.matchQuery("type", "video"));
        } else {
            boolQuery.mustNot(QueryBuilders.matchQuery("type", "video"));
        }
        //只查询主条目
        boolQuery.mustNot(QueryBuilders.matchQuery("flag", "0"));
        //删除状态查询
        if (subjectInfo.getDeleteFlag() != 0) {
            boolQuery.must(QueryBuilders.matchQuery("deleteFlag", "1"));
        } else {
            boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        }

        //专题库类别筛选
        if (subjectInfo.getClassificationType() != null) {
            boolQuery.must(QueryBuilders.termQuery("classificationType", subjectInfo.getClassificationType()));
        }

        //正负面
        if (subjectInfo.getOrientation() != null) {
            boolQuery.must(QueryBuilders.termQuery("orientation", subjectInfo.getOrientation()));
        }

        //时间过滤筛选
        if (StringUtils.isNotBlank(subjectInfo.getStartTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(subjectInfo.getStartTime())));
        }
        if (StringUtils.isNotBlank(subjectInfo.getEndTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(subjectInfo.getEndTime())));
        } else {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(new Date()))));
        }
        if (StringUtils.isNotBlank(crawler)) {
            boolQuery.must(QueryBuilders.matchQuery("source", crawler));
        }
        Set<String> relationIds = new HashSet<>();
        if (StringUtils.isNotEmpty(labelIds)) {
            relationIds.add(labelIds);
        }
        if (CollectionUtils.isNotEmpty(socialCreditCodeList)) {
            relationIds.addAll(socialCreditCodeList);
        }
        if (CollectionUtils.isNotEmpty(relationIds)) {
            BoolQueryBuilder nestedBoolQueryBuilder = QueryBuilders.boolQuery();
            for (String relationId : relationIds) {
                TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationId", relationId);
                nestedBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            }
            boolQuery.must(nestedBoolQueryBuilder);
        }
        if (subjectInfo.getLabelName() != null) {
            BoolQueryBuilder nestedBoolQueryBuilder = QueryBuilders.boolQuery();
            TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationName", subjectInfo.getLabelName());
            nestedBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            boolQuery.must(nestedBoolQueryBuilder);
        }
        if (StringUtils.isNotBlank(sourceId)) {
            boolQuery.must(QueryBuilders.termQuery("sid", sourceId));
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHit[] searchHits = searchResponse.getHits().getHits();
        List<DisplayInfo> list = new ArrayList<>();
        for (SearchHit hit : searchHits) {
            String queryInfo = hit.getSourceAsString();
            DisplayInfo info = JSONUtil.toBean(queryInfo, DisplayInfo.class);
            info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
            String index = hit.getIndex();
            info.setIndex(index);
            info.setSimilarNumber(0);
            list.add(info);
        }
        Map<String, Integer> map = getSimilarNumber(list.stream().map(DisplayInfo::getId).collect(Collectors.toList()));
        for (DisplayInfo displayInfo : list) {
            displayInfo.setSimilarNumber(map.get(displayInfo.getId()));
        }
        IPage<DisplayInfo> pageData = new Page<>(offset, pageSize, searchResponse.getHits().getTotalHits().value);

        pageData.setRecords(list);
        return pageData;
    }

    /**
     * 获取相似文章数
     *
     * @param articleIdList 资讯id
     * @return
     */
    private Map<String, Integer> getSimilarNumber(List<String> articleIdList) {
        Map<String, Integer> map = new HashMap<>();
        articleIdList.add("70694089423478814");
        articleIdList.add("71056913337311317");
        articleIdList.add("71056913337311282");
        Map<String, String> markmap = getMark(articleIdList);
        List<String> markList = new ArrayList<>(markmap.keySet());
        SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("repeatMark", markList));
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_mark")
                .field("repeatMark")
                .order(BucketOrder.count(false))
                .size(1);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();
            Terms groupSource = aggregations.get("group_mark");
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(buckets)) {
                for (Terms.Bucket bucket : buckets) {
                    map.put(markmap.get(bucket.getKeyAsString()), (int) bucket.getDocCount());
                }

            }


        } catch (Exception e) {
            e.printStackTrace();
        }
        return map;
    }

    private Map<String, String> getMark(List<String> articleIdList) {
        Map<String, String> map = new HashMap<>();
        String mark = null;
        SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("articleId", articleIdList));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = response.getHits().getHits();
            if (hits != null && hits.length != 0) {
                for (SearchHit hit : hits) {
                    String queryInfo = hit.getSourceAsString();
                    RepeatHold info = JSONUtil.toBean(queryInfo, RepeatHold.class);
                    map.put(info.getRepeatMark(), info.getArticleId());
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return map;
    }

    /**
     * 根据标题获取相关推荐文章
     *
     * @param title
     * @return
     */
    public IPage<SpecialInformation> queryRecommendList(String title, Integer pageNo, Integer pageSize) throws IOException {
        String[] indexs = EsIndexUtil.getIndexLatelyTwoYear(Constants.ES_DATA_FOR_SUBJECT);
        SearchRequest searchRequest = new SearchRequest(indexs);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        searchSourceBuilder.sort("score", SortOrder.DESC);
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        List<Integer> checkStatus = new ArrayList<>();
        checkStatus.add(1);
        boolQuery.must(QueryBuilders.termsQuery("checkStatus", checkStatus));
        boolQuery.must(QueryBuilders.matchQuery("title", title));
        boolQuery.mustNot(QueryBuilders.matchQuery("type", "video"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        searchSourceBuilder.collapse(new CollapseBuilder("sourceAddress.keyword"));
        SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHit[] searchHits = searchResponse.getHits().getHits();
        List<SpecialInformation> list = new ArrayList<>();
        for (SearchHit hit : searchHits) {
            String queryInfo = hit.getSourceAsString();
            SpecialInformation info = JSON.parseObject(queryInfo, SpecialInformation.class);
            info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
            list.add(info);
        }
        IPage<SpecialInformation> pageData = new Page<>(pageNo, pageSize, searchResponse.getHits().getTotalHits().value);
        pageData.setRecords(list);
        return pageData;
    }

}

