package com.zzsn.event.controller;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.LabelEntity;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.service.ColumnService;
import com.zzsn.event.service.EsStatisticsService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.SubjectDisplayServive;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.util.RestUtil;
import com.zzsn.event.vo.*;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.index.query.WildcardQueryBuilder;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramInterval;
import org.elasticsearch.search.aggregations.bucket.histogram.Histogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 报告数据接口
 *
 * @author lkg
 * @date 2023/3/29
 */
@RestController
@RequestMapping("/datapull/report/api")
public class ReportApiController {

    @Autowired
    private RestHighLevelClient client;
    @Autowired
    private ColumnService columnService;
    @Autowired
    private IEventService eventService;
    @Autowired
    private EsStatisticsService esStatisticsService;
    @Autowired
    private SubjectDisplayServive subjectDisplayServive;

    @Value("${subject.analysis-url:}")
    private String analysisUrl;


    /**
     * 所有资讯类数据的分页列表
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param pageNo    当前页
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/2/28
     */
    @GetMapping("/informationList")
    public Result<?> informationList(@RequestParam String startTime, @RequestParam String endTime,
                                     @RequestParam(defaultValue = "1") Integer pageNo,
                                     @RequestParam(defaultValue = "10") Integer pageSize) {
        List<ReportDataVo> informationList = columnService.informationList(startTime, endTime, pageNo, pageSize);
        return Result.OK(informationList);
    }


    /**
     * 专题数据分页列表
     *
     * @param subjectId 专题id
     * @param pageNo    当前页
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/2/28
     */
    @GetMapping("/subjectDataList")
    public Result<?> subjectDataList(@RequestParam String subjectId,
                                     @RequestParam(defaultValue = "1") Integer pageNo,
                                     @RequestParam(defaultValue = "10") Integer pageSize) {
        List<ReportDataVo> dataList = columnService.subjectDataList(subjectId, pageNo, pageSize);
        return Result.OK(dataList);
    }

    /**
     * 信息总量走势图(来源)
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/2/2
     */
    @GetMapping("/source")
    public Result<?> source(@RequestParam(required = false) String column,
                            @RequestParam(required = false) boolean ynGroup,
                            @RequestParam(required = false) String subjectId,
                            @RequestParam String startTime, @RequestParam String endTime) {
        List<CountVO> source = columnService.source(column, ynGroup, subjectId, startTime, endTime);
        for (CountVO countVO : source) {
            supply(countVO, startTime, endTime);
        }
        return Result.OK(source);
    }


    /**
     * 信息属性占比
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/2/2
     */
    @GetMapping("/orientation")
    public Result<?> orientation(@RequestParam(required = false) String column,
                                 @RequestParam(required = false) boolean ynGroup,
                                 @RequestParam(required = false) String subjectId,
                                 @RequestParam String startTime, @RequestParam String endTime) {
        List<CountVO> list = new ArrayList<>();
        String labelTypeId = "1631119596744265729";
        List<LabelEntity> labelEntities = eventService.listByType(labelTypeId);
        AtomicLong total = new AtomicLong();
        labelEntities.forEach(e -> {
            CompletableFuture<CountVO> async = CompletableFuture.supplyAsync(() -> {
                CountVO countVO = columnService.orientation(column, true, subjectId, startTime, endTime, e.getId());
                countVO.setKey(e.getName());
                total.addAndGet(Long.parseLong(countVO.getCount()));
                supply(countVO, startTime, endTime);
                return countVO;
            });
            try {
                CountVO countVO = async.get();
                list.add(countVO);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        });
        for (CountVO countVO : list) {
            long value = Long.parseLong(countVO.getCount());
            long totalCount = total.get();
            String divide = CalculateUtil.divide(String.valueOf(value), String.valueOf(totalCount));
            String percentage = "0%";
            if (org.apache.commons.lang3.StringUtils.isNotEmpty(divide)) {
                percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
            }
            countVO.setPercentage(percentage);
        }
        return Result.OK(list);
    }

    /**
     * 媒体分布
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/2/2
     */
    @GetMapping("/media")
    public Result<?> media(@RequestParam(required = false) String column,
                           @RequestParam(required = false) boolean ynGroup,
                           @RequestParam(required = false) String subjectId,
                           @RequestParam String startTime, @RequestParam String endTime) {
        List<MediaVO> media = columnService.media(column, ynGroup, subjectId, startTime, endTime);
        return Result.OK(media);
    }

    /**
     * 平台活跃度
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/2/2
     */
    @GetMapping("/origin")
    public Result<?> origin(@RequestParam(required = false) String column,
                            @RequestParam(required = false) boolean ynGroup,
                            @RequestParam(required = false) String subjectId,
                            @RequestParam String startTime, @RequestParam String endTime) {
        List<CountVO> origin = columnService.origin(column, ynGroup, subjectId, startTime, endTime);
        return Result.OK(origin);
    }

    /**
     * 影响力
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/2/2
     */
    @GetMapping("/influence")
    public Result<?> influence(@RequestParam(required = false) String column,
                               @RequestParam(required = false) boolean ynGroup,
                               @RequestParam(required = false) String subjectId,
                               @RequestParam String startTime, @RequestParam String endTime) {
        List<CountVO> influence = columnService.influence(column, ynGroup, subjectId, startTime, endTime);
        return Result.OK(influence);
    }


    /**
     * 关键词云
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param size      热词个数
     * @author lkg
     * @date 2024/2/2
     */
    @GetMapping("/hotWords")
    public Result<?> hotWords(@RequestParam(required = false) String column,
                              @RequestParam(required = false) boolean ynGroup,
                              @RequestParam(required = false) String subjectId,
                              @RequestParam String startTime, @RequestParam String endTime,
                              @RequestParam(defaultValue = "20") Integer size) {
        List<CountVO> horWords = columnService.horWords(column, ynGroup, subjectId, startTime, endTime, size);
        return Result.OK(horWords);
    }

    /**
     * 信息传播路径
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/1/29
     */
    @GetMapping("/propagationPath")
    public String propagationPath(@RequestParam String subjectId) {
        JSONObject variables = new JSONObject();
        variables.put("subjectId", subjectId);
        String url = analysisUrl + "/analysis/propagationPath";
        return RestUtil.get(url, variables).toJSONString();
    }

    /**
     * 传播力
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/2/27
     */
    @GetMapping("/spread")
    public Result<?> spread(@RequestParam String subjectId) {
        Map<String, Object> map = new HashMap<>();
        Subject subject = new Subject();
        Date timeEnable = subject.getTimeEnable();
        Date timeDisable = subject.getTimeDisable();
        DateTime beforeOneHour = DateUtil.offset(timeDisable, DateField.HOUR, -1);
        Long count = columnService.count(null, null, subjectId, DateUtil.formatDateTime(beforeOneHour), DateUtil.formatDateTime(timeDisable));
        map.put("increaseBefore", count);
        Map<String, Object> maxValueMap = columnService.maxValue(subjectId);
        map.put("hourTime", maxValueMap.get("hourTime"));
        map.put("maxValue", maxValueMap.get("maxValue"));
        long hours = DateUtil.between(timeEnable, timeDisable, DateUnit.HOUR);
        map.put("hours", hours);
        String spreadSpeed = CalculateUtil.divide(count.toString(), String.valueOf(hours), 2);
        //传播速度
        map.put("spreadSpeed", spreadSpeed);
        List<MediaVO> media = columnService.media(null, null, subjectId, null, null);
        StringBuilder mediaStr = new StringBuilder("其中");
        StringBuilder stringBuilder = new StringBuilder();
        int total = 0;
        for (MediaVO mediaVO : media) {
            String name = mediaVO.getKey();
            Long mediaNum = mediaVO.getMediaNum();
            total += mediaNum;
            stringBuilder.append(",").append(name).append(mediaNum).append("家");
        }
        mediaStr.append(total).append("家媒体参与报道,");
        if (stringBuilder.length() > 0) {
            mediaStr.append(stringBuilder.substring(1));
        }
        map.put("media", mediaStr.toString());
        return Result.OK(map);
    }


    //补充缺失的时间
    private void supply(CountVO countVO, String startTime, String endTime) {
        List<CountVO> list = new ArrayList<>();
        List<CountVO> children = countVO.getChildren();
        Map<String, CountVO> map = children.stream().collect(Collectors.toMap(CountVO::getKey, item -> item, (k1, k2) -> k2));
        DateTime startDate = DateUtil.parseDateTime(startTime);
        DateTime endDate = DateUtil.parseDateTime(endTime);
        List<DateTime> rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.DAY_OF_YEAR);
        for (DateTime dateTime : rangeToList) {
            String date = DateUtil.format(dateTime, "yyyy-MM-dd");
            if (map.containsKey(date)) {
                list.add(map.get(date));
            } else {
                CountVO vo = new CountVO();
                vo.setKey(date);
                vo.setCount("0");
                list.add(vo);
            }
        }
        countVO.setChildren(list);
    }


    /*------------------------------------------------------------------以下为---真实测试模板对应的数据接口---------------------------------------------------------------------------*/

    /**
     * 获取专题信息列表（最近10条）
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2023/3/29
     */
    @GetMapping("/dataOfSubject")
    public Result<?> dataOfSubject(@RequestParam("subjectId") String subjectId,
                                   @RequestParam(value = "startTime", required = false) String startTime,
                                   @RequestParam(value = "endTime", required = false) String endTime) {
        List<ReportDataVo> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(subjectId, startTime, endTime, 10);
        //按时间倒序
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = null;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (searchResponse != null) {
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                ReportDataVo info = JSON.parseObject(queryInfo, ReportDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                list.add(info);
            }
        }
        return Result.OK(list);
    }

    /**
     * 获取前几来源（饼图+表格）
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param size      前几
     * @author lkg
     * @date 2023/3/29
     */
    @GetMapping("/topOrigin")
    public Result<?> groupByOrigin(@RequestParam("subjectId") String subjectId,
                                   @RequestParam(value = "startTime", required = false) String startTime,
                                   @RequestParam(value = "endTime", required = false) String endTime,
                                   @RequestParam(value = "size", defaultValue = "10") Integer size) {
        List<Map<String, Object>> data = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(subjectId, startTime, endTime, 0);
        //按来源分组
        TermsAggregationBuilder originAggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .order(BucketOrder.count(false))
                .size(size); //返回的最大数据
        searchSourceBuilder.aggregation(originAggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = null;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (searchResponse != null) {
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取按来源分组集合
            Terms groupOrigin = aggregations.get("group_origin");
            List<? extends Terms.Bucket> originBuckets = groupOrigin.getBuckets();
            for (Terms.Bucket originBucket : originBuckets) {
                Map<String, Object> dataMap = new HashMap<>();
                dataMap.put("name", originBucket.getKeyAsString());
                dataMap.put("value", originBucket.getDocCount());
                data.add(dataMap);
            }
        }
        return Result.OK(data);
    }

    /**
     * 柱状图
     *
     * @param subjectId 专题id
     * @param size      前几
     * @author lkg
     * @date 2023/3/29
     */
    @GetMapping("/histogram")
    public Result<?> histogram(@RequestParam("subjectId") String subjectId, @RequestParam(value = "size", defaultValue = "10") Integer size) {
        List<Map<String, Object>> result = new ArrayList<>();
        List<Map<String, Object>> list = groupByOriginDate(subjectId, size);
        Map<String, List<Map<String, Object>>> originMap = list.stream().collect(Collectors.groupingBy(e -> e.get("name").toString()));
        for (Map.Entry<String, List<Map<String, Object>>> nameEntry : originMap.entrySet()) {
            Map<String, Object> data = new HashMap<>();
            String name = nameEntry.getKey();
            List<Map<String, Object>> value = nameEntry.getValue();
            data.put("name", name);
            for (Map<String, Object> map : value) {
                data.put(map.get("date").toString(), map.get("value"));
            }
            result.add(data);
        }
        return Result.OK(result);
    }

    /**
     * 折线图和双轴图
     *
     * @param subjectId 专题id
     * @param size      前几
     * @author lkg
     * @date 2023/3/29
     */
    @GetMapping("/discount")
    public Result<?> discount(@RequestParam("subjectId") String subjectId, @RequestParam(value = "size", defaultValue = "10") Integer size) {
        List<Map<String, Object>> ratioList = new ArrayList<>();
        List<Map<String, Object>> list = groupByOriginDate(subjectId, size);
        Map<String, List<Map<String, Object>>> dateMap = list.stream().collect(Collectors.groupingBy(e -> e.get("date").toString()));
        for (Map.Entry<String, List<Map<String, Object>>> dateEntry : dateMap.entrySet()) {
            Map<String, Object> ratio = new HashMap<>();
            String date = dateEntry.getKey();
            ratio.put("date", date);
            List<Map<String, Object>> value = dateEntry.getValue();
            for (Map<String, Object> map : value) {
                String name = map.get("name").toString();
                if ("网易新闻".equals(name)) {
                    ratio.put(name, map.get("value"));
                }
                if ("新浪财经".equals(name)) {
                    ratio.put(name, map.get("value"));
                }
            }
            Object sina = ratio.get("新浪财经");
            Object netEase = ratio.get("网易新闻");
            String divide = CalculateUtil.divide(sina == null ? null : sina.toString(), netEase == null ? null : netEase.toString(), 2);
            ratio.put("占比", CalculateUtil.multiply(divide, "100"));
            ratioList.add(ratio);
        }
        return Result.OK(ratioList);
    }

    /**
     * 现根据来源分组，再根据月份分组
     *
     * @param subjectId 专题id
     * @param size      前几
     * @author lkg
     * @date 2023/3/29
     */
    public List<Map<String, Object>> groupByOriginDate(String subjectId, Integer size) {
        List<Map<String, Object>> data = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(subjectId, null, null, 0);
        //按来源分组
        TermsAggregationBuilder originAggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .order(BucketOrder.count(false))
                .subAggregation(AggregationBuilders.dateHistogram("group_month")
                        .field("publishDate")
                        .calendarInterval(DateHistogramInterval.MONTH)
                        .format("yyyy-MM")).size(size); //返回的最大数据
        searchSourceBuilder.aggregation(originAggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = null;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (searchResponse != null) {
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取按来源分组集合
            Terms groupOrigin = aggregations.get("group_origin");
            List<? extends Terms.Bucket> originBuckets = groupOrigin.getBuckets();
            for (Terms.Bucket originBucket : originBuckets) {
                String originName = originBucket.getKeyAsString();
                long originCount = originBucket.getDocCount();
                Aggregations monthAggregations = originBucket.getAggregations();
                ParsedDateHistogram groupMonth = monthAggregations.get("group_month");
                for (Histogram.Bucket monthBucket : groupMonth.getBuckets()) {
                    Map<String, Object> map = new HashMap<>();
                    String month = monthBucket.getKeyAsString();
                    long count = monthBucket.getDocCount();
                    map.put("name", originName);
                    map.put("totalCount", originCount);
                    map.put("date", month);
                    map.put("value", count);
                    data.add(map);
                }
            }
        }
        return data;
    }


    private SearchSourceBuilder getSearchSourceBuilder(String subjectId, String startTime, String endTime, Integer originSize) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(originSize);//不输出原始数据
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        //专题筛选
        TermQueryBuilder termQuery = QueryBuilders.termQuery("subjectId", subjectId);
        boolQuery.must(termQuery);
        //时间范围筛选
        if (StringUtils.isNotBlank(startTime)) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(startTime)));
        }
        if (StringUtils.isNotBlank(endTime)) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(endTime)));
        } else {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.formatDateTime(new Date()))));
        }
        //过滤掉origin为空的数据
        WildcardQueryBuilder wildcardQueryBuilder = QueryBuilders.wildcardQuery("origin", "*");
        //主条目且未删除
        boolQuery.mustNot(QueryBuilders.matchQuery("flag", 0));
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        boolQuery.must(wildcardQueryBuilder);
        searchSourceBuilder.query(boolQuery);
        return searchSourceBuilder;
    }
    /**
     * 3.4 总体分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/1/24
     */
    @GetMapping("/total")
    public Result<?> totalAndMax(@RequestParam String subjectId, @RequestParam String startTime, @RequestParam String endTime) {
        Map<String, Object> map = esStatisticsService.totalAndMax(subjectId, null, null);
        long hours = DateUtil.between(DateUtil.parseDateTime(startTime), DateUtil.parseDateTime(endTime), DateUnit.HOUR);
        map.put("duration", hours);
        Object count = map.get("totalCount");
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(hours), 2);
        map.put("spread", divide);
//        String mainReport = esStatisticsService.mainReport(subjectId);
        map.put("mainReport", "mainReport");
        return Result.OK(map);
    }

    /**
     * 3.7 事件资讯列表展示
     *
     * @param video
     * @param pageNo        当前页
     * @param pageSize      返回条数
     * @param column        排序字段
     * @param order         排序方式
     * @param crawler
     * @param labelIds
     * @param sourceId
     * @author lkg
     * @date 2024/1/12
     */
    @ApiOperation(value = "专题信息列表-分页列表查询", notes = "专题信息列表-分页列表查询")
    @GetMapping(value = "/listArticle")
    public Result<?> queryPageList(
                                   @RequestParam(name = "subjectId", defaultValue = "") String subjectId,
                                   @RequestParam(name = "video", defaultValue = "") String video,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                   @RequestParam(name = "column", defaultValue = "common") String column,
                                   @RequestParam(name = "order", defaultValue = "desc") String order,
                                   @RequestParam(name = "crawler", defaultValue = "") String crawler,
                                   @RequestParam(name = "labelIds", required = false) String labelIds,
                                   @RequestParam(name = "sourceId", required = false) String sourceId) throws Exception {
        SubjectInfoVo subjectInfoVo=new SubjectInfoVo();
        subjectInfoVo.setSubjectId(subjectId);
        List<String> socialCreditCodeList = new ArrayList<>();
        //获取数据
        IPage<DisplayInfo> pageList = subjectDisplayServive.frontListByPage(subjectInfoVo, video, pageNo, pageSize, column, order, crawler, labelIds, socialCreditCodeList, sourceId);
        return Result.OK(pageList);
    }


}
