package com.zzsn.event.es;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.net.URLDecoder;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.google.common.base.Charsets;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.entity.SubjectStatisticInfo;
import com.zzsn.event.service.*;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.util.EsIndexUtil;
import com.zzsn.event.util.EsOpUtil;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.DisplayInfo;
import com.zzsn.event.vo.es.Label;
import com.zzsn.event.vo.es.SpecialInformation;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.client.HttpAsyncResponseConsumerFactory;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.text.Text;
import org.elasticsearch.index.query.*;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.aggregations.Aggregation;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramInterval;
import org.elasticsearch.search.aggregations.bucket.histogram.Histogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.nested.NestedAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.nested.ParsedNested;
import org.elasticsearch.search.aggregations.bucket.terms.IncludeExclude;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.Cardinality;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.collapse.CollapseBuilder;
import org.elasticsearch.search.fetch.subphase.FetchSourceContext;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightBuilder;
import org.elasticsearch.search.fetch.subphase.highlight.HighlightField;
import org.elasticsearch.search.sort.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.IOException;
import java.time.LocalDate;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * @author lkg
 * @description: es查询
 * @date 2022/7/7 9:57
 */
@Slf4j
@Service
public class EsService {

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private SubjectService subjectService;
    @Autowired
    private IEventService eventService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private ISubjectKeywordsMapService subjectKeywordsMapService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private EsOpUtil esOpUtil;

    /**
     * 获取专题下的资讯
     *
     * @param subjectId   专题id
     * @param startDate   开始时间
     * @param endDate     结束时间
     * @param fetchFields 抓取的字段
     */
    public List<SubjectDataVo> getDataBySubjectId(String subjectId, String startDate, String endDate, String[] fetchFields) {
        List<SubjectDataVo> allList = new ArrayList<>();
        AtomicBoolean flag = new AtomicBoolean(true);
        AtomicInteger pageNo = new AtomicInteger(1);
        int size = 300;
        do {
            CompletableFuture<List<SubjectDataVo>> async = CompletableFuture.supplyAsync(() -> {
                SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
                //设置需要获取的字段
                searchSourceBuilder.fetchSource(fetchFields, null);
                //设置返回条数
                int currentPage = pageNo.get();
                searchSourceBuilder.from((currentPage - 1) * size);
                searchSourceBuilder.size(size);
                //创建查询对象
                BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
                boolQuery.must(QueryBuilders.matchQuery("subjectId", subjectId));
                if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
                    RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("publishDate");
                    if (StringUtils.isNotBlank(startDate)) {
                        rangeQueryBuilder.gte(EsDateUtil.esFieldDateFormat(startDate));
                    }
                    if (StringUtils.isNotBlank(endDate)) {
                        rangeQueryBuilder.lte(EsDateUtil.esFieldDateFormat(endDate));
                    }
                    boolQuery.filter(rangeQueryBuilder);
                }
                WildcardQueryBuilder wildcardQueryBuilder = QueryBuilders.wildcardQuery("origin", "*");
                boolQuery.must(wildcardQueryBuilder);
                //未删除状态查询
                boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
                List<SubjectDataVo> subjectDataVos = formatData(getEsResult(searchSourceBuilder, boolQuery, subjectId));
                if (subjectDataVos.size() < size) {
                    flag.set(false);
                } else {
                    pageNo.getAndIncrement();
                }
                log.info("本次循环-{}，数据量为-{}", currentPage, subjectDataVos.size());
                return subjectDataVos;
            });
            try {
                List<SubjectDataVo> subjectDataVos = async.get();
                allList.addAll(subjectDataVos);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } while (flag.get());
        return allList;
    }

    /**
     * 获取专题下的资讯数量
     *
     * @param subjectId 专题id
     * @param startDate 开始时间
     * @param endDate   结束时间
     */
    public int count(String subjectId, String startDate, String endDate) {
        long count = 0L;
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
            if (StringUtils.isNotBlank(startDate)) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(startDate)));
            }
            if (StringUtils.isNotBlank(endDate)) {
                boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(endDate)));
            }
        }
        SearchHits searchHits = getEsResult(searchSourceBuilder, boolQuery, subjectId);
        if (searchHits != null) {
            count = searchHits.getTotalHits().value;
        }
        return (int) count;
    }

    /**
     * 根据查询条件分页获取资讯
     *
     * @param subjectId   专题id
     * @param startDate   开始时间
     * @param endDate     结束时间
     * @param fetchFields 抓取的字段
     * @param pageNo      当前页
     * @param pageSize    页面大小
     * @author lkg
     */
    public List<SubjectDataVo> pageList(String subjectId, String startDate, String endDate, String[] fetchFields, Integer sort, Integer pageNo, Integer pageSize) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        //按时间排序
        if (sort == 1) {
            searchSourceBuilder.sort("publishDate", SortOrder.ASC);
        } else if (sort == 2) {
            searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        }
        //设置需要获取的字段
        searchSourceBuilder.fetchSource(fetchFields, null);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        //某专题下的信息
        boolQuery.must(QueryBuilders.matchPhraseQuery("subjectId", subjectId));
        //时间过滤
        if (StringUtils.isNotEmpty(startDate) || StringUtils.isNotEmpty(endDate)) {
            RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("publishDate");
            if (StringUtils.isNotBlank(startDate)) {
                rangeQueryBuilder.gte(EsDateUtil.esFieldDateFormat(startDate));
            }
            if (StringUtils.isNotBlank(endDate)) {
                rangeQueryBuilder.lte(EsDateUtil.esFieldDateFormat(endDate));
            }
            boolQuery.filter(rangeQueryBuilder);
        }
        return formatData(getEsResult(searchSourceBuilder, boolQuery, subjectId));
    }

    /**
     * 专题资讯分页列表-门户
     *
     * @param subjectIdList 专题id集合
     * @param searchWord    搜索词
     * @param position      搜索位置(title-标题;content-内容)
     * @param category      匹配度(1-模糊;2-精确)
     * @param column        排序字段
     * @param order         排序方式(asc-正序;desc-倒序)
     * @param pageNo        当前页
     * @param pageSize      返回条数
     * @author lkg
     * @date 2024/4/10
     */
    public IPage<SubjectDataVo> frontListByPage(List<String> subjectIdList, String searchWord, String position, Integer category,
                                                String labelId, String column, String order, int pageNo, int pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        if (column.equals("score")) {
            if (order.equals("asc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else if (order.equals("desc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
            }
        }
        String[] fetchFields = new String[]{"id", "subjectId", "title", "summary", "origin", "publishDate", "sourceAddress", "author", "labels"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        if (StringUtils.isNotBlank(searchWord)) {
            if (category == 1) {
                boolQuery.must(QueryBuilders.matchQuery(position, searchWord));
            } else if (category == 2) {
                boolQuery.must(QueryBuilders.matchPhraseQuery(position, searchWord));
            }
        }
        if (StringUtils.isNotBlank(labelId)) {
            boolQuery.must(QueryBuilders.nestedQuery("labels", QueryBuilders.termQuery("labels.relationId", labelId), ScoreMode.None));
        }
        boolQuery.mustNot(QueryBuilders.termQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        IPage<SubjectDataVo> pageData = new Page<>();
        try {
            List<SubjectDataVo> list = new ArrayList<>();
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = JSONUtil.toBean(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                String index = hit.getIndex();
                info.setIndex(index);
                list.add(info);
            }
            pageData = new Page<>(pageNo, pageSize, searchResponse.getHits().getTotalHits().value);
            pageData.setRecords(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return pageData;
    }

    /**
     * 查看资讯详情
     *
     * @param index 索引名称
     * @param id    资讯id
     * @author lkg
     * @date 2024/4/10
     */
    public SubjectDataVo queryInfo(String index, String id) {
        SubjectDataVo subjectDataVo = new SubjectDataVo();
        try {
            if (StringUtils.isBlank(index)) {
                index = Constants.SUBJECT_INDEX;
            }
            SearchRequest searchRequest = new SearchRequest(index);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            searchSourceBuilder.size(1);
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("id", id));
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse;
            try {
                searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = searchResponse.getHits();
                if (hits.getTotalHits().value > 0) {
                    SearchHit hit = hits.getAt(0);
                    String sourceAsString = hit.getSourceAsString();
                    subjectDataVo = JSON.parseObject(sourceAsString, SubjectDataVo.class);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return subjectDataVo;
    }

    /**
     * 根据标题获取相关推荐文章
     *
     * @param id       资讯id
     * @param title    标题
     * @param pageNo   当前页
     * @param pageSize 返回个数
     * @author lkg
     * @date 2024/4/10
     */
    public List<SubjectDataVo> queryRecommendList(String subjectId, String id, String title, Integer pageNo, Integer pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        int offset = (pageNo - 1) * pageSize;
        searchSourceBuilder.from(offset);
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = new String[]{"id", "subjectId", "title", "origin", "publishDate", "sourceAddress"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        //boolQuery.must(QueryBuilders.termQuery("checkStatus", 1));
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        boolQuery.must(QueryBuilders.matchQuery("title", title));
        boolQuery.mustNot(QueryBuilders.termQuery("id", id));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        searchSourceBuilder.collapse(new CollapseBuilder("sourceAddress.keyword"));
        searchSourceBuilder.collapse(new CollapseBuilder("title.keyword"));
        List<SubjectDataVo> list = new ArrayList<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String index = hit.getIndex();
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = JSON.parseObject(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                info.setIndex(index);
                list.add(info);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }


    /**
     * 获取相似文章数
     *
     * @param subjectId     专题id
     * @param articleIdList 资讯id集合
     * @author lkg
     * @date 2024/4/10
     */
    public Map<String, Integer> getSimilarNumber(String subjectId, List<String> articleIdList) {
        Map<String, Integer> map = new HashMap<>();
        Map<String, String> markMap = getMark(subjectId, articleIdList);
        if (MapUtil.isNotEmpty(markMap)) {
            List<String> markList = new ArrayList<>(markMap.keySet());
            SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termsQuery("repeatMark", markList));
            searchSourceBuilder.size(0);
            searchSourceBuilder.trackTotalHits(true);
            TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_mark")
                    .field("repeatMark")
                    .order(BucketOrder.count(false))
                    .size(articleIdList.size());
            searchSourceBuilder.aggregation(aggregationBuilder);
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                Aggregations aggregations = response.getAggregations();
                Terms groupSource = aggregations.get("group_mark");
                List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
                if (CollectionUtils.isNotEmpty(buckets)) {
                    for (Terms.Bucket bucket : buckets) {
                        map.put(markMap.get(bucket.getKeyAsString()), (int) bucket.getDocCount());
                    }

                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }

    /**
     * 获取资讯所在蔟(蔟中存储重复数据信息)
     *
     * @param subjectId     专题id
     * @param articleIdList 资讯id集合
     * @author lkg
     * @date 2024/4/10
     */
    private Map<String, String> getMark(String subjectId, List<String> articleIdList) {
        Map<String, String> map = new HashMap<>();
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            List<String> idList = new ArrayList<>();
            for (String articleId : articleIdList) {
                idList.add(articleId.replace(subjectId, ""));
            }
            SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termsQuery("articleId", idList));
            boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHit[] hits = response.getHits().getHits();
                if (hits != null && hits.length != 0) {
                    for (SearchHit hit : hits) {
                        String queryInfo = hit.getSourceAsString();
                        RepeatHold info = JSONUtil.toBean(queryInfo, RepeatHold.class);
                        map.put(info.getRepeatMark(), info.getSubjectId() + info.getArticleId());
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }


    /**
     * 按标题匹配数据并按来源分组
     *
     * @param title       标题
     * @param publishDate 发布时间
     * @author lkg
     * @date 2024/4/11
     */
    public List<String> originList(String title, String publishDate) {
        List<String> originList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //只返回分组聚合结果，不返回具体数据
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.matchQuery("title", title));
        boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gt(EsDateUtil.esFieldDateFormat(publishDate)));
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .size(20)
                .order(BucketOrder.count(false));
        searchSourceBuilder.query(boolQuery);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupOrigin = aggregations.get("group_origin");
            List<? extends Terms.Bucket> buckets = groupOrigin.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String origin = bucket.getKeyAsString();
                if (StringUtils.isNotEmpty(origin)) {
                    originList.add(Constants.getRealOrigin(origin));
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return originList;
    }

    /**
     * 报告导出的资讯信息列表
     *
     * @param subjectIdList 专题id集合
     * @param searchWord    搜索词
     * @param position      搜索位置(title-标题;content-内容)
     * @param category      匹配度(1-模糊;2-精确)
     * @param articleIdList 资讯id集合
     * @param column        排序字段
     * @param order         排序方式(asc-正序;desc-倒序)
     * @param type          导出方式(1-摘要;2-正文)
     * @param pageSize      返回条数
     * @author lkg
     * @date 2024/4/11
     */
    public List<SubjectDataVo> exportDataList(List<String> subjectIdList, String searchWord, String position, Integer category,
                                              List<String> articleIdList, String column, String order, Integer type, Integer pageSize) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            pageSize = articleIdList.size();
        }
        //设置分页参数
        searchSourceBuilder.size(pageSize);
        if (column.equals("score")) {
            if (order.equals("asc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            } else if (order.equals("desc")) {
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            }
        } else if (column.equals("publishDate")) {
            if (order.equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.DESC));
            } else if (order.equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                searchSourceBuilder.sort(SortBuilders.scoreSort().order(SortOrder.ASC));
            }
        }
        String[] fetchFields = new String[]{"id", "title", "summary", "origin", "publishDate", "sourceAddress"};
        if (type == 2) {
            fetchFields = new String[]{"id", "title", "summary", "content", "origin", "publishDate", "sourceAddress"};
        }
        searchSourceBuilder.fetchSource(fetchFields, null);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        if (CollectionUtils.isNotEmpty(articleIdList)) {
            boolQuery.must(QueryBuilders.termsQuery("id", articleIdList));
        } else {
            boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
            if (StringUtils.isNotEmpty(searchWord)) {
                if (category == 1) {
                    boolQuery.must(QueryBuilders.matchQuery(position, searchWord));
                } else if (category == 2) {
                    boolQuery.must(QueryBuilders.matchPhraseQuery(position, searchWord));
                }
            }
            boolQuery.mustNot(QueryBuilders.termQuery("deleteFlag", "1"));
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        List<SubjectDataVo> list = new ArrayList<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                SubjectDataVo info = JSONUtil.toBean(queryInfo, SubjectDataVo.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                list.add(info);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 事件对应采集库的资讯分页列表
     *
     * @param eventDataCondition 条件
     * @author lkg
     * @date 2024/5/6
     */
    public IPage<EventDataVO> collectPageList(InfoDataSearchCondition eventDataCondition) {
        String index = Constants.COLLECT_INDEX;
        if ("1".equals(eventDataCondition.getIsSubject())) {
            Subject byId = subjectService.getById(eventDataCondition.getSubjectId());
            //更新开始、结束时间
            if (byId.getTimeEnable() != null) {
                String timeEnable = DateUtil.format(byId.getTimeEnable(), "yyyy-MM-dd");
                if (StringUtils.isEmpty(eventDataCondition.getStartTime()) || eventDataCondition.getStartTime().compareTo(timeEnable) < 0) {
                    eventDataCondition.setStartTime(timeEnable);
                }
            }
            if (byId.getTimeDisable() != null) {
                String timeDisable = DateUtil.format(byId.getTimeDisable(), "yyyy-MM-dd");
                if (StringUtils.isEmpty(eventDataCondition.getEndTime()) || eventDataCondition.getEndTime().compareTo(timeDisable) > 0) {
                    eventDataCondition.setEndTime(timeDisable);
                }
            }
            //以专题指定的索引为准
            if (StringUtils.isNotEmpty(byId.getEsIndex())) {
                index = byId.getEsIndex();
            }
        }
        SearchRequest searchRequest = new SearchRequest(index);
        //根据检索开始结束时间确定索引范围
        String firstIndex = LocalDate.now().minusMonths(23).toString();
        if (Constants.COLLECT_INDEX.equals(index) && StringUtils.isNotEmpty(eventDataCondition.getStartTime()) && eventDataCondition.getStartTime().compareTo(firstIndex) > 0) {
            String[] indexs = EsIndexUtil.getIndexMonth(Constants.COLLECT_INDEX, eventDataCondition.getStartTime(), eventDataCondition.getEndTime());
            searchRequest = new SearchRequest(indexs);
        }
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        String[] fetchFields = eventDataCondition.getFetchFields();
        String[] excludeFields = eventDataCondition.getExcludeFields();
        if (fetchFields != null && fetchFields.length > 0) {
            searchSourceBuilder.fetchSource(fetchFields, null);
        } else if (excludeFields != null && excludeFields.length > 0) {
            searchSourceBuilder.fetchSource(null, excludeFields);
        }
        //设置分页参数
        Integer pageNo = eventDataCondition.getPageNo();
        Integer pageSize = eventDataCondition.getPageSize();
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        //默认按照时间倒序排列
        if (eventDataCondition.getColumn().equals("publishDate")) {
            if (eventDataCondition.getOrder().equals("desc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.DESC);
            } else if (eventDataCondition.getOrder().equals("asc")) {
                searchSourceBuilder.sort("publishDate", SortOrder.ASC);
            }
        }
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        if ("1".equals(eventDataCondition.getIsAll())) {
            List<String> subjectIdList = new ArrayList<>();
            //判断是否为专题
            if (!"1".equals(eventDataCondition.getIsSubject())) {
                //不是专题
                //查询类别id的所有明细id
                List<String> typeIds = new ArrayList<>();
                Integer category = eventDataCondition.getCategory();
                if (StringUtils.isNotEmpty(eventDataCondition.getSubjectId()) && !"0".equals(eventDataCondition.getSubjectId())) {
                    typeIds = subjectTypeService.belowIdList(eventDataCondition.getSubjectId(), category);
                }
                if (category == 1) {
                    subjectIdList = subjectTypeMapService.selectSubjectByTypeIds(typeIds);
                } else if (category == 2) {
                    subjectIdList = subjectTypeMapService.selectEventByTypeIds(typeIds);
                }
            } else {
                subjectIdList.add(eventDataCondition.getSubjectId());
            }
            //事件真正绑定的信息源id集合
            List<String> infoSourceIdList = new ArrayList<>();
            if (CollectionUtils.isNotEmpty(subjectIdList)) {
                infoSourceIdList = subjectInfoSourceMapService.subjectRealBindInfoSourceList(subjectIdList);
                //查询专题绑定的关键词组
                List<String> wordsIdList = eventService.bindKeyWordsIdList(subjectIdList);
                infoSourceIdList.addAll(wordsIdList);
            }
            //装配信息源的条件
            if (CollectionUtils.isNotEmpty(infoSourceIdList)) {
                boolQuery.must(QueryBuilders.termsQuery("sid", infoSourceIdList.stream().filter(StringUtils::isNotEmpty).collect(Collectors.toList())));
            } else {
                return new Page<>();
            }
        } else {
            boolQuery.must(QueryBuilders.termQuery("sid", eventDataCondition.getSourceId()));
        }
        //高级查询数据处理
        BoolQueryBuilder superQuery = buildSuperQuery(eventDataCondition.getSuperQueryMatchType(), eventDataCondition.getSuperQueryParams());
        if (superQuery != null) {
            boolQuery.must(superQuery);
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getOrigin())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("origin", eventDataCondition.getOrigin()));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getSearch())) {
            boolQuery.must(QueryBuilders.multiMatchQuery(eventDataCondition.getSearch(), "title", "content"));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getContent())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("content", eventDataCondition.getContent()));
        }
        if (StringUtils.isNotEmpty(eventDataCondition.getTitle())) {
            boolQuery.must(QueryBuilders.matchPhraseQuery("title", eventDataCondition.getTitle()));
        }
        if (StringUtils.isNotBlank(eventDataCondition.getVideo())) {
            boolQuery.must(QueryBuilders.matchQuery("type", "video"));
        } else {
            boolQuery.mustNot(QueryBuilders.matchQuery("type", "video"));
        }
        //时间过滤筛选
        if (StringUtils.isNotBlank(eventDataCondition.getStartTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(eventDataCondition.getStartTime())));
        }
        if (StringUtils.isNotBlank(eventDataCondition.getEndTime())) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(eventDataCondition.getEndTime())));
        } else {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(new Date()))));
        }
        if (eventDataCondition.getLabelName() != null) {
            BoolQueryBuilder nestedBoolQueryBuilder = QueryBuilders.boolQuery();
            TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationName", eventDataCondition.getLabelName());
            nestedBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
            boolQuery.must(nestedBoolQueryBuilder);
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        IPage<EventDataVO> pageData = new Page<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            List<EventDataVO> list = new ArrayList<>();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                EventDataVO info = JSON.parseObject(queryInfo, EventDataVO.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                String hitIndex = hit.getIndex();
                info.setIndex(hitIndex);
                list.add(info);
            }
            pageData = new Page<>(pageNo, pageSize, searchResponse.getHits().getTotalHits().value);
            pageData.setRecords(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return pageData;
    }


    /**
     * 事件对应专题库的资讯分页列表
     *
     * @param searchCondition 检索条件
     * @author lkg
     * @date 2024/5/6
     */
    public IPage<SpecialInformation> pageListByCondition(InfoDataSearchCondition searchCondition, List<String> subjectIdList) throws IOException {
        String minDate = subjectService.getMinCreateTime(subjectIdList);
        String[] indexs = EsIndexUtil.getIndexIntervalYearStr(Constants.SUBJECT_INDEX, minDate);
        SearchRequest searchRequest = new SearchRequest(indexs);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = searchCondition.getFetchFields();
        String[] excludeFields = searchCondition.getExcludeFields();
        if (fetchFields != null && fetchFields.length > 0) {
            searchSourceBuilder.fetchSource(fetchFields, null);
        } else if (excludeFields != null && excludeFields.length > 0) {
            searchSourceBuilder.fetchSource(null, excludeFields);
        }
        //设置分页参数
        Integer pageNo = searchCondition.getPageNo();
        Integer pageSize = searchCondition.getPageSize();
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        searchSourceBuilder.size(pageSize);
        //排序规则
        String column = searchCondition.getColumn();
        String order = searchCondition.getOrder();
        if (searchCondition.getTopSortValid() == 1) {
            searchSourceBuilder.sort("topNum", SortOrder.DESC);
        }
        if (StringUtils.isEmpty(column)) {
            searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        } else {
            SortOrder sortOrder = SortOrder.DESC;
            if (order.equals("asc")) {
                sortOrder = SortOrder.ASC;
            }
            switch (column) {
                case "score":
                    searchSourceBuilder.sort("score", sortOrder);
                    searchSourceBuilder.sort("publishDate", sortOrder);
                    break;
                case "publishDate":
                    searchSourceBuilder.sort("publishDate", sortOrder);
                    searchSourceBuilder.sort("score", sortOrder);
                    break;
                case "yqgzzk":
                    searchSourceBuilder.sort("dataFrom", SortOrder.DESC);
                    searchSourceBuilder.sort("topNum", SortOrder.DESC);
                    searchSourceBuilder.sort("score", SortOrder.DESC);
                    break;
                case "title":
                case "origin":
                    searchSourceBuilder.sort(column + ".keyword", sortOrder);
                    break;
                case "summary":
                case "content":
                    NestedSortBuilder nestedSortBuilder = new NestedSortBuilder("sortField");
                    nestedSortBuilder.setFilter(QueryBuilders.termQuery("sortField.fieldType", column));
                    FieldSortBuilder fieldSortBuilder = SortBuilders.fieldSort("sortField.fieldKeyword")
                            .order(sortOrder)
                            .sortMode(SortMode.MAX)
                            .setNestedSort(nestedSortBuilder);
                    searchSourceBuilder.sort(fieldSortBuilder);
                    break;
                case "labels":
                    NestedSortBuilder nestedLabelsSortBuilder = new NestedSortBuilder("labels");
                    FieldSortBuilder labelSortBuilder = SortBuilders.fieldSort("labels.relationId")
                            .order(sortOrder)
                            .sortMode(SortMode.MAX)
                            .setNestedSort(nestedLabelsSortBuilder);
                    searchSourceBuilder.sort(labelSortBuilder);
                    break;
                default:
                    NestedSortBuilder nestedWordSortBuilder = new NestedSortBuilder("sortField");
                    nestedWordSortBuilder.setFilter(QueryBuilders.termsQuery("sortField.fieldType", Arrays.asList(column.split(","))));
                    FieldSortBuilder fieldWordSortBuilder = SortBuilders.fieldSort("sortField.fieldLong")
                            .order(sortOrder)
                            .sortMode(SortMode.SUM)
                            .setNestedSort(nestedWordSortBuilder);
                    searchSourceBuilder.sort(fieldWordSortBuilder);
                    break;
            }
        }
        //构建es查询条件
        BoolQueryBuilder boolQuery;
        //判断ids字段是否为空，若不为空，则表示按照id勾选
        List<String> ids = searchCondition.getIds();
        if (CollectionUtils.isNotEmpty(ids)) {
            boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termsQuery("id", ids));
            List<String> sourceIdList = searchCondition.getSourceIdList();
            if (CollectionUtils.isNotEmpty(sourceIdList)) {
                boolQuery.must(QueryBuilders.termsQuery("sid.keyword", sourceIdList));
            }
        } else {
            boolQuery = buildQuery(searchCondition, subjectIdList);
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        SearchHit[] searchHits = searchResponse.getHits().getHits();
        List<SpecialInformation> list = new ArrayList<>();
        for (SearchHit hit : searchHits) {
            String queryInfo = hit.getSourceAsString();
            SpecialInformation info = JSONUtil.toBean(queryInfo, SpecialInformation.class);
            info.setDbIndex(hit.getIndex());
            info.setIndex(hit.getIndex());
            list.add(info);
        }
        IPage<SpecialInformation> pageData = new Page<>(pageNo, pageSize, searchResponse.getHits().getTotalHits().value);
        pageData.setRecords(list);
        return pageData;
    }

    /**
     * 判断采集信息是否已经进入专题库
     *
     * @param subjectId 专题id
     * @param ids       采集信息id字符串，用逗号拼接
     * @author lkg
     * @date 2024/12/24
     */
    public List<String> listBeInStorage(String subjectId, String ids) {
        List<String> list = new ArrayList<>();
        List<String> idList = Arrays.asList(ids.split(","));
        List<String> beInStorageList = idList.stream().map(uniqueCode -> subjectId + uniqueCode).collect(Collectors.toList());
        String minDate = subjectService.getMinCreateTime(Arrays.asList(subjectId));
        String[] indexs = EsIndexUtil.getIndexIntervalYearStr(Constants.SUBJECT_INDEX, minDate);
        SearchRequest searchRequest = new SearchRequest(indexs);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("id", beInStorageList));
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.query(boolQuery);
        searchSourceBuilder.fetchSource("id", null);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                DisplayInfo info = JSONUtil.toBean(queryInfo, DisplayInfo.class);
                list.add(info.getId().replace(subjectId, ""));
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 获取信息源时间段内的采集量
     *
     * @param sid       信息源id
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @param type      分类(1-不聚合;2-聚合)
     * @author lkg
     * @date 2024/4/25
     */
    public NumVO count(String sid, String startDate, String endDate, Integer type) {
        NumVO countVO = new NumVO();
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("sid", sid));
        if (StringUtils.isNotBlank(startDate) || StringUtils.isNotBlank(endDate)) {
            RangeQueryBuilder rangeQueryBuilder = QueryBuilders.rangeQuery("createDate");
            if (StringUtils.isNotBlank(startDate)) {
                rangeQueryBuilder.from(EsDateUtil.esFieldDateFormat(startDate));
            }
            if (StringUtils.isNotBlank(endDate)) {
                rangeQueryBuilder.to(EsDateUtil.esFieldDateFormat(endDate));
            }
            boolQuery.filter(rangeQueryBuilder);
        }
        searchSourceBuilder.query(boolQuery);
        if (type == 2) {
            DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_date")
                    .field("createDate")
                    .calendarInterval(DateHistogramInterval.DAY)
                    .format("yyyy-MM-dd");
            searchSourceBuilder.aggregation(aggregation);
        }
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            long num = searchResponse.getHits().getTotalHits().value;
            countVO.setCount(num);
            if (type == 2) {
                List<NumVO> list = new ArrayList<>();
                Aggregations aggregations = searchResponse.getAggregations();
                ParsedDateHistogram groupHour = aggregations.get("group_date");
                List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
                if (CollectionUtils.isNotEmpty(buckets)) {
                    for (Histogram.Bucket bucket : buckets) {
                        NumVO vo = new NumVO();
                        vo.setKey(bucket.getKeyAsString());
                        vo.setCount(bucket.getDocCount());
                        list.add(vo);
                    }
                }
                countVO.setChildren(list);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return countVO;
    }

    public Map<String, Map<String, Long>> regetFromEs(List<String> sidList, String startDate, String endDate) {
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.size(0);
        //时间范围筛选
        RangeQueryBuilder rangeQueryBuilder = QueryBuilders
                .rangeQuery("createDate")
                .from(startDate)
                .to(endDate);
        searchSourceBuilder.query(rangeQueryBuilder);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("sid", sidList));
        boolQuery.must(rangeQueryBuilder);
        //按时间，sid分组
        //es 查询天的时间格式
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_sid")
                .size(10000)
                .field("sid")
                .subAggregation(AggregationBuilders.dateHistogram("group_date")
                        .field("createDate")
                        .calendarInterval(DateHistogramInterval.DAY)
                        .format("yyyy-MM-dd"));
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        Map<String, Map<String, Long>> map = new HashMap<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取id分组集合
            Terms parsedStringTerms = aggregations.get("group_sid");

            List<? extends Terms.Bucket> buckets = parsedStringTerms.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                //key的数据，过滤sid为空的
                String key = bucket.getKey().toString();
                if (key == null || "null".equals(key) || "".equals(key)) {
                    continue;
                }
                //获取时间分组集合
                Aggregations bucketAggregations = bucket.getAggregations();
                ParsedDateHistogram groupDate = bucketAggregations.get("group_date");
                List<? extends Histogram.Bucket> dateBucket = groupDate.getBuckets();
                Map<String, Long> temMap = new HashMap<>();
                for (Histogram.Bucket date : dateBucket) {
                    String dateStr = date.getKeyAsString();
                    long num = date.getDocCount();
                    temMap.put(dateStr, num);
                }
                map.put(key, temMap);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return map;
    }


    /**
     * 媒体观点/专家观点信息列表
     *
     * @param eventId    事件id
     * @param relationId 关联标签id
     * @author lkg
     * @date 2024/10/10
     */
    public List<EventViewVO> viewpointList(String eventId, String relationId) {
        List<EventViewVO> dataList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //排序方式 按发布时间升序
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = new String[]{"id", "title", "origin", "publishDate", "subjectId", "sourceAddress", "labels"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", eventId));
        boolQuery.must(QueryBuilders.nestedQuery("labels", QueryBuilders.termQuery("labels.relationId", relationId), ScoreMode.None));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = searchResponse.getHits().getHits();
            if (hits != null && hits.length > 0) {
                for (SearchHit hit : hits) {
                    String index = hit.getIndex();
                    String sourceAsString = hit.getSourceAsString();
                    EventViewVO eventViewVO = JSON.parseObject(sourceAsString, EventViewVO.class);
                    eventViewVO.setPublishDate(EsDateUtil.esFieldDateMapping(eventViewVO.getPublishDate()));
                    eventViewVO.setIndex(index);
                    List<Label> labels = eventViewVO.getLabels();
                    String professionName = null;
                    for (Label label : labels) {
                        String labelRelationId = label.getRelationId();
                        if (StringUtils.isEmpty(labelRelationId)) {
                            professionName = label.getRelationName();
                            break;
                        }
                    }
                    eventViewVO.setProfessionName(professionName);
                    dataList.add(eventViewVO);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return dataList;
    }

    /**
     * 获取专题下置顶的最大值
     *
     * @param index     es缩影
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/19
     */
    public int getTopNum(String index, String subjectId) {
        int topNum = 0;
        SearchRequest searchRequest;
        if (StringUtils.isEmpty(index)) {
            index = Constants.SUBJECT_INDEX;
        }
        searchRequest = new SearchRequest(index);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //设置分页参数
        searchSourceBuilder.size(1);
        searchSourceBuilder.from(0);
        //根据topNum正序查找，查询置顶数据
        searchSourceBuilder.sort("topNum", SortOrder.DESC);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            assert searchResponse != null;
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            //在查询基础上+1,在入库
            if (searchHits.length > 0) {
                Map<String, Object> sourceAsMap = searchHits[0].getSourceAsMap();
                Object topNumObj = sourceAsMap.get("topNum");
                if (topNumObj != null) {
                    topNum = Integer.parseInt(topNumObj.toString());
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return topNum;
    }

    //调用判重接口
    public boolean duplicationByTitleOrSourceAddress(DisplayInfo displayInfo) {
        String minDate = subjectService.getMinCreateTime(Arrays.asList(displayInfo.getSubjectId()));
        String[] indexs = EsIndexUtil.getIndexIntervalYearStr(Constants.SUBJECT_INDEX, minDate);
        SearchRequest searchRequest = new SearchRequest(indexs);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        BoolQueryBuilder shouldBoolQuery = QueryBuilders.boolQuery();
        QueryBuilder builder = QueryBuilders.termQuery("title.keyword", displayInfo.getTitle());
        shouldBoolQuery.should(builder);
        String sourceAddress = displayInfo.getSourceAddress();
        if (StringUtils.isNotEmpty(sourceAddress)) {
            QueryBuilder builder2 = QueryBuilders.termQuery("sourceAddress.keyword", sourceAddress);
            shouldBoolQuery.should(builder2);
        }
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", displayInfo.getSubjectId()));
        boolQuery.must(shouldBoolQuery);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        long count = 0;
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            count = searchResponse.getHits().getTotalHits().value;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return count == 0;
    }

    /**
     * 获取选择的专题库资讯数量
     *
     * @param searchCondition 条件封装
     * @author lkg
     * @date 2024/7/8
     */
    public int getCount(InfoDataSearchCondition searchCondition) {
        int count = 0;
        List<String> ids = searchCondition.getIds();
        if (CollectionUtils.isNotEmpty(ids)) {
            count = ids.size();
        } else {
            String minDate = subjectService.getMinCreateTime(Arrays.asList(searchCondition.getSubjectId()));
            String[] indexs = EsIndexUtil.getIndexIntervalYearStr(Constants.SUBJECT_INDEX, minDate);
            SearchRequest searchRequest = new SearchRequest(indexs);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //默认最大数量是10000，设置为true后，显示准确数量
            searchSourceBuilder.trackTotalHits(true);
            searchSourceBuilder.size(0);
            //创建查询对象
            List<String> subjectIdList = new ArrayList<>();
            //判断是否是专题
            if ("1".equals(searchCondition.getIsSubject())) {
                if (StringUtils.isNotEmpty(searchCondition.getSubjectId())) {
                    subjectIdList.add(searchCondition.getSubjectId());
                }
            } else {
                //该id其实是专题类别id
                //查询类别id的所有明细id
                Integer category = searchCondition.getCategory();
                String subjectTypeId = searchCondition.getSubjectId();
                List<String> typeIds = subjectTypeService.belowIdList(subjectTypeId, category);
                if (category == 1) {
                    subjectIdList = subjectTypeMapService.selectSubjectByTypeIds(typeIds);
                } else if (category == 2) {
                    subjectIdList = subjectTypeMapService.selectEventByTypeIds(typeIds);
                }
            }
            BoolQueryBuilder boolQuery = buildQuery(searchCondition, subjectIdList);
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
                count = (int) searchResponse.getHits().getTotalHits().value;
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return count;
    }

    /**
     * 获取数据(专题数据-已入库)
     *
     * @param searchCondition 查询条件封装
     */
    public List<SpecialInformation> informationList(InfoDataSearchCondition searchCondition) {
        String minDate = subjectService.getMinCreateTime(Arrays.asList(searchCondition.getSubjectId()));
        String[] indexs = EsIndexUtil.getIndexIntervalYearStr(Constants.SUBJECT_INDEX, minDate);
        SearchRequest searchRequest = new SearchRequest(indexs);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        String column = searchCondition.getColumn();
        String order = searchCondition.getOrder();
        if (StringUtils.isEmpty(column)) {
            searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        } else {
            switch (column) {
                case "topNum":
                    searchSourceBuilder.sort("topNum", SortOrder.DESC);
                    break;
                case "score":
                    if (order.equals("asc")) {
                        searchSourceBuilder.sort("score", SortOrder.ASC);
                    } else if (order.equals("desc")) {
                        searchSourceBuilder.sort("score", SortOrder.DESC);
                    }
                    break;
                case "publishDate":
                    if (order.equals("desc")) {
                        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                    } else if (order.equals("asc")) {
                        searchSourceBuilder.sort("publishDate", SortOrder.ASC);
                    }
                    break;
                default:
                    searchSourceBuilder.sort("publishDate", SortOrder.DESC);
                    break;
            }
        }
        String[] fetchFields = searchCondition.getFetchFields();
        if (fetchFields != null && fetchFields.length > 0) {
            searchSourceBuilder.fetchSource(fetchFields, null);
        }
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //设置分页参数
        Integer pageNo = searchCondition.getPageNo();
        Integer pageSize = searchCondition.getPageSize();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        //判断ids字段是否为空，若不为空，则表示按照id勾选
        List<String> ids = searchCondition.getIds();
        if (CollectionUtils.isNotEmpty(ids)) {
            boolQuery.must(QueryBuilders.termsQuery("id", ids));
            List<String> sourceIdList = searchCondition.getSourceIdList();
            if (CollectionUtils.isNotEmpty(sourceIdList)) {
                boolQuery.must(QueryBuilders.termsQuery("sid.keyword", sourceIdList));
            }
        } else {
            Integer num = searchCondition.getNum();
            if (num != null) {
                pageSize = num;
            }
            List<String> subjectIdList = new ArrayList<>();
            //判断是否是专题
            if ("1".equals(searchCondition.getIsSubject())) {
                if (StringUtils.isNotEmpty(searchCondition.getSubjectId())) {
                    subjectIdList.add(searchCondition.getSubjectId());
                }
            } else {
                //该id其实是专题类别id
                //查询类别id的所有明细id
                Integer category = searchCondition.getCategory();
                String subjectTypeId = searchCondition.getSubjectId();
                List<String> typeIds = subjectTypeService.belowIdList(subjectTypeId, category);
                if (category == 1) {
                    subjectIdList = subjectTypeMapService.selectSubjectByTypeIds(typeIds);
                } else if (category == 2) {
                    subjectIdList = subjectTypeMapService.selectEventByTypeIds(typeIds);
                }
            }
            boolQuery = buildQuery(searchCondition, subjectIdList);
        }
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        List<SpecialInformation> dataList = new ArrayList<>();
        try {
            RequestOptions.Builder builder = RequestOptions.DEFAULT.toBuilder();
            builder.setHttpAsyncResponseConsumerFactory(
                    new HttpAsyncResponseConsumerFactory
                            //修改为500MB
                            .HeapBufferedResponseConsumerFactory(500 * 1024 * 1024));
            SearchResponse searchResponse = client.search(searchRequest, builder.build());
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                SpecialInformation info = JSON.parseObject(queryInfo, SpecialInformation.class);
                info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
                info.setDbIndex(hit.getIndex());
                dataList.add(info);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return dataList;
    }


    /**
     * 专题资讯按地域统计
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/23
     */
    public List<CountVO> getRegionStatistic(String subjectId) {
        List<CountVO> data = new ArrayList<>();
        List<LabelTypeVO> regionList = labelEntityService.firstLevelRegions();
        List<String> regionIds = new ArrayList<>();
        Map<String, Long> regionMap = new HashMap<>();
        for (Node node : regionList) {
            regionIds.add(node.getId());
        }
        //根据地区的id从ES查询相关的记录
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        //统计没有删除的
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        NestedQueryBuilder nestedQueryBuilder = QueryBuilders
                .nestedQuery("labels", QueryBuilders.termsQuery("labels.relationId", regionIds), ScoreMode.None);
        boolQuery.must(nestedQueryBuilder);
        searchSourceBuilder.query(boolQuery);
        NestedAggregationBuilder nestedAggregationBuilder = AggregationBuilders.nested("labels", "labels")
                .subAggregation(AggregationBuilders.terms("group_code")
                        .field("labels.relationId").size(6000)
                        .order(BucketOrder.count(false))//true 正序； false 倒序
                );
        searchSourceBuilder.aggregation(nestedAggregationBuilder);

        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            ParsedNested labels = aggregations.get("labels");
            Aggregations labelsAggregations = labels.getAggregations();
            //获取按企业信用代码分组集合
            Terms groupCode = labelsAggregations.get("group_code");
            List<? extends Terms.Bucket> codeBuckets = groupCode.getBuckets();
            for (Terms.Bucket code : codeBuckets) {
                regionMap.put(code.getKeyAsString(), code.getDocCount());
            }
            //转换成地区的名字
            for (LabelTypeVO region : regionList) {
                String name = region.getName();
                String id = region.getId();
                CountVO countVo = new CountVO();
                countVo.setId(id);
                countVo.setKey(name);
                if (regionMap.containsKey(id)) {
                    countVo.setCount(String.valueOf(regionMap.get(id)));
                } else {
                    countVo.setCount("0");
                }
                data.add(countVo);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return data;
    }

    public List<SubjectDataVo> listByIds(List<String> ids) {
        List<SubjectDataVo> dataList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("id", ids));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = searchResponse.getHits().getHits();
            if (hits != null && hits.length > 0) {
                for (SearchHit hit : hits) {
                    String sourceAsString = hit.getSourceAsString();
                    SubjectDataVo subjectDataVo = JSONObject.parseObject(sourceAsString, SubjectDataVo.class);
                    dataList.add(subjectDataVo);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return dataList;
    }

    /**
     * 来源占比-专题分析页
     *
     * @param subjectId 专题id
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @param size      返回个数
     * @author lkg
     * @date 2024/12/23
     */
    public List<CountVO> groupByOrigin(String subjectId, String rangField, String startDate, String endDate, int size) {
        List<CountVO> data = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        if (StringUtils.isNotEmpty(startDate)) {
            boolQuery.must(QueryBuilders.rangeQuery(rangField).from(startDate));
        }
        if (StringUtils.isNotEmpty(endDate)) {
            boolQuery.must(QueryBuilders.rangeQuery(rangField).to(endDate));
        }
        //专题筛选
        if (StringUtils.isNotEmpty(subjectId)) {
            boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
        }
        //过滤掉origin为空的数据
        WildcardQueryBuilder wildcardQueryBuilder = QueryBuilders.wildcardQuery("origin", "*");
        boolQuery.must(wildcardQueryBuilder);
        searchSourceBuilder.query(boolQuery);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .order(BucketOrder.count(false))//true 正序； false 倒序
                .size(size);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取按来源分组集合
            Terms groupOrigin = aggregations.get("group_origin");
            List<? extends Terms.Bucket> originBuckets = groupOrigin.getBuckets();
            for (Terms.Bucket originBucket : originBuckets) {
                CountVO childrenVO = new CountVO();
                childrenVO.setKey(originBucket.getKeyAsString());
                childrenVO.setValue(originBucket.getDocCount());
                data.add(childrenVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return data;
    }


    /**
     * 专题下-采集到的数据按时间分析
     *
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @param type      时间类型
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/23
     */
    public List<CountVO> getCollectedData(String startDate, String endDate, String type, String subjectId) {
        SearchRequest searchRequest = new SearchRequest(Constants.COLLECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        List<String> subjectIdList = new ArrayList<>();
        subjectIdList.add(subjectId);
        List<String> infoSourceIdList = subjectInfoSourceMapService.subjectRealBindInfoSourceList(subjectIdList);
        //查询专题绑定的关键词组
        List<KeyWordsPage> list = subjectKeywordsMapService.bindKeyWordsList(subjectId, "1");
        for (KeyWordsPage keyWordsPage : list) {
            infoSourceIdList.add(keyWordsPage.getId());
        }
        if (CollectionUtils.isNotEmpty(infoSourceIdList)) {
            BoolQueryBuilder childQuery = QueryBuilders.boolQuery();
            //装配信息源的条件
            childQuery.should(QueryBuilders.termsQuery("sid", infoSourceIdList));
            boolQuery.must(childQuery);
        }
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        //时间范围筛选
        RangeQueryBuilder rangeQueryBuilder = QueryBuilders
                .rangeQuery("publishDate")
                .from(startDate)
                .to(endDate);
        boolQuery.filter(rangeQueryBuilder);
        searchSourceBuilder.query(boolQuery);
        DateHistogramAggregationBuilder aggregationBuilder = AggregationBuilders.dateHistogram("group_day")
                .field("publishDate")
                .calendarInterval(DateHistogramInterval.DAY)
                .format("yyyy-MM-dd");
        if ("month".equals(type)) {
            aggregationBuilder = AggregationBuilders.dateHistogram("group_month")
                    .field("publishDate")
                    .calendarInterval(DateHistogramInterval.MONTH)
                    .format("yyyy-MM");
        }
        if ("hour".equals(type)) {
            aggregationBuilder = aggregationBuilder
                    .subAggregation(AggregationBuilders.dateHistogram("group_hour")
                            .field("publishDate")
                            .calendarInterval(DateHistogramInterval.HOUR)
                            .format("HH:00:00"));
        }
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse = null;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return formatGroupData(searchResponse, type);
    }

    /**
     * 专题下-已入库的数据按时间分析
     *
     * @param startDate   开始时间
     * @param endDate     结束时间
     * @param type        时间类型
     * @param subjectId   专题id
     * @param isFreeCheck 是否免审核
     * @param checkStatus 审核状态
     * @author lkg
     * @date 2024/12/24
     */
    public List<CountVO> getSubjectDatabaseByTime(String startDate, String endDate, String type, String subjectId, Integer isFreeCheck, Integer checkStatus) {
        //构建查询（聚合）条件
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setSubjectId(subjectId);
        searchCondition.setStartTime(startDate);
        searchCondition.setEndTime(endDate);
        searchCondition.setIsFreeCheck(isFreeCheck);
        searchCondition.setCheckStatus(checkStatus);
        searchCondition.setGroupType(type);
        return groupByDate(searchCondition);
    }

    /**
     * 概述分析，信息源统计
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/30
     */
    public List<CountVO> overView(InfoDataSearchCondition searchCondition) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //构建查询语句
        List<String> subjectIds = new ArrayList<>();
        subjectIds.add(searchCondition.getSubjectId());
        BoolQueryBuilder boolQuery = buildQuery(searchCondition, subjectIds);
        searchSourceBuilder.query(boolQuery);
        NestedAggregationBuilder nestedAggregationBuilder = AggregationBuilders.nested("labels", "labels")
                .subAggregation(AggregationBuilders.terms("groupTag")
                        .field("labels.relationId")
                        .size(searchCondition.getIncludeValues().length)
                        .includeExclude(new IncludeExclude(searchCondition.getIncludeValues(), searchCondition.getExcludeValues()))
                        .subAggregation(AggregationBuilders.cardinality("sidCount").field("labels.sourceId.keyword")));
        searchSourceBuilder.aggregation(nestedAggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        long totalCount = 0;
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            totalCount = searchResponse.getHits().getTotalHits().value;
            Aggregations aggregations = searchResponse.getAggregations();
            ParsedNested labels = aggregations.get("labels");
            Aggregations labelsAggregations = labels.getAggregations();
            Terms groupTag = labelsAggregations.get("groupTag");
            List<? extends Terms.Bucket> buckets = groupTag.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                CountVO countVO = new CountVO();
                String key = bucket.getKeyAsString();
                countVO.setName(key);
                Aggregations natureAggregations = bucket.getAggregations();
                Cardinality sidCount = natureAggregations.get("sidCount");
                long value = sidCount.getValue();
                countVO.setValue(value);
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        CountVO countVO = new CountVO();
        countVO.setName("total");
        countVO.setValue(totalCount);
        list.add(0, countVO);
        return list;
    }

    /**
     * 按日期分组统计
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    public List<CountVO> groupByDate(InfoDataSearchCondition searchCondition) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //构建查询语句
        List<String> subjectIds = new ArrayList<>();
        subjectIds.add(searchCondition.getSubjectId());
        BoolQueryBuilder boolQuery = buildQuery(searchCondition, subjectIds);
        searchSourceBuilder.query(boolQuery);
        DateHistogramAggregationBuilder aggregationBuilder = null;
        String groupType = searchCondition.getGroupType();
        if ("year".equalsIgnoreCase(groupType)) {
            aggregationBuilder = AggregationBuilders.dateHistogram("groupYear")
                    .field("publishDate")
                    .calendarInterval(DateHistogramInterval.YEAR)
                    .format("yyyy");
        }
        if ("month".equalsIgnoreCase(groupType)) {
            aggregationBuilder = AggregationBuilders.dateHistogram("groupMonth")
                    .field("publishDate")
                    .calendarInterval(DateHistogramInterval.MONTH)
                    .format("yyyy-MM");
        }
        if ("day".equalsIgnoreCase(groupType))
            aggregationBuilder = AggregationBuilders.dateHistogram("groupDay")
                    .field("publishDate")
                    .calendarInterval(DateHistogramInterval.DAY)
                    .format("yyyy-MM-dd");
        if ("hour".equalsIgnoreCase(groupType)) {
            aggregationBuilder = AggregationBuilders.dateHistogram("groupHour")
                    .field("publishDate")
                    .calendarInterval(DateHistogramInterval.HOUR)
                    .format("HH:00:00");
        }
        if (aggregationBuilder != null) {
            TermsAggregationBuilder groupLibraryType = AggregationBuilders.terms("groupLibraryType")
                    .field("classificationType");
            if (!Objects.isNull(searchCondition.getIncludeValues()) || !Objects.isNull(searchCondition.getExcludeValues())) {
                if (!Objects.isNull(searchCondition.getIncludeValues())) {
                    groupLibraryType.size(searchCondition.getIncludeValues().length);
                }
                groupLibraryType.includeExclude(new IncludeExclude(searchCondition.getIncludeValues(), searchCondition.getExcludeValues()));
            }
            aggregationBuilder.subAggregation(groupLibraryType);
            searchSourceBuilder.aggregation(aggregationBuilder);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse = null;
            try {
                searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            } catch (Exception e) {
                e.printStackTrace();
            }
            list = formatGroupData(searchResponse, groupType);
        }
        return list;
    }

    /**
     * 按资讯类型分组统计
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    public List<CountVO> groupByClassificationType(InfoDataSearchCondition searchCondition) {
        return groupByTerm(searchCondition, "group", "classificationType", false);
    }

    /**
     * 按信息源分组统计
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    public List<CountVO> groupBySourceId(InfoDataSearchCondition searchCondition) {
        return groupByTerm(searchCondition, "groupSid", "sid.keyword", false);
    }

    /**
     * 按信息源标签分组统计
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/12/26
     */
    public List<CountVO> groupByInfoSourceTag(InfoDataSearchCondition searchCondition) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //构建查询语句
        List<String> subjectIds = new ArrayList<>();
        subjectIds.add(searchCondition.getSubjectId());
        BoolQueryBuilder boolQuery = buildQuery(searchCondition, subjectIds);
        searchSourceBuilder.query(boolQuery);
        NestedAggregationBuilder nestedAggregationBuilder = AggregationBuilders.nested("labels", "labels")
                .subAggregation(AggregationBuilders.terms("groupTag")
                        .field("labels.relationId")
                        .size(10)
                        .includeExclude(new IncludeExclude(searchCondition.getIncludeValues(), searchCondition.getExcludeValues())));
        searchSourceBuilder.aggregation(nestedAggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            ParsedNested labels = aggregations.get("labels");
            Aggregations labelsAggregations = labels.getAggregations();
            Terms groupTag = labelsAggregations.get("groupTag");
            List<? extends Terms.Bucket> buckets = groupTag.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                CountVO countVO = new CountVO();
                String key = bucket.getKeyAsString();
                countVO.setName(key);
                long value = bucket.getDocCount();
                countVO.setValue(value);
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 统计关键词数量
     *
     * @param subjectId
     * @param size
     * @return
     */
    public List<CountVO> keywordsCount(String subjectId, List<String> excludeKeywords, Integer size) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //构建查询语句
        List<String> subjectIds = new ArrayList<>();
        subjectIds.add(subjectId);
        BoolQueryBuilder boolQuery = buildQuery(new InfoDataSearchCondition(), subjectIds);
        searchSourceBuilder.query(boolQuery);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("groupKeywords")
                .field("keyWordsList.keyword")
                .size(size)
                .order(BucketOrder.count(false));
        aggregationBuilder.includeExclude((new IncludeExclude(null, String.join(",", excludeKeywords).split(","))));
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupSource = aggregations.get("groupKeywords");
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String key = bucket.getKeyAsString();
                long count = bucket.getDocCount();
                CountVO countVO = new CountVO();
                countVO.setName(key);
                countVO.setValue(count);
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }


    /**
     * 获取专题下资讯最早/最晚的发布时间
     *
     * @param subjectIdList 专题id集合
     * @param sortOrder     排序方式
     * @author lkg
     * @date 2025/2/21
     */
    public String getFirstTime(List<String> subjectIdList, String sortOrder) {
        String firstTime = null;
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setColumn("publishDate");
        searchCondition.setOrder(sortOrder);
        searchCondition.setFetchFields(new String[]{"publishDate"});
        searchCondition.setDeleteFlag(0);
        searchCondition.setCategory(1);
        BoolQueryBuilder boolQueryBuilder = buildQuery(searchCondition, subjectIdList);
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(1);
        searchSourceBuilder.query(boolQueryBuilder);
        if ("asc".equals(sortOrder)) {
            searchSourceBuilder.sort("publishDate", SortOrder.ASC);
        } else {
            searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        }
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits searchHits = searchResponse.getHits();
            if (searchHits != null) {
                SearchHit hit = searchHits.getHits()[0];
                Map<String, Object> sourceAsMap = hit.getSourceAsMap();
                String publishDateStr = sourceAsMap.get("publishDate").toString();
                firstTime = EsDateUtil.esFieldDateMapping(publishDateStr).substring(0, 10);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return firstTime;
    }

    /**
     * 专题信息数量统计
     *
     * @param subjectIds  专题id集合
     * @param checkStatus 审核状态
     * @param type        是否多层聚合(先按专题，再按发布时间)
     * @author lkg
     * @date 2025/3/25
     */
    public List<StatisticVO> subjectStatisticInfo(List<String> subjectIds, Integer checkStatus, Integer type) {
        List<StatisticVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        if (checkStatus != null) {
            searchCondition.setCheckStatus(checkStatus);
        }
        BoolQueryBuilder boolQuery = buildQuery(searchCondition, subjectIds);
        searchSourceBuilder.query(boolQuery);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_subject");
        aggregationBuilder.field("subjectId.keyword");
        if (!Objects.isNull(searchCondition.getIncludeValues()) || !Objects.isNull(searchCondition.getExcludeValues())) {
            if (!Objects.isNull(searchCondition.getIncludeValues())) {
                aggregationBuilder.size(searchCondition.getIncludeValues().length);
            }
            aggregationBuilder.includeExclude(new IncludeExclude(searchCondition.getIncludeValues(), searchCondition.getExcludeValues()));
        }
        if (null != type && type == 1) {
            DateHistogramAggregationBuilder format = AggregationBuilders.dateHistogram("groupDay")
                    .field("publishDate")
                    .calendarInterval(DateHistogramInterval.DAY)
                    .format("yyyy-MM-dd")
                    .order(BucketOrder.key(false));
            aggregationBuilder.subAggregation(format);
        }
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupSubject = aggregations.get("group_subject");
            List<? extends Terms.Bucket> buckets = groupSubject.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                StatisticVO statisticVO = new StatisticVO();
                String subjectId = bucket.getKeyAsString();
                long count = bucket.getDocCount();
                statisticVO.setSubjectId(subjectId);
                statisticVO.setCount((int) count);
                if (null != type && type == 1) {
                    ParsedDateHistogram groupDay = bucket.getAggregations().get("groupDay");
                    List<? extends Histogram.Bucket> groupDayBuckets = groupDay.getBuckets();
                    if (CollectionUtils.isNotEmpty(groupDayBuckets)) {
                        Histogram.Bucket groupDayBucket = groupDayBuckets.get(0);
                        String latestDataDate = groupDayBucket.getKeyAsString();
                        statisticVO.setPublishDate(DateUtil.stringToDate(latestDataDate, "yyyy-MM-dd"));
                    }
                }
                list.add(statisticVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (CollectionUtils.isNotEmpty(list)) {
            if (list.size() < subjectIds.size()) {
                for (String subjectId : subjectIds) {
                    if (!list.stream().map(StatisticVO::getSubjectId).collect(Collectors.toList()).contains(subjectId)) {
                        StatisticVO statisticVO = new StatisticVO();
                        statisticVO.setSubjectId(subjectId);
                        statisticVO.setCount(0);
                        list.add(statisticVO);
                    }
                }
            }
        } else {
            list = subjectIds.stream().map(subjectId -> {
                StatisticVO statisticVO = new StatisticVO();
                statisticVO.setSubjectId(subjectId);
                statisticVO.setCount(0);
                return statisticVO;
            }).collect(Collectors.toList());
        }
        return list;
    }

    public IPage<SpecialInformation> searchForReplaceList(InfoDataSearchCondition searchCondition) {
        List<String> subjectIds = new ArrayList<>();
        subjectIds.add(searchCondition.getSubjectId());
        String minCreateTime = subjectService.getMinCreateTime(subjectIds);
        String[] indexArr = EsIndexUtil.getIndexIntervalYearStr(Constants.SUBJECT_INDEX, minCreateTime);
        SearchRequest searchRequest = new SearchRequest(indexArr);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        String[] fetchFields = new String[]{"id", "title", "origin", "content"};
        searchSourceBuilder.fetchSource(fetchFields, null);
        //设置高亮
        HighlightBuilder highlightBuilder = new HighlightBuilder();
        highlightBuilder.preTags("<font style = 'color:red;'>");
        highlightBuilder.postTags("</font>");
        List<HighlightBuilder.Field> fields = highlightBuilder.fields();
        fields.add(new HighlightBuilder.Field("title").numOfFragments(0));
        fields.add(new HighlightBuilder.Field("origin").numOfFragments(0));
        fields.add(new HighlightBuilder.Field("content").numOfFragments(0));
        searchSourceBuilder.highlighter(highlightBuilder);
        //设置分页参数
        Integer pageNo = searchCondition.getPageNo();
        Integer pageSize = searchCondition.getPageSize();
        searchSourceBuilder.from((pageNo - 1) * pageSize);
        searchSourceBuilder.size(pageSize);
        BoolQueryBuilder boolQuery = buildQuery(searchCondition, subjectIds);
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        IPage<SpecialInformation> pageData = new Page<>();
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] searchHits = searchResponse.getHits().getHits();
            List<SpecialInformation> list = new ArrayList<>();
            for (SearchHit hit : searchHits) {
                String queryInfo = hit.getSourceAsString();
                SpecialInformation info = JSONUtil.toBean(queryInfo, SpecialInformation.class);
                Map<String, HighlightField> highlightFields = hit.getHighlightFields();
                if (highlightFields.containsKey("title")) {
                    Text[] titles = highlightFields.get("title").getFragments();
                    StringBuilder titleHighlight = new StringBuilder();
                    for (Text title : titles) {
                        titleHighlight.append(title.string());
                    }
                    info.setTitle(titleHighlight.toString());
                }
                if (highlightFields.containsKey("origin")) {
                    Text[] origins = highlightFields.get("origin").getFragments();
                    StringBuilder originHighlight = new StringBuilder();
                    for (Text origin : origins) {
                        originHighlight.append(origin.string());
                    }
                    info.setOrigin(originHighlight.toString());
                }
                if (highlightFields.containsKey("content")) {
                    Text[] contents = highlightFields.get("content").getFragments();
                    StringBuilder contentHighlight = new StringBuilder();
                    for (Text content : contents) {
                        contentHighlight.append(content.string());
                    }
                    info.setContent(contentHighlight.toString());
                }
                list.add(info);
            }
            pageData = new Page<>(pageNo, pageSize, searchResponse.getHits().getTotalHits().value);
            pageData.setRecords(list);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return pageData;
    }

    /**
     * term方式聚合分组
     *
     * @param searchCondition 筛选条件
     * @param groupName       分组名称
     * @param field           分组字段
     * @param sort            排序方式 true正序；false倒序
     * @author lkg
     * @date 2024/12/27
     */
    private List<CountVO> groupByTerm(InfoDataSearchCondition searchCondition, String groupName, String field, boolean sort) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //构建查询语句
        List<String> subjectIds = new ArrayList<>();
        if (StringUtils.isNotBlank(searchCondition.getSubjectId())) {
            subjectIds.add(searchCondition.getSubjectId());
        }
        BoolQueryBuilder boolQuery = buildQuery(searchCondition, subjectIds);
        searchSourceBuilder.query(boolQuery);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms(groupName);
        aggregationBuilder.field(field);
        if (!Objects.isNull(searchCondition.getIncludeValues()) || !Objects.isNull(searchCondition.getExcludeValues())) {
            if (!Objects.isNull(searchCondition.getIncludeValues())) {
                aggregationBuilder.size(searchCondition.getIncludeValues().length);
            }
            aggregationBuilder.includeExclude(new IncludeExclude(searchCondition.getIncludeValues(), searchCondition.getExcludeValues()));
        }
        aggregationBuilder.order(BucketOrder.count(sort));
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupSource = aggregations.get(groupName);
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String key = bucket.getKeyAsString();
                long count = bucket.getDocCount();
                CountVO countVO = new CountVO();
                countVO.setName(key);
                countVO.setValue(count);
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 构建通用的es查询语句
     *
     * @param searchCondition 检索条件
     * @param subjectIdList   专题id集合
     * @author lkg
     * @date 2024/12/25
     */
    private BoolQueryBuilder buildQuery(InfoDataSearchCondition searchCondition, List<String> subjectIdList) {
        //判断是否需要查询发布时间为空的数据
        boolean timeFlag = false;
        if(StringUtils.isEmpty(searchCondition.getStartTime()) && StringUtils.isEmpty(searchCondition.getEndTime())) {
            //前端没有穿时间参数时，需要查询发布时间为空的数据
            timeFlag = true;
        }
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        if (CollectionUtils.isNotEmpty(subjectIdList)) {
            Integer category = searchCondition.getCategory();
            boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
            //单个专题/事件，默认加上专题/事件的时间范围
            if (subjectIdList.size() == 1) {
                String configStartTime = null;
                String configEndTime = null;
                String id = subjectIdList.get(0);
                if (category == 1) {
                    Subject subject = subjectService.getById(id);
                    if (!Objects.isNull(subject)) {
                        Date timeEnable = subject.getTimeEnable();
                        if (!Objects.isNull(timeEnable)) {
                            configStartTime = DateUtil.dateToString(timeEnable);
                        }
                        Date timeDisable = subject.getTimeDisable();
                        if (!Objects.isNull(timeDisable)) {
                            configEndTime = DateUtil.dateToString(timeDisable);
                        }
                    }
                } else if (category == 2) {
                    Event event = eventService.getById(id);
                    if (!Objects.isNull(event)) {
                        Date startTime = event.getStartTime();
                        if (!Objects.isNull(startTime)) {
                            configStartTime = DateUtil.dateToString(startTime);
                        }
                        Date endTime = event.getEndTime();
                        if (!Objects.isNull(endTime)) {
                            configEndTime = DateUtil.dateToString(endTime);
                        }
                    }
                }
                //时间过滤筛选-专题配置
                if (StringUtils.isNotEmpty(configStartTime) && (StringUtils.isEmpty(searchCondition.getStartTime())||configStartTime.compareTo(searchCondition.getStartTime())>0)) {
                    searchCondition.setStartTime(configStartTime);
                }
                if(StringUtils.isEmpty(configEndTime)){
                    configEndTime = EsDateUtil.esFieldDateFormat(DateUtil.dateToString(new Date()));
                }
                if (StringUtils.isEmpty(searchCondition.getEndTime())||configEndTime.compareTo(searchCondition.getEndTime())<0) {
                    searchCondition.setEndTime(configEndTime);
                }
            }
        }
        //高级查询数据处理
        BoolQueryBuilder superQuery = buildSuperQuery(searchCondition.getSuperQueryMatchType(), searchCondition.getSuperQueryParams());
        if (superQuery != null) {
            boolQuery.must(superQuery);
        }
        List<SearchWordVO> searchWordList = searchCondition.getSearchWordList();
        //研究中心 搜索词方式
        if (CollectionUtils.isNotEmpty(searchWordList)) {
            handlerSearchWordList(boolQuery, searchWordList);
        } else {//平台 搜索词方式
            if (StringUtils.isNotEmpty(searchCondition.getSearch())) {
                boolQuery.must(QueryBuilders.multiMatchQuery(searchCondition.getSearch(), "title", "content"));
            }
            if (StringUtils.isNotEmpty(searchCondition.getContent())) {
                boolQuery.must(QueryBuilders.matchPhraseQuery("content", searchCondition.getContent()));
            }
            if (StringUtils.isNotEmpty(searchCondition.getTitle())) {
                boolQuery.must(QueryBuilders.matchPhraseQuery("title", searchCondition.getTitle()));
            }
            if (StringUtils.isNotEmpty(searchCondition.getOrigin())) {
                boolQuery.must(QueryBuilders.matchPhraseQuery("origin", searchCondition.getOrigin()));
            }
            if (searchCondition.getLabelName() != null) {
                BoolQueryBuilder nestedBoolQueryBuilder = QueryBuilders.boolQuery();
                MatchPhraseQueryBuilder relationNameQuery = QueryBuilders.matchPhraseQuery("labels.relationName", searchCondition.getLabelName());
                nestedBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationNameQuery, ScoreMode.None));
                boolQuery.must(nestedBoolQueryBuilder);
            }
        }
        if (searchCondition.getLabelMark() != null) {
            BoolQueryBuilder nestedBoolQueryBuilder = QueryBuilders.boolQuery();
            MatchPhraseQueryBuilder relationNameQuery = QueryBuilders.matchPhraseQuery("labels.labelMark", searchCondition.getLabelMark());
            nestedBoolQueryBuilder.should(QueryBuilders.nestedQuery("labels", relationNameQuery, ScoreMode.None));
            boolQuery.must(nestedBoolQueryBuilder);
        }
        //资讯状态-研究中心(0-全部;1-模型推荐;2-精选;3-待定;4-移除),和checkStatus、deleteFlag、isFreeCheck互斥
        Integer status = searchCondition.getStatus();
        if (status != null) {
            if (status == 1) {
                boolQuery.must(QueryBuilders.termQuery("checkStatus", 0));
                boolQuery.must(QueryBuilders.termQuery("isFreeCheck", 1));
                boolQuery.mustNot(QueryBuilders.termQuery("deleteFlag", 1));
            } else if (status == 2) {
                boolQuery.must(QueryBuilders.termQuery("checkStatus", 1));
                boolQuery.mustNot(QueryBuilders.termQuery("deleteFlag", 1));
            } else if (status == 3) {
                boolQuery.must(QueryBuilders.termQuery("checkStatus", 3));
                boolQuery.mustNot(QueryBuilders.termQuery("deleteFlag", 1));
            } else if (status == 4) {
                boolQuery.must(QueryBuilders.termQuery("deleteFlag", 1));
            }
        } else {
            Integer checkStatus = searchCondition.getCheckStatus();
            Integer deleteFlag = searchCondition.getDeleteFlag();
            if (checkStatus != null) {
                if (checkStatus == 0) {
                    BoolQueryBuilder checkStatusBoolQuery = QueryBuilders.boolQuery();
                    checkStatusBoolQuery.should(QueryBuilders.termQuery("checkStatus", checkStatus));
                    checkStatusBoolQuery.should(QueryBuilders.boolQuery().mustNot(QueryBuilders.existsQuery("checkStatus")));
                    boolQuery.must(checkStatusBoolQuery);
                } else {
                    boolQuery.must(QueryBuilders.termQuery("checkStatus", checkStatus));
                }
            } else {
                if (deleteFlag == null) {
                    deleteFlag = 0;
                }
            }
            //删除状态查询
            if (deleteFlag != null) {
                if (deleteFlag != 0) {
                    boolQuery.must(QueryBuilders.termQuery("deleteFlag", "1"));
                } else {
                    boolQuery.mustNot(QueryBuilders.termQuery("deleteFlag", "1"));
                }
            }
            //免审核状态
            if (searchCondition.getIsFreeCheck() != null) {
                boolQuery.must(QueryBuilders.termQuery("isFreeCheck", 1));
            }
        }
        if (StringUtils.isNotBlank(searchCondition.getVideo())) {
            boolQuery.must(QueryBuilders.matchQuery("type", "video"));
        } else {
            boolQuery.mustNot(QueryBuilders.matchQuery("type", "video"));
        }
        //专题库类别筛选
        if (CollectionUtils.isNotEmpty(searchCondition.getClassificationType())) {
            boolQuery.must(QueryBuilders.termsQuery("classificationType", searchCondition.getClassificationType()));
        }
        //是否原创
        if (StringUtils.isNotEmpty(searchCondition.getOriginality())) {
            boolQuery.must(QueryBuilders.matchQuery("originality", searchCondition.getOriginality()));
        }
        //得分范围筛选
        Integer minScore = searchCondition.getMinScore();
        if (minScore != null) {
            boolQuery.filter(QueryBuilders.rangeQuery("score").gte(minScore));
        }
        Integer maxScore = searchCondition.getMaxScore();
        if (maxScore != null) {
            boolQuery.filter(QueryBuilders.rangeQuery("score").lte(maxScore));
        }
        //正文长度筛选
        Integer minContentLength = searchCondition.getMinContentLength();
        if (minContentLength != null) {
            NestedQueryBuilder nestedQueryBuilder = QueryBuilders
                    .nestedQuery("sortField", QueryBuilders.rangeQuery("sortField.fieldLong").gte(minContentLength), ScoreMode.None);
            boolQuery.filter(nestedQueryBuilder);
        }
        Integer maxContentLength = searchCondition.getMaxContentLength();
        if (maxContentLength != null) {
            NestedQueryBuilder nestedQueryBuilder = QueryBuilders
                    .nestedQuery("sortField", QueryBuilders.rangeQuery("sortField.fieldLong").lte(maxContentLength), ScoreMode.None);
            boolQuery.filter(nestedQueryBuilder);
        }
        //时间过滤筛选-前端传参
        if(StringUtils.isEmpty(searchCondition.getStartTime()) && StringUtils.isEmpty(searchCondition.getEndTime())){
            BoolQueryBuilder publishDateQueryBuilder = QueryBuilders.boolQuery();
            BoolQueryBuilder configDateQueryBuilder = QueryBuilders.boolQuery();
            //时间过滤筛选-专题配置
            if (StringUtils.isNotBlank(searchCondition.getStartTime())) {
                configDateQueryBuilder.must(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(searchCondition.getStartTime())));
            }
            if (StringUtils.isNotBlank(searchCondition.getEndTime())) {
                configDateQueryBuilder.must(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(searchCondition.getEndTime())));
            } else {
                configDateQueryBuilder.must(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(DateUtil.dateToString(new Date()))));
            }
            publishDateQueryBuilder.should(configDateQueryBuilder);
            if(timeFlag){
                publishDateQueryBuilder.should(QueryBuilders.boolQuery().mustNot(QueryBuilders.existsQuery("publishDate")));
            }
            boolQuery.must(publishDateQueryBuilder);
        }
        String crawler = searchCondition.getCrawler();
        if (StringUtils.isNotBlank(crawler)) {
            boolQuery.must(QueryBuilders.termQuery("source.keyword", crawler));
        }
        //组合标签查询(不同类标签之间是与的关系，同一类标签之间是或的关系)，示例: "a1,a2;c1,c2;e1,e2"
        String composeSearchLabelIds = searchCondition.getComposeSearchLabelIds();
        if (StringUtils.isNotEmpty(composeSearchLabelIds)) {
            //分号隔开的标签查询用且的关系
            String[] split = composeSearchLabelIds.split(";");
            for (String items : split) {
                List<String> ids = Arrays.asList(items.split(","));
                NestedQueryBuilder nestedQueryBuilder = QueryBuilders
                        .nestedQuery("labels", QueryBuilders.termsQuery("labels.relationId", ids), ScoreMode.None);
                boolQuery.must(nestedQueryBuilder);
            }
        }
        //关联标签id集合
        List<String> labelIds = searchCondition.getLabelIds();
        if (CollectionUtils.isNotEmpty(labelIds)) {
            Set<String> relationIdSet = new HashSet<>(labelIds);
            TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", relationIdSet);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        //正面标签
        String tendencyLabelId = searchCondition.getTendencyLabelId();
        if (StringUtils.isNotBlank(tendencyLabelId)) {
            TermQueryBuilder relationIdQuery = QueryBuilders.termQuery("labels.relationId", tendencyLabelId);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        //地域标签
        List<String> areaLabelIds = searchCondition.getAreaLabelIds();
        if (CollectionUtils.isNotEmpty(areaLabelIds)) {
            Set<String> relationIdSet = new HashSet<>(areaLabelIds);
            TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", relationIdSet);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        //企业标签
        List<String> enterpriseLabelTypeIds = searchCondition.getEnterpriseLabelTypeIds();
        List<String> socialCreditCodeList = searchCondition.getSocialCreditCodeList();
        if (CollectionUtils.isNotEmpty(socialCreditCodeList) || CollectionUtils.isNotEmpty(enterpriseLabelTypeIds)) {
            if (CollectionUtils.isEmpty(socialCreditCodeList)) {
                socialCreditCodeList = commonService.codesByLabels(enterpriseLabelTypeIds);
            }
            Set<String> codeSet = new HashSet<>(socialCreditCodeList);
            TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", codeSet);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        //信息源标签
        List<Label> labelList = searchCondition.getLabelList();
        if (CollectionUtils.isNotEmpty(labelList)) {
            List<String> collect = labelList.stream().map(label -> label.getLabelMark() + "-" + label.getRelationId()).collect(Collectors.toList());
            TermsQueryBuilder relationIdQuery = QueryBuilders.termsQuery("labels.relationId", collect);
            boolQuery.must(QueryBuilders.nestedQuery("labels", relationIdQuery, ScoreMode.None));
        }
        String sourceId = searchCondition.getSourceId();
        if (StringUtils.isNotBlank(sourceId)) {
            boolQuery.must(QueryBuilders.termQuery("sid.keyword", sourceId));
        }
        //待删除的关键词列表
        if (CollectionUtils.isNotEmpty(searchCondition.getKeywordList())) {
            TermsQueryBuilder sortFieldQuery = QueryBuilders.termsQuery("sortField.fieldType", searchCondition.getKeywordList());
            boolQuery.must(QueryBuilders.nestedQuery("sortField", sortFieldQuery, ScoreMode.None));
        }
        return boolQuery;
    }


    //格式化 按时间分组数据统计信息
    private List<CountVO> formatGroupData(SearchResponse searchResponse, String groupType) {
        List<CountVO> data = new ArrayList<>();
        if (searchResponse != null) {
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取日期分组集合
            ParsedDateHistogram parsedDateHistogram = null;
            if ("year".equalsIgnoreCase(groupType)) {
                parsedDateHistogram = aggregations.get("groupYear");
            } else if ("month".equalsIgnoreCase(groupType)) {
                parsedDateHistogram = aggregations.get("groupMonth");
            } else if ("day".equalsIgnoreCase(groupType)) {
                parsedDateHistogram = aggregations.get("groupDay");
            } else if ("hour".equalsIgnoreCase(groupType)) {
                parsedDateHistogram = aggregations.get("groupHour");
            }
            if (parsedDateHistogram != null) {
                List<? extends Histogram.Bucket> buckets = parsedDateHistogram.getBuckets();
                for (Histogram.Bucket bucket : buckets) {
                    CountVO countVO = new CountVO();
                    countVO.setName(bucket.getKeyAsString());
                    countVO.setValue(bucket.getDocCount());
                    List<CountVO> children = new ArrayList<>();
                    Aggregations typeAgg = bucket.getAggregations();
                    Terms groupLibraryType = typeAgg.get("groupLibraryType");
                    List<? extends Terms.Bucket> typeBuckets = groupLibraryType.getBuckets();
                    for (Terms.Bucket typeBucket : typeBuckets) {
                        CountVO child = new CountVO();
                        child.setName(typeBucket.getKeyAsString());
                        child.setValue(typeBucket.getDocCount());
                        children.add(child);
                    }
                    countVO.setChildren(children);
                    data.add(countVO);
                }
            }
        }
        return data;
    }

    //格式化数据
    private List<SubjectDataVo> formatData(SearchHits searchHits) {
        List<SubjectDataVo> list = new ArrayList<>();
        if (searchHits != null) {
            SearchHit[] hits = searchHits.getHits();
            for (SearchHit hit : hits) {
                String index = hit.getIndex();
                String sourceAsString = hit.getSourceAsString();
                SubjectDataVo subjectDataVo = JSON.parseObject(sourceAsString, SubjectDataVo.class);
                subjectDataVo.setPublishDate(EsDateUtil.esFieldDateMapping(subjectDataVo.getPublishDate()));
                subjectDataVo.setIndex(index);
                list.add(subjectDataVo);
            }
        }
        return list;
    }

    //获取es查询结果
    private SearchHits getEsResult(SearchSourceBuilder searchSourceBuilder, BoolQueryBuilder boolQuery, String subjectId) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        //未删除
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            return searchResponse.getHits();
        } catch (Exception e) {
            log.info("获取专题-{}-下资讯，查询es库失败", subjectId);
            e.printStackTrace();
        }
        return null;
    }

    private void handlerSearchWordList(BoolQueryBuilder boolQuery, List<SearchWordVO> searchWordList) {
        // 非条件处理
        List<SearchWordVO> notList = searchWordList.stream().filter(ask -> "NOT".equalsIgnoreCase(ask.getSearchLogicRelationship())).collect(Collectors.toList());
        handlerSearchWordMustQuery(boolQuery, notList, 2);
        // 或和与处理 （a and b）or (c and d) or (e and f)
        List<SearchWordVO> otherList = searchWordList.stream().filter(ask -> !"NOT".equalsIgnoreCase(ask.getSearchLogicRelationship())).collect(Collectors.toList());
        List<SearchWordVO> andList = searchWordList.stream().filter(ask -> "AND".equalsIgnoreCase(ask.getSearchLogicRelationship())).collect(Collectors.toList());
        List<SearchWordVO> orList = searchWordList.stream().filter(ask -> "OR".equalsIgnoreCase(ask.getSearchLogicRelationship())).collect(Collectors.toList());
        if (andList.size() == otherList.size()) {//全是 and
            handlerSearchWordMustQuery(boolQuery, andList, 1);
        } else if (orList.size() == otherList.size()) { //全是 or
            handlerSearchWordShouldQuery(boolQuery, orList);
        } else { //and 和 or 都有
            handlerSearchWordMustAndShouldQuery(boolQuery, otherList);
        }
    }

    private void handlerSearchWordMustAndShouldQuery(BoolQueryBuilder allBoolQuery, List<SearchWordVO> searchWordList) {
        // 根据OR将List分组
        List<List<SearchWordVO>> queryList = new ArrayList<>();
        List<SearchWordVO> tempList = new ArrayList<>();
        for (int i = 0; i < searchWordList.size(); i++) {
            SearchWordVO ask = searchWordList.get(i);
            String searchLogicRelationship = ask.getSearchLogicRelationship();
            if ("OR".equalsIgnoreCase(searchLogicRelationship)) {
                // 放入之前数据
                queryList.add(tempList);
                // 创建新数据放入自己
                tempList = new ArrayList<>();
                tempList.add(ask);
            } else {
                tempList.add(ask);
            }
            // 如果最后一个元素未处理放入
            if (i == searchWordList.size() - 1) {
                queryList.add(tempList);
            }
        }
        BoolQueryBuilder childrenBoolQuery = QueryBuilders.boolQuery();
        // 分别处理每一段
        for (List<SearchWordVO> list : queryList) {
            // 每一段内部是and
            BoolQueryBuilder innerBoolQuery = QueryBuilders.boolQuery();
            handlerSearchWordMustQuery(innerBoolQuery, list, 1);
            childrenBoolQuery.should(innerBoolQuery);
        }
        allBoolQuery.must(childrenBoolQuery);
    }


    /**
     * 关键词必须/必须不满足条件封装(must/must_not)
     *
     * @param allBoolQuery   总查询bool
     * @param searchWordList 搜索词
     * @param type           1-必须;2-必须不
     * @author lkg
     * @date 2024/12/25
     */
    private void handlerSearchWordMustQuery(BoolQueryBuilder allBoolQuery, List<SearchWordVO> searchWordList, Integer type) {
        if (CollectionUtils.isNotEmpty(searchWordList)) {
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            for (SearchWordVO ask : searchWordList) {
                String searchInfo = ask.getSearchInfo();
                Integer searchScope = ask.getSearchScope();
                String searchAccuracy = ask.getSearchAccuracy();
                if (searchScope == 1) {
                    buildSplitWordMustQuery(boolQuery, searchInfo, "title", searchAccuracy);
                } else if (searchScope == 2) {
                    buildSplitWordMustQuery(boolQuery, searchInfo, "content", searchAccuracy);
                } else if (searchScope == 3) {
                    if ("精确".equals(searchAccuracy)) {
                        if (searchInfo.contains("|")) {
                            BoolQueryBuilder splitWordQuery = QueryBuilders.boolQuery();
                            for (String word : searchInfo.split("\\|")) {
                                MultiMatchQueryBuilder multiMatchQuery = QueryBuilders.multiMatchQuery(word, "title", "content");
                                multiMatchQuery.type(MultiMatchQueryBuilder.Type.PHRASE);
                                splitWordQuery.should(multiMatchQuery);
                            }
                            boolQuery.must(splitWordQuery);
                        } else {
                            MultiMatchQueryBuilder multiMatchQuery = QueryBuilders.multiMatchQuery(searchInfo, "title", "content");
                            multiMatchQuery.type(MultiMatchQueryBuilder.Type.PHRASE);
                            boolQuery.must(multiMatchQuery);
                        }
                    } else if ("模糊".equals(searchAccuracy)) {
                        if (searchInfo.contains("|")) {
                            BoolQueryBuilder splitWordQuery = QueryBuilders.boolQuery();
                            for (String word : searchInfo.split("\\|")) {
                                MultiMatchQueryBuilder multiMatchQuery = QueryBuilders.multiMatchQuery(word, "title", "content");
                                splitWordQuery.should(multiMatchQuery);
                            }
                            boolQuery.must(splitWordQuery);
                        } else {
                            boolQuery.must(QueryBuilders.multiMatchQuery(searchInfo, "title", "content"));
                        }
                    }
                } else if (searchScope == 4) {
                    buildSplitWordMustQuery(boolQuery, searchInfo, "origin", searchAccuracy);
                }
            }
            if (type == 1) {
                allBoolQuery.must(boolQuery);
            } else if (type == 2) {
                allBoolQuery.mustNot(boolQuery);
            }
        }
    }

    /**
     * 关键词应该满足条件封装(should)
     *
     * @param allBoolQuery   总查询bool
     * @param searchWordList 搜索词
     * @author lkg
     * @date 2024/12/25
     */
    private void handlerSearchWordShouldQuery(BoolQueryBuilder allBoolQuery, List<SearchWordVO> searchWordList) {
        if (CollectionUtils.isNotEmpty(searchWordList)) {
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            for (SearchWordVO ask : searchWordList) {
                String searchInfo = ask.getSearchInfo();
                Integer searchScope = ask.getSearchScope();
                String searchAccuracy = ask.getSearchAccuracy();
                if (searchScope == 1) {
                    buildSplitWordShouldQuery(boolQuery, searchInfo, "title", searchAccuracy);
                } else if (searchScope == 2) {
                    buildSplitWordShouldQuery(boolQuery, searchInfo, "content", searchAccuracy);
                } else if (searchScope == 3) {
                    if ("精确".equals(searchAccuracy)) {
                        if (searchInfo.contains("|")) {
                            for (String word : searchInfo.split("\\|")) {
                                MultiMatchQueryBuilder multiMatchQuery = QueryBuilders.multiMatchQuery(word, "title", "content");
                                multiMatchQuery.type(MultiMatchQueryBuilder.Type.PHRASE);
                                boolQuery.should(multiMatchQuery);
                            }
                        } else {
                            MultiMatchQueryBuilder multiMatchQuery = QueryBuilders.multiMatchQuery(searchInfo, "title", "content");
                            multiMatchQuery.type(MultiMatchQueryBuilder.Type.PHRASE);
                            boolQuery.should(multiMatchQuery);
                        }
                    } else if ("模糊".equals(searchAccuracy)) {
                        if (searchInfo.contains("|")) {
                            for (String word : searchInfo.split("\\|")) {
                                boolQuery.should(QueryBuilders.multiMatchQuery(word, "title", "content"));
                            }
                        } else {
                            boolQuery.should(QueryBuilders.multiMatchQuery(searchInfo, "title", "content"));
                        }
                    }
                } else if (searchScope == 4) {
                    buildSplitWordShouldQuery(boolQuery, searchInfo, "origin", searchAccuracy);
                }
            }
            allBoolQuery.must(boolQuery);
        }
    }

    private void buildSplitWordMustQuery(BoolQueryBuilder boolQuery, String searchInfo, String fieldName, String searchAccuracy) {
        if ("精确".equals(searchAccuracy)) {
            if (searchInfo.contains("|")) {
                BoolQueryBuilder splitWordQuery = QueryBuilders.boolQuery();
                for (String word : searchInfo.split("\\|")) {
                    splitWordQuery.should(QueryBuilders.matchPhraseQuery(fieldName, word));
                }
                boolQuery.must(splitWordQuery);
            } else {
                boolQuery.must(QueryBuilders.matchPhraseQuery(fieldName, searchInfo));
            }
        } else if ("模糊".equals(searchAccuracy)) {
            if (searchInfo.contains("|")) {
                BoolQueryBuilder splitWordQuery = QueryBuilders.boolQuery();
                for (String word : searchInfo.split("\\|")) {
                    splitWordQuery.should(QueryBuilders.matchQuery(fieldName, word));
                }
                boolQuery.must(splitWordQuery);
            } else {
                boolQuery.must(QueryBuilders.matchQuery(fieldName, searchInfo));
            }
        }
    }

    private void buildSplitWordShouldQuery(BoolQueryBuilder boolQuery, String searchInfo, String fieldName, String searchAccuracy) {
        if ("精确".equals(searchAccuracy)) {
            if (searchInfo.contains("|")) {
                for (String word : searchInfo.split("\\|")) {
                    boolQuery.should(QueryBuilders.matchPhraseQuery(fieldName, word));
                }
            } else {
                boolQuery.must(QueryBuilders.matchPhraseQuery(fieldName, searchInfo));
            }
        } else if ("模糊".equals(searchAccuracy)) {
            if (searchInfo.contains("|")) {
                for (String word : searchInfo.split("\\|")) {
                    boolQuery.should(QueryBuilders.matchQuery(fieldName, word));
                }
            } else {
                boolQuery.must(QueryBuilders.matchQuery(fieldName, searchInfo));
            }
        }
    }


    /**
     * 高级检索条件封装
     *
     * @param superQueryMatchType 规则
     * @param superQueryParams    条件（url转码后的json）
     * @author lkg
     * @date 2025/3/13
     */
    private BoolQueryBuilder buildSuperQuery(String superQueryMatchType, String superQueryParams) {
        if (StringUtils.isNotEmpty(superQueryParams)) {
            try {
                BoolQueryBuilder superBuilder = QueryBuilders.boolQuery();
                String params = URLDecoder.decode(superQueryParams, Charsets.UTF_8);
                JSONArray array = JSON.parseArray(params);
                for (int i = 0; i < array.size(); i++) {
                    JSONObject item = array.getJSONObject(i);
                    BoolQueryBuilder builder2 = QueryBuilders.boolQuery();
                    String type = item.getString("type");
                    String rule = item.getString("rule");
                    String field = item.getString("field");
                    String val = item.getString("val");
                    if ("date".equals(type)) { //日期类型处理
                        if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from(val + "T00:00:00").to(val + "T23:59:59"));
                        }
                        if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.rangeQuery(field).from(val + "T00:00:00").to(val + "T23:59:59"));
                        } else if ("gt".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from((val + "T23:59:59")));
                        } else if ("ge".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).from((val + "T00:00:00")));
                        } else if ("lt".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).to((val + "T00:00:00")));
                        } else if ("le".equals(rule)) {
                            builder2.must(QueryBuilders.rangeQuery(field).to((val + "T23:59:59")));
                        }
                    } else if ("list".equals(type)) { //列表（数组）类查询
                        if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("in".equals(rule)) {
                            builder2.must(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("not_in".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("keyWordsGroup".equals(rule)) {
                            if (StringUtils.isNotEmpty(val)) {
                                BoolQueryBuilder wordAllBoolQuery = QueryBuilders.boolQuery();
                                String[] andWords = val.split("\\+");
                                for (String words : andWords) {
                                    String replace = words.replace("(", "").replace(")", "");
                                    BoolQueryBuilder wordOrBoolQuery = QueryBuilders.boolQuery();
                                    for (String word : replace.split("\\|")) {
                                        wordOrBoolQuery.should(QueryBuilders.termQuery(field + ".keyword", word));
                                    }
                                    wordAllBoolQuery.must(wordOrBoolQuery);
                                }
                                builder2.must(wordAllBoolQuery);
                            }
                        }
                    } else if ("string".equals(type)) { //文本类查询
                        if ("fullText".equals(field)) {
                            BoolQueryBuilder fullTextBoolQuery = multiMatchQuery(val);
                            if ("like".equals(rule)) {
                                builder2.must(fullTextBoolQuery);
                            } else if ("not_like".equals(rule)) {
                                builder2.mustNot(fullTextBoolQuery);
                            } else if ("keyWordsGroup".equals(rule)) {
                                BoolQueryBuilder wordAllBoolQuery = QueryBuilders.boolQuery();
                                String[] andWords = val.split("\\+");
                                for (String words : andWords) {
                                    String replace = words.replace("(", "").replace(")", "");
                                    BoolQueryBuilder wordOrBoolQuery = QueryBuilders.boolQuery();
                                    for (String word : replace.split("\\|")) {
                                        wordOrBoolQuery.should(multiMatchQuery(word));
                                    }
                                    wordAllBoolQuery.must(wordOrBoolQuery);
                                }
                                builder2.must(wordAllBoolQuery);
                            }
                        } else if ("eq".equals(rule)) {
                            builder2.must(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("ne".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termQuery(field + ".keyword", val));
                        } else if ("in".equals(rule)) {
                            builder2.must(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("not_in".equals(rule)) {
                            builder2.mustNot(QueryBuilders.termsQuery(field + ".keyword", val.split(" |\\+|,|，|;|；")));
                        } else if ("like".equals(rule)) {
                            builder2.must(QueryBuilders.matchPhrasePrefixQuery(field, val));
                        } else if ("not_like".equals(rule)) {
                            builder2.mustNot(QueryBuilders.matchPhrasePrefixQuery(field, val));
                        } else if ("keyWordsGroup".equals(rule)) {
                            BoolQueryBuilder wordAllBoolQuery = QueryBuilders.boolQuery();
                            String[] andWords = val.split("\\+");
                            for (String words : andWords) {
                                String replace = words.replace("(", "").replace(")", "");
                                BoolQueryBuilder wordOrBoolQuery = QueryBuilders.boolQuery();
                                for (String word : replace.split("\\|")) {
                                    wordOrBoolQuery.should(QueryBuilders.matchPhraseQuery(field, word));
                                }
                                wordAllBoolQuery.must(wordOrBoolQuery);
                            }
                            builder2.must(wordAllBoolQuery);
                        }
                    }
                    if ("or".equals(superQueryMatchType)) {
                        superBuilder.should(builder2);
                    } else {
                        superBuilder.must(builder2);
                    }
                }
                return superBuilder;
            } catch (Exception e) {
                log.error("高级查询条件封装失败,e:{},params:{}", e.getMessage(), superQueryParams);
            }
        }
        return null;
    }

    private BoolQueryBuilder multiMatchQuery(String word) {
        BoolQueryBuilder fullTextBoolQuery = QueryBuilders.boolQuery();
        String[] fullTextFields = new String[]{"title", "summary", "content"};
        for (String fullTextField : fullTextFields) {
            fullTextBoolQuery.should(QueryBuilders.matchPhraseQuery(fullTextField, word));
        }
        return fullTextBoolQuery;
    }

    /**
     * 根据id查询标签列表
     *
     * @param index
     * @param id
     * @return
     */
    public List<Label> getLabelsById(String index, String id) {
        try {
            GetRequest request = new GetRequest(index, id);
            request.fetchSourceContext(new FetchSourceContext(true, new String[]{"labels", "title"}, null));
            request.realtime(true);
            GetResponse documentFields = client.get(request, RequestOptions.DEFAULT);
            String labelsStr = JSONArray.toJSONString(documentFields.getSourceAsMap().get("labels"));
            return JSONArray.parseArray(labelsStr, Label.class);
        } catch (Exception e) {
            log.error("根据数据id获取标签失败，id：{},e:{}", id, e.getMessage());
        }
        return null;
    }

    /**
     * 根据id更新标签列表
     *
     * @param index
     * @param id
     * @param labels
     */
    public void updateLabelsById(String index, String id, List<Label> labels) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("labels", labels);
        esOpUtil.docUpdateById(index, id, jsonObject.toString());
    }
}
