package com.zzsn.event.external.controller;

import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.external.entity.ExternalSubjectInfoSourceMap;
import com.zzsn.event.external.service.ExternalSubjectInfoSourceMapService;
import com.zzsn.event.external.vo.SubjectInfoVO;
import com.zzsn.event.external.vo.SubjectVO;
import com.zzsn.event.llm.LlmService;
import com.zzsn.event.service.CommonService;
import com.zzsn.event.service.InformationService;
import com.zzsn.event.service.SubjectService;
import com.zzsn.event.service.SubjectSimpleService;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.util.ObjectUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.vo.InfoDataSearchCondition;
import com.zzsn.event.vo.SubjectDetailVO;
import com.zzsn.event.vo.SubjectSimpleVO;
import com.zzsn.event.vo.TranslateVO;
import com.zzsn.event.vo.es.DisplayInfo;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * @author lkg
 * @date 2025/8/12
 */
@Slf4j
@RestController
@RequestMapping("/external")
public class ExternalController {


    @Autowired
    private SubjectSimpleService subjectSimpleService;
    @Autowired
    private IXxlJobInfoService xxlJobInfoService;
    @Autowired
    private ExternalSubjectInfoSourceMapService externalSubjectInfoSourceMapService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private LlmService llmService;
    @Autowired
    private InformationService informationService;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    SubjectService subjectService;

    @Value("${caiji.keywordCrawler.url:}")
    private String keywordCrawlerUrl;
    @Value("${translate.url:}")
    private String TRANSLATE_URL;

    /**
     * 创建专题
     *
     * @param subjectVO 参数
     * @author lkg
     * @date 2025/1/9
     */
    @PostMapping("/createSubject")
    public Result<?> createSubject(@RequestBody SubjectVO subjectVO) {
        subjectVO.setDataScope("1");
        SubjectSimpleVO subjectSimpleVO = new SubjectSimpleVO();
        BeanUtils.copyProperties(subjectVO, subjectSimpleVO);
        Subject subject = subjectSimpleService.createExternalSubject(subjectSimpleVO);
        CompletableFuture.runAsync(() -> {
            //插入xxlJob
            xxlJobInfoService.subjectInsert(subject);
        });
        return Result.OK(subject.getId());
    }

    /**
     * 编辑专题
     *
     * @param subjectVO 参数
     * @author lkg
     * @date 2025/1/9
     */
    @PostMapping("/updateSubject")
    public Result<?> updateSubject(@RequestBody SubjectVO subjectVO) {
        subjectVO.setDataScope("1");
        SubjectSimpleVO subjectSimpleVO = new SubjectSimpleVO();
        BeanUtils.copyProperties(subjectVO, subjectSimpleVO);
        subjectSimpleService.editExternalSubject(subjectSimpleVO);
        return Result.OK();
    }

    /**
     * 采集词推荐
     *
     * @param remark 专题描述
     * @author lkg
     * @date 2025/8/12
     */
    @GetMapping("/crawlerWordRecommend")
    public Result<?> crawlerWordRecommend(@RequestParam String remark) {
        String prompt = "你是一个专业信息采集助手，根据用户专题描述生成关键词表达式：\n" +
                "1. 用 + 表示与关系（必须同现），| 表示或关系（任一出现），用()分组\n" +
                "2. 执行三步操作：\n" +
                "   (1) 提取核心实体（如企业/产品名）\n" +
                "   (2) 将抽象词拆解为3-5个具体场景词（如\"动向\"→\"上升|下降|波动|结构调整\"）\n" +
                "   (3) 扩展同义词/行业关联词\n" +
                "3. 排除时间词，+号不超过2个，单组|内词不超过5个\n" +
                "4. 输出仅含表达式\n" +
                "\n" +
                "示例： \n" +
                "描述：苹果新款iPhone发布引发市场反响\n" +
                "输出：(苹果|iPhone)+(発売|価格|予約状況|市場シェア|消費者反応)";
        String crawlerWord = llmService.model("qwen", "qwen-max-latest", prompt, remark);
        return Result.OK(crawlerWord);
    }

    /**
     * 绑定信息源分页列表
     *
     * @param subjectId 专题id
     * @param pageNo    页码
     * @param pageSize  每页返回条数
     * @author lkg
     * @date 2025/8/12
     */
    @GetMapping("/bindSourcePageList")
    public Result<?> bindSourceList(@RequestParam String subjectId,
                                    @RequestParam(defaultValue = "1") Integer pageNo,
                                    @RequestParam(defaultValue = "10") Integer pageSize) {
        Page<ExternalSubjectInfoSourceMap> page = new Page<>(pageNo, pageSize);
        LambdaQueryWrapper<ExternalSubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(ExternalSubjectInfoSourceMap::getSubjectId, subjectId)
                .eq(ExternalSubjectInfoSourceMap::getType, 1);
        Page<ExternalSubjectInfoSourceMap> list = externalSubjectInfoSourceMapService.page(page, queryWrapper);
        return Result.OK(list);
    }


    /**
     * 下载导入信息源
     *
     * @author lkg
     * @date 2024/06/21
     */
    @GetMapping("/downloadInfoSourceTemplate")
    public void downloadResearchTemplate(HttpServletResponse response) {
        String filePath = "subjectDataImport/导入信息源模板.xlsx";
        commonService.downloadTemplate(response, filePath);
    }


    /**
     * 批量导入绑定得信息源
     *
     * @author lkg
     * @date 2025/8/12
     */
    @PostMapping("/importInfoSource")
    public Result<?> importInfoSource(HttpServletRequest request) {
        String subjectId = request.getParameter("subjectId");
        if (StringUtils.isBlank(subjectId)) {
            return Result.FAIL(500, "专题id不能为空");
        }
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        if (fileMap.size() < 1) {
            return Result.FAIL(500, "请上传excel文件");
        }
        MultipartFile multipartFile = fileMap.get(new ArrayList<>(fileMap.keySet()).get(0));
        int index = multipartFile.getOriginalFilename().lastIndexOf(".");
        String fileSuffix = multipartFile.getOriginalFilename().substring(index + 1);
        if ("xls".equals(fileSuffix) || "xlsx".equals(fileSuffix)) {
            try {
                ExcelReader reader = ExcelUtil.getReader(multipartFile.getInputStream());
                Map<String, String> header = new HashMap<>();
                header.put("信息源名称", "infoSourceName");
                header.put("信息源地址", "infoSourceUrl");
                reader.setHeaderAlias(header);
                List<ExternalSubjectInfoSourceMap> infoSourceList = reader.read(0, 1, ExternalSubjectInfoSourceMap.class);
                infoSourceList.forEach(infoSource -> {
                    infoSource.setSubjectId(subjectId);
                    infoSource.setType(1);
                });
                externalSubjectInfoSourceMapService.saveBatch(infoSourceList);
            } catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            return Result.FAIL(500, "不支持的文件类型");
        }
        return Result.OK();
    }

    /**
     * 绑定信息源
     *
     * @param externalSubjectInfoSourceMap 参数
     * @author lkg
     * @date 2025/8/12
     */
    @PostMapping("/bindSource")
    public Result<?> bindSource(@RequestBody ExternalSubjectInfoSourceMap externalSubjectInfoSourceMap) {
        externalSubjectInfoSourceMap.setType(1);
        externalSubjectInfoSourceMapService.save(externalSubjectInfoSourceMap);
        return Result.OK();
    }

    /**
     * 解绑信息源
     *
     * @param ids id集合
     * @author lkg
     * @date 2025/8/12
     */
    @GetMapping("/unBindSource")
    public Result<?> bindSource(@RequestParam List<String> ids) {
        externalSubjectInfoSourceMapService.removeByIds(ids);
        return Result.OK();
    }

    /**
     * 专题详情
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/1/9
     */
    @GetMapping("/queryInfo")
    public Result<?> queryInfo(@RequestParam String subjectId) {
        SubjectInfoVO subjectInfoVO = new SubjectInfoVO();
        SubjectDetailVO subjectDetailVO = subjectSimpleService.queryInfo(subjectId);
        BeanUtils.copyProperties(subjectDetailVO, subjectInfoVO);
        LambdaQueryWrapper<ExternalSubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(ExternalSubjectInfoSourceMap::getSubjectId, subjectId)
                .eq(ExternalSubjectInfoSourceMap::getType, 1);
        subjectInfoVO.setInfoSourceList(externalSubjectInfoSourceMapService.list(queryWrapper));
        return Result.OK(subjectInfoVO);
    }


    /**
     * 配置信息推送给采集
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/8/12
     */
    @GetMapping("/pushToCaiji")
    public Result<?> pushToCaiji(@RequestParam String subjectId) {
        SubjectInfoVO subjectInfoVO = new SubjectInfoVO();
        SubjectDetailVO subjectDetailVO = subjectSimpleService.queryInfo(subjectId);
        BeanUtils.copyProperties(subjectDetailVO, subjectInfoVO);
        LambdaQueryWrapper<ExternalSubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(ExternalSubjectInfoSourceMap::getSubjectId, subjectId)
                .eq(ExternalSubjectInfoSourceMap::getType, 1);
        subjectInfoVO.setInfoSourceList(externalSubjectInfoSourceMapService.list(queryWrapper));
        try {
            HttpUtil.doPost(keywordCrawlerUrl, JSONObject.from(subjectInfoVO), 30000);
            log.info("【{}】-通知元搜索采集数据", subjectDetailVO.getSubjectName());
        } catch (IOException e) {
            e.printStackTrace();
        }
        return Result.OK();
    }


    /**
     * 汇总摘要
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/8/16
     */
    @GetMapping("/gatherSummary")
    public Result<?> gatherSummary(@RequestParam String subjectId, @RequestParam String language) {
        String key = "GATHER_SUMMARY::" + subjectId;
        if (redisUtil.hasKey(key)) {
            Object data = redisUtil.get(key);
            if (data instanceof Map) {
                Map<String, Object> map = (Map<String, Object>) data;
                List<String> ids = JSON.parseArray(map.get("idList").toString(), String.class);
                InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
                searchCondition.setSubjectId(subjectId);
                searchCondition.setCategory(2);
                searchCondition.setColumn("score");
                searchCondition.setPageSize(20);
                IPage<DisplayInfo> pageList = informationService.subjectPageList(searchCondition);
                List<DisplayInfo> records = pageList.getRecords();
                if (CollectionUtils.isNotEmpty(records)) {
                    List<String> idList = new ArrayList<>();
                    records.forEach(record -> idList.add(record.getId()));
                    if (!CollectionUtils.isEqualCollection(idList, ids)) {
                        Map<String, Object> resultMap = generateSummary(subjectId, records, language);
                        return Result.OK(resultMap);
                    }
                }
            }
            return Result.OK(data);
        } else {
            List<DisplayInfo> dataList = getDataList(subjectId);
            return Result.OK(generateSummary(subjectId,dataList, language));
        }
    }

    /**
     * 获取汇总摘要
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/8/16
     */
    @GetMapping("/getGatherSummary")
    public Result<?> getGatherSummary(@RequestParam String subjectId) {
        return Result.OK(redisUtil.get("GATHER_SUMMARY::" + subjectId));
    }

    /**
     * 翻译
     *
     * @param params 参数
     * @author lkg
     * @date 2024/8/21
     */
    @PostMapping("/translate")
    public Result<?> translate(@RequestBody JSONObject params) {
        String language = params.getString("language");
        String summary = params.getString("summary");
        TranslateVO translateVO = new TranslateVO();
        translateVO.setFrom("auto");
        translateVO.setTo(language);
        List<String> content = new ArrayList<>();
        content.add(summary);
        translateVO.setContent(content);
        List<String> data = new ArrayList<>();
        try {
            String s = HttpUtil.doPost(TRANSLATE_URL, ObjectUtil.objectToJSONObject(translateVO), 10000);
            data = JSONArray.parseArray(s, String.class);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Result.OK(data);
    }


    private List<DisplayInfo> getDataList(String subjectId){
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setSubjectId(subjectId);
        searchCondition.setCategory(2);
        searchCondition.setColumn("score");
        searchCondition.setPageSize(20);
        IPage<DisplayInfo> pageList = informationService.subjectPageList(searchCondition);
        return pageList.getRecords();
    }

    private Map<String, Object> generateSummary(String subjectId,List<DisplayInfo> records, String language) {
        Map<String, Object> resultMap = null;
        if (CollectionUtils.isNotEmpty(records)) {
            JSONObject params = new JSONObject();
            List<String> idList = new ArrayList<>();
            List<JSONObject> summaryList = new ArrayList<>();
            records.forEach(record -> {
                idList.add(record.getId());
                JSONObject summary = new JSONObject();
                summary.put("summary", record.getSummary());
                summaryList.add(summary);
            });
            Subject subject = subjectService.getById(subjectId);
            params.put("subject", subject.getSubjectName());
            params.put("summaryList", summaryList);
            String languageCN = "中文简体";
            if ("zh-CN".equals(language)) {
                languageCN = "中文简体";
            } else if ("en".equals(language)) {
                languageCN = "英文";
            } else if ("ja".equals(language)) {
                languageCN = "日文";
            }
            String prompt = "将用户上传的多篇资讯摘要文本和主题整合为一段连贯、精炼的综合摘要（约500字）。要求：\n" +
                    "1）将输入的json解析，subject是主题，summary是摘要，提炼核心主题与高度共识的关键发现；\n" +
                    "2）客观指出存在的分歧点及其原因；\n" +
                    "3）简述观点或技术的演进趋势；\n" +
                    "4）语言需严谨中性，直接输出整合后的段落，无需分节分段或小标题。\n" +
                    "\n" +
                    "约束：\n" +
                    "1.输出禁止分段\n" +
                    "2.必须输出" + languageCN + "的摘要\n" +
                    "\n" +
                    "使用示例：\n" +
                    "输入：\n" +
                    "{\"subject\":\"机器学习可解释性\",\"summaryList\":[{\"summary\":\"提出梯度解释法...验证准确率提升15%\"},{\"summary\":\"对比LIME与SHAP...指出计算效率缺陷\"},{\"summary\":\"医疗领域应用研究...模型透明度不足影响临床采纳\"}]}\n" +
                    "输出（示例段落）：\n" +
                    "多篇关于机器学习可解释性（XAI）的研究聚焦于提升复杂模型透明度。核心共识在于有效解释技术（如梯度解释法、LIME、SHAP）能增强用户信任并提升决策准确率。然而，在方法实用性上存在分歧，部分研究指出主流技术存在显著计算效率缺陷，尤其在资源受限场景；同时，医疗领域应用突显透明度不足是阻碍临床采纳的关键瓶颈。研究趋势显示从开发基础解释方法（如早期梯度法）转向评估实际场景效能及效率优化。当前主要知识缺口在于缺乏统一标准评估解释的可靠性与临床相关性，亟需开发兼顾高效性和领域适配性的XAI解决方案。";
            String model = llmService.model("qwen", "qwen-max-latest", prompt, JSONObject.toJSONString(params));
            resultMap = new HashMap<>();
            resultMap.put("idList", idList);
            resultMap.put("language", language);
            resultMap.put("model", model);
            redisUtil.set("GATHER_SUMMARY::" + subjectId, JSONObject.toJSONString(resultMap));
        }
        return resultMap;
    }
}
