package com.zzsn.event.service.impl;

import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.LabelEntity;
import com.zzsn.event.service.EsStatisticsService;
import com.zzsn.event.service.LabelEntityService;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.vo.CountVO;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramInterval;
import org.elasticsearch.search.aggregations.bucket.histogram.Histogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CompletableFuture;

/**
 * 舆情信息统计 es查询工具类
 *
 * @author lkg
 * @date 2024/1/19
 */
@Service
public class EsStatisticsServiceImpl implements EsStatisticsService {

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private LabelEntityService labelEntityService;

    @Override
    public Map<String, String> totalAndMax(String subjectId, String startTime, String endTime, Integer type) {
        Map<String, String> map = new HashMap<>();
        long totalCount = 0L;
        long max = 0L;
        String maxTime = null;
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, false);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_hour").field("publishDate");
        if (type == 1) {
            aggregation.calendarInterval(DateHistogramInterval.HOUR)
                    .format("yyyy-MM-dd HH")
                    .order(BucketOrder.count(false));
        } else if (type == 2) {
            aggregation.calendarInterval(DateHistogramInterval.DAY)
                    .format("yyyy-MM-dd")
                    .order(BucketOrder.count(false));
        }
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            totalCount = response.getHits().getTotalHits().value;
            Aggregations aggregations = response.getAggregations();
            ParsedDateHistogram groupHour = aggregations.get("group_hour");
            List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                Histogram.Bucket bucket = buckets.get(0);
                max = bucket.getDocCount();
                maxTime = bucket.getKeyAsString();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        map.put("totalCount", String.valueOf(totalCount));
        map.put("max", String.valueOf(max));
        map.put("maxTime", maxTime);
        return map;
    }

    @Override
    public List<CountVO> orientationCount(String subjectId, String startTime, String endTime) {
        List<CountVO> list = new ArrayList<>();
        String labelTypeId = "1631119596744265729";
        List<LabelEntity> labelEntities = labelEntityService.listByType(labelTypeId);
        for (LabelEntity labelEntity : labelEntities) {
            CompletableFuture<CountVO> async = CompletableFuture.supplyAsync(() -> {
                String labelId = labelEntity.getId();
                String name = labelEntity.getName();
                CountVO countVO = new CountVO();
                countVO.setName(name);
                long count = 0L;
                SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
                SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, labelId, startTime, endTime, false);
                searchSourceBuilder.size(0);
                searchSourceBuilder.trackTotalHits(true);
                searchRequest.source(searchSourceBuilder);
                try {
                    SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                    count = response.getHits().getTotalHits().value;
                } catch (Exception e) {
                    e.printStackTrace();
                }
                countVO.setValue(count);
                return countVO;
            });
            try {
                CountVO countVO = async.get();
                list.add(countVO);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return list;
    }

    @Override
    public CountVO orientation(String subjectId, String labelId, String startTime, String endTime, Integer type) {
        CountVO countVO = new CountVO();
        LabelEntity labelEntity = labelEntityService.getById(labelId);
        countVO.setLabelId(labelEntity.getId());
        countVO.setName(labelEntity.getName());
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, labelId, startTime, endTime, false);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_day")
                .field("publishDate");
        if (type == 1) {
            aggregation.calendarInterval(DateHistogramInterval.HOUR).format("yyyy-MM-dd HH");
        } else if (type == 2) {
            aggregation.calendarInterval(DateHistogramInterval.DAY).format("yyyy-MM-dd");
        }
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = response.getHits().getTotalHits().value;
            countVO.setValue(value);
            List<CountVO> list = new ArrayList<>();
            Aggregations aggregations = response.getAggregations();
            ParsedDateHistogram groupHour = aggregations.get("group_day");
            List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                for (Histogram.Bucket bucket : buckets) {
                    CountVO vo = new CountVO();
                    vo.setName(bucket.getKeyAsString());
                    vo.setValue(bucket.getDocCount());
                    list.add(vo);
                }
            }
            countVO.setChildren(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return countVO;
    }

    @Override
    public List<CountVO> origin(String subjectId, String startTime, String endTime) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, true);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .order(BucketOrder.count(false))
                .size(20);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();
            Terms groupSource = aggregations.get("group_origin");
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                String[] splitArr = new String[]{"-","－"};
                for (Terms.Bucket bucket : buckets) {
                    boolean flag = false;
                    long count = bucket.getDocCount();
                    //获取真正的来源
                    String originName = bucket.getKeyAsString();
                    String split = null;
                    boolean contain = false;
                    for (String s : splitArr) {
                        if (originName.contains(s)) {
                            contain = true;
                            split = s;
                            break;
                        }
                    }
                    if (contain) {
                        int index = originName.indexOf(split);
                        originName = originName.substring(0, index);
                    }
                    //先判断是否存在，不存在则新增
                    for (CountVO countVO : list) {
                        if (countVO.getName().equals(originName)) {
                            flag = true;
                            countVO.setValue(countVO.getValue() + count);
                            break;
                        }
                    }
                    if (!flag) {
                        if (list.size() < 10) {
                            CountVO countVO = new CountVO();
                            countVO.setName(originName);
                            countVO.setValue(count);
                            list.add(countVO);
                        } else {
                            break;
                        }
                    }
                }
                long totalCount = list.stream().mapToLong(CountVO::getValue).sum();
                list.sort((arg0, arg1) -> (int) (arg1.getValue() - arg0.getValue()));
                long percentageSum = 0;
                for (int i = 0; i < list.size(); i++) {
                    CountVO countVO = list.get(i);
                    if (i < list.size() - 1) {
                        String percentage = getPercentage(countVO.getValue(), totalCount);
                        countVO.setPercentage(percentage);
                        percentageSum = percentageSum + Long.parseLong(percentage.replace("%", ""));
                    } else {
                        if (totalCount > 0) {
                            countVO.setPercentage((100 - percentageSum) + "%");
                        } else {
                            countVO.setPercentage("0%");
                        }
                    }
                    countVO.setOrder(i + 1);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    @Override
    public String mainReport(String subjectId) {
        String title = null;
        String mark = getMark(subjectId);
        if (StringUtils.isNotEmpty(mark)) {
            SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
            SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
            //创建查询对象
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
            boolQuery.must(QueryBuilders.termQuery("repeatMark", mark));
            boolQuery.must(QueryBuilders.termQuery("saveType", 1));
            searchSourceBuilder.query(boolQuery);
            searchRequest.source(searchSourceBuilder);
            try {
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHit[] hits = response.getHits().getHits();
                if (hits.length > 0) {
                    SearchHit hit = hits[0];
                    title = hit.getSourceAsMap().get("title").toString();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return title;
    }

    /**
     * 获取专题下重复资讯数最多的蔟
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/4/10
     */
    private String getMark(String subjectId) {
        String mark = null;
        SearchRequest searchRequest = new SearchRequest(Constants.ES_REPEAT_OLD);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("subjectId", subjectId));
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_mark")
                .field("repeatMark")
                .order(BucketOrder.count(false))
                .size(1);
        searchSourceBuilder.query(boolQuery);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();
            Terms groupSource = aggregations.get("group_mark");
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                Terms.Bucket bucket = buckets.get(0);
                mark = bucket.getKeyAsString();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return mark;
    }


    /**
     * 获取占比
     *
     * @param count      数量
     * @param totalCount 总数量
     * @author lkg
     * @date 2024/1/25
     */
    private String getPercentage(long count, long totalCount) {
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(totalCount));
        String percentage = "0%";
        if (StringUtils.isNotEmpty(divide)) {
            percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
        }
        return percentage;
    }

    @Override
    public List<CountVO> flowData(String subjectId, String startTime, String endTime, Integer type) {
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(subjectId, null, startTime, endTime, false);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_day")
                .field("publishDate");
        if (type == 1) {
            aggregation.calendarInterval(DateHistogramInterval.HOUR).format("yyyy-MM-dd HH");
        } else if (type == 2) {
            aggregation.calendarInterval(DateHistogramInterval.DAY).format("yyyy-MM-dd");
        }
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);
        List<CountVO> list = new ArrayList<>();
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();
            ParsedDateHistogram groupHour = aggregations.get("group_day");
            List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                for (Histogram.Bucket bucket : buckets) {
                    CountVO vo = new CountVO();
                    vo.setName(bucket.getKeyAsString());
                    vo.setValue(bucket.getDocCount());
                    list.add(vo);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    @Override
    public long totalCount(List<String> eventIdList, String startTime, String endTime) {
        long count = 0;
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        String eventIds = String.join(",", eventIdList);
        SearchSourceBuilder searchSourceBuilder = formatSourceBuilder(eventIds, null, startTime, endTime, false);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            count = response.getHits().getTotalHits().value;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return count;
    }


    /**
     * 构造查询条件
     *
     * @param subjectId  专题id
     * @param relationId 关联id(可以是标签id，企业信用代码等)
     * @param startTime  开始时间
     * @param endTime    结束时间
     * @param originFlag 是否过滤来源字段(origin)为空的数据
     * @author lkg
     * @date 2024/1/23
     */
    private SearchSourceBuilder formatSourceBuilder(String subjectId, String relationId, String startTime, String endTime, Boolean originFlag) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", Arrays.asList(subjectId.split(","))));
        //标签id
        if (StringUtils.isNotEmpty(relationId)) {
            BoolQueryBuilder relationIdQuery = QueryBuilders.boolQuery();
            relationIdQuery.must(QueryBuilders.nestedQuery("labels",
                    QueryBuilders.termsQuery("labels.relationId", Arrays.asList(relationId.split(","))),
                    ScoreMode.None));
            boolQuery.must(relationIdQuery);
        }
        //情报时间过滤筛选
        if (StringUtils.isNotBlank(startTime)) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(startTime)));
        }
        if (StringUtils.isNotBlank(endTime)) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(endTime)));
        }
        if (originFlag) {
            boolQuery.mustNot(QueryBuilders.termQuery("origin.keyword", ""));
        }
        //未删除状态查询
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        return searchSourceBuilder;
    }
}
