package com.zzsn.event.task;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.service.EsStatisticsService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.util.RedisUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 *
 *
 * @author lkg
 * @date 2024/4/10
 */
@Slf4j
@Component
public class EventHotTask {

    @Autowired
    private IEventService eventService;
    @Autowired
    private EsStatisticsService esStatisticsService;

    @Autowired
    private RedisUtil redisUtil;

    @Value("${scoreRule.weekScore}")
    Integer weekScore;
    @Value("${scoreRule.monthScore}")
    Integer monthScore;
    @Value("${scoreRule.yearScore}")
    Integer yearScore;
    @Value("${scoreRule.beforeYearScore}")
    Integer beforeYearScore;

    /**
     * 专题热度计算
     * 每三个小时执行一次
     *
     * @author lkg
     * @date 2024/4/10
     */
    @Scheduled(cron = "0 0 0/3 * * ?")
    public void hot() {
        List<Event> list = eventService.list();
        for (Event event : list) {
            CompletableFuture.runAsync(() -> {
                String id = event.getId();
                //total
                Integer totalScore = computeScore(id);
                Integer totalHot = getHot(Constants.MAX_TOTAL_HOT_SCORE + "::" + id, totalScore);
                event.setTotalHot(totalHot);
                /*//media
                Integer mediaScore = 0;
                Integer mediaHot = getHot(Constants.MAX_MEDIA_HOT_SCORE + "::" + id, mediaScore);
                //wechat
                Integer wechatScore = 0;
                Integer wechatHot = getHot(Constants.MAX_WECHAT_HOT_SCORE + "::" + id, wechatScore);
                //others
                Integer otherScore = 0;
                Integer otherHot = getHot(Constants.MAX_OTHER_HOT_SCORE + "::" + id, otherScore);
                event.setMediaHot(mediaHot);
                event.setWechatHot(wechatHot);
                event.setOtherHot(otherHot);*/
                eventService.updateById(event);
                log.info("专题-{}-热度计算完成", event.getEventName());
            });
        }
    }


    private Integer getHot(String key, Integer score) {
        Object total = redisUtil.get(key);
        if (null != total) {
            Integer maxTotalScore = (Integer) total;
            if (score > maxTotalScore) {
                redisUtil.set(key, score);
                return 100;
            } else {
                return score / maxTotalScore * 100;
            }
        } else {
            redisUtil.set(key, score);
            return 100;
        }
    }

    /**
     * 计算热度
     *
     * @param eventId 事件id
     */
    private int computeScore(String eventId) {
        Date date = new Date();
        DateTime beforeWeekDay = DateUtil.offsetWeek(date, -1);
        DateTime beforeMonthDay = DateUtil.offsetMonth(date, -1);
        DateTime beforeYearDay = DateUtil.offset(date, DateField.YEAR, -1);
        List<String> ids = new ArrayList<>();
        ids.add(eventId);
        // 一周之内
        long weekCount = esStatisticsService.totalCount(ids, DateUtil.formatDateTime(beforeWeekDay), DateUtil.formatDateTime(date));
        // 一周到一月之内
        long monthCount = esStatisticsService.totalCount(ids, DateUtil.formatDateTime(beforeMonthDay), DateUtil.formatDateTime(beforeWeekDay));
        // 一月到一年之内
        long yearInCount = esStatisticsService.totalCount(ids, DateUtil.formatDateTime(beforeYearDay), DateUtil.formatDateTime(beforeMonthDay));
        // 一年之外
        long yearOutCount = esStatisticsService.totalCount(ids, null, DateUtil.formatDateTime(beforeYearDay));
        return (int) (weekCount * weekScore + monthCount * monthScore + yearInCount * yearScore + yearOutCount * beforeYearScore);
    }
}
