package com.zzsn.event.controller.common;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.InfoSourceGroup;
import com.zzsn.event.service.*;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.vo.*;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * 专题/事件关系绑定
 *
 * @author lkg
 * @date 2024/12/17
 */
@RestController
@RequestMapping("/common/relation")
public class RelationBindController {

    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private InfoSourceGroupService infoSourceGroupService;

    /**
     * 项目列表
     *
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/projectList")
    public Result<?> projectList() {
        List<Node> projectList = commonService.projectList();
        return Result.OK(projectList);
    }

    /**
     * 专题/事件绑定的信息源-列表页
     *
     * @param infoSourceCondition 筛选条件
     * @param subjectId           id/分类id
     * @param category            类别(1-专题;2-事件)
     * @param pageNo              当前页
     * @param pageSize            返回条数
     * @author lkg
     * @date 2024/5/7
     */
    @GetMapping(value = "/bindInfoSourceList")
    public Result<?> bindInfoSourceList(InfoSourceCondition infoSourceCondition,
                                        @RequestParam(name = "subjectId", defaultValue = "0") String subjectId,
                                        @RequestParam(name = "category") Integer category,
                                        @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                        @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<String> subjectIdList = new ArrayList<>();
        if (!"0".equals(subjectId)) {
            //查询类别id的所有明细id
            if (StringUtils.isNotEmpty(subjectId)) {
                List<String> typeIds = subjectTypeService.belowIdList(subjectId, category);
                if (category == 1) {
                    subjectIdList = subjectTypeMapService.selectSubjectByTypeIds(typeIds);
                } else if (category == 2) {
                    subjectIdList = subjectTypeMapService.selectEventByTypeIds(typeIds);
                }
            }
            subjectIdList.add(subjectId);
        }
        IPage<InfoSourceVo> pageList = infoSourceService.bindInfoSourceList(infoSourceCondition, subjectIdList, pageNo, pageSize);
        return Result.OK(pageList);
    }


    /**
     * 信息源组搜索列表(专题/事件配置页)
     *
     * @param searchWord 搜索词
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/group/searchList")
    public Result<?> searchList(@RequestParam String searchWord){
        LambdaQueryWrapper<InfoSourceGroup> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.like(InfoSourceGroup::getGroupName,searchWord);
        List<InfoSourceGroup> list = infoSourceGroupService.list(queryWrapper);
        return Result.OK(list);
    }

    /**
     * 信息源组和分类组成的树型结构(专题/事件配置页)
     *
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/groupAndTypeTree")
    public Result<?> groupAndTypeTree(){
        List<GroupTreeVO> tree = infoSourceGroupService.GroupAndTypeTree();
        return Result.OK(tree);
    }

    /**
     * 专题/事件信息源绑定(专题/事件配置页)
     *
     * @param subjectPage 参数
     * @author lkg
     * @date 2024/12/18
     */
    @PostMapping("/infoSourceBind")
    public Result<?> infoSourceBind(@RequestBody SubjectPage subjectPage) {
        subjectInfoSourceMapService.infoSourceBind(subjectPage);
        return Result.OK();
    }

    /**
     * 专题/事件信息源绑定删除(专题/事件配置页)
     *
     * @param subjectPage 参数
     * @return
     */
    @PostMapping(value = "/deleteBindInfoSource")
    public Result<?> deleteBindInfoSource(@RequestBody SubjectPage subjectPage) {
        subjectInfoSourceMapService.deleteBind(subjectPage);
        return Result.OK("删除成功!");
    }

    /**
     * 专题/事件绑定的信息源组列表(专题/事件配置页)
     *
     * @param subjectId 专题/事件id
     * @author lkg
     * @date 2024/12/18
     */
    @GetMapping(value = "/bindInfoSourceGroup")
    public Result<?> bindInfoSource(@RequestParam(name = "subjectId") String subjectId) {
        JSONObject infoSourceGroupPages = subjectInfoSourceMapService.bindInfoSource(subjectId);
        return Result.OK(infoSourceGroupPages);
    }


    /**
     * 编辑专题/事件和信息源组的绑定关系
     *
     * @param infoSourceGroupPage 参数
     * @return
     */
    @PostMapping(value = "/modifyInfoSourceInfo")
    public Result<?> modifyInfoSourceInfo(@RequestBody InfoSourceGroupPage infoSourceGroupPage) {
        if (StrUtil.isBlank(infoSourceGroupPage.getSubjectId()) || StrUtil.isBlank(infoSourceGroupPage.getId())
                || ObjectUtil.isEmpty(infoSourceGroupPage.getSourceType())){
            return Result.OK("请检查参数");
        }
        subjectInfoSourceMapService.modifyInfoSourceInfo(infoSourceGroupPage);
        return Result.OK();
    }

    /**
     * 专题/事件绑定的关键词组分页列表-列表页
     *
     * @param id        专题/事件id/分类id
     * @param category  类别(1-专题;2-事件)
     * @param groupName 词组名称
     * @param wordName  关键词
     * @param pageNo    当前页
     * @param pageSize  每页返回条数
     * @author lkg
     * @date 2024/12/17
     */
    @GetMapping(value = "/bindKeyWordsList")
    public Result<?> bindKeyWordsList(@RequestParam String id,
                                      @RequestParam Integer category,
                                      @RequestParam(required = false) String groupName,
                                      @RequestParam(required = false) String wordName,
                                      @RequestParam(defaultValue = "1") Integer pageNo,
                                      @RequestParam(defaultValue = "10") Integer pageSize) {
        List<String> subjectIdList = new ArrayList<>();
        if (!"0".equals(id)) {
            List<String> typeIds = subjectTypeService.belowIdList(id, category);
            if (category == 1) {
                subjectIdList = subjectTypeMapService.selectSubjectByTypeIds(typeIds);
            } else if (category == 2) {
                subjectIdList = subjectTypeMapService.selectEventByTypeIds(typeIds);
            }
        }
        subjectIdList.add(id);
        IPage<KeyWordsPage> page = keyWordsService.bindKeyWordsList(subjectIdList, groupName, wordName, pageNo, pageSize);
        return Result.OK(page);
    }


}
