package com.zzsn.event.controller;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.entity.SubjectKeywordsGroupRelation;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CronUtil;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.entity.XxlJobInfo;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.CompletableFuture;

/**
 * 平台事件管理页
 *
 * @author lkg
 * @date 2024/5/7
 */
@Slf4j
@RestController
@RequestMapping("/subject/manage")
public class SubjectManageController {

    @Autowired
    private SubjectService subjectService;
    @Autowired
    private ISubjectKeywordsMapService subjectKeywordsMapService;
    @Autowired
    private IXxlJobInfoService xxlJobInfoService;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private ISubjectModelMapService subjectModelMapService;
    @Autowired
    private EsService esService;
    @Autowired
    SubjectKeywordsGroupRelationService subjectKeywordsGroupRelationService;


    @Value("${hotWords.extractUrl}")
    private String extractHotWordsUrl;

    /**
     * 专题列表-资讯转换时使用
     *
     * @author lkg
     * @date 2024/12/24
     */
    @GetMapping("/listAll")
    public Result<?> subjectList() {
        List<Subject> list = subjectService.list();
        return Result.OK(list);
    }


    /**
     * 分页列表查询
     *
     * @param subjectCondition
     * @param pageNo
     * @param pageSize
     * @return
     */
    @GetMapping(value = "/listByTypeId")
    public Result<?> queryPageListByTypeId(SubjectCondition subjectCondition,
                                           @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                           @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<SubjectPage> pageList = subjectService.pageList(subjectCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    @PostMapping("/recommendWordList")
    public Result<?> recommendWordList(MultipartFile[] files,String words){
        //// TODO: 2024/12/24
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("article","12");
        jsonObject.put("word","12");
        return Result.OK(jsonObject);
    }

    /**
     * 添加
     *
     * @param subjectPage
     * @return
     */
    @PostMapping(value = "/add")
    public Result<?> add(@RequestBody SubjectPage subjectPage) {
        Subject subject = subjectService.saveMain(subjectPage);
        //插入xxlJob
        xxlJobInfoService.subjectInsert(subject);
        return Result.OK("添加成功！");
    }

    /**
     * 编辑
     *
     * @param subjectPage
     * @return
     */
    @PostMapping(value = "/edit")
    public Result<?> edit(@RequestBody SubjectPage subjectPage) {
        Subject byId = subjectService.getById(subjectPage.getId());
        subjectService.updateMain(subjectPage);
        CompletableFuture.runAsync(() -> {
            //更新xxljob
            List<String> subjectCodes = new ArrayList<>();
            subjectCodes.add(subjectPage.getSubjectCode());
            String cron = "";
            if (StrUtil.isNotBlank(subjectPage.getUnit()) && ObjectUtil.isNotEmpty(subjectPage.getSpace())) {
                cron = CronUtil.generateCron(subjectPage.getUnit(), subjectPage.getSpace());
            }
            xxlJobInfoService.keyWordsUpdate(subjectCodes, String.valueOf(subjectPage.getStatus()));
            xxlJobInfoService.update(Wrappers.<XxlJobInfo>lambdaUpdate().eq(XxlJobInfo::getInfoSourceCode, subjectPage.getSubjectCode()).set(XxlJobInfo::getJobCron, cron));
            //判断是否提取热词
            extractHotWords(subjectPage);
            if (!Objects.equals(byId.getStatus(), subjectPage.getStatus()) && subjectPage.getStatus() == 1) {
                kafkaTemplate.send("subjectModel", subjectPage.getSubjectCode());
            }
        });
        return Result.OK("编辑成功!");
    }

    /**
     * 通过id删除
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/delete")
    public Result<?> delete(@RequestParam(name = "id") String id) {
        subjectService.deleteMain(id);
        CompletableFuture.runAsync(() -> {
            //删除xxljob里面的信息
            LambdaQueryWrapper<Subject> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(Subject::getId, id);
            Subject subject = subjectService.getOne(queryWrapper);
            xxlJobInfoService.deleteByInfosourceCode(subject.getSubjectCode());
        });
        return Result.OK("删除成功!");
    }

    /**
     * 批量删除
     *
     * @param ids
     * @return
     */
    @GetMapping(value = "/deleteBatch")
    public Result<?> deleteBatch(@RequestParam(name = "ids") String ids) {
        String[] idArr = ids.split(",");
        for (String id : idArr) {
            subjectService.deleteMain(id);
        }
        CompletableFuture.runAsync(() -> {
            for (String id : idArr) {
                //删除xxljob里面的信息
                LambdaQueryWrapper<Subject> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.eq(Subject::getId, id);
                Subject subject = subjectService.getOne(queryWrapper);
                xxlJobInfoService.deleteByInfosourceCode(subject.getSubjectCode());
            }
        });
        return Result.OK("批量删除成功!");
    }

    /**
     * 通过id查询
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/queryById")
    public Result<?> queryById(@RequestParam(name = "id") String id) {
        SubjectPage subject = subjectService.getSubjectById(id);
        List<SubjectPage> pageList = new ArrayList<>();
        pageList.add(subject);
        IPage<SubjectPage> pageData = new Page<>(1, 10, 1);
        pageData.setRecords(pageList);
        return Result.OK(pageData);
    }


    /**
     * 专题直接绑定信息源-批量(专题列表页)
     *
     * @param directBindInfoVO 参数封装
     * @author lkg
     * @date 2024/4/24
     */
    @PostMapping("/directBindInfoSource")
    public Result<?> directBindInfoSource(@RequestBody DirectBindInfoVO directBindInfoVO) {
        subjectService.directBindInfoSource(directBindInfoVO);
        return Result.OK();
    }

    /**
     * 专题直接解绑信息源-批量(专题列表页)
     *
     * @param directBindInfoVO 参数封装
     * @author lkg
     * @date 2024/4/24
     */
    @PostMapping("/directRemoveInfoSource")
    public Result<?> directRemoveInfoSource(@RequestBody DirectBindInfoVO directBindInfoVO) {
        subjectService.directRemoveInfoSource(directBindInfoVO);
        return Result.OK();
    }

    /**
     * 专题直接屏蔽的信息源列表(不包括专题屏蔽的信息源组下的信息源)(专题列表页)
     *
     * @param infoSourceCondition 筛选条件
     * @param subjectId           专题id
     * @param pageNo              当前页
     * @param pageSize            返回条数
     * @author lkg
     * @date 2024/4/30
     */
    @GetMapping("/unBindInfoSourceList")
    public Result<?> unBindInfoSourceList(InfoSourceCondition infoSourceCondition,
                                          @RequestParam(name = "subjectId", defaultValue = "0") String subjectId,
                                          @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                          @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<InfoSourceVo> pageList = infoSourceService.unBindSourcePageList(infoSourceCondition, subjectId, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 删除专题下已屏蔽的信息源(即恢复绑定关系)(专题列表页)
     *
     * @param subjectId 专题id
     * @param sourceId  信息源id
     * @author lkg
     * @date 2024/4/30
     */
    @GetMapping("/removeUnBindInfoSource")
    public Result<?> removeUnBindInfoSource(@RequestParam String subjectId, @RequestParam String sourceId) {
        subjectService.removeUnBindInfoSource(subjectId, sourceId);
        return Result.OK();
    }

    /**
     * 专题绑定关键词组列表-专题配置页
     *
     * @param subjectId 专题id
     * @return
     */
    @GetMapping(value = "/bindKeyWordsList")
    public Result<?> bindKeyWordsList(@RequestParam String subjectId) {
        subjectKeywordsMapService.bindKeyWordsList(subjectId);
        return Result.OK();
    }

    /**
     * 专题绑定关键词-专题配置页
     *
     * @param subjectPage 参数
     * @return
     */
    @PostMapping(value = "/keyWordsBind")
    public Result<?> keyWordsBind(@RequestBody SubjectPage subjectPage) {
        subjectService.keyWordsBind(subjectPage);
        return Result.OK();
    }

    /**
     * 专题绑定关键词的作用范围-专题配置页
     *
     * @param subjectKeywordsMap 参数
     * @return
     */
    @PostMapping(value = "/keyWordsBindScope")
    public Result<?> keyWordsBindScope(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
        subjectKeywordsMapService.updateById(subjectKeywordsMap);
        return Result.OK();
    }

    /**
     * 新增/编辑词组关系
     *
     * @param subjectKeywordsGroupRelation 词组关系
     * @author lkg
     * @date 2024/12/24
     */
    @PostMapping("/keyword/relation/save")
    public Result<?> add(@RequestBody SubjectKeywordsGroupRelation subjectKeywordsGroupRelation) {
        return subjectKeywordsGroupRelationService.add(subjectKeywordsGroupRelation);
    }

    /**
     * 获取不通类型的词组之间的关系数据
     *
     * @param subjectId    专题id
     * @param relationType 1：采集词词组关系，2：过滤词词组关系，3排除词组词组关系
     * @author lkg
     * @date 2024/12/24
     */
    @GetMapping("/keyword/relation/getBySubjectIdAndType")
    public Result<?> getBySubjectIdAndType(String subjectId, String relationType) {
        return Result.OK(subjectKeywordsGroupRelationService.getOne(Wrappers.<SubjectKeywordsGroupRelation>lambdaQuery()
                .eq(SubjectKeywordsGroupRelation::getSubjectId, subjectId)
                .eq(SubjectKeywordsGroupRelation::getRelationType, relationType)
        ));
    }

    /**
     * 获取专题下所有的词组关系
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/24
     */
    @GetMapping("/keyword/relation/getBySubjectId")
    public Result<?> getBySubjectId(String subjectId) {
        return Result.OK(subjectKeywordsGroupRelationService.list(Wrappers.<SubjectKeywordsGroupRelation>lambdaQuery()
                .eq(SubjectKeywordsGroupRelation::getSubjectId, subjectId)
        ));
    }

    /**
     * 删除专题和关键词的绑定关系-专题配置页
     *
     * @param subjectPage
     * @return
     */
    @PostMapping(value = "/deleteKeyWordsBind")
    public Result<?> deleteKeyWordsBind(@RequestBody SubjectPage subjectPage) {
        subjectService.deleteBindKeyWords(subjectPage);
        return Result.OK();
    }

    /**
     * 专题搜索引擎绑定-专题配置页
     *
     * @param subjectPage
     * @return
     */
    @PostMapping(value = "/searchEngineBind")
    public Result<?> searchEngineBind(@RequestBody SubjectPage subjectPage) {
        subjectService.searchEngineBind(subjectPage);
        return Result.OK("编辑成功!");
    }

    /**
     * 专题绑定的搜索引擎列表-专题配置页
     *
     * @return
     */
    @GetMapping(value = "/bindSearchEngineList")
    public Result<?> bindSearchEngineList(SearchEnginesVo searchEnginesVo) {
        List<SearchEnginesVo> searchEngineList = subjectService.bindSearchEngineList(searchEnginesVo);
        return Result.OK(searchEngineList);
    }


    /**
     * 专题配置页左侧列表-专题配置页
     *
     * @param subjectType
     * @return
     */
    @GetMapping(value = "/modelList")
    public Result<?> modelList(@RequestParam(name = "subjectType", defaultValue = "0") String subjectType) {
        List<ModelVO> allList = commonService.allList(subjectType);
        return Result.OK(allList);
    }

    /**
     * 模型绑定（包含去重、筛选、标签模型）-专题配置页
     *
     * @param subjectPage 参数
     * @return
     */
    @PostMapping(value = "/modelBind")
    public Result<?> modelBind(@RequestBody SubjectPage subjectPage) {
        return subjectService.modelBind(subjectPage);
    }

    /**
     * 专题绑定的模型信息列表-专题配置页
     *
     * @param subjectId 专题id
     * @param type      模型分类
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/bindModelList")
    public Result<?> bindModelList(@RequestParam String subjectId, @RequestParam(required = false) String type) {
        List<PythonModelVo> pythonModelVoList = subjectModelMapService.bindModelByType(subjectId, type);
        return Result.OK(pythonModelVoList);
    }

    /**
     * 数据资源绑定-专题配置页
     *
     * @param subjectPage 参数
     * @author lkg
     * @date 2024/12/18
     */
    @PostMapping(value = "/sourceBind")
    public Result<?> sourceBindOrUpdate(@RequestBody SubjectPage subjectPage) {
        Subject subject = new Subject();
        BeanUtils.copyProperties(subjectPage, subject);
        subjectService.updateById(subject);
        return Result.OK();
    }

    /**
     * 模型绑定（包含去重、筛选、标签模型）--检查是否需要中断-专题配置页
     * todo
     *
     * @param subjectPage 参数
     * @return
     */
    @PostMapping(value = "/checkIsBreak")
    public Result<?> checkIsBreak(@RequestBody SubjectPage subjectPage) {
        return subjectService.checkIsBreak(subjectPage);
    }

    /**
     * 前端发送请求，传参数给python
     * todo
     */
    @PostMapping(value = "/sendPythonInfo")
    public Result<?> sendPythonInfo(@RequestBody SubjectPage subjectPage) {
        subjectService.send(subjectPage.getId(), null);
        return Result.OK("编辑成功!");
    }


    /**
     * 热点地图-专题分析页
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/subjectRegionStatistic")
    public Result<?> subjectRegionStatistic(@RequestParam String subjectId) {
        List<CountVO> countVoList = esService.getRegionStatistic(subjectId);
        return Result.OK(countVoList);
    }

    /**
     * 来源占比-专题分析页
     *
     * @param subjectId 专题id
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @param size      返回个数
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/originRatio")
    public Result<?> originRatio(@RequestParam(value = "size", defaultValue = "10") int size,
                                 @RequestParam(value = "startDate", required = false) String startDate,
                                 @RequestParam(value = "endDate", required = false) String endDate,
                                 @RequestParam(value = "subjectId", required = false) String subjectId) {
        List<CountVO> countVOS = esService.groupByOrigin(subjectId, "publishDate",startDate, endDate, size);
        return Result.OK(countVOS);
    }

    /**
     * @param startDate 统计开始日期
     * @param endDate   统计结束日期
     * @param subjectId 专题id
     * @描述 获取专题统计信息
     * @作者 hejinyu
     * @创建时间 2024/12/13
     **/
    @GetMapping("/listSubjectCountInfo")
    public Result<?> listSubjectCountInfo(@RequestParam String startDate, @RequestParam String endDate, @RequestParam String subjectId) {
        //获取统计天数
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        LocalDate localStartDate = LocalDate.parse(startDate, formatter);
        //查询有查结束日期当日的数据，所以获取日期数组时得多算一天
        LocalDate localEndDate = LocalDate.parse(endDate, formatter).plusDays(1);
        long days = ChronoUnit.DAYS.between(localStartDate, localEndDate);
        String type = "day";
        //参数日期间隔只有一天时，做日报表
        if (days == 1) {
            type = "hour";
        }
        if (days > 180) {
            type = "month";
        }
        Map<String, List<CountVO>> resultMap = new HashMap<>();
        //已采集量
        List<CountVO> collectedData = esService.getCollectedData(startDate, endDate, type, subjectId);
        resultMap.put("collectedData", collectedData);
        //已入库量
        List<CountVO> stockdata = esService.getSubjectDatabaseByTime(startDate, endDate, type, subjectId, null, null);
        resultMap.put("stockdata", stockdata);
        //免审核入库的数据
        List<CountVO> freeCheckdata = esService.getSubjectDatabaseByTime(startDate, endDate, type, subjectId, 1, null);
        resultMap.put("freeCheckdata", freeCheckdata);
        //审核通过的数据
        List<CountVO> checkeddata = esService.getSubjectDatabaseByTime(startDate, endDate, type, subjectId, null, 1);
        resultMap.put("checkeddata", checkeddata);
        return Result.OK(resultMap);
    }

    private void extractHotWords(SubjectPage subjectPage) {
        if (StringUtils.isNotEmpty(subjectPage.getYnExtractHotWords()) && "1".equals(subjectPage.getYnExtractHotWords())) {
            Map<String, String> param = new HashMap<>();
            param.put("status", "0,1,2,3");
            param.put("timeFiled", "ALL");
            param.put("subjectId", subjectPage.getId());
            HttpUtil.doGet(extractHotWordsUrl, param, "utf-8");
        }
    }
}
