package com.zzsn.event.task;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.entity.Event;
import com.zzsn.event.service.EsStatisticsService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.vo.HotVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 *
 *
 * @author lkg
 * @date 2024/4/10
 */
@Slf4j
@Component
public class EventHotTask {

    @Autowired
    private IEventService eventService;
    @Autowired
    private EsStatisticsService esStatisticsService;

    @Value("${scoreRule.weekScore}")
    Integer weekScore;
    @Value("${scoreRule.monthScore}")
    Integer monthScore;
    @Value("${scoreRule.yearScore}")
    Integer yearScore;
    @Value("${scoreRule.beforeYearScore}")
    Integer beforeYearScore;

    /**
     * 专题热度计算
     * 每三个小时执行一次
     *
     * @author lkg
     * @date 2024/4/10
     */
    @Scheduled(cron = "0 0 0/3 * * ?")
    public void hot() {
        LambdaQueryWrapper<Event> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(Event::getFacePublic,1).eq(Event::getPublishStatus,1);
        List<Event> list = eventService.list(queryWrapper);
        List<HotVO> countList = new ArrayList<>();
        for (Event event : list) {
            CompletableFuture<HotVO> async = CompletableFuture.supplyAsync(() -> {
                HotVO hotVO = new HotVO();
                String id = event.getId();
                int count = computeScore(id);
                hotVO.setId(id);
                hotVO.setCount(count);
                return hotVO;
            });
            try {
                HotVO hotVO = async.get();
                countList.add(hotVO);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        countList.sort(Comparator.comparingInt(HotVO::getCount));
        Integer min = countList.get(0).getCount();
        Integer max = countList.get(countList.size() - 1).getCount();
        Map<String, List<HotVO>> hotMap = countList.stream().collect(Collectors.groupingBy(HotVO::getId));
        //最小-最大归一化(Min-Max Normalization)
        for (Event event : list) {
            List<HotVO> hotVOS = hotMap.get(event.getId());
            if (CollectionUtils.isNotEmpty(hotVOS)) {
                HotVO hotVO = hotVOS.get(0);
                Integer hot = normalize(hotVO.getCount(), min, max);
                event.setTotalHot(hot);
                eventService.updateById(event);
                log.info("专题-{}-热度计算完成", event.getEventName());
            }
        }
    }


    //归一化计算
    private Integer normalize(Integer value, Integer min,Integer max) {
        int normalizeValue = 0;
       if (value > 0) {
           int param1 = (value - min) * 100;
           int param2 = (max - min);
           String divide = CalculateUtil.divide(String.valueOf(param1), String.valueOf(param2), 0);
           if (StringUtils.isNotEmpty(divide)) {
               normalizeValue = Integer.parseInt(divide);
           }
       }
       return normalizeValue;
    }

    /**
     * 计算热度
     *
     * @param eventId 事件id
     */
    private int computeScore(String eventId) {
        Date date = new Date();
        DateTime beforeWeekDay = DateUtil.offsetWeek(date, -1);
        DateTime beforeMonthDay = DateUtil.offsetMonth(date, -1);
        DateTime beforeYearDay = DateUtil.offset(date, DateField.YEAR, -1);
        List<String> ids = new ArrayList<>();
        ids.add(eventId);
        // 一周之内
        long weekCount = esStatisticsService.totalCount(ids, DateUtil.formatDateTime(beforeWeekDay), DateUtil.formatDateTime(date));
        // 一周到一月之内
        long monthCount = esStatisticsService.totalCount(ids, DateUtil.formatDateTime(beforeMonthDay), DateUtil.formatDateTime(beforeWeekDay));
        // 一月到一年之内
        long yearInCount = esStatisticsService.totalCount(ids, DateUtil.formatDateTime(beforeYearDay), DateUtil.formatDateTime(beforeMonthDay));
        // 一年之外
        long yearOutCount = esStatisticsService.totalCount(ids, null, DateUtil.formatDateTime(beforeYearDay));
        return (int) (weekCount * weekScore + monthCount * monthScore + yearInCount * yearScore + yearOutCount * beforeYearScore);
    }
}
