package com.zzsn.event.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Event;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.EventSimpleService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * 事件
 *
 * @Author: jeecg-boot
 * @Date: 2024-03-14
 * @Version: V1.0
 */
@Slf4j
@Api(tags = "事件")
@RestController
@RequestMapping("/data")
public class EventDataController {
    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;
    @Autowired
    private EventSimpleService eventSimpleService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;


    /**
     * 事件分页列表-门户
     *
     * @param category   栏目类型(1-全部;2-自动追踪;3-人工创建;4-我创建的)
     * @param eventId    事件id
     * @param eventName  事件名称
     * @param eventType  事件分类id
     * @param labelField 搜索字段
     * @param labelName  搜索词
     * @param order      排序字段
     * @param orderType  排序方式
     * @param pageNo     偏移量
     * @param pageSize   返回条数
     * @author lkg
     * @date 2024/4/8
     */
    @ApiOperation(value = "事件-分页列表查询", notes = "事件-分页列表查询")
    @GetMapping(value = "/pageList")
    public Result<?> pageList(@RequestParam(name = "category") Integer category,
                              @RequestParam(name = "eventId", required = false) String eventId,
                              @RequestParam(name = "eventName", required = false) String eventName,
                              @RequestParam(name = "eventType", required = false) String eventType,
                              @RequestParam(name = "labelField", required = false) String labelField,
                              @RequestParam(name = "labelName", required = false) String labelName,
                              @RequestParam(name = "order", required = false) String order,
                              @RequestParam(name = "orderType", required = false) String orderType,
                              @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        String username = null;
        if (category == 4) {
            UserVo currentUser = UserUtil.getLoginUser();
            username = currentUser.getUsername();
        }
        IPage<EventFrontVO> pageList = eventService.frontPageList(category, username, eventId, eventName, eventType, labelField, labelName, order, orderType, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 新增事件(页面简化版)
     *
     * @param addEventVO 事件信息
     * @author lkg
     * @date 2024/7/22
     */
    @PostMapping("/simpleSaveEvent")
    public Result<?> addEvent(@RequestBody AddEventVO addEventVO) {
        Event event = eventSimpleService.simpleSave(addEventVO);
        iXxlJobInfoService.subjectInsert(event);
        return Result.OK();
    }


    /**
     * 事件资讯分页列表
     *
     * @param subjectId  事件id
     * @param searchWord 搜索词
     * @param position   搜索位置(title-标题;content-内容)
     * @param category   匹配度(1-模糊;2-精确)
     * @param column     排序字段
     * @param order      排序方式(asc-正序;desc-倒序)
     * @param pageNo     当前页
     * @param pageSize   返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @ApiOperation(value = "专题信息列表-分页列表查询", notes = "专题信息列表-分页列表查询")
    @GetMapping(value = "/listArticle")
    public Result<?> queryPageList(@RequestParam(name = "subjectId", required = false) String subjectId,
                                   @RequestParam(name = "searchWord", required = false) String searchWord,
                                   @RequestParam(name = "position", required = false) String position,
                                   @RequestParam(name = "category", required = false) Integer category,
                                   @RequestParam(name = "column", defaultValue = "publishDate") String column,
                                   @RequestParam(name = "order", defaultValue = "desc") String order,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<String> subjectIdList = new ArrayList<>();
        if (StringUtils.isNotEmpty(subjectId)) {
            subjectIdList.add(subjectId);
        } else {
            List<EventExcelVO> frontList = eventService.frontList(null, null);
            frontList.forEach(e -> subjectIdList.add(e.getId()));
        }
        //获取数据
        IPage<SubjectDataVo> pageList = esService.frontListByPage(subjectIdList, searchWord, position, category, column, order, pageNo, pageSize);
        return Result.OK(pageList);
    }


    /**
     * 资讯详情
     *
     * @return
     */
    @GetMapping(value = "/articleDetail")
    public Result<?> articleDetail(@RequestParam(required = false) String index, @RequestParam String id) {
        if (StringUtils.isEmpty(index)) {
            index = Constants.SUBJECT_INDEX;
        }
        SubjectDataVo subjectDataVo = esService.queryInfo(index, id);
        return Result.OK(subjectDataVo);
    }

    /**
     * 单篇文章热词
     *
     * @param index  索引名称
     * @param id     资讯id
     * @param number 热词数量
     * @author lkg
     * @date 2024/4/10
     */
    @ApiOperation(value = "单篇文章热词", notes = "单篇文章热词")
    @GetMapping(value = "/hotWords")
    public Result<?> articleList(@RequestParam(value = "index", required = false) String index, @RequestParam("id") String id,
                                 @RequestParam(name = "number", defaultValue = "200") Integer number) {
        if (StringUtils.isEmpty(index)) {
            index = Constants.SUBJECT_INDEX;
        }
        List<StatisticsKeyWordVo> words = eventService.hotWords(index, id, number);
        return Result.OK(words);
    }

    /**
     * 相关推荐
     *
     * @param subjectId 专题id
     * @param id        资讯id
     * @param title     标题
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping(value = "/recommendList")
    public Result<?> recommendList(@RequestParam(name = "subjectId") String subjectId,
                                   @RequestParam(name = "id") String id,
                                   @RequestParam(name = "title") String title,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<SubjectDataVo> recommendList = esService.queryRecommendList(subjectId, id, title, pageNo, pageSize);
        return Result.OK(recommendList);
    }
}
