package com.zzsn.event.service.impl;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.obs.services.model.PutObjectResult;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.*;
import com.zzsn.event.enums.AnalysisColumnEnum;
import com.zzsn.event.es.EsService;
import com.zzsn.event.llm.LlmService;
import com.zzsn.event.service.*;
import com.zzsn.event.util.*;
import com.zzsn.event.util.ppt.ApiAuthAlgorithm;
import com.zzsn.event.util.ppt.CreateResponse;
import com.zzsn.event.util.ppt.PPTUtil;
import com.zzsn.event.util.ppt.ProgressResponse;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.SpecialInformation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lkg
 * @description:
 * @date 2022/7/20 11:32
 */
@Slf4j
@Service
public class AnalysisServiceImpl implements AnalysisService {

    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private LlmService llmService;
    @Autowired
    private EventAnalysisVersionService eventAnalysisVersionService;
    @Autowired
    private EventAnalysisVersionRecordService eventAnalysisVersionRecordService;
    @Autowired
    private SysDictItemService dictItemService;
    @Autowired
    private ObsUtil obsUtil;
    @Autowired
    private EventLlmConfigService eventLlmConfigService;
    @Autowired
    private PythonUtil pythonUtil;

    /*
     * 优先级：事件脉络 > 伪事件脉络 > 资讯
     * 1.若事件脉络资讯数量少于展示伪事件脉络的阈值(6)，若有伪事件脉络就展示，若无则根据发布时间倒序后，默认取前15条资讯为事件脉络。
     * 2.若事件脉络资讯数量少于展示伪事件脉络的阈值(6)但不为空；
     * 2.1 若有伪事件脉络就展示，若无则展示事件脉络；
     * 2.2 若有伪事件脉络就展示，若无则根据发布时间倒序后，默认取前15条资讯为事件脉络。
     * 3.若事件脉络资讯数量大于/等于展示伪事件脉络的阈值(6)，则直接展示
     *
     * 伪事件脉络/资讯作为事件脉络的情况一般是资讯数量少/刚建立/时间脉络抽取失败的事件专题，才会触发。
     */
    @Override
    public List<SubjectAnalysis> eventContext(String eventId, int fakeNum) {
        //专题下的事件脉络
        List<SubjectAnalysis> list = getList(eventId, 2);
        if (list.size() < fakeNum) {
            //专题下的伪事件脉络
            List<SubjectAnalysis> fakeList = getList(eventId, 3);
            if (CollectionUtils.isEmpty(fakeList)) {
                if (CollectionUtils.isEmpty(list)) {
                    List<SubjectAnalysis> finalList = new ArrayList<>();
                    List<SubjectDataVo> dataList = esService.pageList(eventId, null, null, Constants.FETCH_FIELDS_STATISTIC, 2, 1, 15);
                    dataList.forEach(e -> {
                        String dataId = e.getId();
                        SubjectAnalysis subjectAnalysis = new SubjectAnalysis();
                        BeanUtils.copyProperties(e, subjectAnalysis);
                        subjectAnalysis.setId(null);
                        subjectAnalysis.setDataId(dataId);
                        subjectAnalysis.setPublishDate(DateUtil.stringToDate(e.getPublishDate(), "yyyy-MM-dd HH:mm:ss"));
                        finalList.add(subjectAnalysis);
                    });
                    list = finalList;
                }
            } else {
                list = fakeList;
            }
        }
        return list;
    }

    @Override
    public List<JSONObject> eventContext(String eventId) {
        List<JSONObject> dataList = new ArrayList<>();
        LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectAnalysis::getSubjectId, eventId).eq(SubjectAnalysis::getCategory, 4);
        List<SubjectAnalysis> list = subjectAnalysisService.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            Map<String, List<SubjectAnalysis>> map = list.stream().collect(Collectors.groupingBy(e -> DateUtil.format(e.getPublishDate(), "yyyy-MM-dd")));
            for (Map.Entry<String, List<SubjectAnalysis>> entry : map.entrySet()) {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("date", entry.getKey());
                jsonObject.put("dataList", entry.getValue());
                dataList.add(jsonObject);
            }
        }
        return dataList;
    }

    @Override
    public List<CountVO> wordCloud(String eventId, String startTime, String endTime, Integer size) {
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setCategory(2);
        searchCondition.setSubjectId(eventId);
        searchCondition.setStartTime(startTime);
        searchCondition.setEndTime(endTime);
        //排除词
        Set<String> excludeKeywords = commonService.getExcludeKeywords(eventId);
        searchCondition.setExcludeValues(excludeKeywords.toArray(new String[0]));
        return esService.groupByTerm(searchCondition, "groupKeyWord", "keyWordsList.keyword", false, size);
    }

    @Override
    public List<CountVO> wordTrend(String eventId, String startTime, String endTime) {
        int count = esService.count(eventId, startTime, endTime);
        if (count == 0) {
            return new ArrayList<>();
        }
        Event event = eventService.getById(eventId);
        Date beginTime = event.getStartTime();
        if (StringUtils.isNotEmpty(startTime)) {
            beginTime = DateUtil.stringToDate(startTime, "yyyy-MM-dd HH:mm:ss");
        }
        Date finishTime;
        if (StringUtils.isEmpty(endTime)) {
            finishTime = event.getEndTime();
            if (Objects.isNull(event.getEndTime())) {
                finishTime = new Date();
            }
        } else {
            finishTime = DateUtil.stringToDate(endTime, "yyyy-MM-dd HH:mm:ss");
        }
        long between = DateUtil.betweenTwoDate(beginTime, finishTime);
        String groupType;
        if (between <= 30) {
            groupType = "day";
        } else if (between <= 180) {
            groupType = "week";
        } else {
            groupType = "month";
        }
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setCategory(2);
        searchCondition.setSubjectId(eventId);
        searchCondition.setStartTime(DateUtil.dateToString(beginTime));
        searchCondition.setEndTime(DateUtil.dateToString(finishTime));
        //排除词
        Set<String> excludeKeywords = commonService.getExcludeKeywords(eventId);
        searchCondition.setExcludeValues(excludeKeywords.toArray(new String[0]));
        return esService.keywordTrend(searchCondition, groupType);
    }

    @Override
    public List<CoOccurrenceVO> coOccurrence(String eventId, String startTime, String endTime) {
        List<CoOccurrenceVO> list = new ArrayList<>();
        int count = esService.count(eventId, startTime, endTime);
        if (count == 0) {
            return list;
        }
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setCategory(2);
        searchCondition.setSubjectId(eventId);
        searchCondition.setStartTime(startTime);
        searchCondition.setEndTime(endTime);
        String[] fetchFields = new String[]{"id", "keyWordsList"};
        searchCondition.setFetchFields(fetchFields);
        int pageNo = 1;
        int size = 300;
        searchCondition.setPageSize(size);
        boolean flag = true;
        //排除词
        Set<String> excludeKeywords = commonService.getExcludeKeywords(eventId);
        List<List<String>> wordList = new ArrayList<>();
        do {
            searchCondition.setPageNo(pageNo);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            if (CollectionUtils.isNotEmpty(informationList)) {
                for (SpecialInformation information : informationList) {
                    List<String> keyWordsList = information.getKeyWordsList();
                    if (CollectionUtils.isEmpty(keyWordsList)) {
                        continue;
                    }
                    keyWordsList = keyWordsList.stream().filter(e -> !excludeKeywords.contains(e)).collect(Collectors.toList());
                    if (CollectionUtils.isNotEmpty(keyWordsList) && keyWordsList.size() > 2) {
                        wordList.add(keyWordsList);
                    }
                }
                if (informationList.size() < size) {
                    flag = false;
                } else {
                    pageNo++;
                }
            } else {
                flag = false;
            }
        } while (flag);
        Set<String> keyWordSet = new HashSet<>();
        wordList.forEach(keyWordSet::addAll);
        searchCondition.setIncludeValues(keyWordSet.toArray(new String[0]));
        List<CountVO> groupKeyWord = esService.groupByTerm(searchCondition, "groupKeyWord", "keyWordsList.keyword", false, size);
        Map<String, Long> wordFrequencyMap = groupKeyWord.stream().collect(Collectors.toMap(CountVO::getName, CountVO::getValue));
        Map<String, Integer> map = HotWordUtil.calculateCoOccurrence_list(wordList);
        //根据共现次数过滤，支取次数大于等于10的
        //Map<String, Integer> filterMap = map.entrySet().stream().filter(e -> e.getValue() >= 10).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        //根据共现次数倒序排序，并取前100
        LinkedHashMap<String, Integer> filterMap = map.entrySet().stream().sorted(Map.Entry.comparingByValue(Comparator.reverseOrder()))
                .limit(100)
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (oldValue, newValue) -> oldValue, LinkedHashMap::new));
        for (Map.Entry<String, Integer> entry : filterMap.entrySet()) {
            CoOccurrenceVO coOccurrenceVO = new CoOccurrenceVO();
            String key = entry.getKey();
            Integer coOccurrenceNum = entry.getValue();
            coOccurrenceVO.setCoOccurrenceNum((long) coOccurrenceNum);
            CountVO word_1 = new CountVO();
            word_1.setName(key.split("&")[0]);
            word_1.setValue(wordFrequencyMap.getOrDefault(key.split("&")[0], 0L));
            coOccurrenceVO.setWord_1(word_1);
            CountVO word_2 = new CountVO();
            word_2.setName(key.split("&")[1]);
            word_2.setValue(wordFrequencyMap.getOrDefault(key.split("&")[1], 0L));
            coOccurrenceVO.setWord_2(word_2);
            list.add(coOccurrenceVO);
        }
        return list;
    }

    @Override
    public PropagationPathVo propagationPath(String eventId) {
        PropagationPathVo top = null;
        int count = esService.count(eventId, null, null);
        if (count > 0) {
            Event event = eventService.getById(eventId);
            String subjectName = event.getEventName();
            //获取专题数据
            String startDate = null;
            String endDate = null;
            if (event.getStartTime() != null) {
                startDate = DateUtil.dateToString(event.getStartTime(), "yyyy-MM-dd HH:mm:ss");
            }
            if (event.getEndTime() != null) {
                endDate = DateUtil.dateToString(event.getEndTime(), "yyyy-MM-dd HH:mm:ss");
            }
            List<SubjectDataVo> specialDataList = esService.getDataBySubjectId(eventId, startDate, endDate, Constants.FETCH_FIELDS_STATISTIC);
            if (CollectionUtils.isNotEmpty(specialDataList)) {
                //用于来源去重
                List<String> allOriginList = new ArrayList<>();
                top = new PropagationPathVo();
                top.setName(subjectName);
                top.setTime(DateUtil.dateToString(event.getStartTime()));
                //获取发布时间最早的前10的资讯(来源不重复)
                List<PropagationPathVo> secondList = new ArrayList<>();
                List<SubjectDataVo> earlyList = topN(specialDataList, Constants.FAKE_NUM);
                earlyList.forEach(e -> allOriginList.add(Constants.getRealOrigin(e.getOrigin())));
                for (SubjectDataVo subjectDataVo : earlyList) {
                    String origin = subjectDataVo.getOrigin();
                    String time = subjectDataVo.getPublishDate();
                    if (StringUtils.isNotEmpty(origin)) {
                        PropagationPathVo second = new PropagationPathVo();
                        second.setName(Constants.getRealOrigin(origin));
                        second.setTime(time);
                        secondList.add(second);
                        List<String> thirdList = esService.originList(subjectDataVo.getTitle(), subjectDataVo.getPublishDate());
                        thirdList.removeAll(allOriginList);
                        List<PropagationPathVo> lastList = new ArrayList<>();
                        if (thirdList.size() > 3) {
                            thirdList = thirdList.subList(0, 3);
                        }
                        for (String s : thirdList) {
                            PropagationPathVo third = new PropagationPathVo();
                            third.setName(s);
                            lastList.add(third);
                        }
                        second.setChildren(lastList);
                        allOriginList.addAll(thirdList);
                    }
                }
                top.setChildren(secondList);
            }
        }
        return top;
    }

    @Override
    public String llmResult(EventVO event,String content,Boolean webSearch, String startTime, String endTime, EventLlmConfig llmConfig,Map<Integer,String> extraContentParams) {
        if (llmConfig == null) {
            return null;
        }
        String result = llmService.model(llmConfig.getLlmName(),null,webSearch, llmConfig.getLlmPrompt(), content);
        AnalysisColumnEnum analysisColumnEnum = AnalysisColumnEnum.getByCode(llmConfig.getColumnCode());
        String startHeader = null;
        if (analysisColumnEnum != null) {
            String resType = analysisColumnEnum.getResType();
            if (resType.equals("array")) {
                startHeader = "[";
            } else if (resType.equals("object")) {
                startHeader = "{";
            }
        }
        if (startHeader != null && !result.startsWith(startHeader)) {
            if (result.startsWith("```json")) {
                result = result.substring(7, result.length() - 3);
            } else if (result.startsWith("```")) {
                result = result.substring(3, result.length() - 3);
            } else if (result.contains(startHeader)){
                result = result.substring(result.indexOf(startHeader));
            }
        }
        result = result.replaceAll("```json", "").replaceAll("```","");
        if (extraContentParams != null
                && (llmConfig.getColumnCode().equals(AnalysisColumnEnum.IMPACT_ASSESSMENT.getCode())
                || llmConfig.getColumnCode().equals(AnalysisColumnEnum.DRIVING_FACTORS.getCode()))) {
            extraContentParams.put(llmConfig.getColumnCode(),result);
        }
        if (llmConfig.getColumnCode().equals(AnalysisColumnEnum.IMPACT_ASSESSMENT.getCode())) {
            EventLlmConfig detailConfig = eventLlmConfigService.getConfig(event.getId(), AnalysisColumnEnum.IMPACT_ASSESSMENT_DETAIL.getCode());
            List<JSONObject> impactList = JSON.parseArray(result, JSONObject.class);
            for (JSONObject impact : impactList) {
                try {
                    JSONObject params = new JSONObject();
                    params.put("eventName", event.getEventName());
                    params.put("eventSummary", event.getEventDescribe());
                    if (StringUtils.isNotEmpty(event.getKeyIssue())) {
                        params.put("keyIssue", event.getKeyIssue());
                    }
                    params.put("impactOutline", impact);
                    String impactDetail = llmService.model(detailConfig.getLlmName(),null, false,detailConfig.getLlmPrompt(), params.toJSONString());
                    impact.put("impactDetail", impactDetail);
                    log.info("{}-事件分析【{}-{}详情】重新生成逻辑完成。", event.getEventName(), AnalysisColumnEnum.IMPACT_ASSESSMENT.getName(),impact.getString("theme"));
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            result = JSON.toJSONString(impactList);
        } else if (llmConfig.getColumnCode().equals(AnalysisColumnEnum.DRIVING_FACTORS.getCode())) {
            JSONObject params = new JSONObject();
            params.put("drivingFactor", result);
            result = JSON.toJSONString(params);
        }
        return result;
    }

    @Override
    public String llmResult(String eventId,String content,Boolean webSearch,  String startTime, String endTime, Integer columnCode) {
        EventVO event = eventService.queryInfo(eventId);
        EventLlmConfig llmConfig = eventLlmConfigService.getConfig(eventId, columnCode);
        return llmResult(event,content,webSearch, startTime, endTime, llmConfig,null);
    }

    @Override
    public void regenerate(String eventId,String versionId) {
        final List<EventLlmConfig> configList = eventLlmConfigService.getConfigList(eventId);
        String today = DateUtil.dateToString(new Date());
        //保存版本信息
        if (StringUtils.isEmpty(versionId)) {
            versionId = saveVersion(eventId, today);
        }
        final EventVO event = eventService.queryInfo(eventId);
        String eventName = event.getEventName();
        log.info("{}-事件分析重新生成逻辑开始。。。", eventName);
        int attempts = 0;
        do {
            String finalVersionId = versionId;
            Thread thread = new Thread(() -> generate(event, finalVersionId, configList));
            thread.start();
            try {
                Thread.sleep(1000 * 30);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            LambdaQueryWrapper<EventAnalysisVersionRecord> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(EventAnalysisVersionRecord::getVersionId, versionId);
            List<EventAnalysisVersionRecord> list = eventAnalysisVersionRecordService.list(queryWrapper);
            if (list.size() < 1) {
                thread.interrupt();
                attempts ++;
            } else {
                break;
            }
        } while (attempts < 3);
    }


    private void generate(EventVO event,String versionId,List<EventLlmConfig> configList){
        String eventId = event.getId();
        String eventName = event.getEventName();
        String eventDescribe = event.getEventDescribe();
        //核心摘要
        Integer ynManualEdit = event.getYnManualEdit();
        if (ynManualEdit == 0) {
            String startTime = event.getStartTime();
            String endTime = event.getEndTime();
            String contentTimePart = "\n关注时间:" + startTime + "~";
            if (StringUtils.isEmpty(endTime)) {
                contentTimePart += "至今";
            } else {
                contentTimePart += endTime;
            }
            String content  = "事件标题；" + eventName + "\n采集关键词:" + event.getKeywordsVO().getKeyword();
            if (StringUtils.isEmpty(event.getKeyIssue())) {
                content += contentTimePart;
            } else {
                content += "\n关心问题:" + event.getKeyIssue() + contentTimePart;
            }

            //历史核心摘要是否为空
            boolean empty = StringUtils.isEmpty(eventDescribe);
            EventLlmConfig config = configList.stream().filter(e -> e.getColumnCode().equals(AnalysisColumnEnum.CORE_SUMMARY.getCode())).findFirst().orElse(new EventLlmConfig());
            eventDescribe = this.llmResult(event,content, false,null, null, config,null);
            if (empty) {
                event.setEventDescribe(eventDescribe);
                LambdaUpdateWrapper<Event> update = Wrappers.lambdaUpdate();
                update.set(Event::getEventDescribe, eventDescribe).eq(Event::getId, eventId);
                eventService.update(update);
            }
        }
        eventAnalysisVersionRecordService.modify(versionId,AnalysisColumnEnum.CORE_SUMMARY, eventDescribe);
        log.info("{}-事件分析【{}】重新生成逻辑完成。", eventName, AnalysisColumnEnum.CORE_SUMMARY.getName());
        //大模型相关逻辑生成的结果(影响评估,举措建议等)
        List<EventLlmConfig> collect = configList.stream()
                .filter(e -> !e.getColumnCode().equals(AnalysisColumnEnum.CORE_SUMMARY.getCode())
                        && !e.getColumnCode().equals(AnalysisColumnEnum.IMPACT_ASSESSMENT_DETAIL.getCode()))
                .collect(Collectors.toList());
        String content  = "事件标题；" + eventName + "\n事件摘要:" + event.getEventDescribe();
        if (StringUtils.isNotEmpty(event.getKeyIssue())) {
            content += "\n关心问题:" + event.getKeyIssue();
        }
        //额外的输入参数
        Map<Integer,String> extraContentParams = new HashMap<>();
        for (EventLlmConfig config : collect) {
            String realContent = content;
            try {
                Integer columnCode = config.getColumnCode();
                AnalysisColumnEnum analysisColumnEnum = AnalysisColumnEnum.getByCode(columnCode);
                if (analysisColumnEnum == null) {
                    continue;
                }
                //应对建议需要把 驱动因素和影响评估作为输入
                if (Objects.equals(columnCode, AnalysisColumnEnum.MEASURE_SUGGEST.getCode())) {
                    if (extraContentParams.containsKey(AnalysisColumnEnum.IMPACT_ASSESSMENT.getCode())) {
                        realContent += "\n影响评估:" + extraContentParams.get(AnalysisColumnEnum.IMPACT_ASSESSMENT.getCode());
                    }
                    if (extraContentParams.containsKey(AnalysisColumnEnum.DRIVING_FACTORS.getCode())) {
                        realContent += "\n驱动因素:" + extraContentParams.get(AnalysisColumnEnum.DRIVING_FACTORS.getCode());
                    }
                }
                boolean webSearch = true;
                if (Objects.equals(columnCode, AnalysisColumnEnum.ENTERPRISE_CASE.getCode())) {
                    webSearch = false;
                    List<SpecialInformation> informationList = modelAnalysisInformationList(eventId, eventName);
                    if (CollectionUtils.isNotEmpty(informationList)) {
                        realContent += "\n参考资讯:" + JSON.toJSONString(informationList);
                    }
                }
                String llmResult = this.llmResult(event,realContent, webSearch,null, null, config,extraContentParams);
                eventAnalysisVersionRecordService.modify(versionId, analysisColumnEnum, llmResult);
                log.info("{}-事件分析【{}】重新生成逻辑完成。", eventName, config.getColumnName());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public List<SpecialInformation> modelAnalysisInformationList(String eventId, String eventName) {
        List<SpecialInformation> informationList = enterpriseCaseDataList(eventId);
        if (CollectionUtils.isEmpty(informationList)) {
            informationList = webSearchCaseDataList(eventName);
        }
        return informationList;
    }

    /**
     * 事件下企业资讯列表
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/9/19
     */
    private List<SpecialInformation> enterpriseCaseDataList(String eventId) {
        List<SpecialInformation> infoList = new ArrayList<>();
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        String[] fetchFields = new String[]{"id", "title", "content", "origin"};
        searchCondition.setFetchFields(fetchFields);
        searchCondition.setSubjectId(eventId);
        searchCondition.setCategory(2);
        //企业资讯标签
        searchCondition.setComposeSearchLabelIds("1947596156752150530");
        searchCondition.setPageSize(50);
        for (int i = 1; ; i++) {
            searchCondition.setPageNo(i);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            log.info("事件分析-企业案例获取资讯：本次循环-{}，数据量为-{}", i, informationList.size());
            if (CollectionUtils.isNotEmpty(informationList)) {
                List<SpecialInformation> caseFilterList = pythonUtil.enterpriseCaseFilter(informationList);
                if (CollectionUtils.isEmpty(caseFilterList)) {
                    continue;
                }
                infoList.addAll(caseFilterList);
                if (infoList.size() >= 20) {
                    log.info("事件分析-企业案例获取资讯：企业案例资讯数据量为-{}", infoList.size());
                    break;
                }
                if (informationList.size() < 50) {
                    break;
                }
            } else {
                break;
            }
        }
        return infoList;
    }

    /**
     * 智谱接口-联网获取资讯
     *
     * 事件下企业资讯为空时，使用此接口获取数据
     *
     *
     * @param eventName 事件标题
     * @author lkg
     * @date 2025/9/19
     */
    private List<SpecialInformation> webSearchCaseDataList(String eventName) {
        List<SpecialInformation> infoList = new ArrayList<>();
        Map<String,String> headers = new HashMap<>();
        headers.put("Authorization", "c5a53bd5f95a4e37a8997deb5d0c6031.orXyRRPNvZiqRaxF");
        JSONObject params = new JSONObject();
        params.put("search_query", eventName);
        params.put("search_engine", "search_std");
        params.put("search_intent", true);
        try {
            String response = HttpUtil.doPostWithHeader("https://open.bigmodel.cn/api/paas/v4/web_search", params, 30000, headers);
            JSONObject jsonObject = JSONObject.parseObject(response);
            String dataListStr = jsonObject.getString("search_result");
            List<JSONObject> jsonObjects = JSON.parseArray(dataListStr, JSONObject.class);
            for (JSONObject data : jsonObjects) {
                SpecialInformation info = new SpecialInformation();
                info.setId(data.getString("refer"));
                info.setTitle(data.getString("title"));
                info.setContent(data.getString("content"));
                info.setOrigin(data.getString("media"));
                info.setSourceAddress(data.getString("link"));
                info.setPublishDate(data.getString("publish_date"));
                infoList.add(info);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return infoList;
    }



    @Override
    public String exportPPT(String eventId,String versionId, JSONArray content) {
        String pptFile = null;
        Event event = eventService.getById(eventId);
        String eventName = event.getEventName();
        Map<String, Object> params = new HashMap<>();
        params.put("eventName", eventName);
        params.put("dataList", JSON.toJSONString(content));
        List<SysDictItem> dictItemList = dictItemService.listByDictCode("xunfei_ppt");
        long timestamp = System.currentTimeMillis() / 1000;
        for (SysDictItem dictItem : dictItemList) {
            String pptUrl = null;
            //是否应用异常
            boolean appException = false;
            try {
                String appId = dictItem.getItemText();
                String secret = dictItem.getItemValue();
                String signature = ApiAuthAlgorithm.getSignature(appId, secret, timestamp);
                //获取接口得到的大纲
                String outlinePrompt = "请基于提供的json数据，生成大纲。数据如下：" + JSON.toJSONString(params);
                CreateResponse outlineResponse = PPTUtil.createOutline(appId, String.valueOf(timestamp), signature, outlinePrompt);
                if (outlineResponse == null || !outlineResponse.isFlag()) {
                    continue;
                }
                String outline = outlineResponse.getData().getOutline();
                log.info("大纲生成完毕：{}", outline);
                String pptPrompt = "结合大纲及以下正文内容生成ppt。正文内容：" + JSON.toJSONString(params);
                CreateResponse pptResponse = PPTUtil.createByOutline(appId, String.valueOf(timestamp), signature, pptPrompt, outline);
                if (pptResponse == null || !pptResponse.isFlag()) {
                    continue;
                }
                String sid = pptResponse.getData().getSid();
                log.info("PPT生成接口返回的sid：{}", sid);
                boolean flag = true;
                while (flag) {
                    Thread.sleep(3000);
                    ProgressResponse progressResponse = PPTUtil.checkProgress(appId, String.valueOf(timestamp), signature, sid);
                    if (progressResponse == null || !progressResponse.isFlag()) {
                        continue;
                    }
                    String pptStatus = progressResponse.getData().getPptStatus();
                    switch (pptStatus) {
                        case "building":
                            log.info("PPT正在生成中...");
                            break;
                        case "done":
                            pptUrl = progressResponse.getData().getPptUrl();
                            log.info("应用-{},PPT生成成功: {}", appId, pptUrl);
                            flag = false;
                            break;
                        case "build_failed":
                            log.info("PPT生成失败");
                            flag = false;
                            break;
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
                appException = true;
            }
            if (pptUrl != null) {
                try {
                    URL urlPath = new URL(pptUrl);
                    URLConnection connection = urlPath.openConnection();
                    InputStream inputStream = connection.getInputStream();
                    PutObjectResult putObjectResult = obsUtil.uploadFile("event_ppt/" + eventId + "_" + timestamp + ".pptx", inputStream);
                    pptFile = putObjectResult.getObjectKey();
                    break;
                } catch (IOException e) {
                    e.printStackTrace();
                    log.info("PPT上传OBS失败");
                    break;
                }
            } else { //pptFile为空时，不是应用异常，说明是数据问题，直接结束循环
                if (!appException) {
                    log.info("数据问题，导致PPT生成失败");
                    break;
                } else {
                    log.info("应用异常，继续尝试其他应用");
                }
            }
        }
        Result<String> result;
        if (pptFile == null) {
            result = Result.FAIL("ppt生成失败");
        } else {
            result = Result.OK(pptFile);
        }
        LambdaUpdateWrapper<EventAnalysisVersion> update = Wrappers.lambdaUpdate();
        update.set(EventAnalysisVersion::getPptPath, JSON.toJSONString(result)).eq(EventAnalysisVersion::getId, versionId);
        eventAnalysisVersionService.update(update);
        return pptFile;
    }

    private String saveVersion(String eventId, String time) {
        EventAnalysisVersion version = new EventAnalysisVersion();
        version.setEventId(eventId);
        version.setVersionName("版本" + time);
        eventAnalysisVersionService.save(version);
        return version.getId();
    }

    //获取发布时间最早的前N条资讯(来源不重复)
    private List<SubjectDataVo> topN(List<SubjectDataVo> list, Integer num) {
        List<SubjectDataVo> collect = list.stream().filter(e -> StringUtils.isNotEmpty(e.getOrigin())).
                sorted(Comparator.comparing(SubjectDataVo::getPublishDate)).collect(Collectors.toList());
        TreeSet<SubjectDataVo> subjectDataVos = new TreeSet<>(Comparator.comparing(SubjectDataVo::getOrigin));
        for (SubjectDataVo subjectDataVo : collect) {
            subjectDataVos.add(subjectDataVo);
            if (subjectDataVos.size() == num) {
                break;
            }
        }
        return new ArrayList<>(subjectDataVos);
    }

    private List<SubjectAnalysis> getList(String subjectId, Integer category) {
        LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectAnalysis::getSubjectId, subjectId).eq(SubjectAnalysis::getCategory, category).orderByDesc(SubjectAnalysis::getPublishDate);
        return subjectAnalysisService.list(queryWrapper);
    }


    //补充是否主要资讯标识以及排序后截取
    private List<EventContextVO> supplyMain(List<EventContextVO> list) {
        EventContextVO dataVo = null;
        double minDistance = Double.MAX_VALUE;
        for (EventContextVO eventContextVO : list) {
            Double similarity = eventContextVO.getDistance();
            if (similarity == 0) {
                continue;
            }
            if (similarity == 1.0) {
                dataVo = eventContextVO;
                break;
            } else if (similarity < minDistance) {
                minDistance = similarity;
                dataVo = eventContextVO;
            }
        }
        if (dataVo == null) {
            list.get(0).setIsMain(true);
        } else {
            for (EventContextVO eventContextVO : list) {
                if (eventContextVO.getId().equals(dataVo.getId())) {
                    eventContextVO.setIsMain(true);
                    break;
                }
            }
        }
        //先按是否是主事件排序，再按相似度算法(编辑距离)返回值正序排序，最后按时间倒序排序
        list.sort(Comparator.comparing(EventContextVO::getIsMain, Comparator.reverseOrder())
                .thenComparing((o1, o2) -> {
                    Double distance1 = o1.getDistance();
                    Double distance2 = o2.getDistance();
                    if (distance1 == 0 && distance2 == 0) {
                        return 0;
                    }
                    if (distance1 == 0) {
                        return 1;
                    }
                    if (distance2 == 0) {
                        return -1;
                    }
                    return distance1.compareTo(distance2);
                })
                .thenComparing(EventContextVO::getPublishDate, Comparator.reverseOrder()));
        if (list.size() > 3) {
            list = list.subList(0, 3);
        }
        return list;
    }
}
