package com.zzsn.event.controller;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.EventAnalysisReport;
import com.zzsn.event.entity.LabelEntity;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.vo.CountVO;
import com.zzsn.event.vo.SubjectDataVo;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 事件分析
 *
 * @author lkg
 * @date 2024/1/24
 */
@RestController
@RequestMapping("/analysis")
public class EventAnalysisController {

    @Autowired
    private EsStatisticsService esStatisticsService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private EventAnalysisReportService eventAnalysisReportService;
    @Autowired
    private EsService esService;
    @Autowired
    private RedisUtil redisUtil;

    @Value(("${serviceProject.url:}"))
    private String SERVICE_PROJECT_URL;


    /**
     * 3.4 总体分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/1/24
     */
    @GetMapping("/total")
    public Result<?> totalAndMax(@RequestParam String subjectId, @RequestParam String startTime,
                                 @RequestParam(required = false) String endTime,
                                 @RequestParam(defaultValue = "1") Integer type) {
        Map<String, String> map = esStatisticsService.totalAndMax(subjectId, null, null, type);
        if (StringUtils.isEmpty(endTime)) {
            endTime = DateUtil.now();
        }
        long hours = DateUtil.between(DateUtil.parseDateTime(startTime), DateUtil.parseDateTime(endTime), DateUnit.HOUR);
        map.put("duration", String.valueOf(hours));
        Object count = map.get("totalCount");
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(hours), 0, RoundingMode.UP);
        map.put("spread", divide);
        String mainReport = esStatisticsService.mainReport(subjectId);
        map.put("mainReport", mainReport);
        List<String> subjectIdList = new ArrayList<>();
        subjectIdList.add(subjectId);
        DateTime beforeHour = DateUtil.offsetHour(DateUtil.parseDateTime(endTime), -1);
        long lastHour = esStatisticsService.totalCount(subjectIdList, DateUtil.formatDateTime(beforeHour),endTime);
        map.put("lastHour",String.valueOf(lastHour));
        return Result.OK(map);
    }

    /**
     * 热门信息列表
     *
     * @param subjectId 专题id
     * @param size      返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping("/hotList")
    public Result<?> hotList(@RequestParam(name = "subjectId") String subjectId,
                             @RequestParam(name = "size", defaultValue = "10") Integer size) {
        String[] fetchFields = new String[]{"id","subjectId", "title", "origin", "publishDate", "sourceAddress"};
        List<SubjectDataVo> pageList = esService.pageList(subjectId, null, null, fetchFields, 2, 1, size);
        if (CollectionUtils.isNotEmpty(pageList)) {
            List<String> idList = new ArrayList<>();
            pageList.forEach(e -> idList.add(e.getId()));
            Map<String, Integer> similarNumber = esService.getSimilarNumber(subjectId, idList);
            for (SubjectDataVo subjectDataVo : pageList) {
                String dataId = subjectDataVo.getId();
                Integer count = similarNumber.get(dataId);
                if (count == null) {
                    count = 0;
                }
                subjectDataVo.setSimilarNumber(count);
            }
        }
        return Result.OK(pageList);
    }

    /**
     * 3.1 传播路径
     *
     * @param eventId 事件id
     */
    @GetMapping("/propagationPath")
    public Result<?> propagationPath(@RequestParam String eventId) {
        String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.PROPAGATION_KEY + eventId;
        Object cacheObject = redisUtil.get(key);
        if (ObjectUtils.isEmpty(cacheObject)) {
            cacheObject = analysisService.propagationPath(eventId);
        }
        return Result.OK(cacheObject);
    }

    /**
     * 3.2 事件脉络
     *
     * @param subjectId 专题id
     * @param fakeNum   专题事件脉络展示 伪事件脉络 的资讯数量阈值
     * @return com.zzsn.subjectAnalysis.common.Result
     */
    @GetMapping("/eventContext")
    public Result<?> eventContext(@RequestParam String subjectId,
                                  @RequestParam(value = "fakeNum", required = false) Integer fakeNum) {
        if (fakeNum == null) {
            fakeNum = Constants.FAKE_NUM;
        }
        List<SubjectAnalysis> cacheList = analysisService.eventContext(subjectId, fakeNum);
        return Result.OK(cacheList);
    }

    /**
     * 3.8 观点
     *
     * @author lkg
     * @date 2024/1/12
     */
    @ApiOperation(value = "专题信息列表-分页列表查询", notes = "专题信息列表-分页列表查询")
    @GetMapping(value = "/listEventAnalysis")
    public Result<?> listEventAnalysis(@RequestParam(name = "eventId") String eventId,
                                       @RequestParam(name = "type") Integer type) {
        LambdaQueryWrapper<SubjectAnalysis> wrapper = new LambdaQueryWrapper<SubjectAnalysis>()
                .eq(SubjectAnalysis::getSubjectId, eventId)
                .eq(SubjectAnalysis::getCategory, 1)
                .eq(SubjectAnalysis::getType, type);
        List<SubjectAnalysis> list = subjectAnalysisService.list(wrapper);
        return Result.OK(list);
    }

    /**
     * 3.3 热词查询
     *
     * @return
     */
    @ApiOperation(value = "热词查询", notes = "热词查询")
    @GetMapping(value = "/keywordsCount")
    public Object keywordsCount(@RequestParam String subjectId, @RequestParam Integer pageSize) {
        try {
            JSONObject params = new JSONObject();
            params.put("subjectId", subjectId);
            params.put("pageSize", pageSize);
            String url = SERVICE_PROJECT_URL + "event/keywordsCount";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 专题下情感类标签下资讯数量统计
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/5/9
     */
    @GetMapping("/orientationCount")
    public Result<?> orientationCount(@RequestParam String subjectId){
        List<CountVO> countVOS = esStatisticsService.orientationCount(subjectId, null, null);
        return Result.OK(countVOS);
    }


    /**
     * 3.5 情感判断分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/orientation")
    public Result<?> orientation(@RequestParam String subjectId, @RequestParam String startTime,
                                 @RequestParam(required = false) String endTime, @RequestParam Integer type) {
        List<CountVO> list = new ArrayList<>();
        Map<String, String> map = esStatisticsService.totalAndMax(subjectId, null, null, type);
        String totalCount = map.get("totalCount");
        if (!totalCount.equals("0")) {
            String maxTime = map.get("maxTime");
            Map<String, String> timeRangeMap = getTimeRange(startTime, endTime, maxTime, type);
            startTime = timeRangeMap.get("startTime");
            endTime = timeRangeMap.get("endTime");
            String labelTypeId = "1631119596744265729";
            List<LabelEntity> labelEntities = labelEntityService.listByType(labelTypeId);
            AtomicLong total = new AtomicLong();
            String finalStartTime = startTime;
            String finalEndTime = endTime;
            labelEntities.forEach(e -> {
                CompletableFuture<CountVO> async = CompletableFuture.supplyAsync(() -> {
                    CountVO countVO = esStatisticsService.orientation(subjectId, e.getId(), finalStartTime, finalEndTime, type);
                    total.addAndGet(countVO.getValue());
                    supply(countVO, finalStartTime, finalEndTime, type);
                    return countVO;
                });
                try {
                    CountVO countVO = async.get();
                    list.add(countVO);
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            });
            for (CountVO countVO : list) {
                long value = countVO.getValue();
                long totalNum = total.get();
                String divide = CalculateUtil.divide(String.valueOf(value), String.valueOf(totalNum));
                String percentage = "0%";
                if (StringUtils.isNotEmpty(divide)) {
                    percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
                }
                countVO.setPercentage(percentage);
            }
        }
        return Result.OK(list);
    }

    /**
     * 3.6 信息传播走势
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/flowData")
    public Result<?> flowData(@RequestParam String subjectId, @RequestParam String startTime,
                              @RequestParam(required = false) String endTime, @RequestParam Integer type) {
        Map<String, String> map = esStatisticsService.totalAndMax(subjectId, null, null, type);
        String totalCount = map.get("totalCount");
        List<CountVO> list = new ArrayList<>();
        if (!totalCount.equals("0")) {
            String maxTime = map.get("maxTime");
            Map<String, String> timeRangeMap = getTimeRange(startTime, endTime, maxTime, type);
            startTime = timeRangeMap.get("startTime");
            endTime = timeRangeMap.get("endTime");
            List<CountVO> dataList = esStatisticsService.flowData(subjectId, startTime, endTime, type);
            list = supplyChildren(dataList, startTime, endTime, type);
        }
        return Result.OK(list);
    }


    /**
     * 按来源分析
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/origin")
    public Result<?> origin(@RequestParam String subjectId) {
        List<CountVO> list = esStatisticsService.origin(subjectId, null, null);
        return Result.OK(list);
    }

    /**
     * 获取事件分析报告详细信息
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/4/12
     */
    @GetMapping("/reportInfo")
    public Result<?> reportInfo(@RequestParam String eventId){
        LambdaQueryWrapper<EventAnalysisReport> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(EventAnalysisReport::getEventId,eventId);
        EventAnalysisReport one = eventAnalysisReportService.getOne(queryWrapper);
        return Result.OK(one);
    }

    /**
     * 事件分析报告编辑
     *
     * @param report 分析报告信息
     * @author lkg
     * @date 2024/4/12
     */
    @PostMapping("/reportEdit")
    public Result<?> edit(@RequestBody EventAnalysisReport report){
        eventAnalysisReportService.edit(report);
        return Result.OK();
    }

    /**
     * 获取趋势图数据的实际时间范围
     *
     * @param startTime 专题开始时间
     * @param endTime   专题结束时间
     * @param maxTime   峰值所在时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/4/11
     */
    private Map<String, String> getTimeRange(String startTime, String endTime, String maxTime, Integer type) {
        Map<String, String> map = new HashMap<>();
        if (StringUtils.isEmpty(endTime)) {
            endTime = DateUtil.now();
        }
        if (type == 1) {
            DateTime beginOfDay = DateUtil.beginOfDay(DateUtil.parse(maxTime, "yyyy-MM-dd HH"));
            startTime = DateUtil.formatDateTime(beginOfDay);
            DateTime endOfDay = DateUtil.endOfDay(DateUtil.parse(maxTime, "yyyy-MM-dd HH"));
            endTime = DateUtil.formatDateTime(endOfDay);
        } else if (type == 2) {
            long between = DateUtil.betweenDay(DateUtil.parseDateTime(startTime), DateUtil.parseDateTime(endTime), true);
            if (between > 14) {
                DateTime startDate = DateUtil.offsetDay(DateUtil.parseDate(maxTime), -7);
                if (startDate.compareTo(DateUtil.parseDateTime(startTime)) > 0) {
                    startTime = DateUtil.formatDateTime(startDate);
                }
                DateTime endDate = DateUtil.offsetDay(DateUtil.parseDate(maxTime), 7);
                if (endDate.compareTo(DateUtil.parseDateTime(endTime)) < 0){
                    endTime = DateUtil.formatDateTime(endDate);
                }
            }
        }
        map.put("startTime", startTime);
        map.put("endTime", endTime);
        return map;
    }


    //补充缺失的时间
    private void supply(CountVO countVO, String startTime, String endTime, Integer type) {
        List<CountVO> children = countVO.getChildren();
        List<CountVO> list = supplyChildren(children, startTime, endTime, type);
        countVO.setChildren(list);
    }

    //补充缺失的时间
    private List<CountVO> supplyChildren(List<CountVO> children, String startTime, String endTime, Integer type) {
        List<CountVO> list = new ArrayList<>();
        Map<String, CountVO> map = children.stream().collect(Collectors.toMap(CountVO::getName, item -> item, (k1, k2) -> k2));
        DateTime startDate = DateUtil.parseDateTime(startTime);
        DateTime endDate = DateUtil.parseDateTime(endTime);
        List<DateTime> rangeToList = new ArrayList<>();
        String format = null;
        if (type == 1) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.HOUR_OF_DAY);
            format = "yyyy-MM-dd HH";
        } else if (type == 2) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.DAY_OF_YEAR);
            format = "yyyy-MM-dd";
        }
        for (DateTime dateTime : rangeToList) {
            String date = DateUtil.format(dateTime, format);
            if (map.containsKey(date)) {
                list.add(map.get(date));
            } else {
                CountVO vo = new CountVO();
                vo.setName(date);
                vo.setValue(0L);
                list.add(vo);
            }
        }
        return list;
    }
}
