package com.zzsn.event.service;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;

import com.zzsn.event.constant.Constants;
import com.zzsn.event.enums.InfluenceEnum;
import com.zzsn.event.enums.SourceEnum;
import com.zzsn.event.util.CalculateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.vo.CountVO;
import com.zzsn.event.vo.MediaVO;
import com.zzsn.event.vo.ReportDataVo;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.search.join.ScoreMode;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramInterval;
import org.elasticsearch.search.aggregations.bucket.histogram.Histogram;
import org.elasticsearch.search.aggregations.bucket.histogram.ParsedDateHistogram;
import org.elasticsearch.search.aggregations.bucket.nested.NestedAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.nested.ParsedNested;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.Cardinality;
import org.elasticsearch.search.aggregations.metrics.Sum;
import org.elasticsearch.search.aggregations.metrics.SumAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.collapse.CollapseBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.*;

/**
 * todo
 *
 * @author lkg
 * @date 2024/1/30
 */
@Service
public class ColumnService {
    @Autowired
    private RestHighLevelClient client;
    @Autowired
    private IInfoSourceService infoSourceService;

    /**
     * 获取栏目或栏目组下的信息数量
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/1/30
     */
    public Long count(String column, Boolean ynGroup, String subjectId, String startTime, String endTime) {
        long count = 0;
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(column, ynGroup, subjectId, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            count = searchResponse.getHits().getTotalHits().value;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return count;
    }

    /**
     * 按信息源性质分组数量统计
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/1/30
     */
    public List<CountVO> source(String column, Boolean ynGroup, String subjectId, String startTime, String endTime) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(column, ynGroup, subjectId, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_source")
                .field("infoSourceTypeId.keyword")
                .order(BucketOrder.count(false))
                .size(SourceEnum.values().length)
                .subAggregation(AggregationBuilders.dateHistogram("group_day")
                        .field("publishDate")
                        .calendarInterval(DateHistogramInterval.DAY)
                        .format("yyyy-MM-dd")
                );
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        SearchResponse searchResponse;
        try {
            searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = searchResponse.getHits().getTotalHits().value;
            Aggregations aggregations = searchResponse.getAggregations();
            Terms groupSource = aggregations.get("group_source");
            List<? extends Terms.Bucket> sourceBuckets = groupSource.getBuckets();
            for (int i = 0; i < sourceBuckets.size(); i++) {
                Terms.Bucket source = sourceBuckets.get(i);
                CountVO countVO = new CountVO();
                countVO.setOrder(i + 1);
                String keyAsString = source.getKeyAsString();
                countVO.setKey(SourceEnum.getDescription(keyAsString));
                long count = source.getDocCount();
                countVO.setCount(String.valueOf(count));
                String percentage = getPercentage(count, value);
                countVO.setPercentage(percentage);
                List<CountVO> children = new ArrayList<>();
                Aggregations sourceAggregations = source.getAggregations();
                ParsedDateHistogram groupDay = sourceAggregations.get("group_day");
                List<? extends Histogram.Bucket> dayBuckets = groupDay.getBuckets();
                for (Histogram.Bucket day : dayBuckets) {
                    CountVO vo = new CountVO();
                    vo.setKey(day.getKeyAsString());
                    vo.setCount(String.valueOf(day.getDocCount()));
                    children.add(vo);
                }
                countVO.setChildren(children);
                list.add(countVO);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 影响力数量统计-点赞数、评论数、收藏数、分享数
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/1/30
     */
    public List<CountVO> influence(String column, Boolean ynGroup, String subjectId, String startTime, String endTime) {
        List<CountVO> list = new ArrayList<>();
        InfluenceEnum[] influenceEnums = InfluenceEnum.values();
        for (InfluenceEnum influenceEnum : influenceEnums) {
            String field = influenceEnum.getField();
            String description = influenceEnum.getDescription();
            SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
            SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(column, ynGroup, subjectId, startTime, endTime, null);
            searchSourceBuilder.size(0);
            searchSourceBuilder.trackTotalHits(true);
            SumAggregationBuilder sumAggregationBuilder = AggregationBuilders.sum("sum_count").field(field);
            searchSourceBuilder.aggregation(sumAggregationBuilder);
            searchRequest.source(searchSourceBuilder);
            SearchResponse searchResponse;
            try {
                searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
                Aggregations aggregations = searchResponse.getAggregations();
                Sum sumCount = aggregations.get("sum_count");
                double value = sumCount.getValue();
                CountVO countVO = new CountVO();
                countVO.setKey(description);
                countVO.setCount(String.valueOf(value));
                list.add(countVO);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return list;
    }

    /**
     * 情感数量统计
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param labelId   情感标签id
     * @author lkg
     * @date 2024/2/2
     */
    public CountVO orientation(String column, Boolean ynGroup, String subjectId, String startTime, String endTime, String labelId) {
        CountVO countVO = new CountVO();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(column, ynGroup, subjectId, startTime, endTime, labelId);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_day")
                .field("publishDate")
                .calendarInterval(DateHistogramInterval.DAY)
                .format("yyyy-MM-dd");
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = response.getHits().getTotalHits().value;
            countVO.setCount(String.valueOf(value));
            List<CountVO> list = new ArrayList<>();
            Aggregations aggregations = response.getAggregations();
            ParsedDateHistogram groupHour = aggregations.get("group_day");
            List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                for (Histogram.Bucket bucket : buckets) {
                    CountVO vo = new CountVO();
                    vo.setKey(bucket.getKeyAsString());
                    vo.setCount(String.valueOf(bucket.getDocCount()));
                    list.add(vo);
                }
            }
            countVO.setChildren(list);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return countVO;
    }


    /**
     * 媒体分布
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/2/2
     */
    public List<MediaVO> media(String column, Boolean ynGroup, String subjectId, String startTime, String endTime) {
        List<MediaVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(column, ynGroup, subjectId, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_nature")
                .field("infoSourceNatureId.keyword")
                .subAggregation(AggregationBuilders.cardinality("sid_count").field("sid.keyword"));
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long totalCount = response.getHits().getTotalHits().value;
            Aggregations aggregations = response.getAggregations();
            Terms groupNature = aggregations.get("group_nature");
            List<? extends Terms.Bucket> natureBuckets = groupNature.getBuckets();
            for (int i = 0; i < natureBuckets.size(); i++) {
                Terms.Bucket nature = natureBuckets.get(i);
                MediaVO vo = new MediaVO();
                vo.setOrder(i + 1);
                String natureName = infoSourceService.getNatureName(nature.getKeyAsString());
                vo.setKey(natureName);
                long count = nature.getDocCount();
                vo.setCount(count);
                String percentage = getPercentage(count, totalCount);
                vo.setPercentage(percentage);
                Aggregations natureAggregations = nature.getAggregations();
                Cardinality sidCount = natureAggregations.get("sid_count");
                long value = sidCount.getValue();
                vo.setMediaNum(value);
                list.add(vo);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 平台活跃度排名
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/2/2
     */
    public List<CountVO> origin(String column, Boolean ynGroup, String subjectId, String startTime, String endTime) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(column, ynGroup, subjectId, startTime, endTime, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_origin")
                .field("origin.keyword")
                .order(BucketOrder.count(false))
                .size(10);
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            long value = response.getHits().getTotalHits().value;
            Aggregations aggregations = response.getAggregations();
            Terms groupSource = aggregations.get("group_origin");
            List<? extends Terms.Bucket> buckets = groupSource.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                for (int i = 0; i < buckets.size(); i++) {
                    Terms.Bucket bucket = buckets.get(i);
                    CountVO vo = new CountVO();
                    vo.setOrder(i + 1);
                    vo.setKey(bucket.getKeyAsString());
                    long count = bucket.getDocCount();
                    vo.setCount(String.valueOf(count));
                    String percentage = getPercentage(count, value);
                    vo.setPercentage(percentage);
                    list.add(vo);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 热词词云
     *
     * @param column    栏目id或栏目组id,多个逗号隔开
     * @param ynGroup   是否栏目组
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param size      热词个数
     * @author lkg
     * @date 2024/2/2
     */
    public List<CountVO> horWords(String column, Boolean ynGroup, String subjectId, String startTime, String endTime, Integer size) {
        List<CountVO> list = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(column, ynGroup, subjectId, startTime, endTime, null);
        searchRequest.source(searchSourceBuilder);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        TermsAggregationBuilder aggTerms = AggregationBuilders.terms("keyWordsList")
                .field("keyWordsList.keyword")
                .size(size);
        searchSourceBuilder.aggregation(aggTerms);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            Terms keyWordsList = aggregations.get("keyWordsList");
            List<? extends Terms.Bucket> buckets = keyWordsList.getBuckets();
            for (int i = 0; i < buckets.size(); i++) {
                Terms.Bucket bucket = buckets.get(i);
                CountVO countVO = new CountVO();
                countVO.setKey(bucket.getKeyAsString());
                countVO.setCount(String.valueOf(bucket.getDocCount()));
                countVO.setOrder(i + 1);
                list.add(countVO);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return list;
    }

    /**
     * 事件专题信息峰值以及峰值所在点
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/2/28
     */
    public Map<String, Object> maxValue(String subjectId) {
        Map<String, Object> map = new HashMap<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(null, null, subjectId, null, null, null);
        searchSourceBuilder.size(0);
        searchSourceBuilder.trackTotalHits(true);
        DateHistogramAggregationBuilder aggregation = AggregationBuilders.dateHistogram("group_hour")
                .field("publishDate")
                .calendarInterval(DateHistogramInterval.HOUR)
                .format("yyyy-MM-dd HH")
                .order(BucketOrder.count(false));
        searchSourceBuilder.aggregation(aggregation);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();
            ParsedDateHistogram groupHour = aggregations.get("group_hour");
            List<? extends Histogram.Bucket> buckets = groupHour.getBuckets();
            if (CollectionUtils.isNotEmpty(buckets)) {
                Histogram.Bucket bucket = buckets.get(0);
                long count = bucket.getDocCount();
                String key = bucket.getKeyAsString();
                map.put("hourTime", key);
                map.put("maxValue", count);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return map;
    }

    /**
     * @param subjectId 专题
     * @param pageNo    当前页
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/2/28
     */
    public List<ReportDataVo> subjectDataList(String subjectId, Integer pageNo, Integer pageSize) {
        List<ReportDataVo> dataList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(null, null, subjectId, null, null, null);
        int offset = (pageNo - 1) * pageSize;
        searchSourceBuilder.from(offset);
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = response.getHits().getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                ReportDataVo reportDataVo = JSON.parseObject(sourceAsString, ReportDataVo.class);
                dataList.add(reportDataVo);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return dataList;
    }

    /**
     * 所有资讯类数据的分页列表
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param pageNo    当前页
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/2/28
     */
    public List<ReportDataVo> informationList(String startTime, String endTime, Integer pageNo, Integer pageSize) {
        List<ReportDataVo> dataList = new ArrayList<>();
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        SearchSourceBuilder searchSourceBuilder = getSearchSourceBuilder(null, null, null, startTime, endTime, null);
        int offset = (pageNo - 1) * pageSize;
        searchSourceBuilder.from(offset);
        searchSourceBuilder.size(pageSize);
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.sort("publishDate", SortOrder.DESC);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHit[] hits = response.getHits().getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                ReportDataVo reportDataVo = JSON.parseObject(sourceAsString, ReportDataVo.class);
                dataList.add(reportDataVo);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return dataList;
    }

    private SearchSourceBuilder getSearchSourceBuilder(String column, Boolean ynGroup, String subjectId, String startTime, String endTime, String labelId) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //去重 collapse只允许keywords或者numbers类型
        searchSourceBuilder.collapse(new CollapseBuilder("sourceAddress.keyword"));
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        BoolQueryBuilder dataTypeBoolQuery = QueryBuilders.boolQuery();
        dataTypeBoolQuery.should(QueryBuilders.termQuery("dataType.keyword", "yqjc"));
        dataTypeBoolQuery.should(QueryBuilders.boolQuery().mustNot(QueryBuilders.existsQuery("dataType")));
        boolQuery.must(dataTypeBoolQuery);
        if (StringUtils.isNotEmpty(subjectId)) {
            boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
        }
        if (StringUtils.isNotEmpty(startTime)) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").gte(EsDateUtil.esFieldDateFormat(startTime)));
        }
        if (StringUtils.isNotEmpty(endTime)) {
            boolQuery.filter(QueryBuilders.rangeQuery("publishDate").lte(EsDateUtil.esFieldDateFormat(endTime)));
        }
        if (StringUtils.isNotEmpty(labelId)) {
            boolQuery.must(QueryBuilders.nestedQuery("labels", QueryBuilders.termQuery("labels.relationId", labelId), ScoreMode.None));
        }
        //未删除状态查询
        boolQuery.mustNot(QueryBuilders.matchQuery("deleteFlag", "1"));
        searchSourceBuilder.query(boolQuery);
        return searchSourceBuilder;
    }

    /**
     * 获取占比
     *
     * @param count      数量
     * @param totalCount 总数量
     * @author lkg
     * @date 2024/1/25
     */
    private String getPercentage(long count, long totalCount) {
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(totalCount));
        String percentage = "0%";
        if (StringUtils.isNotEmpty(divide)) {
            percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
        }
        return percentage;
    }

}
