package com.zzsn.event.controller;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.*;
import com.zzsn.event.producer.ProduceInfo;
import com.zzsn.event.service.*;
import com.zzsn.event.util.*;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.entity.KeyWords;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 事件
 *
 * @Author: jeecg-boot
 * @Date: 2024-03-14
 * @Version: V1.0
 */
@Slf4j
@Api(tags = "事件")
@RestController
@RequestMapping("/event/event")
public class EventController {
    @Autowired
    private IEventService eventService;
    @Autowired
    private IEventTagService eventTagService;
    @Value(("${serviceProject.url:}"))
    private String SERVICE_PROJECT_URL;

    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;
    @Autowired
    private ProduceInfo produceInfo;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private EsStatisticsService esStatisticsService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private SubjectDisplayServive subjectDisplayServive;

    /**
     * 1.1 分页列表查询
     *
     * @param event
     * @param pageNo
     * @param pageSize
     * @param req
     * @return
     */
    @ApiOperation(value = "事件-分页列表查询", notes = "事件-分页列表查询")
    @GetMapping(value = "/list")
    public Result<?> queryPageList(Event event,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                   HttpServletRequest req) {

        IPage<Event> pageList = eventService.pageList(event, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 1.2 添加
     *
     * @param eventParam
     * @return
     */
    @ApiOperation(value = "事件-添加", notes = "事件-添加")
    @PostMapping(value = "/add")
    public Result<?> add(@RequestBody AddEventParam eventParam) {
        eventParam.setCreateTime(new Date());
        eventParam.setUpdateTime(new Date());
        Event event = eventService.saveMain(eventParam);
        //插入xxlJob
        iXxlJobInfoService.subjectInsert(event);
        return Result.OK("添加成功！");
    }

    /**
     * 1.3 编辑
     *
     * @param addEventParam
     * @return
     */
    @ApiOperation(value = "事件-编辑", notes = "事件-编辑")
    @PutMapping(value = "/edit")
    public Result<?> edit(@RequestBody AddEventParam addEventParam) {
        Event byId = eventService.getById(addEventParam.getId());
        addEventParam.setUpdateTime(new Date());
        eventService.updateMain(addEventParam);
        //更新xxljob
        List<String> subjectCodes = new ArrayList<>();
        subjectCodes.add(addEventParam.getEventCode());
        iXxlJobInfoService.keyWordsUpdate(subjectCodes, String.valueOf(addEventParam.getStatus()));

        //判断是否提取热词
        eventService.extractHotWords(addEventParam);
        if (!Objects.equals(byId.getStatus(), addEventParam.getStatus()) && addEventParam.getStatus() == 1) {
            produceInfo.sendSubjectMsg(addEventParam.getEventCode());
        }
        return Result.OK("编辑成功!");
    }

    /**
     * 1.4 通过id删除
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "事件-通过id删除", notes = "事件-通过id删除")
    @DeleteMapping(value = "/delete")
    public Result<?> delete(@RequestParam(name = "id", required = true) String id) {
        Event event = eventService.getById(id);
        iXxlJobInfoService.deleteByInfosourceCode(event.getEventCode());
        eventService.deleteMain(id);
        return Result.OK("删除成功!");
    }

    /**
     * 1.5 批量删除
     *
     * @param ids
     * @return
     */
    @ApiOperation(value = "事件-批量删除", notes = "事件-批量删除")
    @DeleteMapping(value = "/deleteBatch")
    public Result<?> deleteBatch(@RequestParam(name = "ids", required = true) String ids) {
        this.eventService.removeByIds(Arrays.asList(ids.split(",")));
        return Result.OK("批量删除成功！");
    }

    /**
     * 1.6 通过id查询
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "事件-通过id查询", notes = "事件-通过id查询")
    @GetMapping(value = "/queryById")
    public Result<?> queryById(@RequestParam(name = "id", required = true) String id) {
        Event event = eventService.getById(id);
        String relationEvents = event.getRelationEvents();
        if (null != relationEvents) {
            List<String> split = Arrays.asList(relationEvents.split(","));
            List<Event> list = eventService.list(new LambdaQueryWrapper<Event>().in(Event::getId, split));
            event.setRelatedEventList(list);
        }
        EventTag one = eventTagService.getOne(new LambdaQueryWrapper<EventTag>()
                .eq(EventTag::getEventId, event.getId())
                .last(" limit 1"));
        event.setEventTag(one);
        return Result.OK(event);
    }


    /**
     * 2.1 专题信息源绑定
     */
    @PostMapping("/infoSourceBind")
    public Object infoSourceBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            ;
            String url = SERVICE_PROJECT_URL + "event/infoSource/infoSourceBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.2 查询信息源组的绑定列表
     */
    @GetMapping("/bindList")
    public Object bindList(InfoSourceVo infoSourceVo,
                           @RequestParam(name = "ynBind") Integer ynBind,
                           @RequestParam(name = "groupId") String groupId,
                           @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                           @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        String url = SERVICE_PROJECT_URL + "event/infoSource/bindList";
        Map<String, String> params = ObjectUtil.objectToMap(infoSourceVo);
        params.put("ynBind", ynBind.toString());
        params.put("groupId", groupId);
        params.put("pageNo", pageNo.toString());
        params.put("pageSize", pageSize.toString());
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.3 关键词类别
     */
    @GetMapping("/keywordsType/rootListNoPage")
    public Object keywordsList(@RequestParam(name = "contain", defaultValue = "false") Boolean contain,
                               @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {
        String url = SERVICE_PROJECT_URL + "event/keywordsType/rootListNoPage";
        Map<String, String> params = new HashMap<>();
        params.put("contain", contain.toString());
        params.put("subjectId", subjectId);
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.4 专题关键词绑定
     */
    @PostMapping("/keyWordsBind")
    public Object keyWordsBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            ;
            String url = SERVICE_PROJECT_URL + "event/keyWordsBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.5 专题关键词绑定
     */
    @PostMapping("/keyWordsEdit")
    public Object keyWordsEdit(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectKeywordsMap);
            ;
            String url = SERVICE_PROJECT_URL + "event/keyWords/edit";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.6 标签查询
     */
    @PostMapping("/label/treeList")
    public Object labelTreeList(@RequestBody SubjectKeywordsMap subjectKeywordsMap) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectKeywordsMap);
            ;
            String url = SERVICE_PROJECT_URL + "event/label/treeList";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.7 专题绑定的搜索引擎列表
     *
     * @return
     */
    @ApiOperation(value = "专题绑定的搜索引擎列表", notes = "专题绑定的搜索引擎列表")
    @GetMapping(value = "/bindSearchEngineList")
    public Object bindSearchEngineList(SearchEnginesVo searchEnginesVo) {
        String url = SERVICE_PROJECT_URL + "event/bindSearchEngineList";
        Map<String, String> params = new HashMap<>();
        params.put("searchEnginesVo", searchEnginesVo.toString());
        params.put("subjectId", searchEnginesVo.getSubjectId());
        params.put("type", searchEnginesVo.getType().toString());
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.8 专题搜索引擎绑定
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "专题搜索引擎绑定", notes = "专题搜索引擎绑定")
    @PostMapping(value = "/searchEngineBind")
    public Object searchEngineBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            ;
            String url = SERVICE_PROJECT_URL + "event/searchEngineBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.9 专题绑定的信息源组列表
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "专题绑定/排除的信息源组列表", notes = "专题绑定/排除的信息源组列表")
    @GetMapping(value = "/bindInfoSourceList")
    public Object bindInfoSourceList(@RequestParam(name = "id") String id) {
        String url = SERVICE_PROJECT_URL + "event/bindInfoSourceList";
        Map<String, String> params = new HashMap<>();
        params.put("id", id);
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.10 专题信息源绑定删除
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "专题信息源绑定删除", notes = "专题信息源绑定删除")
    @PostMapping(value = "/deleteBindInfoSource")
    public Object deleteBindInfoSource(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            ;
            String url = SERVICE_PROJECT_URL + "event/deleteBindInfoSource";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.11 专题与信息源关联编辑
     *
     * @param subjectInfoSourceMap
     * @return
     */

    @ApiOperation(value = "专题与信息源关联表-编辑", notes = "专题与信息源关联表-编辑")
    @PutMapping(value = "/SubjectInfoSourceMap/edit")
    public Result<?> edit(@RequestBody SubjectInfoSourceMap subjectInfoSourceMap) {
        subjectInfoSourceMapService.updateMain(subjectInfoSourceMap);
        return Result.OK("编辑成功!");
    }

    /**
     * 2.12 信息源组类别列表查询
     *
     * @param contain
     * @param subjectId
     * @return
     */
    @ApiOperation(value = "信息源组类别-列表查询", notes = "信息源组类别-列表查询")
    @GetMapping(value = "/groupType/rootListNoPage")
    public Object queryPageListNoPage(@RequestParam(name = "contain", defaultValue = "false") Boolean contain,
                                      @RequestParam(name = "keywords", required = false) String keywords,
                                      @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {
        String url = SERVICE_PROJECT_URL + "event/groupType/rootListNoPage";
        Map<String, String> params = new HashMap<>();
        params.put("contain", contain.toString());
        params.put("keywords", keywords);
        params.put("subjectId", subjectId);
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.13 专题绑定的关键词组列表
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "专题绑定的关键词组列表", notes = "专题绑定的关键词组列表")
    @GetMapping(value = "/bindKeyWordsList")
    public Object bindKeyWordsList(@RequestParam(name = "id") String id) {
        String url = SERVICE_PROJECT_URL + "event/bindKeyWordsList";
        Map<String, String> params = new HashMap<>();
        params.put("id", id);
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.14 删除专题关键词绑定
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "删除专题关键词绑定", notes = "删除专题关键词绑定")
    @PostMapping(value = "/deleteKeyWordsBind")
    public Object deleteKeyWordsBind(@RequestBody SubjectPage subjectPage) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(subjectPage);
            ;
            String url = SERVICE_PROJECT_URL + "event/deleteKeyWordsBind";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.15 关键词管理-分页列表查询
     *
     * @param keyWords
     * @param pageNo
     * @param pageSize
     * @return
     */
    @ApiOperation(value = "关键词管理-分页列表查询", notes = "关键词管理-分页列表查询")
    @GetMapping(value = "/keyWords/listByTypeId")
    public Object queryPageListByTypeId(KeyWords keyWords,
                                        @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                        @RequestParam(name = "wordsName", required = false) String wordsName,
                                        @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                        @RequestParam(name = "keyWordsTypeId", defaultValue = "0") String keyWordsTypeId,
                                        @RequestParam(name = "search", defaultValue = "false") Boolean search,
                                        @RequestParam(name = "subjectId", defaultValue = "0") String subjectId) {

        String url = SERVICE_PROJECT_URL + "event/keyWords/listByTypeId";
        Map<String, String> params = new HashMap<>();
        keyWords.setWordsName(wordsName);
        params.put("wordsName", wordsName);
        params.put("pageNo", pageNo.toString());
        params.put("pageSize", pageSize.toString());
        params.put("keyWordsTypeId", keyWordsTypeId);
        params.put("subjectId", subjectId);
        params.put("search", search.toString());
        return HttpUtil.doGet(url, params, "utf-8");
    }

    /**
     * 2.16 具体标签信息列表-树状结构
     *
     * @return org.jeecg.common.api.vo.Result
     * @author lkg
     * @date 2023/1/6
     */
    @PostMapping("/labelEntityTreeList")
    public Object labelEntityTreeList(@RequestBody SysLabelVo sysLabelVo) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(sysLabelVo);
            ;
            String url = SERVICE_PROJECT_URL + "event/labelEntityTreeList";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 2.17 上传icon
     *
     * @return
     */
    @PostMapping(value = "/upload")
    @ResponseBody
    public Result<?> uploadKnowledge(HttpServletRequest request) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        MultipartFile file = multipartRequest.getFile("file");// 获取上传文件对象
        String orgName = file.getOriginalFilename();// 获取文件名
        String url = eventService.upload(file);
        return Result.OK(url);
    }


    /**
     * 3.1 传播路径
     *
     * @param eventId 事件id
     */
    @GetMapping("/propagationPath")
    public Result propagationPath(@RequestParam String eventId) {
        String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.PROPAGATION_KEY + eventId;
        PropagationPathVo pathVo = (PropagationPathVo) redisUtil.get(key);
        if (ObjectUtils.isEmpty(pathVo)) {
            pathVo = analysisService.propagationPath(eventId);
        }
        return Result.OK(pathVo);
    }

    /**
     * 3.2 事件脉络
     *
     * @param subjectId 专题id
     * @param fakeNum   专题事件脉络展示 伪事件脉络 的资讯数量阈值
     * @return com.zzsn.subjectAnalysis.common.Result
     */
    @GetMapping("/eventContext")
    public Result eventContext(@RequestParam String subjectId,
                               @RequestParam(value = "fakeNum", required = false) Integer fakeNum) {
        if (fakeNum == null) {
            fakeNum = Constants.FAKE_NUM;
        }
        List<SubjectAnalysis> cacheList = analysisService.eventContext(subjectId, fakeNum);
        return Result.OK(cacheList);
    }

    /**
     * 3.3 热词查询
     *
     * @return
     */
    @ApiOperation(value = "热词查询", notes = "热词查询")
    @PostMapping(value = "/keywordsCount")
    public Object keywordsCount(@RequestBody SpecialInformationParam param) {
        try {
            JSONObject params = ObjectUtil.objectToJSONObject(param);
            ;
            String url = SERVICE_PROJECT_URL + "event/keywordsCount";
            return HttpUtil.doPost(url, params, 10000);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 3.4 总体分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @author lkg
     * @date 2024/1/24
     */
    @GetMapping("/total")
    public Result<?> totalAndMax(@RequestParam String subjectId, @RequestParam String startTime, @RequestParam String endTime) {
        Map<String, Object> map = esStatisticsService.totalAndMax(subjectId, null, null);
        long hours = DateUtil.between(DateUtil.parseDateTime(startTime), DateUtil.parseDateTime(endTime), DateUnit.HOUR);
        map.put("duration", hours);
        Object count = map.get("totalCount");
        String divide = CalculateUtil.divide(String.valueOf(count), String.valueOf(hours), 2);
        map.put("spread", divide);
//        String mainReport = esStatisticsService.mainReport(subjectId);
        map.put("mainReport", "mainReport");
        return Result.OK(map);
    }


    /**
     * 3.5 情感判断分析
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/orientation")
    public Result<?> orientation(@RequestParam String subjectId, @RequestParam String startTime,
                                 @RequestParam String endTime, @RequestParam Integer type) {
        List<CountVO> list = new ArrayList<>();
        String labelTypeId = "1631119596744265729";
        List<LabelEntity> labelEntities = labelEntityService.listByType(labelTypeId);
        AtomicLong total = new AtomicLong();
        labelEntities.forEach(e -> {
            CompletableFuture<CountVO> async = CompletableFuture.supplyAsync(() -> {
                CountVO countVO = esStatisticsService.orientation(subjectId, e.getId(), startTime, endTime, type);
                total.addAndGet(countVO.getValue());
                supply(countVO, startTime, endTime, type);
                return countVO;
            });
            try {
                CountVO countVO = async.get();
                list.add(countVO);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        });
        for (CountVO countVO : list) {
            long value = countVO.getValue();
            long totalCount = total.get();
            String divide = CalculateUtil.divide(String.valueOf(value), String.valueOf(totalCount));
            String percentage = "0%";
            if (StringUtils.isNotEmpty(divide)) {
                percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
            }
            countVO.setPercentage(percentage);
        }
        return Result.OK(list);
    }

    /**
     * 3.6 趋势
     *
     * @param subjectId 专题id
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      1-按小时;2-按天
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/flowData")
    public Result<?> flowData(@RequestParam String subjectId, @RequestParam String startTime,
                              @RequestParam String endTime, @RequestParam Integer type) {

        AtomicLong total = new AtomicLong();
        CountVO countVO = esStatisticsService.flowData(subjectId, startTime, endTime, type);
        total.addAndGet(countVO.getValue());
        supply(countVO, startTime, endTime, type);
        long value = countVO.getValue();
        long totalCount = total.get();
        String divide = CalculateUtil.divide(String.valueOf(value), String.valueOf(totalCount));
        String percentage = "0%";
        if (StringUtils.isNotEmpty(divide)) {
            percentage = CalculateUtil.percentage(Double.parseDouble(divide), false);
        }
        countVO.setPercentage(percentage);
        return Result.OK(countVO);
    }

    /**
     * 3.6 平台活跃度占比
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/1/25
     */
    @GetMapping("/origin")
    public Result<?> origin(@RequestParam String subjectId) {
        List<CountVO> list = esStatisticsService.origin(subjectId, null, null);
        return Result.OK(list);
    }

    /**
     * 3.7 事件资讯列表展示
     *
     * @param subjectInfoVo 部分筛选条件封装
     * @param userId        用户id
     * @param video
     * @param pageNo        当前页
     * @param pageSize      返回条数
     * @param column        排序字段
     * @param isCustomer    是否为客户 1是 0 否
     * @param order         排序方式
     * @param crawler
     * @param objectType
     * @param objectId
     * @param isSubject
     * @param subjectType
     * @param labelIds
     * @param sourceId
     * @author lkg
     * @date 2024/1/12
     */
    @ApiOperation(value = "专题信息列表-分页列表查询", notes = "专题信息列表-分页列表查询")
    @PostMapping(value = "/listArticle")
    public Result<?> queryPageList(@RequestBody SubjectInfoVo subjectInfoVo,
                                   @RequestParam(name = "userId", defaultValue = "") String userId,
                                   @RequestParam(name = "video", defaultValue = "") String video,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "isCustomer", defaultValue = "0") Integer isCustomer,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                   @RequestParam(name = "column", defaultValue = "common") String column,
                                   @RequestParam(name = "order", defaultValue = "desc") String order,
                                   @RequestParam(name = "crawler", defaultValue = "") String crawler,
                                   @RequestParam(name = "objectType", defaultValue = "", required = false) String objectType,
                                   @RequestParam(name = "objectId", defaultValue = "", required = false) String objectId,
                                   @RequestParam(name = "isSubject", defaultValue = "1") String isSubject,
                                   @RequestParam(name = "subjectType", defaultValue = "1") String subjectType,
                                   @RequestParam(name = "labelIds", required = false) String labelIds,
                                   @RequestParam(name = "sourceId", required = false) String sourceId) throws Exception {
        List<String> socialCreditCodeList = new ArrayList<>();
        //获取数据
        IPage<DisplayInfo> pageList = subjectDisplayServive.frontListByPage(objectType, objectId, userId, subjectInfoVo, video, pageNo, pageSize, column, order, crawler, isSubject, subjectType, labelIds, socialCreditCodeList, sourceId, isCustomer);
        return Result.OK(pageList);
    }


    /**
     * 补充缺失的时间
     */
    private void supply(CountVO countVO, String startTime, String endTime, Integer type) {
        List<CountVO> list = new ArrayList<>();
        List<CountVO> children = countVO.getChildren();
        Map<String, CountVO> map = children.stream().collect(Collectors.toMap(CountVO::getName, item -> item, (k1, k2) -> k2));
        DateTime startDate = DateUtil.parseDateTime(startTime);
        DateTime endDate = DateUtil.parseDateTime(endTime);
        List<DateTime> rangeToList = new ArrayList<>();
        String format = null;
        if (type == 1) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.HOUR_OF_DAY);
            format = "yyyy-MM-dd HH";
        } else if (type == 2) {
            rangeToList = DateUtil.rangeToList(startDate, endDate, DateField.DAY_OF_YEAR);
            format = "yyyy-MM-dd";
        }
        for (DateTime dateTime : rangeToList) {
            String date = DateUtil.format(dateTime, format);
            if (map.containsKey(date)) {
                list.add(map.get(date));
            } else {
                CountVO vo = new CountVO();
                vo.setName(date);
                vo.setValue(0L);
                list.add(vo);
            }
        }
        countVO.setChildren(list);
    }
}
