package com.zzsn.event.controller.common;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.tree.Tree;
import cn.hutool.core.lang.tree.TreeNode;
import cn.hutool.core.lang.tree.TreeUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.InfoSource;
import com.zzsn.event.entity.InfoSourceGroup;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.feign.api.RemoteSystemService;
import com.zzsn.event.feign.entity.SysDict;
import com.zzsn.event.service.*;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.vo.*;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 专题/事件关系绑定
 *
 * @author lkg
 * @date 2024/12/17
 */
@RestController
@RequestMapping("/common/relation")
public class RelationBindController {

    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private InfoSourceGroupService infoSourceGroupService;
    @Autowired
    private SubjectService subjectService;
    @Autowired
    private SubjectDictMapService subjectDictMapService;
    @Autowired
    private SysDictItemService dictItemService;
    @Autowired
    private RemoteSystemService remoteSystemService;

    /**
     * 项目列表
     *
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/projectList")
    public Result<?> projectList() {
        List<Node> projectList = commonService.projectList();
        return Result.OK(projectList);
    }

    /**
     * 信息源分页列表
     *
     * @param infoSourceCondition 筛选条件
     * @param pageNo              当前页
     * @param pageSize            每页返回条数
     * @author lkg
     * @date 2024/12/26
     */
    @GetMapping("/infoSource/pageList")
    public Result<?> infoSourcePageList(InfoSourceCondition infoSourceCondition,
                                        @RequestParam(defaultValue = "1") Integer pageNo,
                                        @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<InfoSource> pageList = infoSourceService.pageList(infoSourceCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 专题/事件绑定的信息源-列表页
     *
     * @param infoSourceCondition 筛选条件
     * @param subjectId           id/分类id
     * @param category            类别(1-专题;2-事件)
     * @param pageNo              当前页
     * @param pageSize            返回条数
     * @author lkg
     * @date 2024/5/7
     */
    @GetMapping(value = "/bindInfoSourceList")
    public Result<?> bindInfoSourceList(InfoSourceCondition infoSourceCondition,
                                        @RequestParam(name = "subjectId", defaultValue = "0") String subjectId,
                                        @RequestParam(name = "category") Integer category,
                                        @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                        @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<String> subjectIdList = new ArrayList<>();
        if (!"0".equals(subjectId)) {
            //查询类别id的所有明细id
            if (StringUtils.isNotEmpty(subjectId)) {
                List<String> typeIds = subjectTypeService.belowIdList(subjectId, category);
                if (category == 1) {
                    subjectIdList = subjectTypeMapService.selectSubjectByTypeIds(typeIds);
                } else if (category == 2) {
                    subjectIdList = subjectTypeMapService.selectEventByTypeIds(typeIds);
                }
            }
            subjectIdList.add(subjectId);
        }
        //查询
        IPage<InfoSourceVo> pageList = null;

        if (CollectionUtil.isNotEmpty(subjectIdList) && subjectIdList.size() == 1 && subjectId.equals(subjectIdList.get(0))) {
            Subject byId = subjectService.getById(subjectId);
            if (ObjectUtil.isNotNull(byId) && StringUtils.isNotEmpty(byId.getDataScope()) && byId.getDataScope().contains("1")) {
                Page<InfoSource> page = new Page<>(pageNo, pageSize);
                return Result.OK(infoSourceService.page(page,
                        Wrappers.<InfoSource>lambdaQuery()
                                .eq(ObjectUtil.isNotNull(infoSourceCondition.getCrawlType()),InfoSource::getCrawlType, infoSourceCondition.getCrawlType())
                                .eq(StrUtil.isNotBlank(infoSourceCondition.getStatus()),InfoSource::getStatus, infoSourceCondition.getStatus())
                                .like(StrUtil.isNotBlank(infoSourceCondition.getWebSiteName()),InfoSource::getWebSiteName, infoSourceCondition.getWebSiteName())
                                .like(StrUtil.isNotBlank(infoSourceCondition.getSiteName()),InfoSource::getSiteName, infoSourceCondition.getSiteName())
                                .like(StrUtil.isNotBlank(infoSourceCondition.getSiteUri()),InfoSource::getSiteUri, infoSourceCondition.getSiteUri())
                                .orderByDesc(InfoSource::getCreateTime)
                ));
            }else {
                pageList = infoSourceService.bindInfoSourcePageList(infoSourceCondition, subjectIdList, pageNo, pageSize);
            }

        }else {
            pageList = infoSourceService.bindInfoSourcePageList(infoSourceCondition, subjectIdList, pageNo, pageSize);
        }

        return Result.OK(pageList);
    }


    /**
     * 信息源组搜索列表(专题/事件配置页)
     *
     * @param searchWord 搜索词
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/group/searchList")
    public Result<?> searchList(@RequestParam String searchWord) {
        LambdaQueryWrapper<InfoSourceGroup> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.like(InfoSourceGroup::getGroupName, searchWord);
        List<InfoSourceGroup> list = infoSourceGroupService.list(queryWrapper);
        return Result.OK(list);
    }

    /**
     * 信息源组和分类组成的树型结构(专题/事件配置页)
     *
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/groupAndTypeTree")
    public Result<?> groupAndTypeTree() {
        List<GroupTreeVO> tree = infoSourceGroupService.GroupAndTypeTree();
        return Result.OK(tree);
    }
    /**
     * 信息源组和分类组成的树型结构(专题/事件配置页)
     *
     * @author lkg
     * @date 2024/12/23
     */
    @GetMapping("/infourceMainGroupAndTypeTree")
    public Result<?> infourceMainGroupAndTypeTree() {
        List<GroupTreeVO> tree = infoSourceGroupService.infourceMainGroupAndTypeTree();
        return Result.OK(tree);
    }

    /**
     * 专题/事件信息源绑定(专题/事件配置页)
     *
     * @param subjectPage 参数
     * @author lkg
     * @date 2024/12/18
     */
    @PostMapping("/infoSourceBind")
    public Result<?> infoSourceBind(@RequestBody SubjectPage subjectPage) {
        subjectInfoSourceMapService.infoSourceBind(subjectPage);
        return Result.OK();
    }

    /**
     * 专题信息源绑定新
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "专题信息源绑定", notes = "专题信息源绑定")
    @PostMapping(value = "/infoSourceBindNew")
    public Result<?> infoSourceBindNew(@RequestBody SubjectPage subjectPage) {

        return subjectService.infoSourceBindNew(subjectPage);
    }
    /**
     * 专题/事件信息源绑定删除(专题/事件配置页)
     *
     * @param subjectPage 参数
     * @return
     */
    @PostMapping(value = "/deleteBindInfoSource")
    public Result<?> deleteBindInfoSource(@RequestBody SubjectPage subjectPage) {
        subjectInfoSourceMapService.deleteBind(subjectPage);
        return Result.OK("删除成功!");
    }

    /**
     * 专题信息源绑定删除
     *
     * @param subjectPage
     * @return
     */
    @ApiOperation(value = "专题信息源绑定删除", notes = "专题信息源绑定删除")
    @PostMapping(value = "/deleteBindInfoSourceNew")
    public Result<?> deleteBindInfoSourceNew(@RequestBody SubjectPage subjectPage) {
        String subjectId = subjectPage.getId();
        List<String> bindIds = subjectPage.getBindIds();
        if (org.springframework.util.StringUtils.isEmpty(subjectId)) {
            return Result.FAIL("专题id不能为空");
        }
        if (StrUtil.isBlank(subjectPage.getSourceBindType())){
            return Result.FAIL("解绑类型不能为空");
        }
        if (StrUtil.isBlank(subjectPage.getSourceType())){
            return Result.FAIL("信息源类型不能为空");
        }
        if(CollectionUtil.isEmpty(bindIds) && ("1".equals(subjectPage.getSourceType()) || "2".equals(subjectPage.getSourceType()))){
            return Result.FAIL("解绑绑定id不能为空");
        }
        if(CollectionUtil.isEmpty(subjectPage.getBindLabels()) && "3".equals(subjectPage.getSourceType() )){
            return Result.FAIL("解绑标签不能为空");
        }


        subjectService.deleteBindNew(subjectPage);
        return Result.OK("删除成功!");
    }

    /**
     * 专题/事件绑定的信息源组列表(专题/事件配置页)
     *
     * @param subjectId 专题/事件id
     * @author lkg
     * @date 2024/12/18
     */
    @GetMapping(value = "/bindInfoSourceGroupList")
    public Result<?> bindInfoSource(@RequestParam(name = "subjectId") String subjectId) {
        JSONObject infoSourceGroupPages = subjectInfoSourceMapService.bindInfoSource(subjectId);
        return Result.OK(infoSourceGroupPages);
    }


    /**
     * 编辑专题/事件和信息源组的绑定关系
     *
     * @param infoSourceGroupPage 参数
     * @return
     */
    @PostMapping(value = "/modifyInfoSourceInfo")
    public Result<?> modifyInfoSourceInfo(@RequestBody InfoSourceGroupPage infoSourceGroupPage) {
        if (StrUtil.isBlank(infoSourceGroupPage.getSubjectId()) || StrUtil.isBlank(infoSourceGroupPage.getId())
                || ObjectUtil.isEmpty(infoSourceGroupPage.getSourceType())) {
            return Result.OK("请检查参数");
        }
        subjectInfoSourceMapService.modifyInfoSourceInfo(infoSourceGroupPage);
        return Result.OK();
    }

    /**
     * 专题/事件绑定的关键词组分页列表-列表页
     *
     * @param id        专题/事件id/分类id
     * @param category  类别(1-专题;2-事件)
     * @param groupName 词组名称
     * @param wordName  关键词
     * @author lkg
     * @date 2024/12/17
     */
    @GetMapping(value = "/bindKeyWordsList")
    public Result<?> bindKeyWordsList(@RequestParam String id,
                                      @RequestParam Integer category,
                                      @RequestParam(required = false) String groupName,
                                      @RequestParam(required = false) String wordName) {
        List<String> subjectIdList = new ArrayList<>();
        if (!"0".equals(id)) {
            List<String> typeIds = subjectTypeService.belowIdList(id, category);
            if (category == 1) {
                subjectIdList = subjectTypeMapService.selectSubjectByTypeIds(typeIds);
            } else if (category == 2) {
                subjectIdList = subjectTypeMapService.selectEventByTypeIds(typeIds);
            }
        }
        subjectIdList.add(id);
        List<KeyWordsPage> bindKeyWordsList = keyWordsService.bindKeyWordsList(subjectIdList, groupName, wordName);
        return Result.OK(bindKeyWordsList);
    }
    /**
     * 专题可绑定的标签树
     * @author yanxin
     * @date 2025/5/7
     */
    @GetMapping("/bindLabelTreeList")
    public Result<?> bindLabelTreeList() {
        Result<List<Tree<String>>> dictTreeList = remoteSystemService.treeListContainsDict();
        //添加老版本人物、企业标签
        List<SysDict> oldLabelList = commonService.selectOldLabelTypeList();
        if (CollectionUtil.isNotEmpty(oldLabelList)){
            SysDict sysDict = new SysDict();
            sysDict.setId("1");
            sysDict.setParentId("0");
            sysDict.setHasChild("1");
            sysDict.setDictName("旧版标签");
            oldLabelList.add(sysDict);
            List<TreeNode<String>> nodeList = getTreeNodes(oldLabelList);
            List<Tree<String>> treeList = TreeUtil.build(nodeList, "0");
            dictTreeList.getResult().addAll(treeList);
        }
        //人物分类
        List<SysDict> leaderLabelList = commonService.selectLeaderTypeList();
        if (CollectionUtil.isNotEmpty(leaderLabelList)){
            SysDict sysDict = new SysDict();
            sysDict.setId("2");
            sysDict.setParentId("0");
            sysDict.setHasChild("1");
            sysDict.setDictName("人物管理-专家库");
            leaderLabelList.add(sysDict);
            List<TreeNode<String>> nodeList = getTreeNodes(leaderLabelList);
            List<Tree<String>> treeList = TreeUtil.build(nodeList, "0");
            dictTreeList.getResult().addAll(treeList);
        }
        return dictTreeList;
    }
    private List<TreeNode<String>> getTreeNodes(List<SysDict> list) {
        List<TreeNode<String>> nodeList = new ArrayList<>();
        list.forEach(e -> {
            TreeNode<String> treeNode = new TreeNode<>();
            treeNode.setExtra(BeanUtil.beanToMap(e,false,true));
            treeNode.setId(e.getId());
            treeNode.setName(e.getDictName());
            treeNode.setParentId(e.getParentId());
            nodeList.add(treeNode);

        });
        return nodeList;
    }
    /**
     * 专题绑定标签(用于人工打标)
     *
     * @param bindLabelParam 绑定参数
     * @author lkg
     * @date 2025/3/28
     */

    @PostMapping("/bindLabels")
    public Result<?> bindLabels(@RequestBody SubjectBindLabelParam bindLabelParam){
        subjectDictMapService.modify(bindLabelParam);
        return Result.OK();
    }

    /**
     * 专题已经绑定的标签(用于人工打标)
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/3/28
     */
    @GetMapping("/boundLabelList")
    public Result<?> boundLabelList(@RequestParam String subjectId){
        List<DictVO> boundList = subjectDictMapService.boundList(subjectId);
        return Result.OK(boundList);
    }

    /**
     * 专题已经绑定的标签(包含标签值)-树型结构----创建专题报告任务时使用
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/3/28
     */
    @GetMapping("/boundLabelTreeList")
    public Result<?> boundLabelTreeList(@RequestParam String subjectId){
        List<DictTreeVO> list = new ArrayList<>();
        List<DictVO> boundList = subjectDictMapService.boundList(subjectId);
        boundList.forEach(dictVO -> {
            DictTreeVO dictTreeVO = new DictTreeVO();
            BeanUtils.copyProperties(dictVO, dictTreeVO);
            dictTreeVO.setYnItem(false);
            list.add(dictTreeVO);
        });
        List<String> dictIds = boundList.stream().map(DictVO::getId).collect(Collectors.toList());
        List<DictTreeVO> itemTreeList = dictItemService.dictItemTreeList(dictIds);
        Map<String, List<DictTreeVO>> dictMap = itemTreeList.stream().collect(Collectors.groupingBy(DictTreeVO::getDictId));
        for (DictTreeVO dictTreeVO : list) {
            if (dictMap.containsKey(dictTreeVO.getId())) {
                List<DictTreeVO> dictTreeVOS = dictMap.get(dictTreeVO.getId());
                dictTreeVO.setChildren(dictTreeVOS);
            }
        }
        return Result.OK(list);
    }
}
