package com.zzsn.event.service.impl;

import cn.hutool.core.map.MapUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.EventRegionMap;
import com.zzsn.event.entity.EventTag;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.mapper.EventMapper;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CodeGenerateUtil;
import com.zzsn.event.util.CronUtil;
import com.zzsn.event.util.HanlpUtil;
import com.zzsn.event.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.bucket.terms.ParsedStringTerms;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.ParsedTopHits;
import org.elasticsearch.search.aggregations.metrics.TopHitsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.jsoup.Jsoup;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.util.*;

/**
 * @Description: 事件
 * @Author: jeecg-boot
 * @Date: 2024-03-14
 * @Version: V1.0
 */
@Slf4j
@Service
public class EventServiceImpl extends ServiceImpl<EventMapper, Event> implements IEventService {
    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private IKeyWordsService iKeyWordsService;
    @Autowired
    private ISubjectInfoSourceMapService iSubjectInfoSourceMapService;
    @Autowired
    private ISubjectKeywordsMapService iSubjectKeywordsMapService;
    @Autowired
    private IEventTagService eventTagService;
    @Autowired
    private EventRegionMapService eventRegionMapService;

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private EsService esService;


    @Override
    public List<SubjectKafkaVo> progressList(Date disableDate) {
        return this.baseMapper.processList(disableDate);
    }

    @Override
    public List<SubjectKafkaVo> eventSubjectList() {
        return this.baseMapper.eventSubjectList();
    }

    @Override
    public EventVO queryInfo(String eventId) {
        return baseMapper.queryInfo(eventId);
    }

    @Override
    public IPage<EventManageVO> pageList(String eventName, Integer eventType, String startTime, String endTime, String order, String orderType, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        List<EventManageVO> pageList = baseMapper.pageList(eventName, eventType, startTime, endTime, order, orderType, offset, pageSize);
        //获取总条数
        Integer count = baseMapper.totalCount(eventName, eventType, startTime, endTime);
        IPage<EventManageVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public IPage<EventFrontVO> frontPageList(String eventName, Integer eventType,String labelField, String labelName,String order, String orderType, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<EventFrontVO> pageList = baseMapper.frontPageList(eventName, eventType,labelField,labelName,type,order,orderType, offset, pageSize);
        if (CollectionUtils.isNotEmpty(pageList)) {
            //获取专题资讯的首发来源
            Map<String, String> map = getFirstMap(pageList);
            if (MapUtil.isNotEmpty(map)) {
                pageList.forEach(e -> {
                    String firstOrigin = map.get(e.getId());
                    e.setFirstOrigin(firstOrigin);
                });
            }
        }
        //获取总条数
        Integer count = baseMapper.frontTotalCount(eventName, eventType,labelField,labelName,type);
        IPage<EventFrontVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public List<EventTopVO> topEventList(String startTime, String endTime, Integer type, Integer pageSize) {
        return baseMapper.topEventList(startTime, endTime, type, 0, pageSize);
    }

    @Override
    public List<EventRegionVO> listByRegion(Integer type, String name) {
        return baseMapper.listByRegion(type, name);
    }

    private Map<String, String> getFirstMap(List<EventFrontVO> pageList) {
        Map<String, String> map = new HashMap<>();
        List<String> eventIdList = new ArrayList<>();
        pageList.forEach(e -> eventIdList.add(e.getId()));
        SearchRequest searchRequest = new SearchRequest(Constants.ES_DATA_FOR_SUBJECT);
        //根据设备查询设备的相关信息
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.from(0);
        searchSourceBuilder.size(0);
        searchSourceBuilder.sort("publishDate", SortOrder.ASC);
        BoolQueryBuilder boolQueryBuilder = new BoolQueryBuilder();
        boolQueryBuilder.must(QueryBuilders.termsQuery("subjectId.keyword", eventIdList));
        //聚合搜索
        TermsAggregationBuilder one = AggregationBuilders.terms("one").field("subjectId.keyword").size(eventIdList.size());
        //ES分组取每组第一条Java写法
        TopHitsAggregationBuilder topHitsAggregationBuilder = AggregationBuilders.topHits("top_docs")
                .sort("publishDate",SortOrder.ASC)
                .size(1);
        one.subAggregation(topHitsAggregationBuilder);
        searchSourceBuilder.aggregation(one);
        searchSourceBuilder.query(boolQueryBuilder);
        searchRequest.source(searchSourceBuilder);
        // 查询ES
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            ParsedStringTerms oneTerms = searchResponse.getAggregations().get("one");
            List<? extends Terms.Bucket> onebuckets = oneTerms.getBuckets();
            for (Terms.Bucket threebucket : onebuckets) {
                ParsedTopHits topDetail = threebucket.getAggregations().get("top_docs");
                SearchHit[] hits = topDetail.getHits().getHits();
                Map<String, Object> latestDocument = hits[0].getSourceAsMap();
                map.put(latestDocument.get("subjectId").toString(), latestDocument.get("origin").toString());
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return map;
    }

    @Override
    @Transactional
    public Event saveMain(AddEventParam addEventParam) {
        String cron;
        Event event = new Event();
        //事件专题的默认分析规则参数-必填
        BeanUtils.copyProperties(addEventParam, event);
        event.setIncreAnaRule(20);
        event.setTotalAnaRule(50);
        event.setTimeAnaRule(5);
        String subjectCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.SUBJECT_DEFAULT.getValue());
        event.setEventCode(subjectCode);
        cron = CronUtil.getRandomCron();
        event.setCron(cron);
        baseMapper.insert(event);
        String eventId = event.getId();
        eventTagService.save(EventTag.builder().eventId(eventId).build());
        //地缘关系绑定
        List<RegionVO> regionList = addEventParam.getRegionList();
        addMap(eventId, regionList);
        return event;
    }

    @Override
    @Transactional
    public void updateMain(AddEventParam addEventParam) {
        Event subject = new Event();
        BeanUtils.copyProperties(addEventParam, subject);
        String eventId = subject.getId();
        baseMapper.updateById(subject);

        //地域关系处理
        LambdaQueryWrapper<EventRegionMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(EventRegionMap::getEventId, eventId);
        eventRegionMapService.remove(queryWrapper);
        List<RegionVO> regionList = addEventParam.getRegionList();
        addMap(eventId, regionList);
    }

    @Override
    @Transactional
    public void deleteMain(String id) {
        //删除与信息源的关联关系
        iSubjectInfoSourceMapService.delete(id);
        //删除与关键词组的关联关系
        iSubjectKeywordsMapService.delete(id);
        //删除地域关系
        LambdaQueryWrapper<EventRegionMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(EventRegionMap::getEventId, id);
        eventRegionMapService.remove(queryWrapper);
        baseMapper.deleteById(id);
    }

    @Override
    public List<ModelVO> modelList() {
        return baseMapper.modelList();
    }

    @Override
    public List<EventVO> eventList(List<String> eventIdList) {
        return baseMapper.eventList(eventIdList);
    }

    @Override
    public List<EventExcelVO> frontList(List<String> eventIdList,Integer size) {
        return baseMapper.frontList(eventIdList,size);
    }

    @Override
    public List<StatisticsKeyWordVo> hotWords(String index,String id, Integer number) {
        SubjectDataVo subjectDataVo = esService.queryInfo(index, id);
        String content = subjectDataVo.getContent();
        List<Map.Entry<String, Integer>> keywordsList = HanlpUtil.extractKeyWordsByText(Jsoup.parse(content).text(), number);
        List<StatisticsKeyWordVo> rn = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(keywordsList)) {
            for (Map.Entry<String, Integer> entry : keywordsList) {
                StatisticsKeyWordVo statisticsKeyWordVo = new StatisticsKeyWordVo();
                statisticsKeyWordVo.setName(entry.getKey());
                statisticsKeyWordVo.setValue(entry.getValue());
                rn.add(statisticsKeyWordVo);
            }
        }
        return rn;
    }

    private void addMap(String eventId, List<RegionVO> regionList) {
        if (CollectionUtils.isNotEmpty(regionList)) {
            List<EventRegionMap> dataList = new ArrayList<>();
            for (RegionVO regionVO : regionList) {
                EventRegionMap eventRegionMap = new EventRegionMap();
                eventRegionMap.setEventId(eventId);
                eventRegionMap.setRegionId(regionVO.getId());
                eventRegionMap.setTopRegionId(regionVO.getTopId());
                eventRegionMap.setType(regionVO.getType());
                eventRegionMap.setCreateTime(new Date());
                dataList.add(eventRegionMap);
            }
            eventRegionMapService.saveBatch(dataList);
        }
    }

}
