package com.zzsn.event.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zzsn.clb.common.model.task.dto.titr.KeyWordsDTO;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.*;
import com.zzsn.event.entity.arrange.ClbModelArrange;
import com.zzsn.event.enums.BindTypeEnum;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.enums.SourceTypeEnum;
import com.zzsn.event.mapper.SubjectMapper;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CodeGenerateUtil;
import com.zzsn.event.util.CronUtil;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.IOException;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @Description: 专题表
 * @Author: jeecg-boot
 * @Date: 2021-12-01
 * @Version: V1.0
 */
@Service
@Slf4j
public class SubjectServiceImpl extends ServiceImpl<SubjectMapper, Subject> implements SubjectService {

    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private RedisUtil redisUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private IProjectSubjectMapService projectSubjectMapService;
    @Autowired
    private ISubjectSearchEnginesMapService subjectSearchEnginesMapService;
    @Autowired
    private ISubjectKeywordsMapService subjectKeywordsMapService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private ISubjectModelMapService subjectModelMapService;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private ClbSubjectModelDetailTaskService subjectModelDetailTaskService;
    @Autowired
    private IAlgorithmModelService modelService;
    @Autowired
    private ClbModelArrangeSubjectMapService clbModelArrangeSubjectMapService;
    @Autowired
    private IClbModelArrangeService clbModelArrangeService;

    @Value("${python.subjectProcessorUrl}")
    private String subjectProcessorUrl;
    @Value("${clb.subject.default.processing.advanceMonth}")
    private Integer defaultAdvanceMonth;


    @Override
    public Page<SubjectPage> pageList(SubjectCondition subjectCondition, Integer pageNo, Integer pageSize) {
        //查询类别id的所有明细id
        List<String> typeIds = new ArrayList<>();
        String subjectTypeId = subjectCondition.getSubjectTypeId();
        if (StringUtils.isNotEmpty(subjectTypeId) && !"0".equals(subjectTypeId)) {
            typeIds = subjectTypeService.belowIdList(subjectTypeId, 1);
        }
        subjectCondition.setTypeIds(typeIds);
        Page<SubjectPage> page = new Page<>(pageNo, pageSize);
        Page<SubjectPage> pageList = baseMapper.pageList(subjectCondition, page);
        List<SubjectPage> records = pageList.getRecords();
        if (CollUtil.isNotEmpty(records)) {
            List<String> idList = records.stream().map(SubjectPage::getId).collect(Collectors.toList());

            //按专题分组，统计绑定关键词组的数量
            Map<String, Integer> keyWordsNumMap = commonService.bindKeyWordsCountList(idList).stream().collect(Collectors.toMap(SubjectPage::getId, SubjectPage::getKeyWordsNum));
            //按专题分组，统计绑定信息源的数量
            //Map<String, List<SubjectSourceVO>> collect = commonService.bindSourceList(idList).stream().collect(Collectors.groupingBy(SubjectSourceVO::getSubjectId));
            Map<String, Integer> sourceCountMap = commonService.bindSourceCount(idList);
//            Map<String, Integer> infoSourceNumMap = new HashMap<>();
//            for (Map.Entry<String, List<SubjectSourceVO>> entry : collect.entrySet()) {
//                String subjectId = entry.getKey();
//                List<SubjectSourceVO> value = entry.getValue();
//                infoSourceNumMap.put(subjectId, value.size());
//            }
            //查询每个专题绑定的信息源数量及关键词数量
            for (SubjectPage subjectPage1 : records) {
                int keyWordsNum = null == keyWordsNumMap.get(subjectPage1.getId()) ? 0 : keyWordsNumMap.get(subjectPage1.getId());
                int infoSourceNum = null == sourceCountMap.get(subjectPage1.getId()) ? 0 : sourceCountMap.get(subjectPage1.getId());
                subjectPage1.setInfoSourceNum(infoSourceNum);
                subjectPage1.setKeyWordsNum(keyWordsNum);
            }

            //查询绑定的流程
            List<ClbModelArrange> clbModelArranges = clbModelArrangeService.listBySubjectIds(idList);
            if (CollectionUtil.isNotEmpty(clbModelArranges)){
                clbModelArranges.forEach(e -> e.setType(SourceTypeEnum.getValueNameByValue(e.getType())));
                Map<String, List<ClbModelArrange>> collect = clbModelArranges.stream().collect(Collectors.groupingBy(ClbModelArrange::getSubjectId));
                records.forEach(e -> {
                    e.setClbModelArranges(collect.get(e.getId()));
                });
            }
        }
        return pageList;
    }

    @Override
    public Page<SubjectPage> researchCenterPageList(SubjectCondition subjectCondition, Integer pageNo, Integer pageSize) {
        Integer facePublic = subjectCondition.getFacePublic();
        Page<SubjectPage> page = new Page<>(pageNo, pageSize);
        if (facePublic != null && facePublic == 1) {
            page = baseMapper.researchCenterFacePageList(subjectCondition, page);
        } else {
            //查询类别id的所有明细id
            List<String> typeIds = new ArrayList<>();
            String subjectTypeId = subjectCondition.getSubjectTypeId();
            if (StringUtils.isNotEmpty(subjectTypeId) && !"0".equals(subjectTypeId)) {
                typeIds = subjectTypeService.researchCenterBelowIdList(subjectTypeId, 1);
            }
            subjectCondition.setTypeIds(typeIds);
            UserVo loginUser = UserUtil.getLoginUser();
            subjectCondition.setUsername(loginUser.getUsername());
            page = baseMapper.researchCenterPageList(subjectCondition, page);
        }
        return page;
    }

    @Override
    public Page<String> visiblePageList(String username,String subjectName, Integer pageNo, Integer pageSize) {
        Page<String> page = new Page<>(pageNo, pageSize);
        return baseMapper.visibleList(username,subjectName, page);
    }

    @Override
    public SubjectDetailVO queryInfo(String subjectId) {
        return baseMapper.queryInfo(subjectId);
    }

    @Override
    @Transactional
    public Subject saveMain(SubjectPage subjectPage) {
        Subject subject = new Subject();
        //事件专题的默认分析规则参数-必填
        BeanUtils.copyProperties(subjectPage, subject);
        String subjectCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.SUBJECT_DEFAULT.getValue());
        subject.setSubjectCode(subjectCode);
        String cron = CronUtil.generateCron(subject.getUnit(), subject.getSpace());
        subject.setCron(cron);
        subject.setEstimateStatus("未启用");
        baseMapper.insert(subject);
        //插入专题-类别、项目的绑定关系
        saveMapMain(subject, subjectPage);
        return subject;
    }

    @Override
    @Transactional
    public void updateMain(SubjectPage subjectPage) {
        Subject oldSubject = this.getById(subjectPage.getId());
        Subject subject = new Subject();
        BeanUtils.copyProperties(subjectPage, subject);
        String cron = CronUtil.generateCron(subject.getUnit(), subject.getSpace());
        subject.setCron(cron);
        baseMapper.updateById(subject);
        //删除专题-类别绑定关系
        subjectTypeMapService.deleteBySubjectId(subject.getId());
        //删除专题-项目的绑定关系
        projectSubjectMapService.deleteBySubjectId(subject.getId());
        //插入新的
        saveMapMain(subject, subjectPage);
        //修改redis缓存，，用于向专题里补充数据
        updateRedisCache(subject, oldSubject);
    }

    @Override
    public void updateStatus(Subject subject) {
        LambdaUpdateWrapper<Subject> updateWrapper = Wrappers.lambdaUpdate();
        String subjectId = subject.getId();
        Integer status = subject.getStatus();
        updateWrapper.set(Subject::getStatus, status).eq(Subject::getId, subjectId);
        if (subject.getStatus() == 1) {
            //判断是否第一次启用
            Subject byId = this.getById(subjectId);
            if (byId.getFirstOpenTime() == null) {
                updateWrapper.set(Subject::getFirstOpenTime, new Date());
                updateWrapper.set(Subject::getEstimateStatus, "预估中");

                //第一次启用时，添加redis缓存，，用于向专题里进数据
                setRedisCache(subjectId);
            }
        }
        this.update(updateWrapper);
        //和python沟通，这块没用了
        /*CompletableFuture.runAsync(()->{
            if (status == 1) {
                send(subject.getId(), "1");
            } else if (status == 0) {
                //向python发起停止处理请求
                send(subject.getId(), "0");
            }
        });*/
    }

    @Override
    public void deleteMain(String subjectId) {
        baseMapper.deleteById(subjectId);
        CompletableFuture.runAsync(() -> {
            //删除与类别的映射
            subjectTypeMapService.deleteBySubjectId(subjectId);
            //删除与信息源的关联关系
            subjectInfoSourceMapService.delete(subjectId);
            //删除与关键词组的关联关系
            subjectKeywordsMapService.delete(subjectId);
            //删除专题-项目的绑定关系
            projectSubjectMapService.deleteBySubjectId(subjectId);
            //删除事件-模型关系
            subjectModelMapService.remove(Wrappers.<SubjectModelMap>lambdaQuery().eq(SubjectModelMap::getSubjectId, subjectId));
            //删除事件-搜索引擎关系
            subjectSearchEnginesMapService.remove(Wrappers.<SubjectSearchEnginesMap>lambdaQuery().eq(SubjectSearchEnginesMap::getSubjectId, subjectId));
            //删除专题/事件-tpu流程关系
            clbModelArrangeSubjectMapService.remove(Wrappers.<ClbModelArrangeSubjectMap>lambdaQuery().eq(ClbModelArrangeSubjectMap::getSubjectId, subjectId));
            //向python发送消息结束
            //send(subjectId, "-1");
        });
    }

    @Override
    public SubjectPage getSubjectById(String subjectId) {
        SubjectPage subjectPage = baseMapper.selectSubjectById(subjectId);
        int keyWordsNum = keyWordsService.bindCount(subjectPage.getId());
        List<String> subjectIds = new ArrayList<>();
        subjectIds.add(subjectPage.getId());
        int infoSourceNum = infoSourceService.bindSourceCount(subjectIds);
        subjectPage.setInfoSourceNum(infoSourceNum);
        subjectPage.setKeyWordsNum(keyWordsNum);
        return subjectPage;
    }

    @Override
    public void directBindInfoSource(DirectBindInfoVO directBindInfoVO) {
        String subjectId = directBindInfoVO.getSubjectId();
        List<String> bindList = directBindInfoVO.getBindList();
        Integer type = BindTypeEnum.INFO_SOURCE.getvalue();
        for (String infoSourceId : bindList) {
            /*
             * 1.先删除该信息源在专题下的所有数据
             * 2.再判断该信息源在专题已绑定的信息源组下，若不在则新增绑定关系；若在则跳过
             */
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId);
            subjectInfoSourceMapService.remove(queryWrapper);
            int num = baseMapper.ynBelowBindGroup(subjectId, infoSourceId);
            if (num == 0) {
                SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                subjectInfoSourceMap.setSourceId(infoSourceId);
                subjectInfoSourceMap.setSubjectId(subjectId);
                subjectInfoSourceMap.setType(type);
                subjectInfoSourceMapService.save(subjectInfoSourceMap);
            }
        }
    }

    @Override
    public void directRemoveInfoSource(DirectBindInfoVO directBindInfoVO) {
        String subjectId = directBindInfoVO.getSubjectId();
        List<String> excludeList = directBindInfoVO.getExcludeList();
        for (String infoSourceId : excludeList) {
            /*
             * 1.先删除该信息源在专题下的所有数据
             * 2.再判断该信息源在专题已排除的信息源组下，若不在则新增绑定关系；若在则跳过
             */
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId);
            subjectInfoSourceMapService.remove(queryWrapper);
            int num = baseMapper.ynBelowExcludeGroup(subjectId, infoSourceId);
            if (num == 0) {
                SubjectInfoSourceMap subjectInfoSourceMap = new SubjectInfoSourceMap();
                subjectInfoSourceMap.setSourceId(infoSourceId);
                subjectInfoSourceMap.setSubjectId(subjectId);
                subjectInfoSourceMap.setType(BindTypeEnum.EXCLUDE_INFO_SOURCE.getvalue());
                subjectInfoSourceMapService.save(subjectInfoSourceMap);
            }
        }
    }

    @Override
    public void removeUnBindInfoSource(String subjectId, String infoSourceId) {
        int num = baseMapper.ynBelowExcludeGroup(subjectId, infoSourceId);
        if (num == 0) {
            LambdaUpdateWrapper<SubjectInfoSourceMap> update = Wrappers.lambdaUpdate();
            subjectInfoSourceMapService.update(update.eq(SubjectInfoSourceMap::getSubjectId, subjectId)
                    .eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getType, 3)
                    .set(SubjectInfoSourceMap::getType, 1));
        } else {
            LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectInfoSourceMap::getSourceId, infoSourceId)
                    .eq(SubjectInfoSourceMap::getSubjectId, subjectId)
                    .eq(SubjectInfoSourceMap::getType, 3);
            subjectInfoSourceMapService.remove(queryWrapper);
        }
    }

    @Override
    public void keyWordsBind(SubjectPage subjectPage) {
        List<String> idList = subjectPage.getKeyWordsIds();
        List<SubjectKeywordsMap> mapList = new ArrayList<>();
        if (idList != null && !idList.isEmpty()) {
            for (String keyWordsId : idList) {
                SubjectKeywordsMap subjectKeywordsMap = new SubjectKeywordsMap();
                subjectKeywordsMap.setKeywordsId(keyWordsId);
                subjectKeywordsMap.setSubjectId(subjectPage.getId());
                subjectKeywordsMap.setType(subjectPage.getType());
                subjectKeywordsMap.setBindingType(subjectPage.getBindingType());
                mapList.add(subjectKeywordsMap);
            }
            subjectKeywordsMapService.saveBatch(mapList);
        }
    }

    @Override
    public void deleteBindKeyWords(SubjectPage subjectPage) {
        List<String> deleteKeyWordsIds = subjectPage.getKeyWordsIds();
        if (deleteKeyWordsIds != null && !deleteKeyWordsIds.isEmpty()) {
            subjectKeywordsMapService.remove(Wrappers.<SubjectKeywordsMap>lambdaQuery()
                    .eq(SubjectKeywordsMap::getSubjectId, subjectPage.getId()).in(SubjectKeywordsMap::getKeywordsId, deleteKeyWordsIds)
                    .eq(StrUtil.isNotBlank(subjectPage.getBindingType()), SubjectKeywordsMap::getBindingType, subjectPage.getBindingType())
            );
            //清除redis里面的绑定关系（包括专题信息，以及搜索引擎）
            for (String keyWordsId : deleteKeyWordsIds) {
                KeyWordsPage keyWordsPage = keyWordsService.getKeyWordsById(keyWordsId);
                redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWordsPage.getWordsCode());
            }
        }
    }

    @Override
    public void searchEngineBind(SubjectPage subjectPage) {
        //先删除后绑定
        subjectSearchEnginesMapService.remove(Wrappers.<SubjectSearchEnginesMap>lambdaQuery().eq(SubjectSearchEnginesMap::getSubjectId, subjectPage.getId()));
        List<String> idList = subjectPage.getSearchEngineIds();
        List<SubjectSearchEnginesMap> mapList = new ArrayList<>();
        if (CollUtil.isNotEmpty(idList)) {
            for (String searchEngineId : idList) {
                SubjectSearchEnginesMap subjectSearchEnginesMap = new SubjectSearchEnginesMap();
                subjectSearchEnginesMap.setSearchEngineId(searchEngineId);
                subjectSearchEnginesMap.setSubjectId(subjectPage.getId());
                mapList.add(subjectSearchEnginesMap);
            }
            subjectSearchEnginesMapService.saveBatch(mapList);
        }
    }

    @Override
    public List<SearchEnginesVo> bindSearchEngineList(SearchEnginesVo searchEnginesVo) {
        List<SearchEnginesVo> searchEnginesVoList = subjectSearchEnginesMapService.bindSearchEngineList(searchEnginesVo);
        for (SearchEnginesVo searchEnginesVo1 : searchEnginesVoList) {
            if (StringUtils.isNotEmpty(searchEnginesVo1.getSubjectId())) {
                searchEnginesVo1.setYn("1");
            }
        }
        return searchEnginesVoList;
    }

    @Override
    public Result<?> modelBind(SubjectPage subjectPage) {
        List<SubjectModelMap> mapList = new ArrayList<>();
        List<String> hisModelids = new ArrayList<>();
        List<PythonModelVo> models = subjectPage.getModels();
        if (CollUtil.isEmpty(models)) {
            return Result.FAIL(500, "请至少配置一个模型");
        }
        for (PythonModelVo model : models) {
            Integer detailType = model.getDetailType();
            if (ObjectUtil.isEmpty(detailType)) {
                return Result.FAIL(501, "请勾选" + model.getModelName() + "的作用范围");
            }
        }
        if (CollUtil.isNotEmpty(models)) {
            List<PythonModelVo> zengLiang = models.stream().filter(f -> ObjectUtil.isNotEmpty(f.getDetailType()) && (f.getDetailType() == 0 || f.getDetailType() == 2)).collect(Collectors.toList());
            List<PythonModelVo> liShi = models.stream().filter(f -> ObjectUtil.isNotEmpty(f.getDetailType()) && (f.getDetailType() == 1 || f.getDetailType() == 2)).collect(Collectors.toList());
            //检查是否有正在处理的模型任务
            List<ClbSubjectModelDetailTask> list = subjectModelDetailTaskService.list(Wrappers.<ClbSubjectModelDetailTask>lambdaQuery()
                    .eq(ClbSubjectModelDetailTask::getSubjectId, subjectPage.getId()).eq(ClbSubjectModelDetailTask::getTaskStatus, 1)
            );

            if (CollUtil.isNotEmpty(list)) {
                //有正在处理的任务
                for (ClbSubjectModelDetailTask clbSubjectModelDetailTask : list) {
                    String modelIds = clbSubjectModelDetailTask.getModelIds();
                    List<String> list1 = Arrays.asList(modelIds.split(","));
                    for (PythonModelVo model : liShi) {
                        if (list1.contains(model.getId())) {
                            return Result.FAIL(201, JSONUtil.toJsonStr(clbSubjectModelDetailTask));
                        }
                    }
                }
            }
            List<SubjectModelMap> bind = subjectModelMapService.list(Wrappers.<SubjectModelMap>lambdaQuery()
                    .select(SubjectModelMap::getModelId)
                    .eq(SubjectModelMap::getSubjectId, subjectPage.getId())
            );
            //判断提取任务模型中的增量和全量数据，插入关系配置表
            if (CollectionUtil.isNotEmpty(zengLiang)) {
                if (CollectionUtil.isNotEmpty(bind)) {
                    List<PythonModelVo> saves = zengLiang.stream().filter(f -> !bind.stream().map(SubjectModelMap::getModelId).collect(Collectors.toList()).contains(f.getId())).collect(Collectors.toList());
                    if (CollectionUtil.isNotEmpty(saves)) {
                        saves.forEach(e -> {
                            SubjectModelMap subjectModelMap = new SubjectModelMap();
                            subjectModelMap.setModelId(e.getId());
                            subjectModelMap.setSubjectId(subjectPage.getId());
                            subjectModelMap.setType(e.getType());
                            subjectModelMap.setSign(1);
                            mapList.add(subjectModelMap);
                        });
                        subjectModelMapService.saveBatch(mapList);
                    }
                } else {
                    zengLiang.forEach(e -> {
                        SubjectModelMap subjectModelMap = new SubjectModelMap();
                        subjectModelMap.setModelId(e.getId());
                        subjectModelMap.setSubjectId(subjectPage.getId());
                        subjectModelMap.setType(e.getType());
                        subjectModelMap.setSign(1);
                        mapList.add(subjectModelMap);
                    });
                    subjectModelMapService.saveBatch(mapList);
                }

            }

            if (CollectionUtil.isNotEmpty(liShi)) {
                liShi.forEach(e -> {
                    if (ObjectUtil.isNotEmpty(e.getDetailType()) && e.getDetailType() != 0) {
                        hisModelids.add(e.getId());
                    }
                });
            }

        }
        //模型中如有历史数据处理任务则发起任务
        if (CollectionUtil.isNotEmpty(hisModelids)) {
            List<AlgorithmModel> modelInfos = modelService.list(Wrappers.<AlgorithmModel>lambdaQuery()
                    .select(AlgorithmModel::getId, AlgorithmModel::getType, AlgorithmModel::getName, AlgorithmModel::getServiceName)
                    .in(AlgorithmModel::getId, hisModelids));
            //处理历史
            ClbSubjectModelDetailTask task = new ClbSubjectModelDetailTask();
            task.setSubjectId(subjectPage.getId());
            task.setTaskStatus(0);
            task.setModelIds(String.join(",", hisModelids));

            Date now = new Date();
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(now);
            // 添加半小时
            calendar.add(Calendar.MINUTE, 30);
            Date halfHourLater = calendar.getTime();

            task.setDetailToTime(halfHourLater);
            if (ObjectUtil.isNotEmpty(subjectPage.getTimeEnable())) {
                task.setDetailFromTime(subjectPage.getTimeEnable());
            }
            if (ObjectUtil.isNotEmpty(subjectPage.getTimeDisable())) {
                task.setDetailToTime(subjectPage.getTimeDisable());
            }
            task.setTaskCode(subjectPage.getTaskCode());
            task.setTaskParam(JSONUtil.toJsonStr(modelInfos));
            task.setAllParam(JSONUtil.toJsonStr(subjectPage));
            subjectModelDetailTaskService.save(task);
            kafkaTemplate.send("subjectModelDetail", JSONUtil.toJsonStr(task));
            return Result.OK(task);
        }
        return Result.OK();
    }

    private Map<String, Integer> codeMap = new HashMap<>();

    @Override
    public Result<?> checkIsBreak(SubjectPage subjectPage) {
        List<PythonModelVo> models = subjectPage.getModels();
        if (CollectionUtil.isEmpty(models)) {
            return Result.FAIL(500, "请至少配置一个模型");
        }
        for (PythonModelVo model : models) {
            Integer detailType = model.getDetailType();
            if (ObjectUtil.isEmpty(detailType)) {
                return Result.FAIL(501, "请勾选" + model.getModelName() + "的作用范围");
            }
        }
        List<PythonModelVo> liShi = models.stream().filter(f -> ObjectUtil.isNotEmpty(f.getDetailType()) && (f.getDetailType() == 1 || f.getDetailType() == 2)).collect(Collectors.toList());
        //检查是否有正在处理的模型任务
        List<ClbSubjectModelDetailTask> list = subjectModelDetailTaskService.list(Wrappers.<ClbSubjectModelDetailTask>lambdaQuery()
                .eq(ClbSubjectModelDetailTask::getSubjectId, subjectPage.getId()).eq(ClbSubjectModelDetailTask::getTaskStatus, 1)
        );

        String code = "";
        String s = cn.hutool.core.date.DateUtil.formatDate(new Date());
        if (ObjectUtil.isEmpty(codeMap.get(s))) {
            codeMap = new HashMap<>();
            codeMap.put(s, 1);
            code = s + " 00" + 1;
        } else {
            codeMap.put(s, codeMap.get(s) + 1);
            code = s + " 00" + +codeMap.get(s);
        }
        if (CollectionUtil.isNotEmpty(list)) {
            //有正在处理的任务
            for (ClbSubjectModelDetailTask clbSubjectModelDetailTask : list) {
                String modelIds = clbSubjectModelDetailTask.getModelIds();
                List<String> list1 = Arrays.asList(modelIds.split(","));
                for (PythonModelVo model : liShi) {
                    if (list1.contains(model.getId())) {
                        Result<Object> error = Result.FAIL(201, "此次任务中的模型处理，有部分模型已经存在于其他任务中，并且未执行完毕");
                        error.setResult(clbSubjectModelDetailTask);
                        return error;
                    }
                }
            }
        }
        ClbSubjectModelDetailTask clbSubjectModelDetailTask = new ClbSubjectModelDetailTask();
        clbSubjectModelDetailTask.setTaskCode(code);
        return Result.OK(clbSubjectModelDetailTask);
    }

    @Override
    public void send(String subjectId, String status) {
        Subject subjectById = this.getById(subjectId);
        if (ObjectUtil.isEmpty(subjectById) || subjectById.getCreateTime().after(DateTime.of("2023-11-30", "yyyy-MM-dd"))) {
            return;
        }
        JSONObject jsonObject = commonService.siteInfo(subjectId, null, 1);
        if ("1".equals(status)) {
            jsonObject.put("subjectStatus", "1");
        } else if ("0".equals(status)) {
            jsonObject.put("subjectStatus", "0");
        } else if ("-1".equals(status)) {
            jsonObject.put("subjectStatus", "-1");
        }
        jsonObject.put("subjectStatusDetails", "");
        //调用python接口
        try {
            Map<String, String> headers = new HashMap<>();
            headers.put("Content-Type", "application/json;charset=UTF-8");
            headers.put("Accept", "application/json");
            headers.put("Authorization", "!0gwY$5S@5V&A_+XEu)");
            HttpUtil.doPostWithHeader(subjectProcessorUrl, jsonObject, 3000, headers);
        } catch (IOException e) {
            log.error("传递专题配置信息异常!{}", e.getMessage(), e);
        }
    }

    private void saveMapMain(Subject subject, SubjectPage subjectPage) {
        if (StringUtils.isNotEmpty(subjectPage.getSubjectTypeId())) {
            SubjectTypeMap subjectTypeMap = new SubjectTypeMap();
            subjectTypeMap.setSubjectId(subject.getId());
            subjectTypeMap.setSysOrgCode(subject.getSysOrgCode());
            subjectTypeMap.setTypeId(subjectPage.getSubjectTypeId());
            subjectTypeMapService.save(subjectTypeMap);
        }
        if (StringUtils.isNotEmpty(subjectPage.getProjectId())) {
            ProjectSubjectMap projectSubjectMap = new ProjectSubjectMap();
            projectSubjectMap.setProjectId(subjectPage.getProjectId());
            projectSubjectMap.setSubjectId(subject.getId());
            projectSubjectMapService.save(projectSubjectMap);
        }
    }

    /**
     * 将专题的时间范围存入redis缓存(专题第一次启用时生效)
     *
     * @param subjectId 专题
     * @author lkg
     * @date 2025/2/7
     */
    private void setRedisCache(String subjectId) {
        Subject subject = this.getById(subjectId);
        Date timeEnable = subject.getTimeEnable();
        Date timeDisable = subject.getTimeDisable();
        //date 转 localdate
        LocalDate start;
        LocalDate end;
        if (ObjectUtil.isNull(timeEnable)) {
            start = LocalDate.now().minusMonths(defaultAdvanceMonth);
        } else {
            start = timeEnable.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        }
        if (ObjectUtil.isNull(timeDisable)) {
            end = LocalDate.now();
        } else {
            end = timeDisable.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        }
        //循环 start 到 end 的每一天
        List<String> dateList = new ArrayList<>();
        for (LocalDate date = start; !date.isAfter(end); date = date.plusDays(1)) {
            // 在这里处理每一天的逻辑
            //格式化date成字符串
            String format = date.format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            dateList.add(format);
        }
        redisUtil.rpushMultipleValues(Constants.HISTORY_DATE_QUEUE + subject.getSubjectCode(), dateList.toArray(new String[0]));
    }

    /**
     * 修改专题时间缓存，用于补充数据（只有启用过的专题才会起作用）
     *
     * @param subject    新专题信息
     * @param oldSubject 旧专题信息
     * @author lkg
     * @date 2025/2/7
     */
    private void updateRedisCache(Subject subject, Subject oldSubject) {
        Date firstOpenTime = oldSubject.getFirstOpenTime();
        if (firstOpenTime != null) {
            Date oldTimeEnable = oldSubject.getTimeEnable();
            Date oldTimeDisable = oldSubject.getTimeDisable();
            LocalDate oldStart;
            LocalDate oldEnd;
            if (ObjectUtil.isNull(oldTimeEnable)) {
                oldStart = firstOpenTime.toInstant().atZone(ZoneId.systemDefault()).toLocalDate().minusMonths(defaultAdvanceMonth);
            } else {
                oldStart = oldTimeEnable.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            }
            if (ObjectUtil.isNull(oldTimeDisable)) {
                oldEnd = LocalDate.now();
            } else {
                oldEnd = oldTimeDisable.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            }
            Date timeEnable = subject.getTimeEnable();
            Date timeDisable = subject.getTimeDisable();
            if (timeEnable != null || timeDisable != null) {
                List<String> newDateList = new ArrayList<>();
                if (timeEnable != null) {
                    LocalDate start = timeEnable.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
                    if (start.isBefore(oldStart)) {
                        for (LocalDate date = start; date.isBefore(oldStart); date = date.plusDays(1)) {
                            // 在这里处理每一天的逻辑
                            //格式化date成字符串
                            String format = date.format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
                            newDateList.add(format);
                        }
                    }
                }
                if (timeDisable != null) {
                    LocalDate end = timeDisable.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
                    if (end.isAfter(oldEnd)) {
                        for (LocalDate date = oldEnd.plusDays(1); !date.isAfter(end); date = date.plusDays(1)) {
                            // 在这里处理每一天的逻辑
                            //格式化date成字符串
                            String format = date.format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
                            newDateList.add(format);
                        }
                    }
                }
                if (CollectionUtils.isNotEmpty(newDateList)) {
                    redisUtil.rpushMultipleValues(Constants.HISTORY_DATE_QUEUE + subject.getSubjectCode(), newDateList.toArray(new String[0]));
                }
            }
        }
    }
}

