package com.zzsn.event.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.zzsn.event.entity.Event;
import com.zzsn.event.service.EsStatisticsService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.vo.EventRegionVO;
import com.zzsn.event.vo.EventTopVO;
import com.zzsn.event.constant.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 事件首页
 *
 * @author lkg
 * @date 2024/4/8
 */
@RestController
@RequestMapping("/home")
public class EventHomeController {

    @Autowired
    private EsStatisticsService esStatisticsService;
    @Autowired
    private IEventService eventService;


    /**
     * 事件数据统计-首页
     *
     * @author lkg
     * @date 2024/1/12
     */
    @GetMapping(value = "/eventData")
    public Result<?> eventData() {
        Map<String, Integer> map = new HashMap<>();
        List<Event> list = eventService.list(new LambdaQueryWrapper<Event>()
                .eq(Event::getFacePublic,1)
                .eq(Event::getPublishStatus,1));
        map.put("eventCount", list.size());
        List<String> eventIdList = list.stream().map(Event::getId).collect(Collectors.toList());
        long total = esStatisticsService.totalCount(eventIdList, null, null);
        map.put("total", (int) total);
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime previousHour = now.minus(1, ChronoUnit.DAYS);
        long totalAdd = esStatisticsService.totalCount(eventIdList, previousHour.format(formatter), now.format(formatter));
        map.put("totalAdd", (int) totalAdd);
        return Result.OK(map);
    }


    /**
     * 热点事件-前10
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param type      分类(1-按发布时间;2.按热度)
     * @param size      返回个数
     * @author lkg
     * @date 2024/4/8
     */
    @GetMapping("/hotEventFlow")
    public Result<?> hotEventFlow(@RequestParam(required = false) String startTime,
                                  @RequestParam(required = false) String endTime,
                                  @RequestParam Integer type,
                                  @RequestParam(defaultValue = "10") Integer size) {
        List<EventTopVO> list = eventService.topEventList(startTime, endTime, type, size);
        return Result.OK(list);
    }

    /**
     * 事件按地域分析
     *
     * @param type       地域类型(1-国际;2-国内)
     * @param regionName 地域名称
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping("/region")
    public Result<?> region(@RequestParam Integer type, @RequestParam(required = false) String regionName) {
        List<Map<String, Object>> dataList = new ArrayList<>();
        List<EventRegionVO> eventRegionVOList = eventService.listByRegion(type, regionName);
        Map<String, List<EventRegionVO>> map = eventRegionVOList.stream().collect(Collectors.groupingBy(EventRegionVO::getRegionName));
        for (Map.Entry<String, List<EventRegionVO>> entry : map.entrySet()) {
            Map<String, Object> regionMap = new HashMap<>();
            regionMap.put("region", entry.getKey());
            regionMap.put("events", entry.getValue());
            regionMap.put("count", entry.getValue().size());
            dataList.add(regionMap);
        }
        return Result.OK(dataList);
    }
}
