package com.zzsn.event.neo4j.util;

import com.zzsn.event.neo4j.vo.GraphInfo;
import com.zzsn.event.neo4j.vo.LinkData;
import com.zzsn.event.neo4j.vo.NodeData;
import lombok.extern.slf4j.Slf4j;
import org.neo4j.driver.*;
import org.neo4j.driver.internal.types.InternalTypeSystem;
import org.neo4j.driver.types.Node;
import org.neo4j.driver.types.Path;
import org.neo4j.driver.types.Relationship;
import org.neo4j.driver.types.Type;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author kongliufeng
 * @create 2022-03-13
 */
@Slf4j
@Component
public class Neo4jUtil {
    @Resource
    public Driver neo4jDriver;



    /**
     * @description 获取执行结果
     *
     * @author kongliufeng
     * @Date 2022-03-13
     */
    public Result executeCypherSql(String cypherSql){
        Result run = null;
        try(Session session = neo4jDriver.session()){
            System.out.println(cypherSql);
            run = session.run(cypherSql);
        }catch (Exception e){
            e.printStackTrace();
            log.error(e.getMessage());
        }
        return run;
    }

    public int getCountTotal(String sql) {
        int total = 0;
        try(Session session = neo4jDriver.session()){
            System.out.println(sql);
            Result result = session.run(sql);
            if(result.hasNext()){
                List<Record> records = result.list();
                for (Record record : records) {
                    total = record.values().get(0).asInt();
                }
            }
        }catch (Exception e){
            e.printStackTrace();
            log.error(e.getMessage());
        }
        return total;
    }

    /**
     * 用于第五版显示
     * match (n:{name=''})  return n
     * 获取指定节点
     * @author kongliufeng
     * @data 2019-12-10 下午1:50:18
     */
    public GraphInfo getNodeAndLink(String cypherSql) {
        Result result = executeCypherSql(cypherSql);
        GraphInfo graphInfo = new GraphInfo();
        if (result == null || !result.hasNext()) {
            return graphInfo;
        }
        // 使用Set避免重复节点和关系
        Set<Node> nodes = new HashSet<>();
        Set<Relationship> relationships = new HashSet<>();

        while (result.hasNext()) {
            Record record = result.next();
            // 处理记录中的每个字段
            for (String key : record.keys()) {
                Value value = record.get(key);
                processValue(value, nodes, relationships);
            }
        }
        graphInfo.setNodes(convertNodes(nodes));
        graphInfo.setLinks(convertRelationships(relationships));

        Set<String> collect = graphInfo.getNodes().stream().map(NodeData::getLabels).flatMap(Set::stream).collect(Collectors.toSet());
        List<HashMap<String,String>> labels = new ArrayList<>();
        for (String label : collect) {
            HashMap<String, String> map = new HashMap<>();
            map.put("name", label);
            labels.add(map);
        }
        graphInfo.setLabels(labels);
        return graphInfo;
    }
    private List<NodeData> convertNodes(Set<Node> nodes) {
        List<NodeData> nodeList = new ArrayList<>();
        for (Node node : nodes) {
            NodeData nodeData = new NodeData();
            nodeData.setId(node.id());
            Set<String> list = new HashSet<>();
            for (String item : node.labels()) {
                list.add(item);
            }
            nodeData.setLabels(list);
            nodeData.setProperties(node.asMap());
            nodeList.add(nodeData);
        }
        return nodeList;
    }

    private List<LinkData> convertRelationships(Set<Relationship> relationships) {
        List<LinkData> linkList = new ArrayList<>();
        for (Relationship rel : relationships) {
            LinkData linkData = new LinkData();
            linkData.setId(rel.id());
            linkData.setType(rel.type());
            linkData.setSourceId(rel.startNodeId());
            linkData.setTargetId(rel.endNodeId());
            linkData.setProperties(rel.asMap());
            linkList.add(linkData);
        }
        return linkList;
    }
    private void processValue(Value value, Set<Node> nodes, Set<Relationship> relationships) {
        Type type = value.type();

        if (type.equals(Types.NODE)) {
            nodes.add(value.asNode());
        } else if (type.equals(Types.RELATIONSHIP)) {
            relationships.add(value.asRelationship());
        } else if (type.equals(Types.PATH)) {
            Path path = value.asPath();
            path.nodes().forEach(nodes::add);
            path.relationships().forEach(relationships::add);
        } else if (type.equals(Types.LIST)) {
            // 递归处理列表中的元素
            value.values().forEach(v -> processValue(v, nodes, relationships));
        } else if (type.equals(Types.MAP)) {
            // 处理Map中的值
            value.asMap().values().forEach(v -> processValue(Values.value(v), nodes, relationships));
        }
    }
    // 静态类型常量
    private static class Types {
        static final Type NODE = InternalTypeSystem.TYPE_SYSTEM.NODE();
        static final Type RELATIONSHIP = InternalTypeSystem.TYPE_SYSTEM.RELATIONSHIP();
        static final Type PATH = InternalTypeSystem.TYPE_SYSTEM.PATH();
        static final Type LIST = InternalTypeSystem.TYPE_SYSTEM.LIST();
        static final Type MAP = InternalTypeSystem.TYPE_SYSTEM.MAP();
    }
}
