package com.zzsn.event.controller.plat;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.IInfoSourceService;
import com.zzsn.event.service.ISubjectTypeService;
import com.zzsn.event.util.*;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 平台事件管理页
 *
 * @author lkg
 * @date 2024/5/7
 */
@Slf4j
@Api(tags = "平台事件管理")
@RestController
@RequestMapping("/plat/manage")
public class PlatEventManageController {


    @Autowired
    private IEventService eventService;
    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private IInfoSourceService infoSourceService;
    @Autowired
    private EsService esService;
    @Autowired
    private EsOpUtil esOpUtil;

    /**
     * 事件页左侧树
     *
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/leftTypeTree")
    public Result<?> leftTree() {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        String userId = null;
        String customerId = null;
        if (category.equals(Constants.COMMON_USER)) {
            userId = currentUser.getUserId();
        } else if (category.equals(Constants.ADMIN_USER)) {
            customerId = currentUser.getCustomerId();
        }
        List<SubjectTreeVO> tree = subjectTypeService.subjectAndTypeTree(userId, customerId);
        return Result.OK(tree);
    }

    /**
     * 事件页左侧树
     *
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/leftCustomerTree")
    public Result<?> leftCustomerTree() {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        String userId = null;
        String customerId = null;
        if (category.equals(Constants.COMMON_USER)) {
            userId = currentUser.getUserId();
        } else if (category.equals(Constants.ADMIN_USER)) {
            customerId = currentUser.getCustomerId();
        }
        List<SubjectTreeVO> tree = subjectTypeService.subjectAndCustomerTree(userId, customerId);
        return Result.OK(tree);
    }


    /**
     * 分页列表-新平台管理
     *
     * @param subjectCondition 筛选条件
     * @param pageNo           当前页
     * @param pageSize         返回条数
     * @author lkg
     * @date 2024/4/28
     */
    @GetMapping("/newPlatPageList")
    public Result<?> newPlatPageList(SubjectCondition subjectCondition,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        if (category.equals(Constants.COMMON_USER)) {
            subjectCondition.setUserId(currentUser.getUserId());
        } else if (category.equals(Constants.ADMIN_USER)) {
            subjectCondition.setCustomerId(currentUser.getCustomerId());
        }
        IPage<EventNewPlatVO> pageList = eventService.newPlatPageList(subjectCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 分页列表-新平台管理
     *
     * @param subjectCondition 筛选条件
     * @param pageNo           当前页
     * @param pageSize         返回条数
     * @author lkg
     * @date 2024/4/28
     */
    @GetMapping("/newPlatCustomerPageList")
    public Result<?> newPlatCustomerPageList(SubjectCondition subjectCondition,
                                             @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                             @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        if (category.equals(Constants.COMMON_USER)) {
            subjectCondition.setUserId(currentUser.getUserId());
            subjectCondition.setCustomerId(currentUser.getCustomerId());
        } else if (category.equals(Constants.ADMIN_USER)) {
            if (StringUtils.isEmpty(subjectCondition.getCustomerId()) || "0".equals(subjectCondition.getCustomerId())) {
                subjectCondition.setCustomerId(currentUser.getCustomerId());
            }
        }
        if ("0".equals(subjectCondition.getCustomerId())) {
            subjectCondition.setCustomerId(null);
        }
        IPage<EventNewPlatVO> pageList = eventService.newPlatCustomerPageList(subjectCondition, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 专题绑定的信息源
     *
     * @param infoSourceCondition 筛选条件
     * @param subjectId           专题id/专题分类id
     * @param pageNo              当前页
     * @param pageSize            返回条数
     * @author lkg
     * @date 2024/5/7
     */
    @GetMapping(value = "/bindInfoSourceList")
    public Result<?> bindInfoSourceList(InfoSourceCondition infoSourceCondition,
                                        @RequestParam(name = "subjectId", defaultValue = "0") String subjectId,
                                        @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                        @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<String> subjectIdList = new ArrayList<>();
        if (!"0".equals(subjectId)) {
            UserVo currentUser = UserUtil.getLoginUser();
            String userId = null;
            if (currentUser.getCategory().equals(Constants.COMMON_USER)) {
                userId = currentUser.getUserId();
            }
            //查询类别id的所有明细id
            if (StringUtils.isNotEmpty(subjectId)) {
                List<String> typeIds = subjectTypeService.belowIdList(subjectId, userId, null);
                subjectIdList = eventService.selectSubjectByTypeIds(userId, typeIds);
            }
            subjectIdList.add(subjectId);
        }
        IPage<InfoSourceVo> pageList = infoSourceService.bindInfoSourceList(infoSourceCondition, subjectIdList, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 专题直接绑定信息源-批量
     *
     * @param directBindInfoVO 参数封装
     * @author lkg
     * @date 2024/4/24
     */
    @PostMapping("/directBindInfoSource")
    public Result<?> directBindInfoSource(@RequestBody DirectBindInfoVO directBindInfoVO) {
        eventService.directBindInfoSource(directBindInfoVO);
        return Result.OK();
    }

    /**
     * 专题直接解绑信息源-批量
     *
     * @param directBindInfoVO 参数封装
     * @author lkg
     * @date 2024/4/24
     */
    @PostMapping("/directRemoveInfoSource")
    public Result<?> directRemoveInfoSource(@RequestBody DirectBindInfoVO directBindInfoVO) {
        eventService.directRemoveInfoSource(directBindInfoVO);
        return Result.OK();
    }

    /**
     * 专题直接屏蔽的信息源列表(不包括专题屏蔽的信息源组下的信息源)
     *
     * @param infoSourceCondition 筛选条件
     * @param subjectId           专题id
     * @param pageNo              当前页
     * @param pageSize            返回条数
     * @author lkg
     * @date 2024/4/30
     */
    @GetMapping("/unBindInfoSourceList")
    public Result<?> unBindInfoSourceList(InfoSourceCondition infoSourceCondition,
                                          @RequestParam(name = "subjectId", defaultValue = "0") String subjectId,
                                          @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                          @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<InfoSourceVo> pageList = infoSourceService.unBindSourcePageList(infoSourceCondition, subjectId, pageNo, pageSize);
        return Result.OK(pageList);
    }

    /**
     * 删除专题下已屏蔽的信息源(即恢复绑定关系)
     *
     * @param subjectId 专题id
     * @param sourceId  信息源id
     * @author lkg
     * @date 2024/4/30
     */
    @GetMapping("/removeUnBindInfoSource")
    public Result<?> removeUnBindInfoSource(@RequestParam String subjectId, @RequestParam String sourceId) {
        eventService.removeUnBindInfoSource(subjectId, sourceId);
        return Result.OK();
    }

    /**
     * 专题关联信息源采集量统计-导出excel
     *
     * @param subjectId 专题id/专题分类id
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @author lkg
     */
    @GetMapping("/exportStatisticsExcel")
    public void exportStatisticsExcel(@RequestParam(name = "subjectId") String subjectId,
                                      @RequestParam(name = "startDate") String startDate,
                                      @RequestParam(name = "endDate") String endDate,
                                      HttpServletResponse response) {
        List<String> subjectIdList = new ArrayList<>();
        try {
            UserVo currentUser = UserUtil.getLoginUser();
            String userId = null;
            if (currentUser.getCategory().equals(Constants.COMMON_USER)) {
                userId = currentUser.getUserId();
            }
            //查询类别id的所有明细id
            if (StringUtils.isNotEmpty(subjectId) && !"0".equals(subjectId)) {
                List<String> typeIds = subjectTypeService.belowIdList(subjectId, userId, null);
                subjectIdList = eventService.selectSubjectByTypeIds(userId, typeIds);
            }
            subjectIdList.add(subjectId);
            List<List<String>> statistics = eventService.subjectStatistics(subjectIdList, startDate, endDate);
            List<String> dateList = DateUtil.betweenDate(startDate, endDate);
            List<String> headers = Stream.of("信息源id", "网站名称", "栏目名称", "栏目地址", "时间段内采集总量").collect(Collectors.toList());
            headers.addAll(dateList);
            XSSFWorkbook workbook = new XSSFWorkbook();
            ExcelExportUtil.exportExcelData(workbook, 0, headers, statistics, "信息源采集信息汇总");
            String name = "statisticsExcel.xls";
            setResponseHeader(response, name);
            ServletOutputStream outputStream = response.getOutputStream();
            workbook.write(outputStream);
            outputStream.flush();
            outputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * 事件对应专题库的资讯分页列表
     *
     * @param eventDataCondition 筛选条件
     * @param sourceId           信息源id
     * @param pageNo             当前页
     * @param pageSize           返回条数
     * @param column             排序字段
     * @param order              排序方式
     * @param isAll              是否全部
     * @param video              是否视频
     * @param crawler            采集器
     * @param isSubject          是否专题
     * @param source             来源(类别/客户)
     * @author lkg
     * @date 2024/5/6
     */
    @GetMapping("/collectPageList")
    public Result<?> collectPageList(EventDataCondition eventDataCondition,
                                     @RequestParam(name = "sourceId", defaultValue = "") String sourceId,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                     @RequestParam(name = "column", defaultValue = "common") String column,
                                     @RequestParam(name = "order", defaultValue = "desc") String order,
                                     @RequestParam(name = "isAll", defaultValue = "0") String isAll,
                                     @RequestParam(name = "video", defaultValue = "") String video,
                                     @RequestParam(name = "crawler", defaultValue = "") String crawler,
                                     @RequestParam(name = "isSubject", defaultValue = "0") String isSubject,
                                     @RequestParam(name = "source", defaultValue = "type") String source) {
        UserVo currentUser = UserUtil.getLoginUser();
        String userId = null;
        String tenantId = currentUser.getCustomerId();
        if (currentUser.getCategory().equals(Constants.COMMON_USER)) {
            userId = currentUser.getUserId();
        }
        IPage<EventDataVO> pageList = esService.collectPageList(eventDataCondition, sourceId, pageNo, pageSize, column, order, isAll, video, crawler, userId, tenantId, isSubject, source);
        return Result.OK(pageList);
    }

    /**
     * 件对应专题库的资讯分页列表
     *
     * @param subjectInfoVo 筛选条件
     * @param userId        用户id
     * @param video         是否视频
     * @param pageNo        当前页
     * @param isCustomer    是否客户
     * @param pageSize      返回条数
     * @param column        排序字段
     * @param order         排序方式
     * @param crawler       采集器
     * @param isSubject     是否换题
     * @param labelIds      关联的标签id
     * @param labelTypeIds  企业标签id
     * @param relationIds   企业信用代码
     * @param sourceId      信息源id
     * @author lkg
     * @date 2024/5/6
     */
    @GetMapping("/subjectPageList")
    public Result<?> subjectPageList(EventDataCondition subjectInfoVo,
                                     @RequestParam(name = "userId", defaultValue = "") String userId,
                                     @RequestParam(name = "video", defaultValue = "") String video,
                                     @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                     @RequestParam(name = "isCustomer", defaultValue = "0") Integer isCustomer,
                                     @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                     @RequestParam(name = "column", defaultValue = "common") String column,
                                     @RequestParam(name = "order", defaultValue = "desc") String order,
                                     @RequestParam(name = "crawler", defaultValue = "") String crawler,
                                     @RequestParam(name = "isSubject", defaultValue = "1") String isSubject,
                                     @RequestParam(name = "labelIds", required = false) String labelIds,
                                     @RequestParam(name = "labelTypeIds", required = false) String labelTypeIds,
                                     @RequestParam(name = "relationIds", required = false) String relationIds,
                                     @RequestParam(name = "sourceId", required = false) String sourceId) throws Exception {
        List<String> socialCreditCodeList = new ArrayList<>();
        if (StringUtils.isNotEmpty(relationIds)) {
            socialCreditCodeList = Arrays.asList(relationIds.split(","));
        } else if (StringUtils.isNotEmpty(labelTypeIds)) {
            socialCreditCodeList = eventService.codesByLabels(Arrays.asList(labelTypeIds.split(",")));
        }
        IPage<DisplayInfo> pageList = esService.subjectPageList(userId, subjectInfoVo, video, pageNo, pageSize, column, order, crawler, isSubject, labelIds, socialCreditCodeList, sourceId, isCustomer);
        return Result.OK(pageList);
    }


    /**
     * 专题绑定的关键词组-分页列表
     *
     * @param id 题id/专题分类id
     * @author lkg
     * @date 2024/5/7
     */
    @GetMapping(value = "/bindKeyWordsList")
    public Result<?> bindKeyWordsList(@RequestParam String id,
                                      @RequestParam(required = false) String groupName,
                                      @RequestParam(required = false) String wordName,
                                      @RequestParam(defaultValue = "1") Integer pageNo,
                                      @RequestParam(defaultValue = "10") Integer pageSize) {
        List<String> subjectIdList = new ArrayList<>();
        try {
            UserVo currentUser = UserUtil.getLoginUser();
            String userId = null;
            if (currentUser.getCategory().equals(Constants.COMMON_USER)) {
                userId = currentUser.getUserId();
            }
            if (StringUtils.isNotEmpty(id) && !"0".equals(id)) {
                List<String> typeIds = subjectTypeService.belowIdList(id, userId, null);
                subjectIdList = eventService.selectSubjectByTypeIds(userId, typeIds);
            }
            subjectIdList.add(id);
        } catch (Exception e) {
            e.printStackTrace();
        }
        IPage<KeyWordsPage> page = eventService.bindKeyWordsList(subjectIdList, groupName, wordName, pageNo, pageSize);
        return Result.OK(page);
    }


    /**
     * 通过id查询(采集库)
     *
     * @param id    资讯id
     * @param index 索引名称
     * @author lkg
     * @date 2024/5/8
     */
    @GetMapping(value = "/collectionQueryById")
    public Result<?> collectionQueryById(@RequestParam String id, @RequestParam String index) {
        DisplayInfo info = (DisplayInfo) esOpUtil.getInfoById(index, id, DisplayInfo.class);
        if (StringUtils.isNotEmpty(info.getContentWithTag())) {
            String contentNoTag = Utility.TransferHTML2TextWithImg(info.getContentWithTag());
            String contentNoTag2 = Utility.dealImg(contentNoTag);
            info.setContent(contentNoTag2);
            info.setContentWithTag(null);
        }
        info.setPublishDate(EsDateUtil.esFieldDateMapping(info.getPublishDate()));
        return Result.OK(info);
    }

    /**
     * 通过id查询(专题库)
     *
     * @param id    资讯id
     * @param index 索引名称
     * @author lkg
     * @date 2024/5/8
     */
    @GetMapping(value = "/queryById")
    public Result<?> queryById(@RequestParam String id, @RequestParam String index) {
        DisplayInfo displayInfo = (DisplayInfo) esOpUtil.getInfoById(index, id, DisplayInfo.class);
        addReadNum(displayInfo,index);
        if (StringUtils.isNotEmpty(displayInfo.getContentWithTag())) {
            String contentNoTag = Utility.TransferHTML2TextWithImg(displayInfo.getContentWithTag());
            String contentNoTag2 = Utility.dealImg(contentNoTag);
            displayInfo.setContent(contentNoTag2);
        }
        displayInfo.setPublishDate(EsDateUtil.esFieldDateMapping(displayInfo.getPublishDate()));
        return Result.OK(displayInfo);
    }

    private void addReadNum(DisplayInfo displayInfo,String index){
        if (displayInfo != null) {
            if (displayInfo.getReadNum() == 0) {
                displayInfo.setReadNum(1L);
            } else {
                displayInfo.setReadNum(displayInfo.getReadNum() + 1);
            }
            esOpUtil.docUpdateById(index, displayInfo.getId(), JSON.toJSONString(displayInfo));
        }
    }


    private void setResponseHeader(HttpServletResponse response, String name) {
        try {
            try {
                name = new String(name.getBytes(), "ISO8859-1");
            } catch (Exception e) {
                e.printStackTrace();
            }
            response.setContentType("application/octet-stream;charset=ISO8859-1");
            response.setHeader("Content-Disposition", "attachment;filename=" + name);
            response.setHeader("Pargam", "no-cache");
            response.setHeader("Cache-Control", "no-cache");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
