package com.zzsn.event.controller.common;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.SubjectUserCondition;
import com.zzsn.event.enums.LabelTypeEnum;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.*;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.util.PythonUtil;
import com.zzsn.event.util.user.AuthUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.DisplayInfo;
import org.apache.commons.lang3.StringUtils;
import org.jsoup.Jsoup;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * 资讯
 *
 * @author lkg
 * @date 2024/12/17
 */
@RestController
@RequestMapping("/info")
public class InformationController {

    @Autowired
    private InformationService informationService;
    @Autowired
    private EsService esService;
    @Autowired
    private CommonService commonService;
    @Autowired
    private SubjectUserConditionService subjectUserConditionService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private SysBaseLabelTypeService sysBaseLabelTypeService;
    @Autowired
    private PythonUtil pythonUtil;


    /**
     * 专题/事件对应采集库的资讯分页列表
     *
     * @param eventDataCondition 筛选条件
     * @author lkg
     * @date 2024/5/6
     */
    @PostMapping("/collectPageList")
    public Result<?> collectPageList(@RequestBody InfoDataSearchCondition eventDataCondition) {
        IPage<EventDataVO> pageList = informationService.collectPageList(eventDataCondition);
        return Result.OK(pageList);
    }

    /**
     * 判断采集信息是否已经进入专题库
     *
     * @param subjectId 专题id
     * @param ids       采集信息id字符串，用逗号拼接
     * @author lkg
     * @date 2024/12/24
     */
    @GetMapping("/listBeInStorage")
    public Result<?> listBeInStorage(@RequestParam String subjectId, @RequestParam String ids) {
        List<String> list = esService.listBeInStorage(subjectId, ids);
        return Result.OK(list);
    }

    /**
     * 通过id查询(采集库)
     *
     * @param id    资讯id
     * @param index 索引名称
     * @author lkg
     * @date 2024/5/8
     */
    @GetMapping(value = "/collectionQueryById")
    public Result<?> collectionQueryById(@RequestParam String id, @RequestParam String index) {
        DisplayInfo info = informationService.queryInfo(1, index, id);
        //todo  浏览记录
        return Result.OK(info);
    }

    /**
     * 获取用户的筛选条件
     *
     * @param relationId 关联id
     * @author lkg
     * @date 2024/2/28
     */
    @GetMapping("/search/condition/list")
    public Result<?> queryInfo(@RequestParam("subjectId") String relationId) {
        UserVo user = UserUtil.getLoginUser();
        String userId = user.getId();
        LambdaQueryWrapper<SubjectUserCondition> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectUserCondition::getUserId, userId)
                .eq(SubjectUserCondition::getRelationId, relationId);
        List<SubjectUserCondition> list = subjectUserConditionService.list(queryWrapper);
        return Result.OK(list);
    }

    /**
     * 新增/编辑筛选条件
     *
     * @param condition 参数封装
     * @author lkg
     * @date 2024/2/28
     */
    @PostMapping("/search/condition/modify")
    public Result<?> modify(@RequestBody SubjectUserCondition condition) {
        UserVo user = UserUtil.getLoginUser();
        String userId = user.getId();
        condition.setUserId(userId);
        subjectUserConditionService.saveOrUpdate(condition);
        return Result.OK();
    }

    /**
     * 删除筛选条件
     *
     * @param id 主键id
     * @author lkg
     * @date 2024/2/29
     */
    @GetMapping("/search/condition/delete")
    public Result<?> delete(@RequestParam String id) {
        subjectUserConditionService.removeById(id);
        return Result.OK();
    }


    /**
     * 专题下模型绑定的标签信息-筛选条件
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2024/12/24
     */
    @GetMapping("/label/listBySubjectId")
    public Result<?> getListBySubjectId(@RequestParam String subjectId) {
        List<String> subjectIds = new ArrayList<>();
        subjectIds.add(subjectId);
        List<LabelModelVo> sysBaseLabelTypeVos = commonService.subjectModelBindLabels(subjectIds);
        return Result.OK(sysBaseLabelTypeVos);
    }


    /**
     * 标签下的标签值信息
     *
     * @param labelVo  筛选条件
     * @param pageNo   当前页
     * @param pageSize 每页返回条数
     * @author lkg
     * @date 2024/12/27
     */
    @PostMapping("/label/entityList")
    public Result<?> entityByLabel(@RequestBody SysLabelVo labelVo,
                                   @RequestParam(defaultValue = "1") Integer pageNo,
                                   @RequestParam(defaultValue = "10") Integer pageSize) {
        Object list = labelEntityService.pageList(labelVo, pageNo, pageSize);
        return Result.OK(list);
    }

    /**
     * 获取企业标签列表-树型结构
     *
     * @author lkg
     * @date 2024/12/27
     */
    @GetMapping("/label/companyLabelTree")
    public Result<?> companyLabelList() {
        List<LabelTypeVO> tree = sysBaseLabelTypeService.labelTypeTree(LabelTypeEnum.ENTERPRISE.getValue());
        return Result.OK(tree);
    }

    /**
     * 事件对应专题库的资讯分页列表
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/5/6
     */
    @PostMapping("/subjectPageList")
    public Result<?> subjectPageList(@RequestBody InfoDataSearchCondition searchCondition) {
        if (StringUtils.isEmpty(searchCondition.getSubjectId())) {
            return Result.FAIL("专题id不能为空");
        }
        UserVo userVo = UserUtil.getLoginUser();
        IPage<DisplayInfo> pageList = informationService.subjectPageList(userVo, searchCondition);
        return Result.OK(pageList);
    }

    /**
     * 保存为数据集
     *
     * @param searchCondition 筛选条件
     * @author lkg
     * @date 2024/5/6
     */
    @PostMapping("/saveAsDataSet")
    public Result<?> saveAsDataSet(@RequestBody InfoDataSearchCondition searchCondition) {
        String dataSetId = searchCondition.getDataSetId();
        if (StringUtils.isEmpty(dataSetId)) {
            return Result.FAIL("数据集id不能为空");
        }
        if (StringUtils.isEmpty(searchCondition.getSubjectId())) {
            return Result.FAIL("专题id不能为空");
        }
        CompletableFuture.runAsync(() -> informationService.saveAsDataSet(searchCondition));
        return Result.OK("数据集保存中。。。");
    }



    /**
     * 通过id查询(专题库)
     *
     * @param id    资讯id
     * @param index 索引名称
     * @author lkg
     * @date 2024/5/8
     */
    @GetMapping(value = "/queryById")
    public Result<?> queryById(@RequestParam String id, @RequestParam String index) {
        DisplayInfo displayInfo = informationService.queryInfo(2, index, id);
        return Result.OK(displayInfo);
    }


    /**
     * 人工添加(采集库到专题库)
     *
     * @param manualAddVOs 参数
     * @author lkg
     * @date 2024/12/19
     */
    @PostMapping(value = "/manualAdd")
    public Result<?> manualAdd(@RequestBody List<ManualAddVO> manualAddVOs) {
        informationService.manualAdd(manualAddVOs);
        return Result.OK();
    }

    /**
     * 新增
     */
    @PostMapping(value = "/add")
    public Result<?> add(@RequestBody JSONObject jsonObject) {
        UserVo userVo = UserUtil.getLoginUser();
        DisplayInfo displayInfo = JSON.parseObject(JSON.toJSONString(jsonObject.get("data")), DisplayInfo.class);
        String contentWithTag = displayInfo.getContentWithTag();
        if (StringUtils.isNotBlank(contentWithTag)) {
            displayInfo.setContent(Jsoup.parse(contentWithTag).text());
        }
        Integer category = (Integer) jsonObject.get("category");
        boolean modelFlag = pythonUtil.judgeDuplicate(displayInfo.getId(), displayInfo.getTitle(), displayInfo.getContent(), displayInfo.getSourceAddress(), displayInfo.getSubjectId());
        if (modelFlag) {
            return Result.FAIL(210, "判重存在重复资讯!");
        } else {
            boolean repeat = informationService.duplicationByTitleOrSourceAddress(displayInfo);
            if (!repeat) {
                return Result.FAIL(211, "专题库存在重复资讯!");
            }
        }
        informationService.add(displayInfo, category, userVo);
        return Result.OK();
    }

    /**
     * 审核
     */
    @PostMapping(value = "/check")
    public Result<?> check(@RequestBody Map<String, Object> map) {
        UserVo userVo = UserUtil.getLoginUser();
        informationService.checkInfo(map, userVo);
        return Result.OK();
    }

    /**
     * 编辑
     *
     * @param jsonObject 参数
     * @author lkg
     * @date 2024/12/19
     */
    @PostMapping(value = "/edit")
    public Result<?> edit(@RequestBody JSONObject jsonObject) {
        UserVo userVo = UserUtil.getLoginUser();
        informationService.updateInfo(jsonObject, userVo);
        return Result.OK();
    }

    /**
     * 批量删除-专题库(逻辑删除)
     *
     * @param map 删除参数
     * @author lkg
     * @date 2024/12/19
     */
    @PostMapping(value = "/deleteBatch")
    public Result<?> deleteBatch(@RequestBody Map<String, Object> map) {
        UserVo userVo = UserUtil.getLoginUser();
        informationService.deleteBatch(map, userVo);
        return Result.OK();
    }

    /**
     * 批量删除-采集库(物理删除)
     *
     * @param map 删除参数
     * @author lkg
     * @date 2024/12/19
     */
    @PostMapping(value = "/removeBatch")
    public Result<?> removeBatch(@RequestBody Map<String, Object> map) {
        informationService.removeBatch(map);
        return Result.OK();
    }

    /**
     * 置顶
     */
    @PostMapping(value = "/top")
    public Result<?> top(@RequestBody SubjectInfoVo subjectInfoVo) {
        UserVo userVo = UserUtil.getLoginUser();
        informationService.topInfo(subjectInfoVo, userVo);
        return Result.OK();
    }

    /**
     * 收藏
     *
     * @param collectionInfo 收藏信息
     * @author lkg
     * @date 2024/12/24
     */
    @PostMapping("/collect")
    public Result<?> collect(@RequestBody CollectionInfo collectionInfo) {
        informationService.collect(collectionInfo);
        return Result.OK();
    }

    /**
     * 单篇文章热词
     *
     * @param index  索引名称
     * @param id     资讯id
     * @param number 热词数量
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping(value = "/hotWords")
    public Result<?> articleList(@RequestParam(value = "index", required = false) String index,
                                 @RequestParam("id") String id,
                                 @RequestParam(name = "number", defaultValue = "200") Integer number) {
        if (StringUtils.isEmpty(index)) {
            index = Constants.SUBJECT_INDEX;
        }
        List<StatisticsKeyWordVo> words = informationService.hotWords(index, id, number);
        return Result.OK(words);
    }

    /**
     * 相关推荐
     *
     * @param subjectId 专题id
     * @param id        资讯id
     * @param title     标题
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping(value = "/recommendList")
    public Result<?> recommendList(@RequestParam(name = "subjectId") String subjectId,
                                   @RequestParam(name = "id") String id,
                                   @RequestParam(name = "title") String title,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<SubjectDataVo> recommendList = informationService.recommendList(subjectId, id, title, pageNo, pageSize);
        return Result.OK(recommendList);
    }
}
