package com.zzsn.event.controller.yjzx;

import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.clb.common.model.task.dto.titr.KeyWordsDTO;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.KeyWords;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.IKeyWordsService;
import com.zzsn.event.service.LabelEntityService;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * 事件后台管理
 *
 * @author lkg
 * @date 2024/4/8
 */
@Slf4j
@Api(tags = "事件后台管理")
@RestController
@RequestMapping("/manage")
public class EventManageController {

    @Autowired
    private IEventService eventService;
    @Autowired
    private LabelEntityService labelEntityService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;

    @Autowired
    private IKeyWordsService keyWordsService;

    @Autowired
    private RedisUtil redisUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;


    @Value("${kafka.topic.event.run:}")
    private String EVENT_MODEL_KAFKA_CHANNEL;

    /**
     * 分页列表-后台管理
     *
     * @param eventName     事件名称
     * @param eventType     事件分类id
     * @param facePublic    是否公开(0-未公开;1-已公开)
     * @param publishStatus 发布状态(0-未发布;1-已发布)
     * @param startTime     开始时间
     * @param endTime       结束时间
     * @param order         排序字段
     * @param orderType     排序方式
     * @param pageNo        偏移量
     * @param pageSize      返回条数
     * @author lkg
     * @date 2024/4/8
     */
    @ApiOperation(value = "事件-分页列表查询", notes = "事件-分页列表查询")
    @GetMapping(value = "/pageList")
    public Result<?> pageList(@RequestParam(name = "eventName", required = false) String eventName,
                              @RequestParam(name = "eventType", required = false) Integer eventType,
                              @RequestParam(name = "facePublic", required = false) Integer facePublic,
                              @RequestParam(name = "publishStatus", required = false) Integer publishStatus,
                              @RequestParam(name = "startTime", required = false) String startTime,
                              @RequestParam(name = "endTime", required = false) String endTime,
                              @RequestParam(name = "order", required = false) String order,
                              @RequestParam(name = "orderType", defaultValue = "asc") String orderType,
                              @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        IPage<EventManageVO> pageList = eventService.pageList(eventName, eventType, facePublic, publishStatus, startTime, endTime, order, orderType, pageNo, pageSize);
        return Result.OK(pageList);
    }


    /**
     * 地域信息-树型结构
     *
     * @param type 类别(1-国际;2-国内)
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping("/regionTree")
    public Result<?> regionTree(@RequestParam Integer type) {
        List<LabelTypeVO> nodes = labelEntityService.regionTree(type);
        return Result.OK(nodes);
    }

    /**
     * 通过id查询
     *
     * @param id 事件id
     * @return
     */
    @GetMapping(value = "/queryById")
    public Result<?> queryById(@RequestParam(name = "id") String id) {
        EventVO eventVO = eventService.queryInfo(id);
        return Result.OK(eventVO);
    }


    /**
     * 1.2 添加
     *
     * @param eventParam
     * @return
     */
    @ApiOperation(value = "事件-添加", notes = "事件-添加")
    @PostMapping(value = "/add")
    public Result<?> add(@RequestBody AddEventParam eventParam) {
        KeywordsVO keywordsVO = eventParam.getKeywordsVO();
        if (keywordsVO != null) {
            UserVo currentUser = UserUtil.getLoginUser();
            eventParam.setCreateBy(currentUser.getUsername());
            Event event = eventService.saveMain(eventParam);
            //新增关键词
            KeyWords keyWords = keyWordsService.saveKeyword(event, keywordsVO.getKeyword(), keywordsVO.getExclusionWord());
            CompletableFuture.runAsync(() -> {
                //插入xxlJob
                iXxlJobInfoService.eventInsert(event);
                //-------------关键词采集相关，需优化，采集那边说目前搜索引擎效果也不好，先这样。
                //关键词入缓存
                KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                BeanUtils.copyProperties(keyWords, redisKeywordDTO);
                redisKeywordDTO.setStartTime(event.getStartTime());
                redisKeywordDTO.setEndTime(event.getEndTime());
                redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWords.getWordsCode(), redisKeywordDTO);
                //插入xxljob
                iXxlJobInfoService.keyWordsInsert(redisKeywordDTO);
                //为了立即响应，关键词新增时放入消息队列
                kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
                //----------------------------------------------------------------------------------------------
                kafkaTemplate.send(EVENT_MODEL_KAFKA_CHANNEL, event.getEventCode());
            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 事件配置校验
     *
     * @param addEventParam 参数
     * @author lkg
     * @date 2025/2/5
     */
    @PostMapping("/configVerification")
    public Result<?> configVerification(@RequestBody AddEventParam addEventParam){
        Boolean verification = eventService.configVerification(addEventParam);
        return Result.OK(verification);
    }

    /**
     * 1.3 编辑
     *
     * @param addEventParam
     * @return
     */
    @ApiOperation(value = "事件-编辑", notes = "事件-编辑")
    @PostMapping(value = "/edit")
    public Result<?> edit(@RequestBody AddEventParam addEventParam) {
        //关键词
        KeywordsVO keywordsVO = addEventParam.getKeywordsVO();
        if (keywordsVO != null) {
            UserVo currentUser = UserUtil.getLoginUser();
            addEventParam.setUpdateTime(new Date());
            addEventParam.setUpdateBy(currentUser.getUsername());
            eventService.updateMain(addEventParam);
            keyWordsService.update(Wrappers.<KeyWords>lambdaUpdate().eq(KeyWords::getId, keywordsVO.getId())
                    .set(KeyWords::getKeyWord, keywordsVO.getKeyword())
                    .set(KeyWords::getExclusionWord, keywordsVO.getExclusionWord()));
            CompletableFuture.runAsync(() -> {
                KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
                BeanUtils.copyProperties(keywordsVO, redisKeywordDTO);
                redisKeywordDTO.setKeyWord(keywordsVO.getKeyword());
                redisKeywordDTO.setStartTime(addEventParam.getStartTime());
                redisKeywordDTO.setEndTime(addEventParam.getEndTime());
                redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
                redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keywordsVO.getWordsCode(), redisKeywordDTO);
                //为了立即响应，关键词编辑时放入消息队列
                kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
            });
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 启用
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/9/12
     */
    @GetMapping("/enable")
    public Result<?> enable(@RequestParam String eventId) {
        eventService.updateStatus(eventId, 1);
        return Result.OK();
    }

    /**
     * 禁用
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2024/9/12
     */
    @GetMapping("/disable")
    public Result<?> disable(@RequestParam String eventId) {
        eventService.updateStatus(eventId, 0);
        return Result.OK();
    }


    /**
     * 清空专题数据
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/2/6
     */
    @GetMapping("/clearData")
    public Result<?> clearData(String subjectId){
        eventService.clearSubjectData(subjectId);
        return Result.OK("正在清空数据");
    }

    /**
     * 通过id删除
     *
     * @param id 事件id
     * @return
     */
    @ApiOperation(value = "事件-通过id删除", notes = "事件-通过id删除")
    @GetMapping(value = "/delete")
    public Result<?> delete(@RequestParam(name = "id") String id) {
        EventVO eventVO = eventService.queryInfo(id);
        eventService.deleteMain(id);
        CompletableFuture.runAsync(() -> {
            iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
            //删除关键词
            KeywordsVO keywordsVO = eventVO.getKeywordsVO();
            if (keywordsVO != null) {
                String wordsCode = keywordsVO.getWordsCode();
                keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
            }
        });
        return Result.OK();
    }

    /**
     * 1.5 批量删除
     *
     * @param ids 事件id集合
     * @return
     */
    @ApiOperation(value = "事件-批量删除", notes = "事件-批量删除")
    @GetMapping(value = "/deleteBatch")
    public Result<?> deleteBatch(@RequestParam(name = "ids") String ids) {
        List<CompletableFuture<Void>> asyncList = new ArrayList<>();
        for (String id : ids.split(",")) {
            asyncList.add(CompletableFuture.runAsync(() -> {
                EventVO eventVO = eventService.queryInfo(id);
                eventService.deleteMain(id);
                iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
                //删除关键词
                KeywordsVO keywordsVO = eventVO.getKeywordsVO();
                if (keywordsVO != null) {
                    String wordsCode = keywordsVO.getWordsCode();
                    keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                    redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                    iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
                }
            }));
        }
        //等待全部执行完
        CompletableFuture.allOf(asyncList.toArray(new CompletableFuture[0])).join();
        return Result.OK();
    }

    /**
     * 1.7 发布
     *
     * @return
     */
    @PostMapping(value = "/publish")
    public Result<?> publish(@RequestBody Event event) {
        Event byId = eventService.getById(event.getId());
        byId.setUpdateTime(new Date());
        byId.setPublishDate(event.getPublishDate());
        byId.setPublishStatus(event.getPublishStatus());
        eventService.updateById(byId);
        return Result.OK();
    }

    /**
     * 模型信息列表
     *
     * @author lkg
     * @date 2024/4/11
     */
    @GetMapping("/modelList")
    public Result<?> modelList() {
        List<ModelVO> modelVOS = eventService.modelList();
        return Result.OK(modelVOS);
    }
}
