package com.zzsn.event.util;

import com.cronutils.builder.CronBuilder;
import com.cronutils.model.Cron;
import com.cronutils.model.CronType;
import com.cronutils.model.definition.CronDefinitionBuilder;
import com.cronutils.model.field.expression.FieldExpressionFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Objects;
import java.util.Random;

/**
 * @author lkg
 * @description: cron表达式工具类
 * @date 2021/6/19 10:13
 */
@Slf4j
public class CronUtil {

    private static final SimpleDateFormat sdf = new SimpleDateFormat("ss mm HH dd MM ? yyyy");
    /**
     * 根据时间间隔生成cron表达式--暂只支持 秒、分、时
     * @param duration 时间间隔(单位：秒)
     * @return java.lang.String
     */
    public static String geneCronByPeriod(Integer duration){
        String cron;
        int hours = (duration % (86400)) / 3600;
        int minutes = ((duration % (86400)) % 3600) / 60;
        if (hours == 0){
            if (minutes == 0){
                cron = cronSecond(duration);
            }else {
                cron = cronMinute(minutes);
            }
        }else {
            cron = cronHour(hours);
        }
        return cron;
    }

    //按秒
    private static String cronSecond(int seconds){
        Cron cron = CronBuilder.cron(CronDefinitionBuilder.instanceDefinitionFor(CronType.QUARTZ))
                .withYear(FieldExpressionFactory.always())
                .withDoW(FieldExpressionFactory.questionMark())
                .withMonth(FieldExpressionFactory.always())
                .withDoM(FieldExpressionFactory.always())
                .withHour(FieldExpressionFactory.always())
                .withMinute(FieldExpressionFactory.always())
                .withSecond(FieldExpressionFactory.every(seconds))
                .instance();
        return cron.asString();
    }

    //按分钟
    private static String cronMinute(int minutes){
        Cron cron = CronBuilder.cron(CronDefinitionBuilder.instanceDefinitionFor(CronType.QUARTZ))
                .withYear(FieldExpressionFactory.always())
                .withDoW(FieldExpressionFactory.questionMark())
                .withMonth(FieldExpressionFactory.always())
                .withDoM(FieldExpressionFactory.always())
                .withHour(FieldExpressionFactory.always())
                .withMinute(FieldExpressionFactory.every(minutes))
                .withSecond(FieldExpressionFactory.on(0))
                .instance();
        return cron.asString();
    }

    //按小时
    private static String cronHour(int hours){
        Cron cron = CronBuilder.cron(CronDefinitionBuilder.instanceDefinitionFor(CronType.QUARTZ))
                .withYear(FieldExpressionFactory.always())
                .withDoW(FieldExpressionFactory.questionMark())
                .withMonth(FieldExpressionFactory.always())
                .withDoM(FieldExpressionFactory.always())
                .withHour(FieldExpressionFactory.every(hours))
                .withMinute(FieldExpressionFactory.on(0))
                .withSecond(FieldExpressionFactory.on(0))
                .instance();
        return cron.asString();
    }

    //当前时间转换为cron表达式
    /***
     *  功能描述：日期转换cron表达式
     * @param date
     * @return
     */
    private static String formatDateByPattern(Date date) {
        String formatTimeStr = null;
        if (Objects.nonNull(date)) {
            formatTimeStr = sdf.format(date);
        }
        return formatTimeStr;
    }

    /***
     * convert Date to cron, eg "0 07 10 15 1 ? 2016"
     * @param date  : 时间点
     * @return
     */
    public static String getCron(Date date, int space) {
        String cron = formatDateByPattern(date);
        //将生成的cron最后一个参数年转化为通配符
        String[] cronArray = cron.split(" ");
        cronArray[6] = "*";
        String result = "";
        if(!StringUtils.isEmpty(space)){
            cronArray[2] = "0/" + space;
        }
        for(int i = 0; i < 3; i++){
            result += cronArray[i] + " ";
        }
        cron = result + "* * ?";
//        for (String s : cronArray) {
//            result += s + " ";
//        }
//        cron = cronExpressionPlusDuration(result, (int)space*60*60);
        return cron;
    }

    /**
     * <p>对cron表达式进行修改，根据传入的时间生成一个新的cron表达式</>
     * 两个表达式之间的间隔为duration
     *
     * @param cron
     * @param duration 单位为秒
     * @return
     */
    private static String cronExpressionPlusDuration(String cron, int duration) {

        String[] cronArray = cron.split(" ");
        int days = duration / 86400;
        int hours = (duration % (86400)) / 3600;
        int minutes = ((duration % (86400)) % 3600) / 60;
        cronArray[2] = String.valueOf(Integer.valueOf(cronArray[2]) + hours);
        cronArray[1] = String.valueOf(Integer.valueOf(cronArray[1]) + minutes);
        /* 对小时和分钟进行合法性校验*/
        if (Integer.valueOf(cronArray[1]) >= 60 || Integer.valueOf(cronArray[2]) >= 24) {
            int extraMinutes = Integer.valueOf(cronArray[1]) % 60;
            int extraHours = (Integer.valueOf(cronArray[2])+ Integer.valueOf(cronArray[1]) / 60)%24;
            cronArray[2] = String.valueOf(extraHours);
            cronArray[1] = String.valueOf(extraMinutes);
        }
        /* 持续时间一天以内*/
        if (duration < 86400 && "*".equals(cronArray[3]) && "*".equals(cronArray[4]) && "?".equals(cronArray[5])) {

        } else if (duration > 86400 && duration < 86400 * 7) {
            cronArray[5] = String.valueOf((Integer.valueOf(cronArray[5]) + days) % 7 + 1);
        } else if (duration > 86400 * 7 && duration < 86400 * 32) {
            if("L".equals(cronArray[3])){
                cronArray[3]=String.valueOf(days);
            }else {
                cronArray[3] = String.valueOf(((Integer.valueOf(cronArray[3]) + days) % 31) + 1);
            }
        }
        String result = "";
        for (String s : cronArray) {
            result += s + " ";
        }
        return result;
    }

    /**
     * 随机生成一个corn表达式
     */
    public static String getRandomCron() {
        int min = 2;
        int max = 12;
        String cron = formatDateByPattern(new Date());
        //将生成的cron最后一个参数年转化为通配符
        String[] cronArray = cron.split(" ");
        Random random = new Random();
        int s = random.nextInt(max) % (max - min + 1) + min;
        int second = random.nextInt(58) % (57) + 2;
        int second1 = random.nextInt(58) % (57) + 2;
        String corn = second + " " + second1 + " " + "0/" + String.valueOf(s) + " " + "* * ?";
        return corn;
    }

    /**
     * 根据设置方式生成cron
     * @param unit (1分；2小时；3日；4月)
     * @param space
     * @return
     */
    public static String generateCron(String unit, Integer space) {
        String cron = formatDateByPattern(new Date());
        //将生成的cron最后一个参数年转化为通配符
        String[] cronArray = cron.split(" ");
        if("1".equals(unit)) {
            cron = cronArray[0] + " " + "1/" + space + " * * * ?";
        }else if("2".equals(unit)){
            cron = cronArray[0] + " " + cronArray[1] + " " + "1/" + space + " * * ?";
        }else if("3".equals(unit)){
            cron = cronArray[0] + " " + cronArray[1] + " " + cronArray[2] + " " + "1/" + space + " * ?";
        }else if("4".equals(unit)){
            cron = cronArray[0] + " " + cronArray[1] + " " + cronArray[2] + " 1 " + "1/" + space + " ?";
        }
        return cron;
    }
}
