package com.zzsn.event.kafka;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.EventAnalysisReport;
import com.zzsn.event.entity.EventContext;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.service.EventAnalysisReportService;
import com.zzsn.event.service.EventContextService;
import com.zzsn.event.service.SubjectAnalysisService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * @author lkg
 * 监听消费kafka消息
 * 消费topic中消息(注意：如果监听的topic不存在则会自动创建)
 * @date 2022/7/15
 */
@Slf4j
@Component
public class KafkaConsumer {
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private EventAnalysisReportService eventAnalysisReportService;
    @Autowired
    private EventContextService eventContextService;


    /**
     * 获取-事件脉络-分析结果数据，并入库
     *
     * @param record 接收的kafka数据
     */
    @KafkaListener(topics = {"eventContextTopic"})
    public void eventContext_new(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isEmpty(value)) {
            return;
        }
        List<eventContextKafkaVO> eventContextKafkaList = JSON.parseArray(value, eventContextKafkaVO.class);
        for (eventContextKafkaVO contextKafkaVO : eventContextKafkaList) {
            String uniqueCode = contextKafkaVO.getUniqueCode();
            String eventId = contextKafkaVO.getEventId();
            LambdaQueryWrapper<EventContext> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(EventContext::getEventId, eventId).eq(EventContext::getUniqueCode, uniqueCode);
            EventContext one = eventContextService.getOne(queryWrapper);
            Integer maxSortNumber = eventContextService.getMaxSortNumber(eventId);
            String keyNodeId;
            if (one == null) {
                EventContext keyNode = new EventContext();
                keyNode.setEventId(eventId);
                keyNode.setTitle(contextKafkaVO.getTitle());
                keyNode.setPublishDate(contextKafkaVO.getPublishDate());
                keyNode.setUniqueCode(uniqueCode);
                keyNode.setYnKeyNode(1);
                keyNode.setParentId("0");
                maxSortNumber = maxSortNumber + 1;
                keyNode.setSortNumber(maxSortNumber);
                eventContextService.save(keyNode);
                keyNodeId = keyNode.getId();
            } else {
                keyNodeId = one.getId();
            }
            List<EventContext> eventContextList = contextKafkaVO.getRelatedArticles();
            if (CollectionUtils.isNotEmpty(eventContextList)) {
                for (int i = 1; i <= eventContextList.size(); i++) {
                    EventContext eventContext = eventContextList.get(i);
                    eventContext.setEventId(eventId);
                    eventContext.setParentId(keyNodeId);
                    eventContext.setSortNumber(maxSortNumber + i);
                    eventContext.setCreateWay(2);
                }
                eventContextService.saveBatch(eventContextList);
            }
        }
    }




    /**
     * 获取-事件脉络-分析结果数据，并入库
     *
     * @param record 接收的kafka数据
     */
    @KafkaListener(topics = {Constants.EVENT_CONTEXT_RECEIVE_TOPIC})
    public void eventContext(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isNotEmpty(value)) {
            String subjectId;
            Integer category = 2;
            try {
                List<SubjectAnalysis> subjectAnalyses = JSON.parseArray(value, SubjectAnalysis.class);
                subjectId = subjectAnalyses.get(0).getSubjectId();
                subjectAnalyses.forEach(e -> {
                    e.setCategory(category);
                    e.setTitle(removeNonBmpUniCodes(e.getTitle()));
                });
                LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.eq(SubjectAnalysis::getSubjectId, subjectId).eq(SubjectAnalysis::getCategory, category);
                int count = subjectAnalysisService.count(queryWrapper);
                if (count > 0) {
                    subjectAnalysisService.remove(queryWrapper);
                }
                subjectAnalysisService.saveBatch(subjectAnalyses);
                log.info("id为-{}-的专题,此次-事件脉络-数据更新完成", subjectId);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 获取-伪事件脉络-分析结果数据，并入库
     *
     * @param record 接收的kafka数据
     */
    @KafkaListener(topics = {Constants.FAKE_EVENT_CONTEXT_RECEIVE_TOPIC})
    public void eventContext_fake(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isNotEmpty(value)) {
            String subjectId;
            Integer category = 3;
            try {
                List<SubjectAnalysis> subjectAnalyses = JSON.parseArray(value, SubjectAnalysis.class);
                subjectId = subjectAnalyses.get(0).getSubjectId();
                subjectAnalyses.forEach(e -> {
                    e.setCategory(category);
                    e.setTitle(removeNonBmpUniCodes(e.getTitle()));
                });
                LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.eq(SubjectAnalysis::getSubjectId, subjectId).eq(SubjectAnalysis::getCategory, category);
                int count = subjectAnalysisService.count(queryWrapper);
                if (count > 0) {
                    subjectAnalysisService.remove(queryWrapper);
                }
                subjectAnalysisService.saveBatch(subjectAnalyses);
                log.info("id为-{}-的专题,此次-伪事件脉络-数据更新完成", subjectId);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 获取事件的分析报告数据
     *
     * @param record 接收的kafka数据
     * @author lkg
     * @date 2024/4/12
     */
    @KafkaListener(topics = {Constants.EVENT_REPORT_RECEIVE_TOPIC})
    public void eventReport(ConsumerRecord<String, String> record) {
        String value = record.value();
        EventAnalysisReport eventAnalysisReport = JSONObject.parseObject(value, EventAnalysisReport.class);
        eventAnalysisReportService.modify(eventAnalysisReport.getEventId(), eventAnalysisReport.getFilePath());
        log.info("id为-{}-的事件,分析报告更新完成", eventAnalysisReport.getEventId());
    }

    //去除特殊的字符，例如表情符
    private String removeNonBmpUniCodes(String str) {
        return StringUtils.isEmpty(str) ? null : str.replaceAll("[^\\u0000-\\uFFFF]", "");
    }
}
