package com.zzsn.event.service.impl;

import cn.hutool.core.map.MapUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.*;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.es.EsService;
import com.zzsn.event.mapper.EventMapper;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CodeGenerateUtil;
import com.zzsn.event.util.CronUtil;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.HanlpUtil;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.util.tree.TreeUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.ParsedStringTerms;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.ParsedTopHits;
import org.elasticsearch.search.aggregations.metrics.TopHitsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.jsoup.Jsoup;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @Description: 事件
 * @Author: jeecg-boot
 * @Date: 2024-03-14
 * @Version: V1.0
 */
@Slf4j
@Service
public class EventServiceImpl extends ServiceImpl<EventMapper, Event> implements IEventService {
    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private ISubjectTypeMapService iSubjectTypeMapService;
    @Autowired
    private ISubjectInfoSourceMapService iSubjectInfoSourceMapService;
    @Autowired
    private ISubjectKeywordsMapService iSubjectKeywordsMapService;
    @Autowired
    private IEventTagService eventTagService;
    @Autowired
    private EventRegionMapService eventRegionMapService;
    @Autowired
    private IProjectSubjectMapService iProjectSubjectMapService;
    @Autowired
    private ISubjectTypeService iSubjectTypeService;
    @Autowired
    private IEventCategoryService eventCategoryService;
    @Autowired
    private ISubjectSearchEnginesMapService subjectSearchEnginesMapService;
    @Autowired
    private ISubjectModelMapService subjectModelMapService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private EventAnalysisReportService eventAnalysisReportService;

    @Resource
    private RestHighLevelClient client;
    @Autowired
    private CommonService commonService;


    @Override
    public List<SubjectKafkaVo> progressList(Date disableDate) {
        return this.baseMapper.processList(disableDate);
    }

    @Override
    public List<SubjectKafkaVo> eventSubjectList() {
        return this.baseMapper.eventSubjectList();
    }

    @Override
    public EventVO queryInfo(String eventId) {
        EventVO eventVO = baseMapper.queryInfo(eventId);
        String relationEvents = eventVO.getRelationEvents();
        if (null != relationEvents) {
            List<String> split = Arrays.asList(relationEvents.split(","));
            List<EventVO> relationEventList = this.eventList(split);
            eventVO.setRelatedEventList(relationEventList);
        }
        EventTag one = eventTagService.getOne(new LambdaQueryWrapper<EventTag>()
                .eq(EventTag::getEventId, eventId)
                .last(" limit 1"));
        eventVO.setEventTag(one);
        List<RegionVO> regionList = eventRegionMapService.regionList(eventId);
        eventVO.setRegionList(regionList);
        KeywordsVO keywordsVO = keyWordsService.keywordInfoByEventId(eventId);
        eventVO.setKeywordsVO(keywordsVO);
        return eventVO;
    }

    @Override
    public IPage<EventManageVO> pageList(String eventName, Integer eventType, Integer facePublic, Integer publishStatus, String startTime, String endTime, String order, String orderType, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        List<EventManageVO> pageList = baseMapper.pageList(eventName, eventType, facePublic, publishStatus, startTime, endTime, order, orderType, offset, pageSize);
        //获取总条数
        Integer count = baseMapper.totalCount(eventName, eventType, facePublic, publishStatus, startTime, endTime);
        IPage<EventManageVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public IPage<EventFrontVO> frontAllPageList(String projectId,String eventId, String eventName, String eventType, String labelField, String labelName, String order, String orderType, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<String> eventTypes = new ArrayList<>();
        if (StringUtils.isNotEmpty(eventType)) {
            List<Node> nodes = eventCategoryService.categoryList();
            eventTypes = TreeUtil.belowList(nodes, eventType, true);
        }
        //获取当前登录人信息
        UserVo currentUser = UserUtil.getLoginUser();
        List<EventFrontVO> pageList = baseMapper.frontAllPageList(projectId, currentUser.getUsername(), eventId,eventName, eventTypes, labelField, labelName, type, order, orderType, offset, pageSize);
        if (CollectionUtils.isNotEmpty(pageList)) {
            Map<String, String> map = getFirstMap(pageList);
            if (MapUtil.isNotEmpty(map)) {
                for (EventFrontVO eventFrontVO : pageList) {
                    //获取专题资讯的首发来源
                    String firstOrigin = map.get(eventFrontVO.getId());
                    eventFrontVO.setFirstOrigin(firstOrigin);
                }
            }
            for (EventFrontVO eventFrontVO : pageList) {
                String createBy = eventFrontVO.getCreateBy();
                eventFrontVO.setOwner(StringUtils.isNotBlank(createBy) && createBy.equals(currentUser.getUsername()));
            }
        }
        //获取总条数
        Integer count = baseMapper.frontAllCount(projectId, currentUser.getUsername(), eventId,eventName, eventTypes, labelField, labelName, type);
        IPage<EventFrontVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public IPage<EventFrontVO> frontOwnerPageList(String projectId, String eventName, String eventType, String labelField, String labelName, String order, String orderType, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<String> eventTypes = new ArrayList<>();
        if (StringUtils.isNotEmpty(eventType)) {
            List<Node> nodes = eventCategoryService.categoryList();
            eventTypes = TreeUtil.belowList(nodes, eventType, true);
        }
        //获取当前登录人信息
        UserVo currentUser = UserUtil.getLoginUser();
        List<EventFrontVO> pageList = baseMapper.frontOwnerPageList(projectId, currentUser.getUsername(), eventName, eventTypes, labelField, labelName, type, order, orderType, offset, pageSize);
        if (CollectionUtils.isNotEmpty(pageList)) {
            Map<String, String> map = getFirstMap(pageList);
            if (MapUtil.isNotEmpty(map)) {
                for (EventFrontVO eventFrontVO : pageList) {
                    //获取专题资讯的首发来源
                    String firstOrigin = map.get(eventFrontVO.getId());
                    eventFrontVO.setFirstOrigin(firstOrigin);
                }
            }
        }
        //获取总条数
        Integer count = baseMapper.frontOwnerCount(projectId, currentUser.getUsername(), eventName, eventTypes, labelField, labelName, type);
        IPage<EventFrontVO> pageData = new Page<>(pageNo, pageSize, count);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public Page<EventNewPlatVO> newPlatPageList(SubjectCondition subjectCondition, Integer pageNo, Integer pageSize) {
        int offset = (pageNo - 1) * pageSize;
        //查询类别id的所有明细id
        List<String> typeIds = new ArrayList<>();
        String subjectTypeId = subjectCondition.getSubjectTypeId();
        if (StringUtils.isNotEmpty(subjectTypeId) && !"0".equals(subjectTypeId)) {
            typeIds = iSubjectTypeService.belowIdList(subjectTypeId, 2);
        }
        subjectCondition.setTypeIds(typeIds);
        //在根据所有明细节点查出专题列表
        List<EventNewPlatVO> pageList = baseMapper.newPlatPageList(subjectCondition, offset, pageSize);
        long t1 = System.currentTimeMillis();
        count(pageList);
        log.info("数量统计总共耗时====" + (System.currentTimeMillis() - t1));
        Page<EventNewPlatVO> pageData = new Page<>(pageNo, pageSize);
        pageData.setRecords(pageList);
        return pageData;
    }

    @Override
    public List<EventTopVO> topEventList(String projectId, String createBy,String startTime, String endTime, Integer type, Integer pageSize) {
        return baseMapper.topEventList(projectId,createBy,startTime, endTime, type, 0, pageSize);
    }

    @Override
    public List<EventRegionVO> listByRegion(String projectId, String createBy,Integer type, String name) {
        return baseMapper.listByRegion(projectId,createBy,type, name);
    }

    private Map<String, String> getFirstMap(List<EventFrontVO> pageList) {
        Map<String, String> map = new HashMap<>();
        List<String> eventIdList = new ArrayList<>();
        pageList.forEach(e -> eventIdList.add(e.getId()));
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        //根据设备查询设备的相关信息
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.from(0);
        searchSourceBuilder.size(0);
        searchSourceBuilder.sort("publishDate", SortOrder.ASC);
        BoolQueryBuilder boolQueryBuilder = new BoolQueryBuilder();
        boolQueryBuilder.must(QueryBuilders.termsQuery("subjectId.keyword", eventIdList));
        //聚合搜索
        TermsAggregationBuilder one = AggregationBuilders.terms("one").field("subjectId.keyword").size(eventIdList.size());
        //ES分组取每组第一条Java写法
        TopHitsAggregationBuilder topHitsAggregationBuilder = AggregationBuilders.topHits("top_docs")
                .sort("publishDate", SortOrder.ASC)
                .size(1);
        one.subAggregation(topHitsAggregationBuilder);
        searchSourceBuilder.aggregation(one);
        searchSourceBuilder.query(boolQueryBuilder);
        searchRequest.source(searchSourceBuilder);
        // 查询ES
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            ParsedStringTerms oneTerms = searchResponse.getAggregations().get("one");
            List<? extends Terms.Bucket> onebuckets = oneTerms.getBuckets();
            for (Terms.Bucket threebucket : onebuckets) {
                ParsedTopHits topDetail = threebucket.getAggregations().get("top_docs");
                SearchHit[] hits = topDetail.getHits().getHits();
                Map<String, Object> latestDocument = hits[0].getSourceAsMap();
                map.put(latestDocument.get("subjectId").toString(), latestDocument.get("origin").toString());
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return map;
    }

    @Override
    @Transactional
    public Event saveMain(AddEventParam addEventParam) {
        Event event = new Event();
        BeanUtils.copyProperties(addEventParam, event);
        event.setStatus(1);
        Date endTime = event.getEndTime();
        if (endTime == null) {
            endTime = DateUtil.addDate(new Date(), 15);
        }
        event.setEndTime(endTime);
        //事件专题的默认分析规则参数-必填
        event.setIncreAnaRule(20);
        event.setTotalAnaRule(50);
        event.setTimeAnaRule(5);
        String subjectCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.EVENT_DEFAULT.getValue());
        event.setEventCode(subjectCode);
        //默认发布
        event.setPublishStatus(1);
        event.setPublishDate(DateUtil.dateToString(new Date()));
        event.setUnit("2");
        event.setSpace(1);
        String cron = CronUtil.generateCron("2", 1);
        event.setCron(cron);
        baseMapper.insert(event);
        String eventId = event.getId();
        CompletableFuture.runAsync(() -> {
            //插入专题-类别、项目的绑定关系
            saveMapMain(event, addEventParam);
            //事件标签
            eventTagService.save(EventTag.builder().eventId(eventId).build());
            //地缘关系绑定
            List<RegionVO> regionList = addEventParam.getRegionList();
            addRegionMap(eventId, regionList);
            //默认绑定搜索引擎、模型
            saveDefaultMap(eventId);
        });
        return event;
    }

    @Override
    @Transactional
    public void updateMain(AddEventParam addEventParam) {
        Event subject = new Event();
        BeanUtils.copyProperties(addEventParam, subject);
        String eventId = subject.getId();
        baseMapper.updateById(subject);
        CompletableFuture.runAsync(() -> {
            //专题-类别、项目的绑定关系处理
            //删除专题-类别绑定关系
            iSubjectTypeMapService.deleteBySubjectId(subject.getId());
            //删除专题-项目的绑定关系
            iProjectSubjectMapService.deleteBySubjectId(subject.getId());
            saveMapMain(subject, addEventParam);
            //地域关系处理
            LambdaQueryWrapper<EventRegionMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(EventRegionMap::getEventId, eventId);
            eventRegionMapService.remove(queryWrapper);
            List<RegionVO> regionList = addEventParam.getRegionList();
            addRegionMap(eventId, regionList);
        });
    }

    @Override
    @Transactional
    public void deleteMain(String id) {
        baseMapper.deleteById(id);
        CompletableFuture.runAsync(() -> {
            //删除与信息源的关联关系
            iSubjectInfoSourceMapService.delete(id);
            //删除与关键词组的关联关系
            iSubjectKeywordsMapService.delete(id);
            //删除地域关系
            eventRegionMapService.remove(Wrappers.<EventRegionMap>lambdaQuery().eq(EventRegionMap::getEventId, id));
            //删除专题-类别绑定关系
            iSubjectTypeMapService.deleteBySubjectId(id);
            //删除专题-项目的绑定关系
            iProjectSubjectMapService.deleteBySubjectId(id);
            //删除事件标签
            eventTagService.remove(Wrappers.<EventTag>lambdaQuery().eq(EventTag::getEventId, id));
            //删除事件-模型关系
            subjectModelMapService.remove(Wrappers.<SubjectModelMap>lambdaQuery().eq(SubjectModelMap::getSubjectId, id));
            //删除事件-搜索引擎关系
            subjectSearchEnginesMapService.remove(Wrappers.<SubjectSearchEnginesMap>lambdaQuery().eq(SubjectSearchEnginesMap::getSubjectId, id));
            //删除事件 观点分析、事件脉络数据
            subjectAnalysisService.remove(Wrappers.<SubjectAnalysis>lambdaQuery().eq(SubjectAnalysis::getSubjectId,id));
            //删除事件 分析报告数据
            eventAnalysisReportService.delete(id);
        });
    }

    @Override
    public List<ModelVO> modelList() {
        return baseMapper.modelList();
    }

    @Override
    public List<EventVO> eventList(List<String> eventIdList) {
        return baseMapper.eventList(eventIdList);
    }

    @Override
    public List<String> bindKeyWordsIdList(List<String> subjectIds) {
        return baseMapper.bindKeyWordsIdList(subjectIds);
    }

    @Override
    public List<EventExcelVO> frontAllList(String projectId, String createBy, List<String> eventIdList,String eventName,
                                           String eventType, String labelField,String labelName, Integer size) {
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<String> eventTypes = new ArrayList<>();
        if (StringUtils.isNotEmpty(eventType)) {
            List<Node> nodes = eventCategoryService.categoryList();
            eventTypes = TreeUtil.belowList(nodes, eventType, true);
        }
        return baseMapper.frontAllList(projectId,createBy,eventIdList,eventName,eventTypes,labelField,labelName,type, size);
    }

    @Override
    public List<EventExcelVO> frontOwnerList(String projectId, String createBy, List<String> eventIdList,String eventName,
                                             String eventType, String labelField,String labelName, Integer size) {
        Integer type = null;
        if (StringUtils.isNotEmpty(labelField) && labelField.equals("event_label")) {
            type = 1;
        }
        List<String> eventTypes = new ArrayList<>();
        if (StringUtils.isNotEmpty(eventType)) {
            List<Node> nodes = eventCategoryService.categoryList();
            eventTypes = TreeUtil.belowList(nodes, eventType, true);
        }
        return baseMapper.frontOwnerList(projectId,createBy,eventIdList,eventName,eventTypes,labelField,labelName,type, size);
    }


    private void addRegionMap(String eventId, List<RegionVO> regionList) {
        if (CollectionUtils.isNotEmpty(regionList)) {
            List<EventRegionMap> dataList = new ArrayList<>();
            for (RegionVO regionVO : regionList) {
                EventRegionMap eventRegionMap = new EventRegionMap();
                eventRegionMap.setEventId(eventId);
                eventRegionMap.setRegionId(regionVO.getId());
                eventRegionMap.setTopRegionId(regionVO.getTopId());
                eventRegionMap.setType(regionVO.getType());
                dataList.add(eventRegionMap);
            }
            eventRegionMapService.saveBatch(dataList);
        }
    }

    private void saveMapMain(Event subject, AddEventParam addEventParam) {
        if (StringUtils.isNotEmpty(addEventParam.getSubjectTypeId())) {
            SubjectTypeMap subjectTypeMap = new SubjectTypeMap();
            subjectTypeMap.setSubjectId(subject.getId());
            subjectTypeMap.setUpdateBy(subject.getUpdateBy());
            subjectTypeMap.setUpdateTime(subject.getUpdateTime());
            subjectTypeMap.setTypeId(addEventParam.getSubjectTypeId());
            iSubjectTypeMapService.save(subjectTypeMap);
        }
        if (StringUtils.isNotEmpty(addEventParam.getProjectId())) {
            ProjectSubjectMap projectSubjectMap = new ProjectSubjectMap();
            projectSubjectMap.setProjectId(addEventParam.getProjectId());
            projectSubjectMap.setSubjectId(subject.getId());
            iProjectSubjectMapService.save(projectSubjectMap);
        }
    }

    @Override
    public void saveDefaultMap(String eventId) {
        //默认引擎
        Map<String, String> defaultSearchEngines = Constants.DEFAULT_SEARCH_ENGINE;
        List<SubjectSearchEnginesMap> enginesMaps = new ArrayList<>();
        for (Map.Entry<String, String> entry : defaultSearchEngines.entrySet()) {
            SubjectSearchEnginesMap enginesMap = new SubjectSearchEnginesMap();
            enginesMap.setSubjectId(eventId);
            enginesMap.setSearchEngineId(entry.getKey());
            enginesMaps.add(enginesMap);
        }
        subjectSearchEnginesMapService.saveBatch(enginesMaps);
        //默认模型
        Map<String, String> defaultModels = Constants.DEFAULT_MODEL;
        List<SubjectModelMap> modelMaps = new ArrayList<>();
        for (Map.Entry<String, String> entry : defaultModels.entrySet()) {
            SubjectModelMap modelMap = new SubjectModelMap();
            modelMap.setSubjectId(eventId);
            modelMap.setModelId(entry.getKey());
            modelMap.setType(entry.getValue());
            modelMap.setSign(1);
            modelMaps.add(modelMap);
        }
        subjectModelMapService.saveBatch(modelMaps);
    }

    //查询每个专题的数量
    private Map<String, Integer> subjectInfoCountMap(List<String> subjectIdList, List<Integer> checkStatusList) {
        Map<String, Integer> map = new HashMap<>(subjectIdList.size());
        SearchRequest searchRequest = new SearchRequest(Constants.SUBJECT_INDEX);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        //默认最大数量是10000，设置为true后，显示准确数量
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.size(0);
        //创建查询对象
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("subjectId.keyword", subjectIdList));
        //默认查询没删除的
        boolQuery.must(QueryBuilders.matchQuery("deleteFlag", 0));
        if (checkStatusList != null && checkStatusList.size() > 0) {
            boolQuery.must(QueryBuilders.termsQuery("checkStatus", checkStatusList));
        }
        searchSourceBuilder.query(boolQuery);
        //分组
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("group_subjectId").field("subjectId.keyword");
        searchSourceBuilder.aggregation(aggregationBuilder);
        searchRequest.source(searchSourceBuilder);
        try {
            SearchResponse searchResponse = client.search(searchRequest, RequestOptions.DEFAULT);
            //获取分组桶
            Aggregations aggregations = searchResponse.getAggregations();
            //获取id分组集合
            Terms parsedStringTerms = aggregations.get("group_subjectId");
            List<? extends Terms.Bucket> buckets = parsedStringTerms.getBuckets();
            for (Terms.Bucket bucket : buckets) {
                String subjectId = bucket.getKeyAsString();
                long collectCount = bucket.getDocCount();
                map.put(subjectId, (int) collectCount);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return map;
    }

    private void count(List<EventNewPlatVO> pageList) {
        if (CollectionUtils.isNotEmpty(pageList)) {
            List<String> idList = pageList.stream().map(EventNewPlatVO::getId).collect(Collectors.toList());
            Map<String, Integer> infoSourceNumMap = new HashMap<>();
            long t1 = System.currentTimeMillis();
            Map<String, Integer> keyWordsNumMap = commonService.bindKeyWordsCountList(idList).stream().collect(Collectors.toMap(SubjectPage::getId, SubjectPage::getKeyWordsNum));
            long t2 = System.currentTimeMillis();
            log.info("关键词数量查询耗时==={}",t2-t1);
            Map<String, List<SubjectSourceVO>> collect = commonService.bindSourceList(idList).stream().collect(Collectors.groupingBy(SubjectSourceVO::getSubjectId));
            long t3 = System.currentTimeMillis();
            log.info("信息源数量查询耗时==={}",t3-t2);
            for (Map.Entry<String, List<SubjectSourceVO>> entry : collect.entrySet()) {
                String subjectId = entry.getKey();
                List<SubjectSourceVO> value = entry.getValue();
                infoSourceNumMap.put(subjectId, value.size());
            }
            long t4 = System.currentTimeMillis();
            Map<String, Integer> subjectInfoCountMap = subjectInfoCountMap(idList, null);
            log.info("资讯数量查询耗时==={}",System.currentTimeMillis()-t4);
            for (EventNewPlatVO newPlatVO : pageList) {
                int keyWordsNum = null == keyWordsNumMap.get(newPlatVO.getId()) ? 0 : keyWordsNumMap.get(newPlatVO.getId());
                int infoSourceNum = null == infoSourceNumMap.get(newPlatVO.getId()) ? 0 : infoSourceNumMap.get(newPlatVO.getId());
                //查询每个专题现有的信息数量
                int subjectInfoCount = null == subjectInfoCountMap.get(newPlatVO.getId()) ? 0 : subjectInfoCountMap.get(newPlatVO.getId());
                newPlatVO.setSubjectInfoNum(subjectInfoCount);
                newPlatVO.setInfoSourceNum(infoSourceNum);
                newPlatVO.setKeyWordsNum(keyWordsNum);
            }
        }
    }
}
