package com.zzsn.event.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.CustomerDataPermissionMap;
import com.zzsn.event.entity.SubjectType;
import com.zzsn.event.entity.SubjectTypeMap;
import com.zzsn.event.entity.SysUserDataPermission;
import com.zzsn.event.service.ICustomerDataPermissionMapService;
import com.zzsn.event.service.ISubjectTypeMapService;
import com.zzsn.event.service.ISubjectTypeService;
import com.zzsn.event.service.ISysUserDataPermissionService;
import com.zzsn.event.util.tree.Node;
import com.zzsn.event.util.tree.TreeUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 事件后台管理-事件分类
 *
 * @author lkg
 * @date 2024/4/29
 */
@Slf4j
@Api(tags = "-事件分类")
@RestController
@RequestMapping("/manage/subjectType")
public class EventTypeController {

    @Autowired
    private ISubjectTypeService subjectTypeService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private ICustomerDataPermissionMapService customerDataPermissionMapService;
    @Autowired
    private ISysUserDataPermissionService sysUserDataPermissionService;




    /**
     * 事件分类列表-树型结构
     *
     * @author lkg
     * @date 2024/4/28
     */
    @GetMapping("/list")
    public Result<?> typeList() {
        //获取当前登录用户
        UserVo currentUser = UserUtil.getLoginUser();
        Integer category = currentUser.getCategory();
        String userId = null;
        String customerId = null;
        if (category.equals(Constants.COMMON_USER)) {
            userId = currentUser.getUserId();
        } else if (category.equals(Constants.ADMIN_USER)) {
            customerId = currentUser.getCustomerId();
        }
        List<Node> nodes = subjectTypeService.enableList(userId,customerId);
        List<Node> tree = TreeUtil.tree(nodes, "0");
        return Result.OK(tree);
    }

    /**
     * 新增专题分类
     *
     * @param subjectType 专题分类
     * @author lkg
     * @date 2024/4/29
     */
    @PostMapping("/add")
    public Result<?> addSubjectType(@RequestBody SubjectType subjectType) {
        UserVo currentUser = UserUtil.getLoginUser();
        String pid = subjectType.getPid();
        if (!"0".equals(pid)) {
            int count = subjectTypeMapService.count(new LambdaQueryWrapper<SubjectTypeMap>().eq(SubjectTypeMap::getTypeId, pid));
            if (count > 0) {
                return Result.FAIL(501, "当前分类下存在专题");
            }
        }
        subjectType.setCustomerId(currentUser.getCustomerId());
        subjectTypeService.add(subjectType);
        //权限
        if (currentUser.getCategory().equals(Constants.ADMIN_USER)) {
            customerDataPermissionMapService.save(new CustomerDataPermissionMap().setCustomerId(currentUser.getCustomerId()).setPermissionId(subjectType.getId()).setCategory(Constants.PERMISSION_SUBJECT));
        }
        if (currentUser.getCategory().equals(Constants.COMMON_USER)) {
            sysUserDataPermissionService.save(new SysUserDataPermission().setUserId(currentUser.getUserId()).setPermissionId(subjectType.getId()).setCategory(Constants.PERMISSION_SUBJECT));
            customerDataPermissionMapService.save(new CustomerDataPermissionMap().setCustomerId(currentUser.getCustomerId()).setPermissionId(subjectType.getId()).setCategory(Constants.PERMISSION_SUBJECT));
        }
        return Result.OK();
    }

    /**
     * 编辑
     *
     * @param subjectType 专题分类
     * @author lkg
     * @date 2024/4/29
     */
    @PostMapping("/edit")
    public Result<?> edit(@RequestBody SubjectType subjectType){
        subjectTypeService.edit(subjectType);
        return Result.OK();
    }

    /**
     * 删除
     *
     * @param id 专题分类id
     * @author lkg
     * @date 2024/4/29
     */
    @GetMapping("/delete")
    public Result<?> delete(@RequestParam String id){
        int count = subjectTypeMapService.count(new LambdaQueryWrapper<SubjectTypeMap>().eq(SubjectTypeMap::getTypeId, id));
        if (count > 0) {
            return Result.FAIL(501, "当前分类下存在专题");
        }
        //删除数据权限关系
        customerDataPermissionMapService.remove(Wrappers.<CustomerDataPermissionMap>lambdaQuery().eq(CustomerDataPermissionMap::getPermissionId, id));
        sysUserDataPermissionService.remove(Wrappers.<SysUserDataPermission>lambdaQuery().eq(SysUserDataPermission::getPermissionId, id));
        subjectTypeService.delete(id);
        return Result.OK();
    }
}
