package com.zzsn.event.controller.common;

import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.EventAnalysisVersion;
import com.zzsn.event.entity.EventAnalysisVersionRecord;
import com.zzsn.event.entity.EventLlmConfig;
import com.zzsn.event.enums.AnalysisColumnEnum;
import com.zzsn.event.service.*;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.vo.EventVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * 大模型配置
 *
 * @author lkg
 * @date 2025/7/28
 */
@Slf4j
@RestController
@RequestMapping("/llm/config")
public class LLmConfigController {

    @Autowired
    private EventLlmConfigService llmConfigService;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private EventAnalysisVersionService eventAnalysisVersionService;
    @Autowired
    private EventAnalysisVersionRecordService eventAnalysisVersionRecordService;
    @Autowired
    private IEventService eventService;

    /**
     * 事件下大模型配置信息
     *
     * @param eventId 事件id
     * @author lkg
     * @date 2025/7/28
     */
    @GetMapping("/list")
    public Result<?> list(@RequestParam String eventId) {
        List<EventLlmConfig> configList = llmConfigService.getConfigList(eventId);
        return Result.OK(configList);
    }

    /**
     * 批量编辑事件的大模型配置信息
     *
     * @param eventLlmConfigList 大模型配置信息
     * @author lkg
     * @date 2025/7/28
     */
    @PostMapping("/modifyBatch")
    public Result<?> modifyBatch(@RequestBody List<EventLlmConfig> eventLlmConfigList) {
        llmConfigService.modifyBatch(eventLlmConfigList);
        return Result.OK();
    }
    /**
     * 编辑事件栏目的大模型配置信息,同时触发大模型更新结果
     *
     * @param eventLlmConfig 大模型配置信息
     * @author lkg
     * @date 2025/7/28
     */
    @PostMapping("/modifySingle")
    public Result<?> modifySingle(@RequestBody EventLlmConfig eventLlmConfig) {
        llmConfigService.modifySingle(eventLlmConfig);
        CompletableFuture.runAsync(() ->{
            AnalysisColumnEnum analysisColumnEnum = AnalysisColumnEnum.getByCode(eventLlmConfig.getColumnCode());
            if (analysisColumnEnum != null) {
                String eventId = eventLlmConfig.getEventId();
                EventVO event = eventService.queryInfo(eventId);
                String eventName = event.getEventName();
                String content  = "事件标题；" + eventName + "\n事件摘要:" + event.getEventDescribe();
                //调用大模型更细结果
                String llmResult = analysisService.llmResult(eventId,content, null, null, eventLlmConfig.getColumnCode());
                EventAnalysisVersion eventAnalysisVersion = eventAnalysisVersionService.latestVersion(eventId);
                String versionId;
                if (eventAnalysisVersion == null) {
                    eventAnalysisVersion = new EventAnalysisVersion();
                    eventAnalysisVersion.setEventId(eventId);
                    eventAnalysisVersion.setVersionName("版本" + DateUtil.dateToString(new Date()));
                    eventAnalysisVersionService.save(eventAnalysisVersion);
                    versionId = eventAnalysisVersion.getId();
                } else {
                    versionId = eventAnalysisVersion.getId();
                }
                EventAnalysisVersionRecord versionRecord = EventAnalysisVersionRecord.of(versionId, analysisColumnEnum, llmResult);
                eventAnalysisVersionRecordService.save(versionRecord);
                log.info("{}-事件分析【{}】重新生成逻辑完成。", eventName, analysisColumnEnum.getName());
            } else {
                log.info("未找到对应的事件栏目【{}】", eventLlmConfig.getColumnCode());
            }
        });
        return Result.OK();
    }
}
