package com.zzsn.event.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.*;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.es.EsService;
import com.zzsn.event.feign.api.RemoteModelService;
import com.zzsn.event.service.*;
import com.zzsn.event.util.CodeGenerateUtil;
import com.zzsn.event.util.EsOpUtil;
import com.zzsn.event.util.PythonUtil;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.Label;
import com.zzsn.event.vo.es.SpecialInformation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * 简化专题创建流程
 *
 * @author lkg
 * @date 2025/1/9
 */
@Slf4j
@Service
public class SubjectSimpleServiceImpl implements SubjectSimpleService {

    @Autowired
    private SubjectService subjectService;
    @Autowired
    private IKeyWordsService keywordWordsService;
    @Autowired
    private IKeywordsTypeMapService keywordsTypeMapService;
    @Autowired
    private SubjectKeywordsGroupRelationService subjectKeywordsGroupRelationService;
    @Autowired
    private ISubjectKeywordsMapService subjectKeywordsMapService;
    @Autowired
    private ClbModelArrangeSubjectMapService clbModelArrangeSubjectMapService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private IProjectSubjectMapService projectSubjectMapService;
    @Autowired
    private ISubjectSearchEnginesMapService subjectSearchEnginesMapService;
    @Autowired
    private ISubjectInfoSourceMapService subjectInfoSourceMapService;
    @Autowired
    private ISubjectModelMapService subjectModelMapService;
    @Autowired
    private ScoreModelService scoreModelService;
    @Autowired
    private ConfigurationMessageService configurationMessageService;
    @Autowired
    private InformationService informationService;

    @Autowired
    private EsOpUtil esOpUtil;
    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private PythonUtil pythonUtil;
    @Autowired(required = false)
    private RemoteModelService remoteModelService;

    @Value("${clb.subject.default.processing.advanceMonth}")
    private Integer defaultAdvanceMonth;

    //关键词默认分类
    private static final String KEYWORDS_TYPE_ID = "1476498704680194050";
    //默认项目
    private static final String PROJECT_ID = "1476527644425682945";


    @Override
    @Transactional
    public Subject createSubject(SubjectSimpleVO subjectSimpleVO) {
        SubjectPage subjectPage = new SubjectPage();
        BeanUtils.copyProperties(subjectSimpleVO, subjectPage);
        subjectPage.setProjectId(PROJECT_ID);
        Subject subject = subjectService.saveMain(subjectPage);
        String subjectId = subject.getId();
        //关键词绑定
        List<SearchWordVO> keywords = subjectSimpleVO.getKeywords();
        modifyKeyword(subjectId, subject.getSubjectName(), keywords);
        //默认绑定tpu流程
        List<ClbModelArrangeSubjectMap> tpuList = new ArrayList<>();
        ClbModelArrangeSubjectMap tpu = new ClbModelArrangeSubjectMap();
        tpu.setSubjectId(subject.getId());
        tpu.setArrangeId("1877652205629173761");
        tpu.setType("baseDateToSubject");
        tpuList.add(tpu);
        ClbModelArrangeSubjectMap tpu1 = new ClbModelArrangeSubjectMap();
        tpu1.setSubjectId(subject.getId());
        tpu1.setArrangeId("1887436365952548866");
        tpu1.setType("yjzxPlatFormToSubject");
        tpuList.add(tpu1);
        clbModelArrangeSubjectMapService.saveBatch(tpuList);
        //默认通用打分配置
        String defaultConfig = "[{\"id\": \"1-1\", \"name\": \"信息源组\", \"children\": [], \"indexWeight\": 10}, {\"id\": \"1-2\", \"name\": \"文章长度\", \"indexWeight\": 15, \"keyWordsTopLimit\": 2500, \"keyWordsLowerLimit\": 500}, {\"id\": \"1-3\", \"name\": \"内容\", \"children\": [{\"id\": \"1-3-1\", \"name\": \"内容-关键词\", \"keyWords\": \"KEY_WORD\", \"indexWeight\": 75, \"titleWeight\": 10, \"keyWordsTopLimit\": 15, \"keyWordsLowerLimit\": 3}]}]";
        List<SearchWordVO> collect = keywords.stream().filter(searchWordVO -> !"NOT".equalsIgnoreCase(searchWordVO.getSearchLogicRelationship())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(collect)) {
            defaultConfig = defaultConfig.replace("KEY_WORD", String.join("|", collect.stream().map(SearchWordVO::getSearchInfo).collect(Collectors.toList())));
        }
        ScoreModel scoreModel = new ScoreModel();
        scoreModel.setSubjectId(subjectId);
        scoreModel.setType("1");
        scoreModel.setData(defaultConfig);
        scoreModelService.save(scoreModel);
        //同步配置到采集
        configurationMessageService.bindInfoSourceSend(subjectId);
        configurationMessageService.bindKeyWordsSend(subjectId);
        return subject;
    }

    @Override
    public Boolean configVerification(SubjectParamsCheckVO subjectParamsCheckVO) {
        SubjectSimpleVO subjectSimpleVO = subjectParamsCheckVO.getSubjectSimpleVO();
        String subjectId = subjectSimpleVO.getId();
        SubjectDetailVO subjectDetailVO = queryInfo(subjectId);
        Date firstOpenTime = subjectDetailVO.getFirstOpenTime();
        //未启用过的可以直接修改,校验通过
        if (firstOpenTime == null) {
            return true;
        } else {
            //判断数据来源和资讯类型是否变化,变化则校验不通过;反之不通过
            if (!subjectSimpleVO.getDataSource().equals(subjectDetailVO.getDataSource()) || !subjectSimpleVO.getLibrary().equals(subjectDetailVO.getLibrary())) {
                return false;
            }
            //判断时间范围是否增加(新时间范围完全包含旧时间范围 标识增加),增加则校验通过，反之不通过
            Date today = new Date();
            Date timeEnable = subjectSimpleVO.getTimeEnable();
            if (timeEnable == null) {
                timeEnable = DateUtil.offsetMonth(firstOpenTime, -defaultAdvanceMonth);
            }
            Date timeDisable = subjectSimpleVO.getTimeDisable();
            if (timeDisable == null) {
                timeDisable = today;
            }
            Date oldTimeEnable = subjectDetailVO.getTimeEnable();
            if (oldTimeEnable == null) {
                oldTimeEnable = DateUtil.offsetMonth(firstOpenTime, -defaultAdvanceMonth);
            }
            Date oldTimeDisable = subjectDetailVO.getTimeDisable();
            if (oldTimeDisable == null) {
                oldTimeDisable = today;
            }
            if (!(!timeEnable.after(oldTimeEnable) && !timeDisable.before(oldTimeDisable))) {
                return false;
            }
            //判断关键词配置是否变化
            List<SearchWordVO> keywordsOld = subjectDetailVO.getKeywords();
            List<SearchWordVO> keywordsNew = subjectSimpleVO.getKeywords();
            boolean judgeKeyword = judgeKeyword(keywordsNew, keywordsOld);
            if (judgeKeyword) {
                return false;
            }
            //判断信息源标签配置是否变化
            boolean judgeInfoSourceLabel = judgeInfoSourceLabel(subjectId, subjectParamsCheckVO.getSubjectSourceTagVO());
            //判断定向信息源是否发生变化(前端传参)
            boolean directSourceChange = subjectParamsCheckVO.isDirectSourceChange();
            return !judgeInfoSourceLabel && !directSourceChange;
        }
    }


    /**
     * 判断关键词是否发生变化 true 是;false 否
     *
     * @param keywordsNew 新词(编辑时前端传参)
     * @param keywordsOld 旧词(已入库)
     * @author lkg
     * @date 2025/2/5
     */
    public boolean judgeKeyword(List<SearchWordVO> keywordsNew, List<SearchWordVO> keywordsOld) {
        boolean flag = false;
        //判断是否存在词组id为空的数据，有的发生变化;无则继续判断内容
        boolean present = keywordsNew.stream().anyMatch(searchWordVO -> StringUtils.isBlank(searchWordVO.getId()));
        if (present) {
            flag = true;
        } else if (keywordsNew.size() == keywordsOld.size()) {
            Map<String, SearchWordVO> oldMap = keywordsOld.stream().collect(Collectors.toMap(SearchWordVO::getId, searchWordVO -> searchWordVO));
            for (SearchWordVO searchWordVO : keywordsNew) {
                if (oldMap.containsKey(searchWordVO.getId())) {
                    SearchWordVO old = oldMap.get(searchWordVO.getId());
                    if (!(old.getSearchInfo().equals(searchWordVO.getSearchInfo())
                            && ((old.getSearchLogicRelationship() == null && searchWordVO.getSearchLogicRelationship() == null)
                            || old.getSearchLogicRelationship().equals(searchWordVO.getSearchLogicRelationship()))
                            && old.getSearchScope().equals(searchWordVO.getSearchScope()))) {
                        flag = true;
                    }
                } else {
                    flag = true;
                }
                if (flag) {
                    break;
                }
            }
        }
        return flag;
    }

    /**
     * 判断信息源配置是否发生变化 true 是;false 否
     *
     * @param subjectId              专题id
     * @param subjectSourceTagVO     信息源标签信息(编辑时前端传参)
     * @author lkg
     * @date 2025/2/5
     */
    private boolean judgeInfoSourceLabel(String subjectId, SubjectSourceTagVO subjectSourceTagVO) {
        boolean flag = false;
        LambdaQueryWrapper<SubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectInfoSourceMap::getSubjectId, subjectId);
        List<SubjectInfoSourceMap> mapList = subjectInfoSourceMapService.list(queryWrapper);
        //判断绑定的信息源标签是否发生变化
        if (CollectionUtils.isEmpty(mapList)) {
            if (subjectSourceTagVO != null) {
                flag = true;
            }
        } else {
            //判断 是否绑定信息源标签
            Set<String> sourceLabelCodeSet = mapList.stream().filter(e -> e.getType() == 12).map(SubjectInfoSourceMap::getSourceItemId).collect(Collectors.toSet());
            if (CollectionUtils.isNotEmpty(sourceLabelCodeSet)) {
                if (subjectSourceTagVO != null) {
                    Set<String> labelItemCodeSetAll = new HashSet<>();
                    List<InfoSourceLabelVO> labelList = subjectSourceTagVO.getLabelList();
                    for (InfoSourceLabelVO infoSourceLabelVO : labelList) {
                        List<InfoSourceLabelItemVO> infoSourceLabelItemList = infoSourceLabelVO.getInfoSourceLabelItemList();
                        Set<String> labelItemCodeSet = infoSourceLabelItemList.stream().map(InfoSourceLabelItemVO::getLabelItemCode).collect(Collectors.toSet());
                        labelItemCodeSetAll.addAll(labelItemCodeSet);
                    }
                    if (!sourceLabelCodeSet.equals(labelItemCodeSetAll)) {
                        flag = true;
                    }
                } else {
                    flag = true;
                }
            }
        }
        return flag;
    }


    @Override
    @Transactional
    public void editSubject(SubjectSimpleVO subjectSimpleVO) {
        SubjectPage subjectPage = new SubjectPage();
        BeanUtils.copyProperties(subjectSimpleVO, subjectPage);
        subjectPage.setProjectId(PROJECT_ID);
        subjectService.updateMain(subjectPage);
        //关键词绑定
        List<SearchWordVO> keywords = subjectSimpleVO.getKeywords();
        modifyKeyword(subjectSimpleVO.getId(), subjectSimpleVO.getSubjectName(), keywords);
        //同步配置到采集
        configurationMessageService.bindInfoSourceSend(subjectSimpleVO.getId());
        configurationMessageService.bindKeyWordsSend(subjectSimpleVO.getId());
    }

    @Override
    public List<SearchWordVO> subjectBindWordInfo(String subjectId) {
        List<SearchWordVO> wordInfoList = subjectKeywordsMapService.subjectBindWordInfo(subjectId);
        Map<String, SearchWordVO> map = wordInfoList.stream().collect(Collectors.toMap(SearchWordVO::getId, item -> item));
        LambdaQueryWrapper<SubjectKeywordsGroupRelation> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectKeywordsGroupRelation::getSubjectId, subjectId);
        List<SubjectKeywordsGroupRelation> relationList = subjectKeywordsGroupRelationService.list(queryWrapper);
        List<String> sortList = new ArrayList<>(relationList.size());
        for (SubjectKeywordsGroupRelation relation : relationList) {
            sortList.addAll(Arrays.asList(relation.getKeywordsGroupIds().split(",")));
            String relationType = relation.getRelationType();
            //paramStr：[{"title":"巴黎奥运会","value":"1830487575308136450"},{"title":"or","value":"|"},{"title":"栏目关键词","value":"1860252747367051265"}]
            String paramsStr = relation.getParamsStr();
            List<JSONObject> params = JSON.parseArray(paramsStr, JSONObject.class);
            if ("2".equals(relationType)) {
                if (params.size() > 1) {
                    for (int i = 1; i <= params.size() - 2; i += 2) {
                        JSONObject param1 = params.get(i);
                        String value = param1.get("value").toString();
                        String relationStr = null;
                        if ("&".equals(value)) {
                            relationStr = "and";
                        } else if ("|".equals(value)) {
                            relationStr = "or";
                        }
                        JSONObject param2 = params.get(i + 1);
                        String id = param2.get("value").toString();
                        map.get(id).setSearchLogicRelationship(relationStr);
                    }
                }
            } else if ("3".equals(relationType)) {
                String valueStr = params.get(0).get("value").toString();
                map.get(valueStr).setSearchLogicRelationship("and");
                for (int i = 1; i < params.size() - 2; i += 2) {
                    JSONObject param = params.get(i + 1);
                    String id = param.get("value").toString();
                    map.get(id).setSearchLogicRelationship("and");
                }
            }
        }
        List<SearchWordVO> wordList = new ArrayList<>(map.values());
        sortByAnotherList(wordList, sortList);
        return wordList;
    }

    @Override
    public SubjectDetailVO queryInfo(String subjectId) {
        SubjectDetailVO subjectDetailVO = subjectService.queryInfo(subjectId);
        if (subjectDetailVO != null) {
            List<SearchWordVO> list = this.subjectBindWordInfo(subjectId);
            subjectDetailVO.setKeywords(list);
        }
        return subjectDetailVO;
    }

    @Override
    public void removeSubject(String subjectId) {
        subjectService.removeById(subjectId);
        CompletableFuture.runAsync(() -> {
            //删除与类别的映射
            subjectTypeMapService.deleteBySubjectId(subjectId);
            //删除与信息源的关联关系
            subjectInfoSourceMapService.delete(subjectId);
            //获取绑定的关键词
            LambdaQueryWrapper<SubjectKeywordsMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectKeywordsMap::getSubjectId, subjectId);
            List<SubjectKeywordsMap> list = subjectKeywordsMapService.list(queryWrapper);
            if (CollectionUtils.isNotEmpty(list)) {
                List<String> keywordIds = list.stream().map(SubjectKeywordsMap::getKeywordsId).collect(Collectors.toList());
                //删除关键词
                keywordWordsService.removeByIds(keywordIds);
                //删除与关键词组的关联关系
                subjectKeywordsMapService.delete(subjectId);
                //删除关键词组关联关系
                subjectKeywordsGroupRelationService.remove(Wrappers.<SubjectKeywordsGroupRelation>lambdaQuery().eq(SubjectKeywordsGroupRelation::getSubjectId, subjectId));
            }
            //删除专题-项目的绑定关系
            projectSubjectMapService.deleteBySubjectId(subjectId);
            //删除事件-模型关系
            subjectModelMapService.remove(Wrappers.<SubjectModelMap>lambdaQuery().eq(SubjectModelMap::getSubjectId, subjectId));
            //删除事件-搜索引擎关系
            subjectSearchEnginesMapService.remove(Wrappers.<SubjectSearchEnginesMap>lambdaQuery().eq(SubjectSearchEnginesMap::getSubjectId, subjectId));
            //删除专题/事件-tpu流程关系
            clbModelArrangeSubjectMapService.remove(Wrappers.<ClbModelArrangeSubjectMap>lambdaQuery().eq(ClbModelArrangeSubjectMap::getSubjectId, subjectId));
            //向python发送消息结束
            //subjectService.send(subjectId, "-1");
        });
    }

    @Override
    public void selected(InfoDataSearchCondition searchCondition) {
        //勾选的主题信息列表
        List<Label> themeList = searchCondition.getThemeList();
        String[] fetchFields = new String[]{"id", "checkStatus","labels"};
        searchCondition.setFetchFields(fetchFields);
        Map<String, List<SpecialInformation>> updateMap = new HashMap<>();
        List<SpecialInformation> informationList = informationService.informationAllList(searchCondition);
        for (SpecialInformation information : informationList) {
            String index = information.getDbIndex();
            information.setCheckStatus(1);
            if (CollectionUtils.isNotEmpty(themeList)) {
                List<Label> labels = information.getLabels();
                //主题标签
                List<Label> collect = labels.stream().filter(e -> "thematic_information_column".equals(e.getLabelMark())).collect(Collectors.toList());
                //其他类型的标签
                labels.removeAll(collect);
                List<String> oldThemeIds = collect.stream().map(Label::getRelationId).collect(Collectors.toList());
                List<String> newThemeIds = themeList.stream().map(Label::getRelationId).collect(Collectors.toList());
                for (Label label : collect) {
                    if (newThemeIds.contains(label.getRelationId())) {
                        labels.add(label);
                    }
                }
                for (Label label : themeList) {
                    if (!oldThemeIds.contains(label.getRelationId())) {
                        labels.add(label);
                    }
                }
                information.setLabels(labels);
            }
            if (updateMap.containsKey(index)) {
                updateMap.get(index).add(information);
            } else {
                List<SpecialInformation> list = new ArrayList<>();
                list.add(information);
                updateMap.put(index, list);
            }
        }
        updateMap.forEach((k, v) -> esOpUtil.docUpdateBulk(k, v));
    }

    @Override
    public void removeSelected(InfoDataSearchCondition searchCondition) {
        String[] fetchFields = new String[]{"id", "checkStatus"};
        searchCondition.setFetchFields(fetchFields);
        Map<String, List<SpecialInformation>> updateMap = new HashMap<>();
        List<SpecialInformation> informationList = informationService.informationAllList(searchCondition);
        for (SpecialInformation information : informationList) {
            String index = information.getDbIndex();
            information.setCheckStatus(0);
            if (updateMap.containsKey(index)) {
                updateMap.get(index).add(information);
            } else {
                List<SpecialInformation> list = new ArrayList<>();
                list.add(information);
                updateMap.put(index, list);
            }
        }
        updateMap.forEach((k, v) -> esOpUtil.docUpdateBulk(k, v));
    }

    @Override
    public void batchRemove(InfoDataSearchCondition searchCondition) {
        String[] fetchFields = new String[]{"id", "deleteFlag"};
        searchCondition.setFetchFields(fetchFields);
        Map<String, List<SpecialInformation>> updateMap = new HashMap<>();
        List<SpecialInformation> informationList = informationService.informationAllList(searchCondition);
        for (SpecialInformation information : informationList) {
            String index = information.getDbIndex();
            information.setDeleteFlag(1);
            if (updateMap.containsKey(index)) {
                updateMap.get(index).add(information);
            } else {
                List<SpecialInformation> list = new ArrayList<>();
                list.add(information);
                updateMap.put(index, list);
            }
        }
        updateMap.forEach((k, v) -> esOpUtil.docUpdateBulk(k, v));
    }

    @Override
    public void clearSubjectData(String subjectId) {
        LambdaUpdateWrapper<Subject> updateWrapper = Wrappers.lambdaUpdate();
        updateWrapper.eq(Subject::getId, subjectId)
                .set(Subject::getStatus, 0)
                .set(Subject::getFirstOpenTime, null)
                .set(Subject::getEstimateStatus, null);
        subjectService.update(updateWrapper);
        CompletableFuture.runAsync(() -> {
            //调用python接口
            pythonUtil.clearDuplicateHistory(Collections.singletonList(subjectId));
            //清空专题数据
            BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
            boolQuery.must(QueryBuilders.termQuery("subjectId.keyword", subjectId));
            esOpUtil.docDeleteByQuery(Constants.SUBJECT_INDEX, boolQuery);
        });
        remoteModelService.removeMiddleDataRequest(subjectId);
        log.info("专题-{}，删除中间库数据请求发送成功", subjectId);
    }

    //目标集合按照另一个集合的顺序排序
    private void sortByAnotherList(List<SearchWordVO> list, List<String> anotherList) {
        list.sort((o1, o2) -> {
            int index1 = anotherList.indexOf(o1.getId());
            int index2 = anotherList.indexOf(o2.getId());
            if (index1 != -1) {
                index1 = list.size() - index1;
            }
            if (index2 != -1) {
                index2 = list.size() - index2;
            }
            return index2 - index1;
        });
    }


    private void modifyKeyword(String subjectId, String subjectName, List<SearchWordVO> keywords) {
        LambdaQueryWrapper<SubjectKeywordsMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectKeywordsMap::getSubjectId, subjectId);
        List<SubjectKeywordsMap> bindList = subjectKeywordsMapService.list(queryWrapper);
        List<SubjectKeywordsMap> filterList = bindList.stream().filter(s -> s.getBindingType().equals("2")).collect(Collectors.toList());
        //前端传过来的编辑的关键词信息
        List<SearchWordVO> updateList = keywords.stream().filter(searchWordVO -> searchWordVO.getId() != null).collect(Collectors.toList());
        List<String> updateIdList = updateList.stream().map(SearchWordVO::getId).collect(Collectors.toList());
        //比较后，获得要删除的关键词组id集合
        List<String> removeIdList = new ArrayList<>();
        for (SubjectKeywordsMap subjectKeywordsMap : filterList) {
            if (!updateIdList.contains(subjectKeywordsMap.getKeywordsId())) {
                removeIdList.add(subjectKeywordsMap.getKeywordsId());
            }
        }
        //编辑后保留下来的关键词
        List<SearchWordVO> retainList = modifyKeywordList(subjectName, keywords, removeIdList);
        //专题-关键词绑定关系
        List<SearchWordVO> addMapIdList = new ArrayList<>();
        for (SearchWordVO searchWordVO : retainList) {
            if (!updateIdList.contains(searchWordVO.getId())) {//过滤掉已存在绑定关系的数据
                addMapIdList.add(searchWordVO);
            }
        }
        handlerSubjectKeywordMap(subjectId, addMapIdList,bindList, retainList, removeIdList);
        //专题下关键词之间的关系
        modifySubjectKeywordsGroupRelation(subjectId, retainList);
    }

    //保存词组关系
    private void modifySubjectKeywordsGroupRelation(String subjectId, List<SearchWordVO> retainList) {
        LambdaQueryWrapper<SubjectKeywordsGroupRelation> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectKeywordsGroupRelation::getSubjectId, subjectId);
        int count = subjectKeywordsGroupRelationService.count(queryWrapper);
        if (count > 0) {
            subjectKeywordsGroupRelationService.remove(queryWrapper);
        }
        List<SubjectKeywordsGroupRelation> subjectKeywordsGroupRelationList = new ArrayList<>();
        List<SearchWordVO> relationList = retainList.stream().filter(searchWordVO -> !"NOT".equalsIgnoreCase(searchWordVO.getSearchLogicRelationship())).collect(Collectors.toList());
        List<SearchWordVO> excludedList = retainList.stream().filter(searchWordVO -> "NOT".equalsIgnoreCase(searchWordVO.getSearchLogicRelationship())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(relationList)) {
            //采集词
            SubjectKeywordsGroupRelation subjectKeywordsGroupRelationCaiJi = handlerGroupRelationMap(subjectId, relationList, "1");
            subjectKeywordsGroupRelationList.add(subjectKeywordsGroupRelationCaiJi);
            //过滤词
            SubjectKeywordsGroupRelation subjectKeywordsGroupRelationGuoLv = handlerGroupRelationMap(subjectId, relationList, "2");
            subjectKeywordsGroupRelationList.add(subjectKeywordsGroupRelationGuoLv);
        }
        if (CollectionUtils.isNotEmpty(excludedList)) {
            SubjectKeywordsGroupRelation subjectKeywordsGroupRelation = handlerGroupRelationMap(subjectId, excludedList, "3");
            subjectKeywordsGroupRelationList.add(subjectKeywordsGroupRelation);
        }
        subjectKeywordsGroupRelationService.saveBatch(subjectKeywordsGroupRelationList);
    }

    //格式化 词组关系数据
    private SubjectKeywordsGroupRelation handlerGroupRelationMap(String subjectId, List<SearchWordVO> relationList, String relationType) {
        SubjectKeywordsGroupRelation subjectKeywordsGroupRelation = new SubjectKeywordsGroupRelation();
        subjectKeywordsGroupRelation.setSubjectId(subjectId);
        subjectKeywordsGroupRelation.setRelationType(relationType);
        List<String> keywordsGroupIds = new ArrayList<>();
        List<JSONObject> paramsStr = new ArrayList<>();
        StringBuilder expressionStr = new StringBuilder();
        for (int i = 0; i < relationList.size(); i++) {
            SearchWordVO searchWordVO = relationList.get(i);
            String id = searchWordVO.getId();
            String wordName = searchWordVO.getWordName();
            keywordsGroupIds.add(id);
            if (i == 0) {
                expressionStr.append(id);
                JSONObject params = new JSONObject();
                params.put("title", wordName);
                params.put("value", id);
                paramsStr.add(params);
            } else {
                String searchLogicRelationship = searchWordVO.getSearchLogicRelationship();
                String relationStr = "|";
                if ("and".equals(searchLogicRelationship)) {
                    relationStr = "&";
                } else if ("or".equals(searchLogicRelationship)) {
                    relationStr = "|";
                }
                expressionStr.append(relationStr).append(id);
                JSONObject relationParam = new JSONObject();
                relationParam.put("title", searchLogicRelationship);
                relationParam.put("value", relationStr);
                paramsStr.add(relationParam);
                JSONObject params = new JSONObject();
                params.put("title", wordName);
                params.put("value", id);
                paramsStr.add(params);
            }
        }
        String expression = expressionStr.toString();
        subjectKeywordsGroupRelation.setKeywordsGroupIds(String.join(",", keywordsGroupIds));
        subjectKeywordsGroupRelation.setExpressionStr(expression);
        subjectKeywordsGroupRelation.setParamsStr(JSON.toJSONString(paramsStr));
        return subjectKeywordsGroupRelation;
    }

    //处理专题-关键词绑定关系
    private void handlerSubjectKeywordMap(String subjectId, List<SearchWordVO> addMapIdList,List<SubjectKeywordsMap> bindList, List<SearchWordVO> retainList, List<String> removeIdList) {
        if (CollectionUtils.isNotEmpty(addMapIdList)) {
            List<SubjectKeywordsMap> subjectKeywordsMapList = new ArrayList<>();
            for (SearchWordVO wordVO : addMapIdList) {
                SubjectKeywordsMap subjectKeywordsMap = new SubjectKeywordsMap();
                subjectKeywordsMap.setKeywordsId(wordVO.getId());
                subjectKeywordsMap.setSubjectId(subjectId);
                subjectKeywordsMap.setType(String.valueOf(wordVO.getSearchScope()));
                String bindType;
                if ("NOT".equalsIgnoreCase(wordVO.getSearchLogicRelationship())) {
                    bindType = "3";
                } else {
                    bindType = "2";
                }
                subjectKeywordsMap.setBindingType(bindType);
                subjectKeywordsMapList.add(subjectKeywordsMap);
            }
            subjectKeywordsMapService.saveBatch(subjectKeywordsMapList);
            //绑定为过滤词的同时，绑定为采集词
            List<SubjectKeywordsMap> collect = subjectKeywordsMapList.stream().filter(e -> "2".equals(e.getBindingType())).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(collect)) {
                collect.forEach(e -> {
                    e.setBindingType("1");
                    e.setId(null);
                });
                subjectKeywordsMapService.saveBatch(collect);
            }
        }
        //更新关键词的作用范围
        if (CollectionUtils.isNotEmpty(retainList)) {
            List<SubjectKeywordsMap> subjectKeywordsMapList = new ArrayList<>();
            for (SearchWordVO wordVO : retainList) {
                for (SubjectKeywordsMap subjectKeywordsMap : bindList) {
                    if (wordVO.getId().equals(subjectKeywordsMap.getKeywordsId())) {
                        subjectKeywordsMap.setType(String.valueOf(wordVO.getSearchScope()));
                        subjectKeywordsMapList.add(subjectKeywordsMap);
                    }
                }
            }
            subjectKeywordsMapService.updateBatchById(subjectKeywordsMapList);
        }
        if (CollectionUtils.isNotEmpty(removeIdList)) {
            LambdaQueryWrapper<SubjectKeywordsMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectKeywordsMap::getSubjectId, subjectId);
            queryWrapper.in(SubjectKeywordsMap::getKeywordsId, removeIdList);
            subjectKeywordsMapService.remove(queryWrapper);
        }
    }

    //处理专题下关键词
    private List<SearchWordVO> modifyKeywordList(String subjectName, List<SearchWordVO> keywords, List<String> removeIdList) {
        List<SearchWordVO> realKeyWordList = new ArrayList<>();
        for (SearchWordVO wordVO : keywords) {
            String id = wordVO.getId();
            KeyWords keyWords = new KeyWords();
            keyWords.setKeyWord(wordVO.getSearchInfo());
            if (StringUtils.isEmpty(id)) {
                keyWords.setStatus("1");
                String wordsCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.KEY_WORDS_DEFAULT.getValue());
                String wordsName = subjectName + wordsCode.substring(wordsCode.indexOf("-"));
                keyWords.setWordsName(wordsName);
                keyWords.setWordsCode(wordsCode);
                keywordWordsService.save(keyWords);
                //关键词组-分类 关系绑定
                KeywordsTypeMap keywordsTypeMap = new KeywordsTypeMap();
                keywordsTypeMap.setTypeId(KEYWORDS_TYPE_ID);
                keywordsTypeMap.setKeywordsId(keyWords.getId());
                keywordsTypeMapService.save(keywordsTypeMap);
                //补充数据，后续逻辑使用
                wordVO.setId(keyWords.getId());
                wordVO.setWordName(wordsName);
            } else {
                keyWords.setId(id);
                keywordWordsService.updateById(keyWords);
            }
            realKeyWordList.add(wordVO);
        }
        if (CollectionUtils.isNotEmpty(removeIdList)) {
            keywordWordsService.removeByIds(removeIdList);
            LambdaQueryWrapper<KeywordsTypeMap> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.in(KeywordsTypeMap::getKeywordsId, removeIdList);
            keywordsTypeMapService.remove(queryWrapper);
        }
        return realKeyWordList;
    }
}
