package com.zzsn.event.service.impl;

import cn.hutool.core.date.DateUnit;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.AnalysisService;
import com.zzsn.event.service.CommonService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.SubjectAnalysisService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.HotWordUtil;
import com.zzsn.event.util.SimilarityUtil;
import com.zzsn.event.vo.*;
import com.zzsn.event.vo.es.SpecialInformation;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lkg
 * @description:
 * @date 2022/7/20 11:32
 */
@Service
public class AnalysisServiceImpl implements AnalysisService {

    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;
    @Autowired
    private CommonService commonService;

    /*
     * 优先级：事件脉络 > 伪事件脉络 > 资讯
     * 1.若事件脉络资讯数量少于展示伪事件脉络的阈值(6)，若有伪事件脉络就展示，若无则根据发布时间倒序后，默认取前15条资讯为事件脉络。
     * 2.若事件脉络资讯数量少于展示伪事件脉络的阈值(6)但不为空；
     * 2.1 若有伪事件脉络就展示，若无则展示事件脉络；
     * 2.2 若有伪事件脉络就展示，若无则根据发布时间倒序后，默认取前15条资讯为事件脉络。
     * 3.若事件脉络资讯数量大于/等于展示伪事件脉络的阈值(6)，则直接展示
     *
     * 伪事件脉络/资讯作为事件脉络的情况一般是资讯数量少/刚建立/时间脉络抽取失败的事件专题，才会触发。
     */
    @Override
    public List<SubjectAnalysis> eventContext(String eventId, int fakeNum) {
        //专题下的事件脉络
        List<SubjectAnalysis> list = getList(eventId, 2);
        if (list.size() < fakeNum) {
            //专题下的伪事件脉络
            List<SubjectAnalysis> fakeList = getList(eventId, 3);
            if (CollectionUtils.isEmpty(fakeList)) {
                if (CollectionUtils.isEmpty(list)) {
                    List<SubjectAnalysis> finalList = new ArrayList<>();
                    List<SubjectDataVo> dataList = esService.pageList(eventId, null, null, Constants.FETCH_FIELDS_STATISTIC, 2, 1, 15);
                    dataList.forEach(e -> {
                        String dataId = e.getId();
                        SubjectAnalysis subjectAnalysis = new SubjectAnalysis();
                        BeanUtils.copyProperties(e, subjectAnalysis);
                        subjectAnalysis.setId(null);
                        subjectAnalysis.setDataId(dataId);
                        subjectAnalysis.setPublishDate(DateUtil.stringToDate(e.getPublishDate(), "yyyy-MM-dd HH:mm:ss"));
                        finalList.add(subjectAnalysis);
                    });
                    list = finalList;
                }
            } else {
                list = fakeList;
            }
        }
        return list;
    }

    @Override
    public List<JSONObject> eventContext(String eventId) {
        List<JSONObject> dataList = new ArrayList<>();
        LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectAnalysis::getSubjectId, eventId).eq(SubjectAnalysis::getCategory, 4);
        List<SubjectAnalysis> list = subjectAnalysisService.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            Map<String, List<SubjectAnalysis>> map = list.stream().collect(Collectors.groupingBy(e -> DateUtil.format(e.getPublishDate(), "yyyy-MM-dd")));
            for (Map.Entry<String, List<SubjectAnalysis>> entry : map.entrySet()) {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("date", entry.getKey());
                jsonObject.put("dataList", entry.getValue());
                dataList.add(jsonObject);
            }
        }
        return dataList;
    }

    @Override
    public List<CountVO> wordCloud(String eventId, Integer size) {
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setCategory(2);
        searchCondition.setSubjectId(eventId);
        //排除词
        Set<String> excludeKeywords = commonService.getExcludeKeywords(eventId);
        searchCondition.setExcludeValues(excludeKeywords.toArray(new String[0]));
        return esService.groupByTerm(searchCondition, "groupKeyWord", "keyWordsList.keyword", false, size);
    }

    @Override
    public List<CountVO> wordTrend(String eventId, String endTime) {
        Event event = eventService.getById(eventId);
        Date startTime = event.getStartTime();
        Date finishTime;
        if (StringUtils.isEmpty(endTime)) {
            finishTime = event.getEndTime();
            if (Objects.isNull(event.getEndTime())) {
                finishTime = new Date();
            }
        } else {
            finishTime = DateUtil.stringToDate(endTime, "yyyy-mm-dd HH:mm:ss");
        }
        long between = DateUtil.betweenTwoDate(startTime, finishTime);
        String groupType;
        if (between <= 30) {
            groupType = "day";
        } else if (between <= 180) {
            groupType = "week";
        } else {
            groupType = "month";
        }
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setCategory(2);
        searchCondition.setSubjectId(eventId);
        searchCondition.setStartTime(DateUtil.dateToString(startTime));
        searchCondition.setEndTime(DateUtil.dateToString(finishTime));
        //排除词
        Set<String> excludeKeywords = commonService.getExcludeKeywords(eventId);
        searchCondition.setExcludeValues(excludeKeywords.toArray(new String[0]));
        return esService.keywordTrend(searchCondition, groupType);
    }

    @Override
    public List<CoOccurrenceVO> coOccurrence(String eventId, String endTime) {
        List<CoOccurrenceVO> list = new ArrayList<>();
        InfoDataSearchCondition searchCondition = new InfoDataSearchCondition();
        searchCondition.setCategory(2);
        searchCondition.setSubjectId(eventId);
        String[] fetchFields = new String[]{"id", "keyWordsList"};
        searchCondition.setFetchFields(fetchFields);
        int pageNo = 1;
        int size = 300;
        searchCondition.setPageSize(size);
        boolean flag = true;
        //排除词
        Set<String> excludeKeywords = commonService.getExcludeKeywords(eventId);
        List<List<String>> wordList = new ArrayList<>();
        do {
            searchCondition.setPageNo(pageNo);
            List<SpecialInformation> informationList = esService.informationList(searchCondition);
            if (CollectionUtils.isNotEmpty(informationList)) {
                for (SpecialInformation information : informationList) {
                    List<String> keyWordsList = information.getKeyWordsList();
                    if (CollectionUtils.isNotEmpty(keyWordsList)) {
                        continue;
                    }
                    keyWordsList = keyWordsList.stream().filter(e -> !excludeKeywords.contains(e)).collect(Collectors.toList());
                    if (CollectionUtils.isNotEmpty(keyWordsList) && keyWordsList.size() > 2) {
                        wordList.add(keyWordsList);
                    }
                }
                if (informationList.size() < size) {
                    flag = false;
                } else {
                    pageNo++;
                }
            } else {
                flag = false;
            }
        } while (flag);
        Set<String> keyWordSet = new HashSet<>();
        wordList.forEach(keyWordSet::addAll);
        searchCondition.setIncludeValues(keyWordSet.toArray(new String[0]));
        List<CountVO> groupKeyWord = esService.groupByTerm(searchCondition, "groupKeyWord", "keyWordsList.keyword", false, size);
        Map<String, Long> wordFrequencyMap = groupKeyWord.stream().collect(Collectors.toMap(CountVO::getName, CountVO::getValue));
        Map<String, Integer> map = HotWordUtil.calculateCoOccurrence_list(wordList);
        Map<String, Integer> filterMap = map.entrySet().stream().filter(e -> e.getValue() > 10).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        for (Map.Entry<String, Integer> entry : filterMap.entrySet()) {
            CoOccurrenceVO coOccurrenceVO = new CoOccurrenceVO();
            String key = entry.getKey();
            Integer coOccurrenceNum = entry.getValue();
            coOccurrenceVO.setCoOccurrenceNum((long) coOccurrenceNum);
            CountVO word_1 = new CountVO();
            word_1.setName(key.split("&")[0]);
            word_1.setValue(wordFrequencyMap.getOrDefault(key.split("&")[0], 0L));
            coOccurrenceVO.setWord_1(word_1);
            CountVO word_2 = new CountVO();
            word_2.setName(key.split("&")[1]);
            word_2.setValue(wordFrequencyMap.getOrDefault(key.split("&")[1], 0L));
            coOccurrenceVO.setWord_2(word_2);
            list.add(coOccurrenceVO);
        }
        return list;
    }

    @Override
    public PropagationPathVo propagationPath(String eventId) {
        PropagationPathVo top = null;
        Event event = eventService.getById(eventId);
        String subjectName = event.getEventName();
        //获取专题数据
        String startDate = null;
        String endDate = null;
        if (event.getStartTime() != null) {
            startDate = DateUtil.dateToString(event.getStartTime(), "yyyy-MM-dd HH:mm:ss");
        }
        if (event.getEndTime() != null) {
            endDate = DateUtil.dateToString(event.getEndTime(), "yyyy-MM-dd HH:mm:ss");
        }
        List<SubjectDataVo> specialDataList = esService.getDataBySubjectId(eventId, startDate, endDate, Constants.FETCH_FIELDS_STATISTIC);
        if (CollectionUtils.isNotEmpty(specialDataList)) {
            //用于来源去重
            List<String> allOriginList = new ArrayList<>();
            top = new PropagationPathVo();
            top.setName(subjectName);
            top.setTime(DateUtil.dateToString(event.getStartTime()));
            //获取发布时间最早的前10的资讯(来源不重复)
            List<PropagationPathVo> secondList = new ArrayList<>();
            List<SubjectDataVo> earlyList = topN(specialDataList, Constants.FAKE_NUM);
            earlyList.forEach(e -> allOriginList.add(Constants.getRealOrigin(e.getOrigin())));
            for (SubjectDataVo subjectDataVo : earlyList) {
                String origin = subjectDataVo.getOrigin();
                String time = subjectDataVo.getPublishDate();
                if (StringUtils.isNotEmpty(origin)) {
                    PropagationPathVo second = new PropagationPathVo();
                    second.setName(Constants.getRealOrigin(origin));
                    second.setTime(time);
                    secondList.add(second);
                    List<String> thirdList = esService.originList(subjectDataVo.getTitle(), subjectDataVo.getPublishDate());
                    thirdList.removeAll(allOriginList);
                    List<PropagationPathVo> lastList = new ArrayList<>();
                    if (thirdList.size() > 3) {
                        thirdList = thirdList.subList(0, 3);
                    }
                    for (String s : thirdList) {
                        PropagationPathVo third = new PropagationPathVo();
                        third.setName(s);
                        lastList.add(third);
                    }
                    second.setChildren(lastList);
                    allOriginList.addAll(thirdList);
                }
            }
            top.setChildren(secondList);
        }
        return top;
    }

    //获取发布时间最早的前N条资讯(来源不重复)
    private List<SubjectDataVo> topN(List<SubjectDataVo> list, Integer num) {
        List<SubjectDataVo> collect = list.stream().filter(e -> StringUtils.isNotEmpty(e.getOrigin())).
                sorted(Comparator.comparing(SubjectDataVo::getPublishDate)).collect(Collectors.toList());
        TreeSet<SubjectDataVo> subjectDataVos = new TreeSet<>(Comparator.comparing(SubjectDataVo::getOrigin));
        for (SubjectDataVo subjectDataVo : collect) {
            subjectDataVos.add(subjectDataVo);
            if (subjectDataVos.size() == num) {
                break;
            }
        }
        return new ArrayList<>(subjectDataVos);
    }

    private List<SubjectAnalysis> getList(String subjectId, Integer category) {
        LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectAnalysis::getSubjectId, subjectId).eq(SubjectAnalysis::getCategory, category).orderByDesc(SubjectAnalysis::getPublishDate);
        return subjectAnalysisService.list(queryWrapper);
    }


    //补充是否主要资讯标识以及排序后截取
    private List<EventContextVO> supplyMain(List<EventContextVO> list) {
        EventContextVO dataVo = null;
        double minDistance = Double.MAX_VALUE;
        for (EventContextVO eventContextVO : list) {
            Double similarity = eventContextVO.getDistance();
            if (similarity == 0) {
                continue;
            }
            if (similarity == 1.0) {
                dataVo = eventContextVO;
                break;
            } else if (similarity < minDistance) {
                minDistance = similarity;
                dataVo = eventContextVO;
            }
        }
        if (dataVo == null) {
            list.get(0).setIsMain(true);
        } else {
            for (EventContextVO eventContextVO : list) {
                if (eventContextVO.getId().equals(dataVo.getId())) {
                    eventContextVO.setIsMain(true);
                    break;
                }
            }
        }
        //先按是否是主事件排序，再按相似度算法(编辑距离)返回值正序排序，最后按时间倒序排序
        list.sort(Comparator.comparing(EventContextVO::getIsMain, Comparator.reverseOrder())
                .thenComparing((o1, o2) -> {
                    Double distance1 = o1.getDistance();
                    Double distance2 = o2.getDistance();
                    if (distance1 == 0 && distance2 == 0) {
                        return 0;
                    }
                    if (distance1 == 0) {
                        return 1;
                    }
                    if (distance2 == 0) {
                        return -1;
                    }
                    return distance1.compareTo(distance2);
                })
                .thenComparing(EventContextVO::getPublishDate, Comparator.reverseOrder()));
        if (list.size() > 3) {
            list = list.subList(0, 3);
        }
        return list;
    }
}
