package com.zzsn.event.service;

import com.cronutils.model.Cron;
import com.zzsn.event.entity.*;
import com.zzsn.event.enums.CodePrefixEnum;
import com.zzsn.event.util.CodeGenerateUtil;
import com.zzsn.event.util.CronUtil;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.vo.AddEventVO;
import com.zzsn.event.vo.RegionVO;
import com.zzsn.event.vo.SubjectKeywordsMap;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 简化版事件新增逻辑
 *
 * @author lkg
 * @date 2024/7/22
 */
@Service
public class EventSimpleService {

    @Autowired
    private CodeGenerateUtil codeGenerateUtil;
    @Autowired
    private IEventService eventService;
    @Autowired
    private ISubjectTypeMapService subjectTypeMapService;
    @Autowired
    private IProjectSubjectMapService projectSubjectMapService;
    @Autowired
    private ISubjectSearchEnginesMapService subjectSearchEnginesMapService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private IKeywordsTypeMapService keywordsTypeMapService;
    @Autowired
    private ISubjectKeywordsMapService subjectKeywordsMapService;
    @Autowired
    private IEventTagService eventTagService;
    @Autowired
    private EventRegionMapService eventRegionMapService;

    //默认搜索引擎
    private final static String[] SEARCH_ENGINES = new String[]{"1539145903551361026","1539146006513135617","1539146115049140225"};

    /**
     * 新增事件(页面简化版)
     *
     * @param addEventVO 事件信息
     * @author lkg
     * @date 2024/7/19
     */
    @Transactional
    public Event simpleSave(AddEventVO addEventVO) {
        Event event = new Event();
        BeanUtils.copyProperties(addEventVO, event);
        event.setIncreAnaRule(20);
        event.setTotalAnaRule(50);
        event.setTimeAnaRule(5);
        String subjectCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.SUBJECT_DEFAULT.getValue());
        event.setEventCode(subjectCode);
        //默认发布
        event.setPublishStatus(1);
        event.setPublishDate(DateUtil.dateToString(new Date()));
        //调度周期，默认2小时
        event.setUnit("2");
        event.setSpace(2);
        event.setCron(CronUtil.generateCron("2",2));
        event.setCreateTime(new Date());
        eventService.save(event);
        String eventId = event.getId();
        saveMapMain(eventId);
        saveKeyword(event, addEventVO.getKeywords(), addEventVO.getExclusionWord());
        //事件标签
        eventTagService.save(EventTag.builder().eventId(eventId).build());
        //地域关系绑定
        List<RegionVO> regionList = addEventVO.getRegionList();
        addRegionMap(eventId, regionList);
        return event;
    }

    private void saveMapMain(String eventId) {
        //专题和分类关系
        SubjectTypeMap subjectTypeMap = new SubjectTypeMap();
        subjectTypeMap.setSubjectId(eventId);
        subjectTypeMap.setCreateTime(new Date());
        //分类默认 研究中心
        subjectTypeMap.setTypeId("1802619433932296193");
        subjectTypeMapService.save(subjectTypeMap);
        //专题和项目关系
        ProjectSubjectMap projectSubjectMap = new ProjectSubjectMap();
        //项目默认 研究中心
        projectSubjectMap.setProjectId("1476527644425682945");
        projectSubjectMap.setSubjectId(eventId);
        projectSubjectMapService.save(projectSubjectMap);
        //专题和搜索引擎关系
        List<SubjectSearchEnginesMap> enginesMaps = new ArrayList<>();
        for (String searchEngine : SEARCH_ENGINES) {
            SubjectSearchEnginesMap subjectSearchEnginesMap = new SubjectSearchEnginesMap();
            subjectSearchEnginesMap.setSearchEngineId(searchEngine);
            subjectSearchEnginesMap.setSubjectId(eventId);
            enginesMaps.add(subjectSearchEnginesMap);
        }
        subjectSearchEnginesMapService.saveBatch(enginesMaps);
    }

    private void saveKeyword(Event event, String keyword, String exclusionWord) {
        if (StringUtils.isNotEmpty(keyword) || StringUtils.isNotEmpty(exclusionWord)) {
            //关键词
            KeyWords keyWords = new KeyWords();
            String wordsCode = codeGenerateUtil.geneCodeNo(CodePrefixEnum.KEY_WORDS_DEFAULT.getValue());
            keyWords.setWordsCode(wordsCode);
            keyWords.setWordsName(event.getEventName());
            keyWords.setKeyWord(keyword);
            keyWords.setExclusionWord(exclusionWord);
            keyWords.setStatus("1");
            keyWords.setCreateTime(new Date());
            keyWordsService.save(keyWords);
            String keyWordsId = keyWords.getId();
            //关键词和分类关系
            KeywordsTypeMap keywordsTypeMap = new KeywordsTypeMap();
            keywordsTypeMap.setKeywordsId(keyWordsId);
            //分类默认 事件专题
            keywordsTypeMap.setTypeId("1802634369064525826");
            keywordsTypeMap.setCreateTime(new Date());
            keywordsTypeMapService.save(keywordsTypeMap);
            //专题和关键词关系
            SubjectKeywordsMap subjectKeywordsMap = new SubjectKeywordsMap();
            subjectKeywordsMap.setKeywordsId(keyWordsId);
            subjectKeywordsMap.setSubjectId(event.getId());
            subjectKeywordsMapService.save(subjectKeywordsMap);
        }
    }

    private void addRegionMap(String eventId, List<RegionVO> regionList) {
        if (CollectionUtils.isNotEmpty(regionList)) {
            List<EventRegionMap> dataList = new ArrayList<>();
            for (RegionVO regionVO : regionList) {
                EventRegionMap eventRegionMap = new EventRegionMap();
                eventRegionMap.setEventId(eventId);
                eventRegionMap.setRegionId(regionVO.getId());
                eventRegionMap.setTopRegionId(regionVO.getTopId());
                eventRegionMap.setType(regionVO.getType());
                eventRegionMap.setCreateTime(new Date());
                dataList.add(eventRegionMap);
            }
            eventRegionMapService.saveBatch(dataList);
        }
    }
}
