package com.zzsn.event.kafka;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.*;
import com.zzsn.event.service.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.annotation.KafkaListener;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;

/**
 * @author lkg
 * 监听消费kafka消息
 * 消费topic中消息(注意：如果监听的topic不存在则会自动创建)
 * @date 2022/7/15
 */
@Slf4j
@Component
public class KafkaConsumer {
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;
    @Autowired
    private IEventService eventService;
    @Autowired
    private EventExtractService eventExtractService;
    @Autowired
    private EventAnalysisReportService eventAnalysisReportService;
    @Autowired
    private IEventCategoryService eventCategoryService;

    /**
     * 获取-观点分析-分析结果数据，并入库
     *
     * @param record 接收的kafka数据
     */
    /*@KafkaListener(topics = {Constants.VIEWPOINT_RECEIVE_TOPIC})
    public void viewPointAnalysis(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isNotEmpty(value)) {
            String subjectId = null;
            try {
                List<SubjectAnalysis> subjectAnalyses = JSON.parseArray(value, SubjectAnalysis.class);
                subjectAnalyses.forEach(e -> e.setTitle(removeNonBmpUniCodes(e.getTitle())));
                Integer category = subjectAnalyses.get(0).getCategory();
                subjectId = subjectAnalyses.get(0).getSubjectId();
                Date analysisDate = subjectAnalyses.get(0).getAnalysisDate();
                LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.eq(SubjectAnalysis::getSubjectId, subjectId).eq(SubjectAnalysis::getCategory, category);
                subjectAnalysisService.remove(queryWrapper);
                subjectAnalysisService.saveBatch(subjectAnalyses);
                //更新专题最近一次观点分析的时间
                Event subject = new Event();
                subject.setId(subjectId);
                subject.setAnalysisTime(analysisDate);
                eventService.updateById(subject);
            } catch (Exception e) {
                e.printStackTrace();
            }
            log.info("id为-{}-的专题,此次-观点分析-数据更新完成", subjectId);
        }
    }*/

    /**
     * 获取-事件脉络-分析结果数据，并入库
     *
     * @param record 接收的kafka数据
     */
    @KafkaListener(topics = {Constants.EVENT_CONTEXT_RECEIVE_TOPIC})
    public void eventContext(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isNotEmpty(value)) {
            try {
                List<SubjectAnalysis> subjectAnalyses = JSON.parseArray(value, SubjectAnalysis.class);
                for (SubjectAnalysis e : subjectAnalyses) {
                    if (exist(e)) {
                        e.setTitle(removeNonBmpUniCodes(e.getTitle()));
                        subjectAnalysisService.save(e);
                    }
                }
                log.info("id为-{}-的专题,此次-事件脉络-数据更新完成", subjectAnalyses.get(0).getSubjectId());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 获取-伪事件脉络-分析结果数据，并入库
     *
     * @param record 接收的kafka数据
     */
    @KafkaListener(topics = {Constants.FAKE_EVENT_CONTEXT_RECEIVE_TOPIC})
    public void eventContext_fake(ConsumerRecord<String, String> record) {
        String value = record.value();
        if (StringUtils.isNotEmpty(value)) {
            String subjectId = null;
            Integer category = 3;
            try {
                List<SubjectAnalysis> subjectAnalyses = JSON.parseArray(value, SubjectAnalysis.class);
                subjectId = subjectAnalyses.get(0).getSubjectId();
                subjectAnalyses.forEach(e -> {
                    e.setCategory(category);
                    e.setTitle(removeNonBmpUniCodes(e.getTitle()));
                });
                LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
                queryWrapper.eq(SubjectAnalysis::getSubjectId, subjectId).eq(SubjectAnalysis::getCategory, category);
                int count = subjectAnalysisService.count(queryWrapper);
                if (count > 0) {
                    subjectAnalysisService.remove(queryWrapper);
                }
                subjectAnalysisService.saveBatch(subjectAnalyses);
            } catch (Exception e) {
                e.printStackTrace();
            }
            log.info("id为-{}-的专题,此次-伪事件脉络-数据更新完成", subjectId);
        }
    }

    /**
     * 获取事件的分析报告数据
     *
     * @param record 接收的kafka数据
     * @author lkg
     * @date 2024/4/12
     */
    @KafkaListener(topics = {Constants.EVENT_REPORT_RECEIVE_TOPIC})
    public void eventReport(ConsumerRecord<String, String> record) {
        String value = record.value();
        EventAnalysisReport eventAnalysisReport = JSONObject.parseObject(value, EventAnalysisReport.class);
        eventAnalysisReportService.modify(eventAnalysisReport.getEventId(), eventAnalysisReport.getFilePath());
        log.info("id为-{}-的事件,分析报告更新完成", eventAnalysisReport.getEventId());
    }


    /**
     * 获取事件挖掘的结果
     *
     * @param record 接收的kafka数据
     * @author lkg
     * @date 2024/8/30
     */
    @KafkaListener(topics = {Constants.EVENT_DIG_RECEIVE_TOPIC})
    public void eventDig(ConsumerRecord<String, String> record) {
        String value = record.value();
        try {
            JSONObject jsonObject = JSONObject.parseObject(value);
            JSONArray resultData = jsonObject.getJSONArray("resultData");
            if (resultData != null && resultData.size() > 0) {
                for (Object item : resultData) {
                    JSONObject eventItem = JSONObject.parseObject(item.toString());
                    String uniqueId = eventItem.get("event_id").toString();
                    String eventTitle = eventItem.get("event_title").toString();
                    String eventType = eventItem.get("event_type").toString();
                    String eventDesc = eventItem.get("event_name").toString();
                    String taskId = eventItem.get("task_id").toString();
                    String projectId = eventItem.get("project_id").toString();
                    String relatedIds = eventItem.getJSONArray("related_id").toString();
                    EventExtract one = eventExtractService.getOne(Wrappers.<EventExtract>lambdaQuery().eq(EventExtract::getUniqueId,uniqueId));
                    if (one == null) {
                        String eventId = saveEventExtract(uniqueId, eventTitle, eventType, eventDesc, relatedIds,taskId,projectId);
                        log.info("挖掘到事件，id-{},名称-{}，并保存成功", eventId, eventTitle);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private String saveEventExtract(String eventId, String eventTitle, String eventType, String eventDesc,String relatedIds,String taskId,String projectId) {
        EventExtract eventExtract = new EventExtract();
        eventExtract.setUniqueId(eventId);
        eventExtract.setEventName(eventTitle);
        if (StringUtils.isNotBlank(eventType)) {
            String[] split = eventType.split("-");
            LambdaQueryWrapper<EventCategory> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(EventCategory::getTypeName, split[split.length - 1]);
            EventCategory eventCategory = eventCategoryService.getOne(queryWrapper);
            if (eventCategory != null) {
                eventExtract.setEventType(eventCategory.getId());
            }
        }
        eventExtract.setEventDescribe(eventDesc);
        eventExtract.setRelatedId(relatedIds);
        eventExtract.setExtractTime(new Date());
        eventExtract.setCreateTime(new Date());
        eventExtract.setCreateBy("event-dig");
        eventExtract.setTaskId(taskId);
        eventExtract.setProjectId(projectId);
        eventExtractService.save(eventExtract);
        return eventExtract.getId();
    }


    private boolean exist(SubjectAnalysis subjectAnalyse) {
        LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(SubjectAnalysis::getCategory, 2).eq(SubjectAnalysis::getDataId, subjectAnalyse.getDataId())
                .eq(SubjectAnalysis::getSubjectId, subjectAnalyse.getSubjectId());
        SubjectAnalysis one = subjectAnalysisService.getOne(queryWrapper);
        return one == null;
    }

    //去除特殊的字符，例如表情符
    private String removeNonBmpUniCodes(String str) {
        return StringUtils.isEmpty(str) ? null : str.replaceAll("[^\\u0000-\\uFFFF]", "");
    }
}
