package com.zzsn.event.controller;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.clb.common.model.task.dto.titr.KeyWordsDTO;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.CustomerDataPermissionMap;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.KeyWords;
import com.zzsn.event.entity.SysUserDataPermission;
import com.zzsn.event.es.EsService;
import com.zzsn.event.service.EventExtractService;
import com.zzsn.event.service.EventSimpleService;
import com.zzsn.event.service.IEventService;
import com.zzsn.event.service.IKeyWordsService;
import com.zzsn.event.util.*;
import com.zzsn.event.util.user.UserUtil;
import com.zzsn.event.util.user.UserVo;
import com.zzsn.event.vo.*;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * 事件
 *
 * @Author: jeecg-boot
 * @Date: 2024-03-14
 * @Version: V1.0
 */
@Slf4j
@Api(tags = "事件")
@RestController
@RequestMapping("/data")
public class EventDataController {
    @Autowired
    private IEventService eventService;
    @Autowired
    private EventExtractService eventExtractService;
    @Autowired
    private EsService esService;
    @Autowired
    private EventSimpleService eventSimpleService;
    @Autowired
    private IXxlJobInfoService iXxlJobInfoService;
    @Autowired
    private IKeyWordsService keyWordsService;
    @Autowired
    private RedisUtil redisUtil;
    @Resource
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private EsOpUtil esOpUtil;

    @Value("${translate.url:}")
    private String TRANSLATE_URL;

    @Value("${clb.eventAdd}")
    private String CLB_EVENT_ADD;

    @Value("${clb.eventEdit}")
    private String CLB_EVENT_EDIT;

    @Value("${clb.eventDelete}")
    private String CLB_EVENT_DELETE;

    /**
     * 事件分页列表-门户
     *
     * @param category   栏目类型(1-全部;2-自动追踪;3-我的)
     * @param eventName  事件名称
     * @param eventType  事件分类id
     * @param labelField 搜索字段(标签)
     * @param labelName  搜索词(标签)
     * @param order      排序字段
     * @param orderType  排序方式
     * @param pageNo     偏移量
     * @param pageSize   返回条数
     * @author lkg
     * @date 2024/4/8
     */
    @ApiOperation(value = "事件-分页列表查询", notes = "事件-分页列表查询")
    @GetMapping(value = "/pageList")
    public Result<?> pageList(@RequestParam(defaultValue = "1476527644425682945") String projectId,
                              @RequestParam(name = "category") Integer category,
                              @RequestParam(name = "eventId", required = false) String eventId,
                              @RequestParam(name = "eventName", required = false) String eventName,
                              @RequestParam(name = "eventType", required = false) String eventType,
                              @RequestParam(name = "labelField", required = false) String labelField,
                              @RequestParam(name = "labelName", required = false) String labelName,
                              @RequestParam(name = "order", required = false) String order,
                              @RequestParam(name = "orderType", required = false) String orderType,
                              @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                              HttpServletRequest request) {
        IPage<EventFrontVO> pageList = new Page<>();
        String tenant = RequestUtil.getTenantByRequest(request);
        if (category == 1){
            pageList = eventService.frontAllPageList(projectId, eventId,eventName, eventType, labelField, labelName, order, orderType, pageNo, pageSize, tenant);
        } else if (category == 2) {
            pageList = eventExtractService.frontDigPageList(projectId,eventName,eventType,pageNo,pageSize);
        } else if (category == 3) {
            pageList = eventService.frontOwnerPageList(projectId, eventName, eventType, labelField, labelName, order, orderType, pageNo, pageSize, tenant);
        }
        return Result.OK(pageList);
    }

    /**
     * 新增事件(页面简化版)
     *
     * @param addEventVO 事件信息
     * @author lkg
     * @date 2024/7/22
     */
    @PostMapping("/simpleSaveEvent")
    public Result<?> addEvent(HttpServletRequest request, @RequestBody AddEventVO addEventVO) {
        KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
        addEventVO.setTenant(RequestUtil.getTenantByRequest(request));
        if (keywordsVO != null) {
            // 调用克虏宝事件新增接口
            UserVo currentUser = UserUtil.getLoginUser();
            JSONObject params = JSONObject.from(addEventVO);
            params.put("username", currentUser.getUsername());
            try {
                String resultStr = HttpUtil.doPost(CLB_EVENT_ADD, params, 1000 * 30);
                JSONObject jsonObject = JSONObject.parseObject(resultStr);
                Integer code = jsonObject.getInteger("code");
                if (code != null && code.equals(Constants.SUCCESS_CODE)) {
                    JSONObject resultMap = jsonObject.getJSONObject("result");
                    String id = resultMap.getString("id");
                    String eventCode = resultMap.getString("eventCode");
                    // 事件id和code使用clb中生成的
                    addEventVO.setId(id);
                    addEventVO.setEventCode(eventCode);
                    Event event = eventSimpleService.simpleSave(addEventVO);
                    KeyWords keyWords = keyWordsService.saveKeyword(event, keywordsVO.getKeyword(), keywordsVO.getExclusionWord());
//                    CompletableFuture.runAsync(()->{
//                        iXxlJobInfoService.subjectInsert(event);
//                        //关键词
//                        KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
//                        BeanUtils.copyProperties(keyWords, redisKeywordDTO);
//                        redisKeywordDTO.setStartTime(event.getStartTime());
//                        redisKeywordDTO.setEndTime(event.getEndTime());
//                        redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
//                        redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keyWords.getWordsCode(), redisKeywordDTO);
//                        //插入xxljob
//                        iXxlJobInfoService.keyWordsInsert(redisKeywordDTO);
//                        //为了立即响应，关键词新增时放入一个首次录入消息队列
//                        kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
//                        kafkaTemplate.send(Constants.EVENT_SUBJECT_MODEL, event.getEventCode());
//                    });
                    return Result.OK();
                } else {
                    log.error("调用克虏宝新增返回失败");
                    return Result.FAIL(500, "保存事件信息失败!");
                }
            } catch (Exception e) {
                log.error("调用克虏宝新增事件异常：{}", e);
                return Result.FAIL(500, "保存事件信息失败!");
            }
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 编辑事件(页面简化版)
     *
     * @param addEventVO 事件信息
     * @author lkg
     * @date 2024/7/22
     */
    @PostMapping("/simpleUpdateEvent")
    public Result<?> updateEvent(HttpServletRequest request, @RequestBody AddEventVO addEventVO) {
        KeywordsVO keywordsVO = addEventVO.getKeywordsVO();
        addEventVO.setTenant(RequestUtil.getTenantByRequest(request));
        if (keywordsVO != null) {
            // 调用克虏宝编辑接口
            UserVo currentUser = UserUtil.getLoginUser();
            JSONObject params = JSONObject.from(addEventVO);
            params.put("username", currentUser.getUsername());
            try{
                String resultStr = HttpUtil.doPost(CLB_EVENT_EDIT, params, 1000 * 30);
                JSONObject jsonObject = JSONObject.parseObject(resultStr);
                Integer code = jsonObject.getInteger("code");
                if (code != null && code.equals(Constants.SUCCESS_CODE)) {
                    // 本地编辑接口
                    eventSimpleService.updateMain(addEventVO);
                    keyWordsService.update(Wrappers.<KeyWords>lambdaUpdate().eq(KeyWords::getId, keywordsVO.getId())
                            .set(KeyWords::getKeyWord, keywordsVO.getKeyword())
                            .set(KeyWords::getExclusionWord, keywordsVO.getExclusionWord()));
//                    CompletableFuture.runAsync(()->{
//                        //关键词
//                        KeyWordsDTO redisKeywordDTO = new KeyWordsDTO();
//                        BeanUtils.copyProperties(keywordsVO, redisKeywordDTO);
//                        redisKeywordDTO.setKeyWord(keywordsVO.getKeyword());
//                        redisKeywordDTO.setStartTime(addEventVO.getStartTime());
//                        redisKeywordDTO.setEndTime(addEventVO.getEndTime());
//                        redisKeywordDTO.setSearchEngines(new ArrayList<>(Constants.DEFAULT_SEARCH_ENGINE.values()));
//                        redisUtil.set(Constants.KEY_WORDS_TO_REDIS_PREFIX + keywordsVO.getWordsCode(), redisKeywordDTO);
//                        //为了立即响应，关键词新增时放入一个首次录入消息队列
//                        kafkaTemplate.send(Constants.KEY_WORDS_COLLECT_TOPIC, JSON.toJSONString(redisKeywordDTO));
//                    });
                } else {
                    log.error("调用克虏宝编辑返回失败");
                    return Result.FAIL(500, "编辑事件信息失败!");
                }
            } catch (Exception e) {
                log.error("调用克虏宝编辑事件异常：{}", e);
                return Result.FAIL(500, "编辑事件信息失败!");
            }
            return Result.OK();
        } else {
            return Result.FAIL(500, "关键词不能为空");
        }
    }

    /**
     * 通过id删除
     *
     * @param id 事件id
     * @return
     */
    @ApiOperation(value = "事件-通过id删除", notes = "事件-通过id删除")
    @GetMapping(value = "/simpleDeleteEvent")
    public Result<?> delete(@RequestParam(name = "id") String id) {
        try {
            // 调用克虏宝删除接口
            Map<String, String> params = new HashMap<>();
            params.put("id", id);
            String resultStr = HttpUtil.doGet(CLB_EVENT_DELETE, params, "utf-8");
            // 调用本地删除
            EventVO eventVO = eventService.queryInfo(id);
            eventService.deleteMain(id);
            CompletableFuture.runAsync(() -> {
                //iXxlJobInfoService.deleteByInfosourceCode(eventVO.getEventCode());
                KeywordsVO keywordsVO = eventVO.getKeywordsVO();
                if (keywordsVO != null) {
                    String wordsCode = keywordsVO.getWordsCode();
                    keyWordsService.remove(Wrappers.<KeyWords>lambdaQuery().eq(KeyWords::getWordsCode, wordsCode));
                    //redisUtil.del(Constants.KEY_WORDS_TO_REDIS_PREFIX + wordsCode);
                    //iXxlJobInfoService.deleteByInfosourceCode(wordsCode);
                }
            });
        } catch (Exception e) {
            log.error("调用克虏宝删除事件异常：{}", e);
            return Result.FAIL(500, "删除事件信息失败!");
        }
        return Result.OK();
    }


    /**
     * 事件资讯分页列表
     *
     * @param projectId  项目id
     * @param subjectId  事件id
     * @param searchWord 搜索词
     * @param position   搜索位置(title-标题;content-内容)
     * @param category   匹配度(1-模糊;2-精确)
     * @param column     排序字段
     * @param order      排序方式(asc-正序;desc-倒序)
     * @param pageNo     当前页
     * @param pageSize   返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @ApiOperation(value = "专题信息列表-分页列表查询", notes = "专题信息列表-分页列表查询")
    @GetMapping(value = "/listArticle")
    public Result<?> queryPageList(@RequestParam(name = "projectId", defaultValue = "1476527644425682945") String projectId,
                                   @RequestParam(name = "subjectId", required = false) String subjectId,
                                   @RequestParam(name = "searchWord", required = false) String searchWord,
                                   @RequestParam(name = "position", required = false) String position,
                                   @RequestParam(name = "category", required = false) Integer category,
                                   @RequestParam(name = "labelId", required = false) String labelId,
                                   @RequestParam(name = "column", defaultValue = "publishDate") String column,
                                   @RequestParam(name = "order", defaultValue = "desc") String order,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                   HttpServletRequest request) {
        String tenant = RequestUtil.getTenantByRequest(request);
        UserVo currentUser = UserUtil.getLoginUser();
        String username = currentUser.getUsername();
        List<String> subjectIdList = new ArrayList<>();
        if (StringUtils.isNotEmpty(subjectId)) {
            subjectIdList.add(subjectId);
        } else {
            List<EventExcelVO> frontList = eventService.frontAllList(projectId,username, null,null,null,null,null,null, tenant);
            frontList.forEach(e -> subjectIdList.add(e.getId()));
        }
        //获取数据
        IPage<SubjectDataVo> pageList = esService.frontListByPage(subjectIdList, searchWord, position, category,labelId, column, order, pageNo, pageSize);
        return Result.OK(pageList);
    }


    /**
     * 资讯详情
     *
     * @return
     */
    @GetMapping(value = "/articleDetail")
    public Result<?> articleDetail(@RequestParam(required = false) String index, @RequestParam String id) {
        if (StringUtils.isEmpty(index)) {
            index = Constants.SUBJECT_INDEX;
        }
        SubjectDataVo subjectDataVo = esService.queryInfo(index, id);
        return Result.OK(subjectDataVo);
    }

    /**
     * 批量删除资讯
     *
     * @param list 信息集合
     * @author lkg
     * @date 2024/9/14
     */
    @PostMapping("/batchDelete")
    public Result<?> batchDelete(@RequestBody List<JSONObject> list){
        Map<String,String> updateFieldMap = new HashMap<>();
        updateFieldMap.put("deleteFlag","1");
        List<CompletableFuture<Void>> asyncList = new ArrayList<>();
        for (JSONObject jsonObject : list) {
            asyncList.add(CompletableFuture.runAsync(()->{
                String id = jsonObject.getString("id");
                String index = jsonObject.getString("index");
                try {
                    esOpUtil.updateByid(index,id,updateFieldMap);
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }));
        }
        //等待全部执行完
        CompletableFuture.allOf(asyncList.toArray(new CompletableFuture[0])).join();
        return Result.OK();
    }

    /**
     * 翻译
     *
     * @param transferData 但翻译文本
     * @author lkg
     * @date 2024/8/21
     */
    @PostMapping("/translate")
    public Result<?> translate(@RequestBody SubjectDataVo transferData){
        TranslateVO translateVO = new TranslateVO();
        translateVO.setFrom("auto");
        translateVO.setTo("zh-CN");
        List<String> content = new ArrayList<>();
        content.add(transferData.getTitle());
        content.add(transferData.getSummary());
        content.add(transferData.getContent());
        translateVO.setContent(content);
        List<String> data = new ArrayList<>();
        try {
            String s = HttpUtil.doPost(TRANSLATE_URL, ObjectUtil.objectToJSONObject(translateVO), 10000);
            data = JSONArray.parseArray(s, String.class);
        } catch (Exception e) {
            log.error("翻译请求失败！");
            e.printStackTrace();
        }
        return Result.OK(data);
    }

    /**
     * 单篇文章热词
     *
     * @param index  索引名称
     * @param id     资讯id
     * @param number 热词数量
     * @author lkg
     * @date 2024/4/10
     */
    @ApiOperation(value = "单篇文章热词", notes = "单篇文章热词")
    @GetMapping(value = "/hotWords")
    public Result<?> articleList(@RequestParam(value = "index", required = false) String index, @RequestParam("id") String id,
                                 @RequestParam(name = "number", defaultValue = "200") Integer number) {
        if (StringUtils.isEmpty(index)) {
            index = Constants.SUBJECT_INDEX;
        }
        List<StatisticsKeyWordVo> words = eventService.hotWords(index, id, number);
        return Result.OK(words);
    }

    /**
     * 相关推荐
     *
     * @param subjectId 专题id
     * @param id        资讯id
     * @param title     标题
     * @param pageSize  返回条数
     * @author lkg
     * @date 2024/4/10
     */
    @GetMapping(value = "/recommendList")
    public Result<?> recommendList(@RequestParam(name = "subjectId") String subjectId,
                                   @RequestParam(name = "id") String id,
                                   @RequestParam(name = "title") String title,
                                   @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                   @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        List<SubjectDataVo> recommendList = esService.queryRecommendList(subjectId, id, title, pageNo, pageSize);
        return Result.OK(recommendList);
    }
}
