package com.zzsn.event.external.controller;

import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelUtil;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.zzsn.event.constant.Result;
import com.zzsn.event.entity.Subject;
import com.zzsn.event.external.entity.ExternalSubjectInfoSourceMap;
import com.zzsn.event.external.service.ExternalSubjectInfoSourceMapService;
import com.zzsn.event.external.vo.SubjectInfoVO;
import com.zzsn.event.external.vo.SubjectVO;
import com.zzsn.event.service.CommonService;
import com.zzsn.event.service.SubjectSimpleService;
import com.zzsn.event.util.HttpUtil;
import com.zzsn.event.util.LLMUtil;
import com.zzsn.event.vo.SubjectDetailVO;
import com.zzsn.event.vo.SubjectSimpleVO;
import com.zzsn.event.xxljob.service.IXxlJobInfoService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * @author lkg
 * @date 2025/8/12
 */
@RestController
@RequestMapping("/external")
public class ExternalController {


    @Autowired
    private SubjectSimpleService subjectSimpleService;
    @Autowired
    private IXxlJobInfoService xxlJobInfoService;
    @Autowired
    private ExternalSubjectInfoSourceMapService externalSubjectInfoSourceMapService;
    @Autowired
    private CommonService commonService;

    @Value("${caiji.keywordCrawler.url:}")
    private String keywordCrawlerUrl;

    /**
     * 创建专题
     *
     * @param subjectVO 参数
     * @author lkg
     * @date 2025/1/9
     */
    @PostMapping("/createSubject")
    public Result<?> createSubject(@RequestBody SubjectVO subjectVO) {
        subjectVO.setDataScope("1");
        SubjectSimpleVO subjectSimpleVO = new SubjectSimpleVO();
        BeanUtils.copyProperties(subjectVO, subjectSimpleVO);
        Subject subject = subjectSimpleService.createExternalSubject(subjectSimpleVO);
        CompletableFuture.runAsync(() -> {
            //插入xxlJob
            xxlJobInfoService.subjectInsert(subject);
        });
        return Result.OK();
    }

    /**
     * 编辑专题
     *
     * @param subjectVO 参数
     * @author lkg
     * @date 2025/1/9
     */
    @PostMapping("/updateSubject")
    public Result<?> updateSubject(@RequestBody SubjectVO subjectVO) {
        SubjectSimpleVO subjectSimpleVO = new SubjectSimpleVO();
        BeanUtils.copyProperties(subjectVO, subjectSimpleVO);
        subjectSimpleService.editExternalSubject(subjectSimpleVO);
        return Result.OK();
    }

    /**
     * 采集词推荐
     *
     * @param remark 专题描述
     * @author lkg
     * @date 2025/8/12
     */
    @GetMapping("/crawlerWordRecommend")
    public Result<?> crawlerWordRecommend(@RequestParam String remark) {
        return Result.OK(LLMUtil.crawlerWord(remark));
    }

    /**
     * 绑定信息源分页列表
     *
     * @param subjectId 专题id
     * @param pageNo    页码
     * @param pageSize  每页返回条数
     * @author lkg
     * @date 2025/8/12
     */
    @GetMapping("/bindSourcePageList")
    public Result<?> bindSourceList(@RequestParam String subjectId,
                                    @RequestParam(defaultValue = "1") Integer pageNo,
                                    @RequestParam(defaultValue = "10") Integer pageSize) {
        Page<ExternalSubjectInfoSourceMap> page = new Page<>(pageNo, pageSize);
        LambdaQueryWrapper<ExternalSubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(ExternalSubjectInfoSourceMap::getSubjectId, subjectId)
                .eq(ExternalSubjectInfoSourceMap::getType, 1);
        Page<ExternalSubjectInfoSourceMap> list = externalSubjectInfoSourceMapService.page(page, queryWrapper);
        return Result.OK(list);
    }


    /**
     * 下载导入信息源
     *
     * @author lkg
     * @date 2024/06/21
     */
    @GetMapping("/downloadInfoSourceTemplate")
    public void downloadResearchTemplate(HttpServletResponse response) {
        String filePath = "subjectDataImport/导入信息源模板.xlsx";
        commonService.downloadTemplate(response, filePath);
    }


    /**
     * 批量导入绑定得信息源
     *
     * @author lkg
     * @date 2025/8/12
     */
    @PostMapping("/importInfoSource")
    public Result<?> importInfoSource(HttpServletRequest request) {
        String subjectId = request.getParameter("subjectId");
        if (StringUtils.isBlank(subjectId)) {
            return Result.FAIL(500, "专题id不能为空");
        }
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        if (fileMap.size() < 1) {
            return Result.FAIL(500, "请上传excel文件");
        }
        MultipartFile multipartFile = fileMap.get(new ArrayList<>(fileMap.keySet()).get(0));
        int index = multipartFile.getOriginalFilename().lastIndexOf(".");
        String fileSuffix = multipartFile.getOriginalFilename().substring(index + 1);
        if ("xls".equals(fileSuffix) || "xlsx".equals(fileSuffix)) {
            try {
                ExcelReader reader = ExcelUtil.getReader(multipartFile.getInputStream());
                Map<String, String> header = new HashMap<>();
                header.put("信息源名称", "infoSourceName");
                header.put("信息源地址", "infoSourceUrl");
                reader.setHeaderAlias(header);
                List<ExternalSubjectInfoSourceMap> infoSourceList = reader.read(0, 1, ExternalSubjectInfoSourceMap.class);
                infoSourceList.forEach(infoSource -> {
                    infoSource.setSubjectId(subjectId);
                    infoSource.setType(1);
                });
                externalSubjectInfoSourceMapService.saveBatch(infoSourceList);
            } catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            return Result.FAIL(500, "不支持的文件类型");
        }
        return Result.OK();
    }

    /**
     * 绑定信息源
     *
     * @param externalSubjectInfoSourceMap 参数
     * @author lkg
     * @date 2025/8/12
     */
    @PostMapping("/bindSource")
    public Result<?> bindSource(@RequestBody ExternalSubjectInfoSourceMap externalSubjectInfoSourceMap) {
        externalSubjectInfoSourceMap.setType(1);
        externalSubjectInfoSourceMapService.save(externalSubjectInfoSourceMap);
        return Result.OK();
    }

    /**
     * 解绑信息源
     *
     * @param ids id集合
     * @author lkg
     * @date 2025/8/12
     */
    @GetMapping("/unBindSource")
    public Result<?> bindSource(@RequestParam List<String> ids) {
        externalSubjectInfoSourceMapService.removeByIds(ids);
        return Result.OK();
    }

    /**
     * 专题详情
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/1/9
     */
    @GetMapping("/queryInfo")
    public Result<?> queryInfo(@RequestParam String subjectId) {
        SubjectInfoVO subjectInfoVO = new SubjectInfoVO();
        SubjectDetailVO subjectDetailVO = subjectSimpleService.queryInfo(subjectId);
        BeanUtils.copyProperties(subjectDetailVO, subjectInfoVO);
        LambdaQueryWrapper<ExternalSubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(ExternalSubjectInfoSourceMap::getSubjectId, subjectId)
                .eq(ExternalSubjectInfoSourceMap::getType, 1);
        subjectInfoVO.setInfoSourceList(externalSubjectInfoSourceMapService.list(queryWrapper));
        return Result.OK(subjectInfoVO);
    }


    /**
     * 配置信息推送给采集
     *
     * @param subjectId 专题id
     * @author lkg
     * @date 2025/8/12
     */
    @GetMapping("/pushToCaiji")
    public Result<?> pushToCaiji(@RequestParam String subjectId) {
        SubjectInfoVO subjectInfoVO = new SubjectInfoVO();
        SubjectDetailVO subjectDetailVO = subjectSimpleService.queryInfo(subjectId);
        BeanUtils.copyProperties(subjectDetailVO, subjectInfoVO);
        LambdaQueryWrapper<ExternalSubjectInfoSourceMap> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(ExternalSubjectInfoSourceMap::getSubjectId, subjectId)
                .eq(ExternalSubjectInfoSourceMap::getType, 1);
        subjectInfoVO.setInfoSourceList(externalSubjectInfoSourceMapService.list(queryWrapper));
        try {
            HttpUtil.doPost(keywordCrawlerUrl, JSONObject.from(subjectInfoVO), 30000);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return Result.OK();
    }
}
