package com.zzsn.event.task;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONWriter;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zzsn.event.constant.Constants;
import com.zzsn.event.entity.Event;
import com.zzsn.event.entity.SubjectAnalysis;
import com.zzsn.event.service.AnalysisService;
import com.zzsn.event.service.EsService;
import com.zzsn.event.service.IEventService;

import com.zzsn.event.service.SubjectAnalysisService;
import com.zzsn.event.util.DateUtil;
import com.zzsn.event.util.EsDateUtil;
import com.zzsn.event.util.RedisUtil;
import com.zzsn.event.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * @author lkg
 * @description: 专题分析定时任务
 * @date 2022/7/8 11:58
 */
@Slf4j
@Component
@EnableScheduling
public class AnalysisTask {

    @Autowired
    private IEventService eventService;
    @Autowired
    private EsService esService;
    @Autowired
    private KafkaTemplate<String, String> kafkaTemplate;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private AnalysisService analysisService;
    @Autowired
    private SubjectAnalysisService subjectAnalysisService;

    private static final String queryString = "{\"bool\":{\"must\":[{\"match\":{\"subjectId\":\"subject_id\"}}],\"must_not\":[{\"match\":{\"flag\":\"0\"}},{\"match\":{\"deleteFlag\":\"1\"}}]}}";
    private static final String beforeQueryString = "{\"bool\":{\"filter\":[{\"range\":{\"publishDate\":{\"gte\":\"start_time\",\"lte\":\"end_time\"}}}],\"must\":[{\"match\":{\"subjectId\":\"subject_id\"}}],\"must_not\":[{\"match\":{\"flag\":\"0\"}},{\"match\":{\"deleteFlag\":\"1\"}}]}}";
    private static final String repeatNumQueryString = "{\"bool\":{\"must\":[{\"match\":{\"subjectId\":\"subject_id\"}},{\"match\":{\"masterEntryId\":\"data_id\"}}],\"must_not\":[{\"match\":{\"id\":\"data_id\"}}]}}";

    /**
     * 每天凌晨0点执行一次
     * 按天发送kafka  获取进行中(未结束)的事件专题列表
     */
    @Scheduled(cron = "0 0 0 * * ?")
//    @Scheduled(cron = "0 * * * * ?")
    public void subjectList() {
        List<SubjectKafkaVo> data = new ArrayList<>();
        Date today = new Date();
        Date disableDate = DateUtil.addDate(today, -1);
        String startTime = EsDateUtil.esFieldDateFormat(DateUtil.dateToString(disableDate, "yyyy-MM-dd 00:00:00"));
        String endTime = EsDateUtil.esFieldDateFormat(DateUtil.dateToString(disableDate, "yyyy-MM-dd 23:59:59"));
        log.info("subjectList");
        List<SubjectKafkaVo> subjects = eventService.progressList(disableDate);
        subjects.forEach(e -> {
            String subjectId = e.getId();
            Integer increCount = esService.count(subjectId, startTime, endTime, "0");
            Integer totalCount = esService.count(subjectId, null, null, "0");
            Date analysisTime = e.getAnalysisTime();
            if (analysisTime == null) {
                analysisTime = e.getTimeEnable();
            }
            Integer betweenTwoDate = DateUtil.betweenTwoDate(analysisTime, today);
            if (increCount.compareTo(e.getIncreAnaRule()) > 0
                    || totalCount.compareTo(e.getTotalAnaRule()) > 0
                    || betweenTwoDate.compareTo(e.getTimeAnaRule()) > 0) {
                e.setIndexName(Constants.SUBJECT_INDEX);
                String query = queryString.replace("subject_id", subjectId);
                e.setQuery(query);
                String beforeQuery = beforeQueryString.replace("subject_id", subjectId).replace("start_time", startTime).replace("end_time", endTime);
                e.setBeforeQuery(beforeQuery);
                String repeatNumQuery = repeatNumQueryString.replace("subject_id", subjectId);
                e.setRepeatNumQuery(repeatNumQuery);
                e.setAnalysisTime(today);
                data.add(e);
            }
        });
        if (CollectionUtils.isNotEmpty(data)) {
            kafkaTemplate.send(Constants.EVENT_ANALYSIS_TOPIC, JSON.toJSONString(data));
        }
        log.info("进行中(未结束)的事件专题列表数据{}发送成功!",data);
    }

    /**
     * 每天凌晨0点5分执行一次
     * 按天发送 事件脉络 所需信息到kafka对应的topic
     * 保证信息采集的及时性，审核人员审核的及时性
     */
    @Scheduled(cron = "0 5 0 * * ?")
//    @Scheduled(cron = "0 * * * * ?")
    public void eventContext() {
        Date today = new Date();
        Date disableDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> subjects = eventService.progressList(disableDate);
        for (SubjectKafkaVo subjectKafkaVo : subjects) {
            String subjectId = subjectKafkaVo.getId();
            Date eventTime = subjectKafkaVo.getEventTime();
            String startDate;
            if (eventTime != null) {
                startDate = DateUtil.dateToString(eventTime, "yyyy-MM-dd HH:mm:ss");
            } else {
                startDate = DateUtil.dateToString(subjectKafkaVo.getTimeEnable(), "yyyy-MM-dd 00:00:00");
            }
            String endDate = DateUtil.dateToString(disableDate, "yyyy-MM-dd 23:59:59");
            List<SubjectDataVo> perDateList = esService.getDataBySubjectId(subjectId, startDate, endDate, "0", Constants.FETCH_FIELDS_DATA, 2, null, null);
            if (perDateList.size()>  30) {
                String keyWord = subjectKafkaVo.getKeyWord();
                List<KafkaDataVo> list = new ArrayList<>();
                format(subjectId, list, perDateList);
                //分批发送
                splitSend(Constants.EVENT_CONTEXT_SEND_TOPIC, subjectId, list, keyWord);
                //更新专题 分析事件脉络-最新资讯时间
                Event subject = new Event();
                subject.setId(subjectId);
                subject.setAnalysisTime(DateUtil.stringToDate(perDateList.get(0).getPublishDate(), "yyyy-MM-dd HH:mm:ss"));
                eventService.updateById(subject);
                log.info("id为-{}-的专题,此次事件脉络({}~{})数据-共{}条,发送成功!", subjectId, startDate.substring(0, 10), endDate.substring(0, 10), list.size());
            }
        }
    }

    /**
     * 每天凌晨0点20分执行一次
     * 发送 伪事件脉络 所需信息到kafka对应的topic
     */
    @Scheduled(cron = "0 20 0 * * ?")
//    @Scheduled(cron = "0 * * * * ?")
    public void eventContext_fake() {
        Date today = new Date();
        Date disableDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> fakeEventIdList = new ArrayList<>();
        List<SubjectKafkaVo> subjectList = eventService.progressList(disableDate);
        subjectList.forEach(e -> {
            LambdaQueryWrapper<SubjectAnalysis> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SubjectAnalysis::getSubjectId, e.getId()).eq(SubjectAnalysis::getCategory, 2);
            int count = subjectAnalysisService.count(queryWrapper);
            if (count < Constants.FAKE_NUM) {
                fakeEventIdList.add(e);
            }
        });
        for (SubjectKafkaVo subjectKafkaVo : fakeEventIdList) {
            List<KafkaDataVo> kafkaDataVoList = new ArrayList<>();
            String subjectId = subjectKafkaVo.getId();
            String keyWord = subjectKafkaVo.getKeyWord();
            List<SubjectDataVo> dataList = esService.getDataBySubjectId(subjectId, null, null, "0", Constants.FETCH_FIELDS_DATA, 2, null, null);
            format(subjectId, kafkaDataVoList, dataList);
            splitSend(Constants.FAKE_EVENT_CONTEXT_SEND_TOPIC, subjectId, kafkaDataVoList, keyWord);
        }
    }

    private void format(String subjectId, List<KafkaDataVo> kafkaDataVoList, List<SubjectDataVo> dataList) {
        dataList.forEach(e -> {
            KafkaDataVo kafkaDataVo = new KafkaDataVo();
            BeanUtils.copyProperties(e, kafkaDataVo);
            List<SubjectDataVo> subjectDataVoList = esService.dataById(subjectId, e.getId());
            kafkaDataVo.setRepeatNum(subjectDataVoList.size());
            kafkaDataVoList.add(kafkaDataVo);
        });
    }

    //防止数据量太大，超过kafka的最大值，所以分批发送，一次20条数据
    private void splitSend(String topic, String subjectId, List<KafkaDataVo> list, String keyWord) {
        List<List<KafkaDataVo>> partition = ListUtils.partition(list, 20);
        partition.forEach(e -> {
            Map<String, Object> map = new HashMap<>();
            map.put("keyword", keyWord);
            map.put("data", e);
            kafkaTemplate.send(topic, subjectId, JSON.toJSONString(map, JSONWriter.Feature.WriteMapNullValue));
        });
    }

    /**
     * 定时生成传播路径
     * 每天凌晨0点10分执行一次
     */
//    @Scheduled(cron = "0 0 0 * * ?")
    @Scheduled(cron = "0 * * * * ?")
    public void propagationPath() {
        Date today = new Date();
        Date deadlineDate = DateUtil.addDate(today, -1);
        List<SubjectKafkaVo> subjects = eventService.eventSubjectList();
        for (SubjectKafkaVo subject : subjects) {
            String subjectId = subject.getId();
            String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.PROPAGATION_KEY + subjectId;
            Date timeDisable = subject.getTimeDisable();
            //已经结束的事件专题，永久缓存
            if (timeDisable != null && deadlineDate.compareTo(timeDisable) > 0) {
                Object cacheObject = redisUtil.get(key);
                if (cacheObject == null) {
                    PropagationPathVo pathVo = analysisService.propagationPath(subjectId);
                    if (ObjectUtils.isNotEmpty(pathVo)) {
                        redisUtil.set(key, pathVo);
                    }
                }
            } else {//已经结束的事件专题，缓存有效期一天
                PropagationPathVo pathVo = analysisService.propagationPath(subjectId);
                if (ObjectUtils.isNotEmpty(pathVo)) {
                    redisUtil.set(key, pathVo, 3600 * 24);
                }
            }
            log.info("专题id为-{}-的专题-传播路径数据-缓存成功!", subjectId);
        }
    }

//

//    /**
//     * 定时生成统计分析数据-最近两周(按最新时间)
//     * 每天凌晨1点执行一次
//     */
//    @Scheduled(cron = "0 0 1 * * ?")
//    public void tw_statisticAnalysis() {
//        Date today = new Date();
//        Date deadlineDate = DateUtil.addDate(today, -1);
//        List<SubjectKafkaVo> subjects = eventService.eventSubjectList();
//        for (SubjectKafkaVo subject : subjects) {
//            String subjectId = subject.getId();
//            List<SubjectDataVo> dataVoList = esService.getDataBySubjectId(subjectId, null, null, "0", Constant.FETCH_FIELDS_STATISTIC, 2, null, null);
//            if (CollectionUtils.isNotEmpty(dataVoList)) {
//                String key = Constants.SUBJECT_ANALYSIS_PRE + Constants.TW_STATISTIC_ANALYSIS_KEY + subjectId;
//                Date timeDisable = subject.getTimeDisable();
//                String latelyTime = EsDateUtil.esFieldDateMapping(dataVoList.get(0).getPublishDate());
//                String twoWeekBeforeTime = DateUtil.getDayBeforeOrAfter(latelyTime, -14);
//                //已经结束的事件专题，永久缓存
//                if (timeDisable != null && deadlineDate.compareTo(timeDisable) > 0) {
//                    Object cacheObject = redisUtil.get(key);
//                    if (cacheObject == null) {
//                        List<StatisticAnalysisVo> statisticAnalysisVos = analysisService.statisticAnalysis(subjectId, twoWeekBeforeTime, latelyTime);
//                        if (CollectionUtils.isNotEmpty(statisticAnalysisVos)) {
//                            redisUtil.set(key, statisticAnalysisVos);
//                        }
//                    }
//                } else { //已经结束的事件专题，缓存有效期一天
//                    List<StatisticAnalysisVo> statisticAnalysisVos = analysisService.statisticAnalysis(subjectId, twoWeekBeforeTime, latelyTime);
//                    if (CollectionUtils.isNotEmpty(statisticAnalysisVos)) {
//                        redisUtil.set(key, statisticAnalysisVos,3600*24);
//                    }
//                }
//                log.info("专题id为-{}-的专题-统计分析数据-最近两周-分析成功!", subjectId);
//            }
//        }
//    }
}
